/*
 *  prefdlg.cpp  -  program preferences dialog
 *  Program:  kalarm
 *  Copyright © 2001-2008 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kalarm.h"

#include <qobjectlist.h>
#include <qlayout.h>
#include <qbuttongroup.h>
#include <qvbox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qpushbutton.h>
#include <qspinbox.h>
#include <qwhatsthis.h>
#include <qtooltip.h>
#include <qstyle.h>
#include <qevent.h>
#include <qtimer.h>

#include <kglobal.h>
#include <klocale.h>
#include <kstandarddirs.h>
#include <kshell.h>
#include <kmessagebox.h>
#include <kaboutdata.h>
#include <kapplication.h>
#include <kiconloader.h>
#include <kcombobox.h>
#include <ktabwidget.h>
#include <kstdguiitem.h>
#include <ksystemtimezone.h>
#include <ksimpleconfig.h>
#ifdef Q_WS_X11
#include <kwin.h>
#endif
#include <kdebug.h>

#include <libkholidays/kholidays.h>

#include "alarmcalendar.h"
#include "alarmresources.h"
#include "alarmtimewidget.h"
#include "colourbutton.h"
#include "editdlg.h"
#include "editdlgtypes.h"
#include "fontcolour.h"
#include "functions.h"
#include "kalarmapp.h"
#include "kamail.h"
#include "label.h"
#include "latecancel.h"
#include "mainwindow.h"
#include "preferences.h"
#include "radiobutton.h"
#include "recurrenceedit.h"
#ifndef WITHOUT_ARTS
#include "sounddlg.h"
#endif
#include "soundpicker.h"
#include "specialactions.h"
#include "stackedwidgets.h"
#include "timeedit.h"
#include "timespinbox.h"
#include "timezonecombo.h"
#include "traywindow.h"
#include "prefdlg_p.moc"
#include "prefdlg.moc"

static const char PREF_DIALOG_NAME[] = "PrefDialog";

// Command strings for executing commands in different types of terminal windows.
// %t = window title parameter
// %c = command to execute in terminal
// %w = command to execute in terminal, with 'sleep 86400' appended
// %C = temporary command file to execute in terminal
// %W = temporary command file to execute in terminal, with 'sleep 86400' appended
static QString xtermCommands[] = {
	QString::fromLatin1("xterm -sb -hold -title %t -e %c"),
	QString::fromLatin1("konsole --noclose -T %t -e ${SHELL:-sh} -c %c"),
	QString::fromLatin1("gnome-terminal -t %t -e %W"),
	QString::fromLatin1("eterm --pause -T %t -e %C"),    // some systems use eterm...
	QString::fromLatin1("Eterm --pause -T %t -e %C"),    // while some use Eterm
	QString::fromLatin1("rxvt -title %t -e ${SHELL:-sh} -c %w"),
	QString::null       // end of list indicator - don't change!
};


/*=============================================================================
= Class KAlarmPrefDlg
=============================================================================*/

KAlarmPrefDlg* KAlarmPrefDlg::mInstance = 0;

void KAlarmPrefDlg::display()
{
	if (!mInstance)
	{
		mInstance = new KAlarmPrefDlg;
		QSize s;
		if (KAlarm::readConfigWindowSize(PREF_DIALOG_NAME, s))
			mInstance->resize(s);
		mInstance->show();
	}
	else
	{
#ifdef Q_WS_X11
		KWin::WindowInfo info = KWin::windowInfo(mInstance->winId(), static_cast<unsigned long>(NET::WMGeometry | NET::WMDesktop));
		KWin::setCurrentDesktop(info.desktop());
#endif
		mInstance->showNormal();   // un-minimise it if necessary
		mInstance->raise();
		mInstance->setActiveWindow();
	}
}

KAlarmPrefDlg::KAlarmPrefDlg()
	: KDialogBase(IconList, i18n("Configure"), Help | Default | Ok | Apply | Cancel, Ok, 0, "PrefDlg", false, true)
{
	setWFlags(Qt::WDestructiveClose);
	setIconListAllVisible(false);
	mTabScrollGroup = new StackedScrollGroup(this, this);

	QVBox* frame = addVBoxPage(i18n("General"), i18n("General"), DesktopIcon("misc"));
	mMiscPage = new MiscPrefTab(mTabScrollGroup, frame);

	frame = addVBoxPage(i18n("Time & Date"), i18n("Time and Date"), DesktopIcon("clock"));
	mTimePage = new TimePrefTab(mTabScrollGroup, frame);

	frame = addVBoxPage(i18n("Storage"), i18n("Alarm Storage"), DesktopIcon("fileopen"));
	mStorePage = new StorePrefTab(mTabScrollGroup, frame);

	frame = addVBoxPage(i18n("Email"), i18n("Email Alarm Settings"), DesktopIcon("mail_generic"));
	mEmailPage = new EmailPrefTab(mTabScrollGroup, frame);

	frame = addVBoxPage(i18n("View"), i18n("View Settings"), DesktopIcon("view_choose"));
	mViewPage = new ViewPrefTab(mTabScrollGroup, frame);

	frame = addVBoxPage(i18n("Edit"), i18n("Default Alarm Edit Settings"), DesktopIcon("edit"));
	mEditPage = new EditPrefTab(mTabScrollGroup, frame);

	restore();
	adjustSize();
}

KAlarmPrefDlg::~KAlarmPrefDlg()
{
	mInstance = 0;
}

// Restore all defaults in the options...
void KAlarmPrefDlg::slotDefault()
{
	kdDebug(5950) << "KAlarmPrefDlg::slotDefault()" << endl;
	mEmailPage->setDefaults();
	mViewPage->setDefaults();
	mEditPage->setDefaults();
	mStorePage->setDefaults();
	mTimePage->setDefaults();
	mMiscPage->setDefaults();
}

void KAlarmPrefDlg::slotHelp()
{
	kapp->invokeHelp("preferences");
}

// Apply the preferences that are currently selected
void KAlarmPrefDlg::slotApply()
{
	kdDebug(5950) << "KAlarmPrefDlg::slotApply()" << endl;
	QString errmsg = mEmailPage->validate();
	if (!errmsg.isEmpty())
	{
		showPage(pageIndex(mEmailPage->parentWidget()));
		if (KMessageBox::warningYesNo(this, errmsg) != KMessageBox::Yes)
		{
			mValid = false;
			return;
		}
	}
	errmsg = mEditPage->validate();
	if (!errmsg.isEmpty())
	{
		showPage(pageIndex(mEditPage->parentWidget()));
		KMessageBox::sorry(this, errmsg);
		mValid = false;
		return;
	}
	mValid = true;
	mEmailPage->apply(false);
	mViewPage->apply(false);
	mEditPage->apply(false);
	mStorePage->apply(false);
	mTimePage->apply(false);
	mMiscPage->apply(false);
	Preferences::syncToDisc();
}

// Apply the preferences that are currently selected
void KAlarmPrefDlg::slotOk()
{
	kdDebug(5950) << "KAlarmPrefDlg::slotOk()" << endl;
	mValid = true;
	slotApply();
	if (mValid)
		KDialogBase::slotOk();
}

// Discard the current preferences and close the dialogue
void KAlarmPrefDlg::slotCancel()
{
	kdDebug(5950) << "KAlarmPrefDlg::slotCancel()" << endl;
	restore();
	KDialogBase::slotCancel();
}

// Discard the current preferences and use the present ones
void KAlarmPrefDlg::restore()
{
	kdDebug(5950) << "KAlarmPrefDlg::restore()" << endl;
	mEmailPage->restore();
	mViewPage->restore();
	mEditPage->restore();
	mStorePage->restore();
	mTimePage->restore();
	mMiscPage->restore();
}

/******************************************************************************
* Return the minimum size for the dialog.
* If the minimum size would be too high to fit the desktop, the tab contents
* are made scrollable.
*/
QSize KAlarmPrefDlg::minimumSizeHint() const
{
	if (!mTabScrollGroup->sized())
	{
		QSize s = mTabScrollGroup->adjustSize();
		if (s.isValid())
		{
			if (mTabScrollGroup->heightReduction())
			{
				s = QSize(s.width(), s.height() - mTabScrollGroup->heightReduction());
				const_cast<KAlarmPrefDlg*>(this)->resize(s);
			}
			return s;
		}
	}
	return KDialogBase::minimumSizeHint();
}

void KAlarmPrefDlg::showEvent(QShowEvent* e)
{
	KDialogBase::showEvent(e);
	mTabScrollGroup->adjustSize(true);
}

/******************************************************************************
*  Called when the dialog's size has changed.
*  Records the new size in the config file.
*/
void KAlarmPrefDlg::resizeEvent(QResizeEvent* re)
{
	if (isVisible())
		KAlarm::writeConfigWindowSize(PREF_DIALOG_NAME, re->size());
	KDialogBase::resizeEvent(re);
}


/*=============================================================================
= Class PrefsTabBase
=============================================================================*/
int PrefsTabBase::mIndentWidth = 0;

PrefsTabBase::PrefsTabBase(StackedScrollGroup* scrollGroup, QVBox* frame)
	: StackedScrollWidget(scrollGroup, frame),
	  mPage(new QVBox(viewport()))
{
	addChild(mPage);
	mPage->setSpacing(KDialog::spacingHint());
	if (!mIndentWidth)
		mIndentWidth = style().subRect(QStyle::SR_RadioButtonIndicator, this).width();
}

void PrefsTabBase::apply(bool syncToDisc)
{
	Preferences::save(syncToDisc);
}



/*=============================================================================
= Class MiscPrefTab
=============================================================================*/

MiscPrefTab::MiscPrefTab(StackedScrollGroup* scrollGroup, QVBox* frame)
	: PrefsTabBase(scrollGroup, frame)
{
	QGroupBox* group = new QGroupBox(i18n("Run Mode"), mPage, "modeGroup");
	QBoxLayout* layout = new QVBoxLayout(group, KDialog::marginHint(), KDialog::spacingHint());
	layout->addSpacing(fontMetrics().height() - KDialog::marginHint() + KDialog::spacingHint());

	// Start at login
	mAutoStart = new QCheckBox(i18n("Start at login"), group, "autoTray");
	connect(mAutoStart, SIGNAL(clicked()), SLOT(slotAutostartClicked()));
	QWhatsThis::add(mAutoStart,
	      i18n("Automatically start KAlarm whenever you start KDE.\n\n"
	           "This option should always be checked unless you intend to discontinue use of KAlarm."));
	layout->addWidget(mAutoStart, 0, Qt::AlignAuto);

	// Warn before quitting
	mQuitWarn = new QCheckBox(i18n("Warn before &quitting"), group, "disableAl");
	QWhatsThis::add(mQuitWarn,
	      i18n("Check to display a warning prompt before quitting KAlarm."));
	layout->addWidget(mQuitWarn, 0, Qt::AlignAuto);

	group->setFixedHeight(group->sizeHint().height());

	// Confirm alarm deletion?
	QHBox* itemBox = new QHBox(mPage);   // this is to allow left adjustment
	mConfirmAlarmDeletion = new QCheckBox(i18n("Con&firm alarm deletions"), itemBox, "confirmDeletion");
	mConfirmAlarmDeletion->setMinimumSize(mConfirmAlarmDeletion->sizeHint());
	QWhatsThis::add(mConfirmAlarmDeletion,
	      i18n("Check to be prompted for confirmation each time you delete an alarm."));
	itemBox->setStretchFactor(new QWidget(itemBox), 1);    // left adjust the controls
	itemBox->setFixedHeight(itemBox->sizeHint().height());

	// Terminal window to use for command alarms
	group = new QGroupBox(i18n("Terminal for Command Alarms"), mPage);
	QWhatsThis::add(group,
	      i18n("Choose which application to use when a command alarm is executed in a terminal window"));
	QGridLayout* grid = new QGridLayout(group, 1, 3, KDialog::marginHint(), KDialog::spacingHint());
	grid->addRowSpacing(0, fontMetrics().height() - KDialog::marginHint() + KDialog::spacingHint());
	int row = 0;

	mXtermType = new QButtonGroup(group);
	mXtermType->hide();
	QString whatsThis = i18n("The parameter is a command line, e.g. 'xterm -e'", "Check to execute command alarms in a terminal window by '%1'");
	int index = 0;
	mXtermFirst = -1;
	for (mXtermCount = 0;  !xtermCommands[mXtermCount].isNull();  ++mXtermCount)
	{
		QString cmd = xtermCommands[mXtermCount];
		QStringList args = KShell::splitArgs(cmd);
		if (args.isEmpty()  ||  KStandardDirs::findExe(args[0]).isEmpty())
			continue;
		QRadioButton* radio = new QRadioButton(args[0], group);
		radio->setMinimumSize(radio->sizeHint());
		mXtermType->insert(radio, mXtermCount);
		if (mXtermFirst < 0)
			mXtermFirst = mXtermCount;   // note the id of the first button
		cmd.replace("%t", kapp->aboutData()->programName());
		cmd.replace("%c", "<command>");
		cmd.replace("%w", "<command; sleep>");
		cmd.replace("%C", "[command]");
		cmd.replace("%W", "[command; sleep]");
		QWhatsThis::add(radio, whatsThis.arg(cmd));
		grid->addWidget(radio, (row = index/3 + 1), index % 3, Qt::AlignAuto);
		++index;
	}

	itemBox = new QHBox(group);
	grid->addMultiCellWidget(itemBox, row + 1, row + 1, 0, 2, Qt::AlignAuto);
	QRadioButton* radio = new QRadioButton(i18n("Other:"), itemBox);
	radio->setFixedSize(radio->sizeHint());
	connect(radio, SIGNAL(toggled(bool)), SLOT(slotOtherTerminalToggled(bool)));
	mXtermType->insert(radio, mXtermCount);
	if (mXtermFirst < 0)
		mXtermFirst = mXtermCount;   // note the id of the first button
	mXtermCommand = new QLineEdit(itemBox);
	QWhatsThis::add(itemBox,
	      i18n("Enter the full command line needed to execute a command in your chosen terminal window. "
	           "By default the alarm's command string will be appended to what you enter here. "
	           "See the KAlarm Handbook for details of special codes to tailor the command line."));

	mPage->setStretchFactor(new QWidget(mPage), 1);    // top adjust the widgets
}

void MiscPrefTab::restore()
{
	mAutoStart->setChecked(Preferences::mAutoStart);
	mQuitWarn->setChecked(Preferences::quitWarn());
	mConfirmAlarmDeletion->setChecked(Preferences::confirmAlarmDeletion());
	QString xtermCmd = Preferences::cmdXTermCommand();
	int id = mXtermFirst;
	if (!xtermCmd.isEmpty())
	{
		for ( ;  id < mXtermCount;  ++id)
		{
			if (mXtermType->find(id)  &&  xtermCmd == xtermCommands[id])
				break;
		}
	}
	mXtermType->setButton(id);
	mXtermCommand->setEnabled(id == mXtermCount);
	mXtermCommand->setText(id == mXtermCount ? xtermCmd : "");
}

void MiscPrefTab::apply(bool syncToDisc)
{
	// First validate anything entered in Other X-terminal command
	int xtermID = mXtermType->selectedId();
	if (xtermID >= mXtermCount)
	{
		QString cmd = mXtermCommand->text();
		if (cmd.isEmpty())
			xtermID = -1;       // 'Other' is only acceptable if it's non-blank
		else
		{
			QStringList args = KShell::splitArgs(cmd);
			cmd = args.isEmpty() ? QString::null : args[0];
			if (KStandardDirs::findExe(cmd).isEmpty())
			{
				mXtermCommand->setFocus();
				if (KMessageBox::warningContinueCancel(this, i18n("Command to invoke terminal window not found:\n%1").arg(cmd))
				                != KMessageBox::Continue)
					return;
			}
		}
	}
	if (xtermID < 0)
	{
		xtermID = mXtermFirst;
		mXtermType->setButton(mXtermFirst);
	}

	Preferences::mAutoStart = mAutoStart->isChecked();
	Preferences::setQuitWarn(mQuitWarn->isChecked());
	Preferences::setConfirmAlarmDeletion(mConfirmAlarmDeletion->isChecked());
	Preferences::mCmdXTermCommand = (xtermID < mXtermCount) ? xtermCommands[xtermID] : mXtermCommand->text();
	PrefsTabBase::apply(syncToDisc);
}

void MiscPrefTab::setDefaults()
{
	mAutoStart->setChecked(Preferences::default_autoStart);
	mQuitWarn->setChecked(Preferences::default_quitWarn);
	mConfirmAlarmDeletion->setChecked(Preferences::default_confirmAlarmDeletion);
	mXtermType->setButton(mXtermFirst);
	mXtermCommand->setEnabled(false);
}

void MiscPrefTab::slotAutostartClicked()
{
	if (!mAutoStart->isChecked()
	&&  KMessageBox::warningYesNo(this,
		                      i18n("You should not uncheck this option unless you intend to discontinue use of KAlarm"),
		                      QString::null, KStdGuiItem::cont(), KStdGuiItem::cancel()
		                     ) != KMessageBox::Yes)
		mAutoStart->setChecked(true);	
}

void MiscPrefTab::slotOtherTerminalToggled(bool on)
{
	mXtermCommand->setEnabled(on);
}


/*=============================================================================
= Class TimePrefTab
=============================================================================*/

TimePrefTab::TimePrefTab(StackedScrollGroup* scrollGroup, QVBox* frame)
	: PrefsTabBase(scrollGroup, frame)
{
	// Get alignment to use in QGridLayout (AlignAuto doesn't work correctly there)
	int alignment = QApplication::reverseLayout() ? Qt::AlignRight : Qt::AlignLeft;

	// Default time zone
	QHBox* box = new QHBox(mPage);   // this is to control the QWhatsThis text display area
	box->setSpacing(KDialog::spacingHint());
	QLabel* labeltz = new QLabel(i18n("Time zone:"), box);
	mTimeZone = new TimeZoneCombo(box);
	mTimeZone->setSizeLimit(15);
	QWhatsThis::add(box,
	      i18n("Select the time zone which KAlarm should use as its default for displaying and entering dates and times."));
	labeltz->setBuddy(mTimeZone);
 
	// Holiday region
	box = new QHBox(mPage);   // this is to control the QWhatsThis text display area
	box->setSpacing(KDialog::spacingHint());
	QLabel* labelhol = new QLabel(i18n("Holiday region:"), box);
	mHolidays = new KComboBox(box);
	mHolidays->setAutoResize(true);
	labelhol->setBuddy(mHolidays);
	QWhatsThis::add(box, i18n("Select which holiday region to use"));
	QMap<QString, QString> holidays;
	QStringList countryList = KHolidays::locations();
	for (QStringList::ConstIterator it = countryList.constBegin();  it != countryList.constEnd();  ++it)
	{
		QString country = *it;
		QString countryFile = locate("locale", "l10n/" + country + "/entry.desktop");
		QString regionName;  // name to display
		if (!countryFile.isEmpty())
		{
			KSimpleConfig config(countryFile, true);   // read-only
			config.setGroup("KCM Locale");
			regionName = config.readEntry("Name");
		}
		if (regionName.isEmpty())
			regionName = country;   // default to file name

		holidays[regionName.lower()] = regionName;   // sort case-insensitively
		mHolidayNames[regionName] = country; // store region for saving to config file
	}
	mHolidays->insertItem(i18n("Do not use holidays", "(None)"));
	mHolidays->insertStringList(holidays.values());

	int w = QMAX(labeltz->sizeHint().width(), labelhol->sizeHint().width());
	labeltz->setFixedWidth(w);
	labelhol->setFixedWidth(w);

	// Start-of-day time
	QHBox* itemBox = new QHBox(mPage);
	box = new QHBox(itemBox);   // this is to control the QWhatsThis text display area
	box->setSpacing(KDialog::spacingHint());
	QLabel* label = new QLabel(i18n("&Start of day for date-only alarms:"), box);
	mStartOfDay = new TimeEdit(box);
	label->setBuddy(mStartOfDay);
	static const QString startOfDayText = i18n("The earliest time of day at which a date-only alarm will be triggered.");
	QWhatsThis::add(box, QString("%1\n\n%2").arg(startOfDayText).arg(TimeSpinBox::shiftWhatsThis()));
	itemBox->setStretchFactor(new QWidget(itemBox), 1);    // left adjust the controls
	itemBox->setFixedHeight(box->sizeHint().height());

	// Working hours
	QGroupBox* group = new QGroupBox(i18n("Working Hours"), mPage);
	QVBoxLayout* layout = new QVBoxLayout(group, KDialog::marginHint(), KDialog::spacingHint());
	layout->addSpacing(fontMetrics().height() - KDialog::marginHint() + KDialog::spacingHint());
	QWidget* daybox = new QWidget(group);   // this is to control the QWhatsThis text display area
	layout->addWidget(daybox);
	QGridLayout* wgrid = new QGridLayout(daybox, 4, 4, 0, KDialog::spacingHint());
	const KLocale* locale = KGlobal::locale();
	for (int i = 0;  i < 7;  ++i)
	{
		int day = KAlarm::localeDayInWeek_to_weekDay(i);
		mWorkDays[i] = new QCheckBox(KAlarm::weekDayName(day, locale), daybox);
		mWorkDays[i]->setFixedSize(mWorkDays[i]->sizeHint());
		wgrid->addWidget(mWorkDays[i], i/3, i%3, alignment);
	}
	daybox->setFixedHeight(daybox->sizeHint().height());
	QWhatsThis::add(daybox, i18n("Check the days in the week which are work days"));

	itemBox = new QHBox(group);
	layout->addWidget(itemBox);
	box = new QHBox(itemBox);   // this is to control the QWhatsThis text display area
	box->setSpacing(KDialog::spacingHint());
	QLabel* startLabel = new QLabel(i18n("Daily start time:"), box);
	mWorkStart = new TimeEdit(box);
	startLabel->setBuddy(mWorkStart);
	QWhatsThis::add(box, i18n("Enter the start time of the working day.\n\n%1").arg(TimeSpinBox::shiftWhatsThis()));
	itemBox->setStretchFactor(new QWidget(itemBox), 1);    // left adjust the controls

	itemBox = new QHBox(group);
	layout->addWidget(itemBox);
	box = new QHBox(itemBox);   // this is to control the QWhatsThis text display area
	box->setSpacing(KDialog::spacingHint());
	QLabel* endLabel = new QLabel(i18n("Daily end time:"), box);
	mWorkEnd = new TimeEdit(box);
	endLabel->setBuddy(mWorkEnd);
	QWhatsThis::add(box, i18n("Enter the end time of the working day.\n\n%1").arg(TimeSpinBox::shiftWhatsThis()));
	itemBox->setStretchFactor(new QWidget(itemBox), 1);    // left adjust the controls
	box->setFixedHeight(box->sizeHint().height());
	w = QMAX(startLabel->sizeHint().width(), endLabel->sizeHint().width());
	startLabel->setFixedWidth(w);
	endLabel->setFixedWidth(w);

	mPage->setStretchFactor(new QWidget(mPage), 1);    // top adjust the widgets
}

void TimePrefTab::restore()
{
	mTimeZone->setTimeZone(Preferences::timeZone());
	setHolidays(Preferences::holidays().location());
	mStartOfDay->setValue(Preferences::mStartOfDay);
	mWorkStart->setValue(Preferences::mWorkDayStart);
	mWorkEnd->setValue(Preferences::mWorkDayEnd);
	setWorkDays(Preferences::mWorkDays);
}

void TimePrefTab::apply(bool syncToDisc)
{
	KTimeZone tz = mTimeZone->timeZone();
	if (tz.isValid())
		Preferences::mTimeZone = tz;
	QString hol = mHolidays->currentItem() ? mHolidayNames[mHolidays->currentText()] : QString::null;
	if (hol != Preferences::holidays().location())
		Preferences::setHolidayRegion(hol);
	int t = mStartOfDay->value();
	Preferences::mStartOfDay.setHMS(t/60, t%60, 0);
	t = mWorkStart->value();
	Preferences::mWorkDayStart.setHMS(t/60, t%60, 0);
	t = mWorkEnd->value();
	Preferences::mWorkDayEnd.setHMS(t/60, t%60, 0);
	Preferences::mWorkDays.fill(false);
	for (int i = 0;  i < 7;  ++i)
		if (mWorkDays[i]->isChecked())
			Preferences::mWorkDays.setBit(KAlarm::localeDayInWeek_to_weekDay(i) - 1, 1);
	PrefsTabBase::apply(syncToDisc);
}

void TimePrefTab::setDefaults()
{
	mTimeZone->setTimeZone(Preferences::default_timeZone());
	setHolidays(Preferences::default_holidays().location());
	mStartOfDay->setValue(Preferences::default_startOfDay);
	mWorkStart->setValue(Preferences::default_workDayStart);
	mWorkEnd->setValue(Preferences::default_workDayEnd);
	setWorkDays(Preferences::default_workDays());
}

void TimePrefTab::setHolidays(const QString& regionCode)
{
	QString region;
	if (!regionCode.isEmpty())
	{
		for (QMap<QString, QString>::ConstIterator it = mHolidayNames.constBegin();  it != mHolidayNames.constEnd();  ++it)
		{
			if (it.data() == regionCode)
			{
				region = it.key();
				break;
			}
		}
	}
	int i = 0;
	if (!region.isEmpty())
		for (i = mHolidays->count();  --i > 0 && mHolidays->text(i) != region; ) ;
	mHolidays->setCurrentItem(i);
}

void TimePrefTab::setWorkDays(const QBitArray& days)
{
	for (int i = 0;  i < 7;  ++i)
	{
		bool x = days.testBit(KAlarm::localeDayInWeek_to_weekDay(i) - 1);
		mWorkDays[i]->setChecked(x);
	}
}


/*=============================================================================
= Class StorePrefTab
=============================================================================*/

StorePrefTab::StorePrefTab(StackedScrollGroup* scrollGroup, QVBox* frame)
	: PrefsTabBase(scrollGroup, frame),
	  mCheckKeepChanges(false)
{
	// Get alignment to use in QGridLayout (AlignAuto doesn't work correctly there)
	int alignment = QApplication::reverseLayout() ? Qt::AlignRight : Qt::AlignLeft;

	// Which resource to save to
	QGroupBox* group = new QButtonGroup(i18n("New Alarms && Templates"), mPage);
	QBoxLayout* layout = new QVBoxLayout(group, KDialog::marginHint(), KDialog::spacingHint());
	layout->addSpacing(fontMetrics().height() - KDialog::marginHint() + KDialog::spacingHint());

	mDefaultResource = new QRadioButton(i18n("Store in default resource"), group, "defResource");
	mDefaultResource->setFixedSize(mDefaultResource->sizeHint());
	QWhatsThis::add(mDefaultResource, i18n("Add all new alarms and alarm templates to the default resources, without prompting."));
	layout->addWidget(mDefaultResource, 0, Qt::AlignAuto);
	mAskResource = new QRadioButton(i18n("Prompt for which resource to store in."), group, "askResource");
	mAskResource->setFixedSize(mAskResource->sizeHint());
	QWhatsThis::add(mAskResource,
	      i18n("When saving a new alarm or alarm template, prompt for which resource to store it in, if there is more than one active resource.\n\n"
	           "Note that archived alarms are always stored in the default archived alarm resource."));
	layout->addWidget(mAskResource, 0, Qt::AlignAuto);

	// Archived alarms
	group = new QGroupBox(i18n("Archived Alarms"), mPage);
	QGridLayout* grid = new QGridLayout(group, 2, 2, KDialog::marginHint(), KDialog::spacingHint());
	grid->setColStretch(1, 1);
	grid->addColSpacing(0, indentWidth());
	grid->addRowSpacing(0, fontMetrics().height() - KDialog::marginHint() + KDialog::spacingHint());
	mKeepArchived = new QCheckBox(i18n("Keep alarms after e&xpiry"), group, "keepArchived");
	mKeepArchived->setFixedSize(mKeepArchived->sizeHint());
	connect(mKeepArchived, SIGNAL(toggled(bool)), SLOT(slotArchivedToggled(bool)));
	QWhatsThis::add(mKeepArchived,
	      i18n("Check to store alarms after expiry or deletion (except deleted alarms which were never triggered)."));
	grid->addMultiCellWidget(mKeepArchived, 1, 1, 0, 1, alignment);

	QHBox* box = new QHBox(group);
	box->setSpacing(KDialog::spacingHint());
	mPurgeArchived = new QCheckBox(i18n("Discard archived alarms after:"), box, "purgeArchived");
	mPurgeArchived->setMinimumSize(mPurgeArchived->sizeHint());
	connect(mPurgeArchived, SIGNAL(toggled(bool)), SLOT(slotArchivedToggled(bool)));
	mPurgeAfter = new SpinBox(box);
	mPurgeAfter->setMinValue(1);
	mPurgeAfter->setLineShiftStep(10);
	mPurgeAfter->setMinimumSize(mPurgeAfter->sizeHint());
	mPurgeAfterLabel = new QLabel(i18n("da&ys"), box);
	mPurgeAfterLabel->setMinimumSize(mPurgeAfterLabel->sizeHint());
	mPurgeAfterLabel->setBuddy(mPurgeAfter);
	QWhatsThis::add(box,
	      i18n("Uncheck to store archived alarms indefinitely. Check to enter how long archived alarms should be stored."));
	grid->addWidget(box, 2, 1, alignment);

	mClearArchived = new QPushButton(i18n("Clear Archived Alarms"), group);
	mClearArchived->setFixedSize(mClearArchived->sizeHint());
	connect(mClearArchived, SIGNAL(clicked()), SLOT(slotClearArchived()));
	QWhatsThis::add(mClearArchived,
	      (AlarmResources::instance()->activeCount(AlarmResource::ARCHIVED, false) <= 1)
	      ? i18n("Delete all existing archived alarms.")
	      : i18n("Delete all existing archived alarms (from the default archived alarm resource only)."));
	grid->addWidget(mClearArchived, 3, 1, alignment);
	group->setFixedHeight(group->sizeHint().height());

	mPage->setStretchFactor(new QWidget(mPage), 1);    // top adjust the widgets
}

void StorePrefTab::restore()
{
	mCheckKeepChanges = false;
	mAskResource->setChecked(Preferences::mAskResource);
	mDefaultResource->setChecked(!Preferences::mAskResource);
	int keepDays = Preferences::mArchivedKeepDays;
	mOldKeepArchived = keepDays;
	setArchivedControls(keepDays);
	mCheckKeepChanges = true;
}

void StorePrefTab::apply(bool syncToDisc)
{
	Preferences::mAskResource = mAskResource->isChecked();
	Preferences::mArchivedKeepDays = !mKeepArchived->isChecked() ? 0 : mPurgeArchived->isChecked() ? mPurgeAfter->value() : -1;
	PrefsTabBase::apply(syncToDisc);
}

void StorePrefTab::setDefaults()
{
	mAskResource->setChecked(Preferences::default_askResource);
	setArchivedControls(Preferences::default_archivedKeepDays);
}

void StorePrefTab::setArchivedControls(int purgeDays)
{
	mKeepArchived->setChecked(purgeDays);
	mPurgeArchived->setChecked(purgeDays > 0);
	mPurgeAfter->setValue(purgeDays > 0 ? purgeDays : 0);
	slotArchivedToggled(true);
}

void StorePrefTab::slotArchivedToggled(bool)
{
	bool keep = mKeepArchived->isChecked();
	if (keep  &&  !mOldKeepArchived  &&  mCheckKeepChanges
	&&  !AlarmResources::instance()->getStandardResource(AlarmResource::ARCHIVED))
	{
		KMessageBox::sorry(this,
		     i18n("A default resource is required in order to archive alarms, but none is currently enabled.\n\n"
		          "If you wish to keep expired alarms, please first use the resources view to select a default "
		          "archived alarms resource."));
		mKeepArchived->setChecked(false);
		return;
	}
	mOldKeepArchived = keep;
	mPurgeArchived->setEnabled(keep);
	mPurgeAfter->setEnabled(keep && mPurgeArchived->isChecked());
	mPurgeAfterLabel->setEnabled(keep);
	mClearArchived->setEnabled(keep);
}

void StorePrefTab::slotClearArchived()
{
	bool single = AlarmResources::instance()->activeCount(AlarmResource::ARCHIVED, false) <= 1;
	if (KMessageBox::warningContinueCancel(this, single ? i18n("Do you really want to delete all archived alarms?")
	                                                    : i18n("Do you really want to delete all alarms in the default archived alarm resource?"))
			!= KMessageBox::Continue)
		return;
	theApp()->purgeAll();
}


/*=============================================================================
= Class EmailPrefTab
=============================================================================*/

EmailPrefTab::EmailPrefTab(StackedScrollGroup* scrollGroup, QVBox* frame)
	: PrefsTabBase(scrollGroup, frame),
	  mAddressChanged(false),
	  mBccAddressChanged(false)
{
	QHBox* box = new QHBox(mPage);
	box->setSpacing(2*KDialog::spacingHint());
	QLabel* label = new QLabel(i18n("Email client:"), box);
	mEmailClient = new ButtonGroup(box);
	mEmailClient->hide();
	RadioButton* radio = new RadioButton(i18n("&KMail"), box, "kmail");
	radio->setMinimumSize(radio->sizeHint());
	mEmailClient->insert(radio, Preferences::KMAIL);
	radio = new RadioButton(i18n("&Sendmail"), box, "sendmail");
	radio->setMinimumSize(radio->sizeHint());
	mEmailClient->insert(radio, Preferences::SENDMAIL);
	connect(mEmailClient, SIGNAL(buttonSet(int)), SLOT(slotEmailClientChanged(int)));
	box->setFixedHeight(box->sizeHint().height());
	QWhatsThis::add(box,
	      i18n("Choose how to send email when an email alarm is triggered.\n"
	           "- '%1': The email is sent automatically via KMail. KMail is started first if necessary.\n"
	           "- '%2': The email is sent automatically. This option will only work if "
	           "your system is configured to use sendmail or a sendmail compatible mail transport agent.").arg(i18n("KMail")).arg(i18n("Sendmail")));

	box = new QHBox(mPage);   // this is to allow left adjustment
	mEmailCopyToKMail = new QCheckBox(i18n("Co&py sent emails into KMail's %1 folder").arg(KAMail::i18n_sent_mail()), box);
	QWhatsThis::add(mEmailCopyToKMail,
	      i18n("After sending an email, store a copy in KMail's %1 folder").arg(KAMail::i18n_sent_mail()));
	box->setStretchFactor(new QWidget(box), 1);    // left adjust the controls
	box->setFixedHeight(box->sizeHint().height());

	box = new QHBox(mPage);   // this is to allow left adjustment
	mEmailQueuedNotify = new QCheckBox(i18n("&Notify when remote emails are queued"), box);
	QWhatsThis::add(mEmailQueuedNotify,
	      i18n("Display a notification message whenever an email alarm has queued an email for sending to a remote system. "
	           "This could be useful if, for example, you have a dial-up connection, so that you can then ensure that the email is actually transmitted."));
	box->setStretchFactor(new QWidget(box), 1);    // left adjust the controls
	box->setFixedHeight(box->sizeHint().height());

	// Your Email Address group box
	QGroupBox* group = new QGroupBox(i18n("Your Email Address"), mPage);
	QGridLayout* grid = new QGridLayout(group, 6, 3, KDialog::marginHint(), KDialog::spacingHint());
	grid->addRowSpacing(0, fontMetrics().height() - KDialog::marginHint() + KDialog::spacingHint());
	grid->setColStretch(2, 1);

	// 'From' email address controls ...
	label = new Label(EditEmailAlarmDlg::i18n_f_EmailFrom(), group);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 1, 0);
	mFromAddressGroup = new ButtonGroup(group);
	mFromAddressGroup->hide();
	connect(mFromAddressGroup, SIGNAL(buttonSet(int)), SLOT(slotFromAddrChanged(int)));

	// Line edit to enter a 'From' email address
	radio = new RadioButton(group);
	mFromAddressGroup->insert(radio, Preferences::MAIL_FROM_ADDR);
	radio->setFixedSize(radio->sizeHint());
	label->setBuddy(radio);
	grid->addWidget(radio, 1, 1);
	mEmailAddress = new QLineEdit(group);
	connect(mEmailAddress, SIGNAL(textChanged(const QString&)), SLOT(slotAddressChanged()));
	QString whatsThis = i18n("Your email address, used to identify you as the sender when sending email alarms.");
	QWhatsThis::add(radio, whatsThis);
	QWhatsThis::add(mEmailAddress, whatsThis);
	radio->setFocusWidget(mEmailAddress);
	grid->addWidget(mEmailAddress, 1, 2);

	// 'From' email address to be taken from Control Centre
	radio = new RadioButton(i18n("&Use address from Control Center"), group);
	radio->setFixedSize(radio->sizeHint());
	mFromAddressGroup->insert(radio, Preferences::MAIL_FROM_CONTROL_CENTRE);
	QWhatsThis::add(radio,
	      i18n("Check to use the email address set in the KDE Control Center, to identify you as the sender when sending email alarms."));
	grid->addMultiCellWidget(radio, 2, 2, 1, 2, Qt::AlignAuto);

	// 'From' email address to be picked from KMail's identities when the email alarm is configured
	radio = new RadioButton(i18n("Use KMail &identities"), group);
	radio->setFixedSize(radio->sizeHint());
	mFromAddressGroup->insert(radio, Preferences::MAIL_FROM_KMAIL);
	QWhatsThis::add(radio,
	      i18n("Check to use KMail's email identities to identify you as the sender when sending email alarms. "
	           "For existing email alarms, KMail's default identity will be used. "
	           "For new email alarms, you will be able to pick which of KMail's identities to use."));
	grid->addMultiCellWidget(radio, 3, 3, 1, 2, Qt::AlignAuto);

	// 'Bcc' email address controls ...
	grid->addRowSpacing(4, KDialog::spacingHint());
	label = new Label(i18n("'Bcc' email address", "&Bcc:"), group);
	label->setFixedSize(label->sizeHint());
	grid->addWidget(label, 5, 0);
	mBccAddressGroup = new ButtonGroup(group);
	mBccAddressGroup->hide();
	connect(mBccAddressGroup, SIGNAL(buttonSet(int)), SLOT(slotBccAddrChanged(int)));

	// Line edit to enter a 'Bcc' email address
	radio = new RadioButton(group);
	radio->setFixedSize(radio->sizeHint());
	mBccAddressGroup->insert(radio, Preferences::MAIL_FROM_ADDR);
	label->setBuddy(radio);
	grid->addWidget(radio, 5, 1);
	mEmailBccAddress = new QLineEdit(group);
	whatsThis = i18n("Your email address, used for blind copying email alarms to yourself. "
	                 "If you want blind copies to be sent to your account on the computer which KAlarm runs on, you can simply enter your user login name.");
	QWhatsThis::add(radio, whatsThis);
	QWhatsThis::add(mEmailBccAddress, whatsThis);
	radio->setFocusWidget(mEmailBccAddress);
	grid->addWidget(mEmailBccAddress, 5, 2);

	// 'Bcc' email address to be taken from Control Centre
	radio = new RadioButton(i18n("Us&e address from Control Center"), group);
	radio->setFixedSize(radio->sizeHint());
	mBccAddressGroup->insert(radio, Preferences::MAIL_FROM_CONTROL_CENTRE);
	QWhatsThis::add(radio,
	      i18n("Check to use the email address set in the KDE Control Center, for blind copying email alarms to yourself."));
	grid->addMultiCellWidget(radio, 6, 6, 1, 2, Qt::AlignAuto);

	group->setFixedHeight(group->sizeHint().height());

	mPage->setStretchFactor(new QWidget(mPage), 1);    // top adjust the widgets
}

void EmailPrefTab::restore()
{
	mEmailClient->setButton(Preferences::mEmailClient);
	mEmailCopyToKMail->setChecked(Preferences::emailCopyToKMail());
	setEmailAddress(Preferences::mEmailFrom, Preferences::mEmailAddress);
	setEmailBccAddress((Preferences::mEmailBccFrom == Preferences::MAIL_FROM_CONTROL_CENTRE), Preferences::mEmailBccAddress);
	mEmailQueuedNotify->setChecked(Preferences::emailQueuedNotify());
	mAddressChanged = mBccAddressChanged = false;
}

void EmailPrefTab::apply(bool syncToDisc)
{
	int client = mEmailClient->id(mEmailClient->selected());
	Preferences::mEmailClient = (client >= 0) ? Preferences::MailClient(client) : Preferences::default_emailClient;
	Preferences::mEmailCopyToKMail = mEmailCopyToKMail->isChecked();
	Preferences::setEmailAddress(static_cast<Preferences::MailFrom>(mFromAddressGroup->selectedId()), mEmailAddress->text().stripWhiteSpace());
	Preferences::setEmailBccAddress((mBccAddressGroup->selectedId() == Preferences::MAIL_FROM_CONTROL_CENTRE), mEmailBccAddress->text().stripWhiteSpace());
	Preferences::setEmailQueuedNotify(mEmailQueuedNotify->isChecked());
	PrefsTabBase::apply(syncToDisc);
}

void EmailPrefTab::setDefaults()
{
	mEmailClient->setButton(Preferences::default_emailClient);
	setEmailAddress(Preferences::default_emailFrom(), Preferences::default_emailAddress);
	setEmailBccAddress((Preferences::default_emailBccFrom == Preferences::MAIL_FROM_CONTROL_CENTRE), Preferences::default_emailBccAddress);
	mEmailQueuedNotify->setChecked(Preferences::default_emailQueuedNotify);
}

void EmailPrefTab::setEmailAddress(Preferences::MailFrom from, const QString& address)
{
	mFromAddressGroup->setButton(from);
	mEmailAddress->setText(from == Preferences::MAIL_FROM_ADDR ? address.stripWhiteSpace() : QString::null);
}

void EmailPrefTab::setEmailBccAddress(bool useControlCentre, const QString& address)
{
	mBccAddressGroup->setButton(useControlCentre ? Preferences::MAIL_FROM_CONTROL_CENTRE : Preferences::MAIL_FROM_ADDR);
	mEmailBccAddress->setText(useControlCentre ? QString::null : address.stripWhiteSpace());
}

void EmailPrefTab::slotEmailClientChanged(int id)
{
	mEmailCopyToKMail->setEnabled(id == Preferences::SENDMAIL);
}

void EmailPrefTab::slotFromAddrChanged(int id)
{
	mEmailAddress->setEnabled(id == Preferences::MAIL_FROM_ADDR);
	mAddressChanged = true;
}

void EmailPrefTab::slotBccAddrChanged(int id)
{
	mEmailBccAddress->setEnabled(id == Preferences::MAIL_FROM_ADDR);
	mBccAddressChanged = true;
}

QString EmailPrefTab::validate()
{
	if (mAddressChanged)
	{
		mAddressChanged = false;
		QString errmsg = validateAddr(mFromAddressGroup, mEmailAddress, KAMail::i18n_NeedFromEmailAddress());
		if (!errmsg.isEmpty())
			return errmsg;
	}
	if (mBccAddressChanged)
	{
		mBccAddressChanged = false;
		return validateAddr(mBccAddressGroup, mEmailBccAddress, i18n("No valid 'Bcc' email address is specified."));
	}
	return QString::null;
}

QString EmailPrefTab::validateAddr(ButtonGroup* group, QLineEdit* addr, const QString& msg)
{
	QString errmsg = i18n("%1\nAre you sure you want to save your changes?").arg(msg);
	switch (group->selectedId())
	{
		case Preferences::MAIL_FROM_CONTROL_CENTRE:
			if (!KAMail::controlCentreAddress().isEmpty())
				return QString::null;
			errmsg = i18n("No email address is currently set in the KDE Control Center. %1").arg(errmsg);
			break;
		case Preferences::MAIL_FROM_KMAIL:
			if (KAMail::identitiesExist())
				return QString::null;
			errmsg = i18n("No KMail identities currently exist. %1").arg(errmsg);
			break;
		case Preferences::MAIL_FROM_ADDR:
			if (!addr->text().stripWhiteSpace().isEmpty())
				return QString::null;
			break;
	}
	return errmsg;
}


/*=============================================================================
= Class EditPrefTab
=============================================================================*/

EditPrefTab::EditPrefTab(StackedScrollGroup* scrollGroup, QVBox* frame)
	: PrefsTabBase(scrollGroup, frame)
{
	// Get alignment to use in QLabel::setAlignment(alignment | Qt::WordBreak)
	// (AlignAuto doesn't work correctly there)
	int alignment = QApplication::reverseLayout() ? Qt::AlignRight : Qt::AlignLeft;

	int groupTopMargin = fontMetrics().height() - KDialog::marginHint() + KDialog::spacingHint();
	QString defsetting   = i18n("The default setting for \"%1\" in the alarm edit dialog.");
	QString soundSetting = i18n("Check to select %1 as the default setting for \"%2\" in the alarm edit dialog.");

	KTabWidget* tabs = new KTabWidget(mPage);
	StackedGroup* tabgroup = new StackedGroup(tabs);
	StackedWidget* topGeneral = new StackedWidget(tabgroup, tabs);
	tabs->addTab(topGeneral, i18n("General"));
	QVBoxLayout* layGeneral = new QVBoxLayout(topGeneral, KDialog::marginHint(), KDialog::spacingHint());
	StackedWidget* topTypes = new StackedWidget(tabgroup, tabs);
	tabs->addTab(topTypes, i18n("Alarm Types"));
	QVBoxLayout* layTypes = new QVBoxLayout(topTypes, KDialog::marginHint(), KDialog::spacingHint());
	StackedWidget* topFontColour = new StackedWidget(tabgroup, tabs);
	tabs->addTab(topFontColour, i18n("Font & Color"));
	QVBoxLayout* layFontColour = new QVBoxLayout(topFontColour, KDialog::marginHint(), KDialog::spacingHint());

	// MISCELLANEOUS
	// Show in KOrganizer
	mCopyToKOrganizer = new QCheckBox(EditAlarmDlg::i18n_g_ShowInKOrganizer(), topGeneral, "defShowKorg");
	mCopyToKOrganizer->setMinimumSize(mCopyToKOrganizer->sizeHint());
	QWhatsThis::add(mCopyToKOrganizer, defsetting.arg(EditAlarmDlg::i18n_ShowInKOrganizer()));
	layGeneral->addWidget(mCopyToKOrganizer, 0, Qt::AlignAuto);

	// Late cancellation
	mLateCancel = new QCheckBox(LateCancelSelector::i18n_n_CancelIfLate(), topGeneral, "defCancelLate");
	mLateCancel->setMinimumSize(mLateCancel->sizeHint());
	QWhatsThis::add(mLateCancel, defsetting.arg(LateCancelSelector::i18n_CancelIfLate()));
	layGeneral->addWidget(mLateCancel, 0, Qt::AlignAuto);

	// Recurrence
	QHBoxLayout* hlayout = new QHBoxLayout(layGeneral);
	QHBox* box = new QHBox(topGeneral);   // this is to control the QWhatsThis text display area
	box->setSpacing(KDialog::spacingHint());
	QLabel* label = new QLabel(i18n("&Recurrence:"), box);
	label->setFixedSize(label->sizeHint());
	mRecurPeriod = new QComboBox(box, "defRecur");
	mRecurPeriod->insertItem(RecurrenceEdit::i18n_NoRecur());
	mRecurPeriod->insertItem(RecurrenceEdit::i18n_AtLogin());
	mRecurPeriod->insertItem(RecurrenceEdit::i18n_HourlyMinutely());
	mRecurPeriod->insertItem(RecurrenceEdit::i18n_Daily());
	mRecurPeriod->insertItem(RecurrenceEdit::i18n_Weekly());
	mRecurPeriod->insertItem(RecurrenceEdit::i18n_Monthly());
	mRecurPeriod->insertItem(RecurrenceEdit::i18n_Yearly());
	mRecurPeriod->setFixedSize(mRecurPeriod->sizeHint());
	label->setBuddy(mRecurPeriod);
	QWhatsThis::add(box,
	      i18n("The default setting for the recurrence rule in the alarm edit dialog."));
	hlayout->addWidget(box);
	hlayout->addStretch();

	// How to handle February 29th in yearly recurrences
	QVBox* vbox = new QVBox(topGeneral);   // this is to control the QWhatsThis text display area
	vbox->setSpacing(KDialog::spacingHint());
	label = new QLabel(i18n("In non-leap years, repeat yearly February 29th alarms on:"), vbox);
	label->setAlignment(alignment | Qt::WordBreak);
	QHBox* itemBox = new QHBox(vbox);
	itemBox->setSpacing(2*KDialog::spacingHint());
	mFeb29 = new QButtonGroup(itemBox);
	mFeb29->hide();
	QWidget* widget = new QWidget(itemBox);
	widget->setFixedWidth(3*KDialog::spacingHint());
	QRadioButton* radio = new QRadioButton(i18n("February 2&8th"), itemBox);
	radio->setMinimumSize(radio->sizeHint());
	mFeb29->insert(radio, Preferences::Feb29_Feb28);
	radio = new QRadioButton(i18n("March &1st"), itemBox);
	radio->setMinimumSize(radio->sizeHint());
	mFeb29->insert(radio, Preferences::Feb29_Mar1);
	radio = new QRadioButton(i18n("Do &not repeat"), itemBox);
	radio->setMinimumSize(radio->sizeHint());
	mFeb29->insert(radio, Preferences::Feb29_None);
	itemBox->setFixedHeight(itemBox->sizeHint().height());
	QWhatsThis::add(vbox,
	      i18n("For yearly recurrences, choose what date, if any, alarms due on February 29th should occur in non-leap years.\n"
	           "Note that the next scheduled occurrence of existing alarms is not re-evaluated when you change this setting."));
	layGeneral->addWidget(vbox, 0, Qt::AlignAuto);

	layGeneral->addStretch();    // top adjust the tab widgets

	// DISPLAY ALARMS
	QGroupBox* group = new QGroupBox(i18n("Display Alarms"), topTypes);
	layTypes->addWidget(group);
	QBoxLayout* layout = new QVBoxLayout(group, KDialog::marginHint(), KDialog::spacingHint());
	layout->addSpacing(groupTopMargin);

	mConfirmAck = new QCheckBox(EditDisplayAlarmDlg::i18n_k_ConfirmAck(), group, "defConfAck");
	mConfirmAck->setMinimumSize(mConfirmAck->sizeHint());
	QWhatsThis::add(mConfirmAck, defsetting.arg(EditDisplayAlarmDlg::i18n_ConfirmAck()));
	layout->addWidget(mConfirmAck, 0, Qt::AlignAuto);

	mAutoClose = new QCheckBox(LateCancelSelector::i18n_i_AutoCloseWinLC(), group, "defAutoClose");
	mAutoClose->setMinimumSize(mAutoClose->sizeHint());
	QWhatsThis::add(mAutoClose, defsetting.arg(LateCancelSelector::i18n_AutoCloseWin()));
	layout->addWidget(mAutoClose, 0, Qt::AlignAuto);

	box = new QHBox(group);
	box->setSpacing(KDialog::spacingHint());
	layout->addWidget(box);
	label = new QLabel(i18n("Reminder &units:"), box);
	mReminderUnits = new QComboBox(box, "defWarnUnits");
	mReminderUnits->insertItem(TimePeriod::i18n_Minutes(), TimePeriod::MINUTES);
	mReminderUnits->insertItem(TimePeriod::i18n_Hours_Mins(), TimePeriod::HOURS_MINUTES);
	mReminderUnits->setFixedSize(mReminderUnits->sizeHint());
	label->setBuddy(mReminderUnits);
	QWhatsThis::add(box,
	      i18n("The default units for the reminder in the alarm edit dialog, for alarms due soon."));
	box->setStretchFactor(new QWidget(box), 1);    // left adjust the control

	mSpecialActionsButton = new SpecialActionsButton(true, box);
	mSpecialActionsButton->setFixedSize(mSpecialActionsButton->sizeHint());

	// SOUND
	QButtonGroup* bgroup = new QButtonGroup(SoundPicker::i18n_Sound(), topTypes, "soundGroup");
	layTypes->addWidget(bgroup);
	layout = new QVBoxLayout(bgroup, KDialog::marginHint(), KDialog::spacingHint());
	layout->addSpacing(groupTopMargin);

	hlayout = new QHBoxLayout(layout, KDialog::spacingHint());
	mSound = new QComboBox(false, bgroup, "defSound");
	mSound->insertItem(SoundPicker::i18n_None());         // index 0
	mSound->insertItem(SoundPicker::i18n_Beep());         // index 1
	mSound->insertItem(SoundPicker::i18n_File());         // index 2
	if (theApp()->speechEnabled())
		mSound->insertItem(SoundPicker::i18n_Speak());  // index 3
	mSound->setMinimumSize(mSound->sizeHint());
	QWhatsThis::add(mSound, defsetting.arg(SoundPicker::i18n_Sound()));
	hlayout->addWidget(mSound);
	hlayout->addStretch(1);

#ifndef WITHOUT_ARTS
	mSoundRepeat = new QCheckBox(i18n("Repea&t sound file"), bgroup, "defRepeatSound");
	mSoundRepeat->setMinimumSize(mSoundRepeat->sizeHint());
	QWhatsThis::add(mSoundRepeat, i18n("sound file \"Repeat\" checkbox", "The default setting for sound file \"%1\" in the alarm edit dialog.").arg(SoundDlg::i18n_Repeat()));
	hlayout->addWidget(mSoundRepeat);
#endif

	box = new QHBox(bgroup);   // this is to control the QWhatsThis text display area
	box->setSpacing(KDialog::spacingHint());
	mSoundFileLabel = new QLabel(i18n("Sound &file:"), box);
	mSoundFileLabel->setFixedSize(mSoundFileLabel->sizeHint());
	mSoundFile = new QLineEdit(box);
	mSoundFileLabel->setBuddy(mSoundFile);
	mSoundFileBrowse = new QPushButton(box);
	mSoundFileBrowse->setPixmap(SmallIcon("fileopen"));
	mSoundFileBrowse->setFixedSize(mSoundFileBrowse->sizeHint());
	connect(mSoundFileBrowse, SIGNAL(clicked()), SLOT(slotBrowseSoundFile()));
	QToolTip::add(mSoundFileBrowse, i18n("Choose a sound file"));
	QWhatsThis::add(box,
	      i18n("Enter the default sound file to use in the alarm edit dialog."));
	box->setFixedHeight(box->sizeHint().height());
	layout->addWidget(box);
	bgroup->setFixedHeight(bgroup->sizeHint().height());

	// COMMAND ALARMS
	group = new QGroupBox(i18n("Command Alarms"), topTypes);
	layTypes->addWidget(group);
	layout = new QVBoxLayout(group, KDialog::marginHint(), KDialog::spacingHint());
	layout->addSpacing(groupTopMargin);
	layout = new QHBoxLayout(layout, KDialog::spacingHint());

	mCmdScript = new QCheckBox(EditCommandAlarmDlg::i18n_p_EnterScript(), group, "defCmdScript");
	mCmdScript->setMinimumSize(mCmdScript->sizeHint());
	QWhatsThis::add(mCmdScript, defsetting.arg(EditCommandAlarmDlg::i18n_EnterScript()));
	layout->addWidget(mCmdScript);
	layout->addStretch();

	mCmdXterm = new QCheckBox(EditCommandAlarmDlg::i18n_w_ExecInTermWindow(), group, "defCmdXterm");
	mCmdXterm->setMinimumSize(mCmdXterm->sizeHint());
	QWhatsThis::add(mCmdXterm, defsetting.arg(EditCommandAlarmDlg::i18n_ExecInTermWindow()));
	layout->addWidget(mCmdXterm);

	// EMAIL ALARMS
	group = new QGroupBox(i18n("Email Alarms"), topTypes);
	layTypes->addWidget(group);
	layout = new QVBoxLayout(group, KDialog::marginHint(), KDialog::spacingHint());
	layout->addSpacing(groupTopMargin);

	// BCC email to sender
	mEmailBcc = new QCheckBox(EditEmailAlarmDlg::i18n_e_CopyEmailToSelf(), group, "defEmailBcc");
	mEmailBcc->setMinimumSize(mEmailBcc->sizeHint());
	QWhatsThis::add(mEmailBcc, defsetting.arg(EditEmailAlarmDlg::i18n_CopyEmailToSelf()));
	layout->addWidget(mEmailBcc, 0, Qt::AlignAuto);

	layTypes->addStretch();    // top adjust the tab widgets

	// FONT / COLOUR TAB
	mFontChooser = new FontColourChooser(topFontColour, 0, false, QStringList(), i18n("Message Font && Color"), true);
	layFontColour->addWidget(mFontChooser);
}

void EditPrefTab::restore()
{
	mAutoClose->setChecked(Preferences::mDefaultAutoClose);
	mConfirmAck->setChecked(Preferences::mDefaultConfirmAck);
	mReminderUnits->setCurrentItem(Preferences::mDefaultReminderUnits);
	mSpecialActionsButton->setActions(Preferences::mDefaultPreAction, Preferences::mDefaultPostAction, Preferences::mDefaultCancelOnPreActionError);
	mSound->setCurrentItem(soundIndex(Preferences::mDefaultSoundType));
	mSoundFile->setText(Preferences::mDefaultSoundFile);
#ifndef WITHOUT_ARTS
	mSoundRepeat->setChecked(Preferences::mDefaultSoundRepeat);
#endif
	mCmdScript->setChecked(Preferences::mDefaultCmdScript);
	mCmdXterm->setChecked(Preferences::mDefaultCmdLogType == Preferences::Log_Terminal);
	mEmailBcc->setChecked(Preferences::mDefaultEmailBcc);
	mCopyToKOrganizer->setChecked(Preferences::mDefaultCopyToKOrganizer);
	mLateCancel->setChecked(Preferences::mDefaultLateCancel);
	mRecurPeriod->setCurrentItem(recurIndex(Preferences::mDefaultRecurPeriod));
	mFeb29->setButton(Preferences::mDefaultFeb29Type);
	mFontChooser->setFgColour(Preferences::mDefaultFgColour);
	mFontChooser->setBgColour(Preferences::mDefaultBgColour);
	mFontChooser->setFont(Preferences::mMessageFont);
}

void EditPrefTab::apply(bool syncToDisc)
{
	Preferences::mDefaultAutoClose        = mAutoClose->isChecked();
	Preferences::mDefaultConfirmAck       = mConfirmAck->isChecked();
	Preferences::mDefaultReminderUnits    = static_cast<TimePeriod::Units>(mReminderUnits->currentItem());
	Preferences::mDefaultPreAction        = mSpecialActionsButton->preAction();
	Preferences::mDefaultPostAction       = mSpecialActionsButton->postAction();
	Preferences::mDefaultCancelOnPreActionError = mSpecialActionsButton->cancelOnError();
	switch (mSound->currentItem())
	{
		case 3:  Preferences::mDefaultSoundType = SoundPicker::SPEAK;      break;
		case 2:  Preferences::mDefaultSoundType = SoundPicker::PLAY_FILE;  break;
		case 1:  Preferences::mDefaultSoundType = SoundPicker::BEEP;       break;
		case 0:
		default: Preferences::mDefaultSoundType = SoundPicker::NONE;       break;
	}
	Preferences::mDefaultSoundFile        = mSoundFile->text();
#ifndef WITHOUT_ARTS
	Preferences::mDefaultSoundRepeat      = mSoundRepeat->isChecked();
#endif
	Preferences::mDefaultCmdScript        = mCmdScript->isChecked();
	Preferences::mDefaultCmdLogType       = (mCmdXterm->isChecked() ? Preferences::Log_Terminal : Preferences::Log_Discard);
	Preferences::mDefaultEmailBcc         = mEmailBcc->isChecked();
	Preferences::mDefaultCopyToKOrganizer = mCopyToKOrganizer->isChecked();
	Preferences::mDefaultLateCancel       = mLateCancel->isChecked() ? 1 : 0;
	switch (mRecurPeriod->currentItem())
	{
		case 6:  Preferences::mDefaultRecurPeriod = RecurrenceEdit::ANNUAL;    break;
		case 5:  Preferences::mDefaultRecurPeriod = RecurrenceEdit::MONTHLY;   break;
		case 4:  Preferences::mDefaultRecurPeriod = RecurrenceEdit::WEEKLY;    break;
		case 3:  Preferences::mDefaultRecurPeriod = RecurrenceEdit::DAILY;     break;
		case 2:  Preferences::mDefaultRecurPeriod = RecurrenceEdit::SUBDAILY;  break;
		case 1:  Preferences::mDefaultRecurPeriod = RecurrenceEdit::AT_LOGIN;  break;
		case 0:
		default: Preferences::mDefaultRecurPeriod = RecurrenceEdit::NO_RECUR;  break;
	}
	int feb29 = mFeb29->selectedId();
	Preferences::mDefaultFeb29Type  = (feb29 >= 0) ? static_cast<Preferences::Feb29Type>(feb29) : Preferences::default_defaultFeb29Type;
	Preferences::mDefaultFgColour = mFontChooser->fgColour();
	Preferences::mDefaultBgColour = mFontChooser->bgColour();
	Preferences::mMessageFont     = mFontChooser->font();
	PrefsTabBase::apply(syncToDisc);
}

void EditPrefTab::setDefaults()
{
	mAutoClose->setChecked(Preferences::default_defaultAutoClose);
	mConfirmAck->setChecked(Preferences::default_defaultConfirmAck);
	mReminderUnits->setCurrentItem(Preferences::default_defaultReminderUnits);
	mSpecialActionsButton->setActions(Preferences::default_defaultPreAction, Preferences::default_defaultPostAction, Preferences::default_defaultCancelOnPreActionError);
	mSound->setCurrentItem(soundIndex(Preferences::default_defaultSoundType));
	mSoundFile->setText(Preferences::default_defaultSoundFile);
#ifndef WITHOUT_ARTS
	mSoundRepeat->setChecked(Preferences::default_defaultSoundRepeat);
#endif
	mCmdScript->setChecked(Preferences::default_defaultCmdScript);
	mCmdXterm->setChecked(Preferences::default_defaultCmdLogType == Preferences::Log_Terminal);
	mEmailBcc->setChecked(Preferences::default_defaultEmailBcc);
	mCopyToKOrganizer->setChecked(Preferences::default_defaultCopyToKOrganizer);
	mLateCancel->setChecked(Preferences::default_defaultLateCancel);
	mRecurPeriod->setCurrentItem(recurIndex(Preferences::default_defaultRecurPeriod));
	mFeb29->setButton(Preferences::default_defaultFeb29Type);
	mFontChooser->setFgColour(Preferences::default_defaultFgColour);
	mFontChooser->setBgColour(Preferences::default_defaultBgColour);
	mFontChooser->setFont(Preferences::default_messageFont());
}

void EditPrefTab::slotBrowseSoundFile()
{
	QString defaultDir;
	QString url = SoundPicker::browseFile(defaultDir, mSoundFile->text());
	if (!url.isEmpty())
		mSoundFile->setText(url);
}

int EditPrefTab::soundIndex(SoundPicker::Type type)
{
	switch (type)
	{
		case SoundPicker::SPEAK:      return 3;
		case SoundPicker::PLAY_FILE:  return 2;
		case SoundPicker::BEEP:       return 1;
		case SoundPicker::NONE:
		default:                      return 0;
	}
}

int EditPrefTab::recurIndex(RecurrenceEdit::RepeatType type)
{
	switch (type)
	{
		case RecurrenceEdit::ANNUAL:   return 6;
		case RecurrenceEdit::MONTHLY:  return 5;
		case RecurrenceEdit::WEEKLY:   return 4;
		case RecurrenceEdit::DAILY:    return 3;
		case RecurrenceEdit::SUBDAILY: return 2;
		case RecurrenceEdit::AT_LOGIN: return 1;
		case RecurrenceEdit::NO_RECUR:
		default:                       return 0;
	}
}

QString EditPrefTab::validate()
{
	if (mSound->currentItem() == SoundPicker::PLAY_FILE  &&  mSoundFile->text().isEmpty())
	{
		mSoundFile->setFocus();
		return i18n("You must enter a sound file when %1 is selected as the default sound type").arg(SoundPicker::i18n_File());;
	}
	return QString::null;
}


/*=============================================================================
= Class ViewPrefTab
=============================================================================*/

ViewPrefTab::ViewPrefTab(StackedScrollGroup* scrollGroup, QVBox* frame)
	: PrefsTabBase(scrollGroup, frame)
{
	int groupTopMargin = fontMetrics().height() - KDialog::marginHint() + KDialog::spacingHint();

	KTabWidget* tabs = new KTabWidget(mPage);
	QWidget* topGeneral = new QWidget(tabs);
	tabs->addTab(topGeneral, i18n("General"));
	QVBoxLayout* layGeneral = new QVBoxLayout(topGeneral, KDialog::marginHint(), KDialog::spacingHint());
	QWidget* topWindows = new QWidget(tabs);
	tabs->addTab(topWindows, i18n("Alarm Windows"));
	QVBoxLayout* layWindows = new QVBoxLayout(topWindows, KDialog::marginHint(), KDialog::spacingHint());

	QHBox* itemBox = new QHBox(topGeneral);
	layGeneral->addWidget(itemBox);
	QHBox* box = new QHBox(itemBox);   // this is to control the QWhatsThis text display area
	box->setSpacing(KDialog::spacingHint());
	// Run-in-system-tray check box
	mShowInSystemTray = new QCheckBox(i18n("Show in system tray"), box, "runTray");
	QWhatsThis::add(box,
	      i18n("Check to show KAlarm's icon in the system tray."
	           " Showing it in the system tray provides easy access and a status indication."));
	itemBox->setStretchFactor(new QWidget(itemBox), 1);    // left adjust the controls
	itemBox->setFixedHeight(box->sizeHint().height());

	QGroupBox* group = new QGroupBox(i18n("System Tray Tooltip"), topGeneral);
	layGeneral->addWidget(group);
	QGridLayout* grid = new QGridLayout(group, 5, 3, KDialog::marginHint(), KDialog::spacingHint());
	grid->setColStretch(2, 1);
	grid->addColSpacing(0, indentWidth());
	grid->addColSpacing(1, indentWidth());
	grid->addRowSpacing(0, fontMetrics().height() - KDialog::marginHint() + KDialog::spacingHint());

	mTooltipShowAlarms = new QCheckBox(i18n("Show next &24 hours' alarms"), group, "tooltipShow");
	mTooltipShowAlarms->setMinimumSize(mTooltipShowAlarms->sizeHint());
	connect(mTooltipShowAlarms, SIGNAL(toggled(bool)), SLOT(slotTooltipAlarmsToggled(bool)));
	QWhatsThis::add(mTooltipShowAlarms,
	      i18n("Specify whether to include in the system tray tooltip, a summary of alarms due in the next 24 hours"));
	grid->addMultiCellWidget(mTooltipShowAlarms, 1, 1, 0, 2, Qt::AlignAuto);

	box = new QHBox(group);
	box->setSpacing(KDialog::spacingHint());
	mTooltipMaxAlarms = new QCheckBox(i18n("Ma&ximum number of alarms to show:"), box, "tooltipMax");
	mTooltipMaxAlarms->setMinimumSize(mTooltipMaxAlarms->sizeHint());
	connect(mTooltipMaxAlarms, SIGNAL(toggled(bool)), SLOT(slotTooltipMaxToggled(bool)));
	mTooltipMaxAlarmCount = new SpinBox(1, 99, 1, box);
	mTooltipMaxAlarmCount->setLineShiftStep(5);
	mTooltipMaxAlarmCount->setMinimumSize(mTooltipMaxAlarmCount->sizeHint());
	QWhatsThis::add(box,
	      i18n("Uncheck to display all of the next 24 hours' alarms in the system tray tooltip. "
	           "Check to enter an upper limit on the number to be displayed."));
	grid->addMultiCellWidget(box, 2, 2, 1, 2, Qt::AlignAuto);

	mTooltipShowTime = new QCheckBox(MainWindow::i18n_m_ShowAlarmTime(), group, "tooltipTime");
	mTooltipShowTime->setMinimumSize(mTooltipShowTime->sizeHint());
	connect(mTooltipShowTime, SIGNAL(toggled(bool)), SLOT(slotTooltipTimeToggled(bool)));
	QWhatsThis::add(mTooltipShowTime,
	      i18n("Specify whether to show in the system tray tooltip, the time at which each alarm is due"));
	grid->addMultiCellWidget(mTooltipShowTime, 3, 3, 1, 2, Qt::AlignAuto);

	mTooltipShowTimeTo = new QCheckBox(MainWindow::i18n_l_ShowTimeToAlarm(), group, "tooltipTimeTo");
	mTooltipShowTimeTo->setMinimumSize(mTooltipShowTimeTo->sizeHint());
	connect(mTooltipShowTimeTo, SIGNAL(toggled(bool)), SLOT(slotTooltipTimeToToggled(bool)));
	QWhatsThis::add(mTooltipShowTimeTo,
	      i18n("Specify whether to show in the system tray tooltip, how long until each alarm is due"));
	grid->addMultiCellWidget(mTooltipShowTimeTo, 4, 4, 1, 2, Qt::AlignAuto);

	box = new QHBox(group);   // this is to control the QWhatsThis text display area
	box->setSpacing(KDialog::spacingHint());
	mTooltipTimeToPrefixLabel = new QLabel(i18n("&Prefix:"), box);
	mTooltipTimeToPrefixLabel->setFixedSize(mTooltipTimeToPrefixLabel->sizeHint());
	mTooltipTimeToPrefix = new QLineEdit(box);
	mTooltipTimeToPrefixLabel->setBuddy(mTooltipTimeToPrefix);
	QWhatsThis::add(box,
	      i18n("Enter the text to be displayed in front of the time until the alarm, in the system tray tooltip"));
	box->setFixedHeight(box->sizeHint().height());
	grid->addWidget(box, 5, 2, Qt::AlignAuto);
	group->setMaximumHeight(group->sizeHint().height());

	group = new QGroupBox(i18n("Alarm List"), topGeneral);
	layGeneral->addWidget(group);
	QHBoxLayout* hlayout = new QHBoxLayout(group, KDialog::marginHint(), 0);
	QVBoxLayout* colourLayout = new QVBoxLayout(hlayout, 0);
	colourLayout->addSpacing(groupTopMargin);

	box = new QHBox(group);    // to group widgets for QWhatsThis text
	box->setSpacing(KDialog::spacingHint()/2);
	colourLayout->addWidget(box);
	QLabel* label1 = new QLabel(i18n("Di&sabled alarm color:"), box);
	box->setStretchFactor(new QWidget(box), 1);
	mDisabledColour = new ColourButton(box);
	label1->setBuddy(mDisabledColour);
	QWhatsThis::add(box,
	      i18n("Choose the text color in the alarm list for disabled alarms."));

	box = new QHBox(group);    // to group widgets for QWhatsThis text
	box->setSpacing(KDialog::spacingHint()/2);
	colourLayout->addWidget(box);
	QLabel* label2 = new QLabel(i18n("Archived alarm color:"), box);
	box->setStretchFactor(new QWidget(box), 1);
	mArchivedColour = new ColourButton(box);
	label2->setBuddy(mArchivedColour);
	QWhatsThis::add(box,
	      i18n("Choose the text color in the alarm list for archived alarms."));
	hlayout->addStretch();

	layGeneral->addStretch();    // top adjust the tab widgets

	mWindowPosition = new ButtonGroup(i18n("Alarm Message Windows"), topWindows, "winGroup");
	layWindows->addWidget(mWindowPosition);
	grid = new QGridLayout(mWindowPosition, 4, 2, KDialog::marginHint(), KDialog::spacingHint());
	grid->setColStretch(1, 1);
	grid->addColSpacing(0, indentWidth());
	grid->addRowSpacing(0, fontMetrics().height() - KDialog::marginHint() + KDialog::spacingHint());

	QString whatsthis = i18n("Choose how to reduce the chance of alarm messages being accidentally acknowledged:\n"
	                         "- Position alarm message windows as far as possible from the current mouse cursor location, or\n"
	                         "- Position alarm message windows in the center of the screen, but disable buttons for a short time after the window is displayed.");
	QRadioButton* radio = new QRadioButton(i18n("Position windows far from mouse cursor"), mWindowPosition, "winAway");
	mWindowPosition->remove(radio);
	mWindowPosition->insert(radio, 0);
	QWhatsThis::add(radio, whatsthis);
	grid->addMultiCellWidget(radio, 1, 1, 0, 1, Qt::AlignAuto);
	radio = new QRadioButton(i18n("Center windows, delay activating window buttons"), mWindowPosition, "winDelay");
	mWindowPosition->remove(radio);
	mWindowPosition->insert(radio, 1);
	QWhatsThis::add(radio, whatsthis);
	grid->addMultiCellWidget(radio, 2, 2, 0, 1, Qt::AlignAuto);
	connect(mWindowPosition, SIGNAL(buttonSet(int)), SLOT(slotWindowPosChanged(int)));

	itemBox = new QHBox(mWindowPosition);
	box = new QHBox(itemBox);   // this is to control the QWhatsThis text display area
	box->setSpacing(KDialog::spacingHint());
	mWindowButtonDelayLabel = new QLabel(i18n("Button activation delay (seconds):"), box);
	mWindowButtonDelay = new QSpinBox(1, 10, 1, box);
	mWindowButtonDelayLabel->setBuddy(mWindowButtonDelay);
	QWhatsThis::add(box,
	      i18n("Enter how long its buttons should remain disabled after the alarm message window is shown."));
	itemBox->setStretchFactor(new QWidget(itemBox), 1);    // left adjust the controls
	grid->addMultiCellWidget(itemBox, 3, 3, 1, 1, Qt::AlignAuto);

	grid->addRowSpacing(4, KDialog::spacingHint());

	mModalMessages = new QCheckBox(i18n("Message &windows have a title bar and take keyboard focus"), mWindowPosition, "modalMsg");
	mModalMessages->setMinimumSize(mModalMessages->sizeHint());
	mWindowPosition->remove(mModalMessages);
	QWhatsThis::add(mModalMessages,
	      i18n("Specify the characteristics of alarm message windows:\n"
	           "- If checked, the window is a normal window with a title bar, which grabs keyboard input when it is displayed.\n"
	           "- If unchecked, the window does not interfere with your typing when "
	           "it is displayed, but it has no title bar and cannot be moved or resized."));
	grid->addMultiCellWidget(mModalMessages, 5, 5, 0, 1, Qt::AlignAuto);

	layWindows->addStretch();    // top adjust the tab widgets
}

void ViewPrefTab::restore()
{
	mDisabledColour->setColor(Preferences::mDisabledColour);
	mArchivedColour->setColor(Preferences::mArchivedColour);
	setTooltip(Preferences::mTooltipAlarmCount,
	           Preferences::mShowTooltipAlarmTime,
	           Preferences::mShowTooltipTimeToAlarm,
	           Preferences::mTooltipTimeToPrefix);
	mShowInSystemTray->setChecked(Preferences::mShowInSystemTray);
	mWindowPosition->setButton(Preferences::mMessageButtonDelay ? 1 : 0);
	mWindowButtonDelay->setValue(Preferences::mMessageButtonDelay);
	mModalMessages->setChecked(Preferences::mModalMessages);
}

void ViewPrefTab::apply(bool syncToDisc)
{
	Preferences::mDisabledColour = mDisabledColour->color();
	Preferences::mArchivedColour = mArchivedColour->color();
	int n = mTooltipShowAlarms->isChecked() ? -1 : 0;
	if (n  &&  mTooltipMaxAlarms->isChecked())
		n = mTooltipMaxAlarmCount->value();
	Preferences::mTooltipAlarmCount      = n;
	Preferences::mShowTooltipAlarmTime   = mTooltipShowTime->isChecked();
	Preferences::mShowTooltipTimeToAlarm = mTooltipShowTimeTo->isChecked();
	Preferences::mTooltipTimeToPrefix    = mTooltipTimeToPrefix->text();
	Preferences::mShowInSystemTray       = mShowInSystemTray->isChecked();
	Preferences::mMessageButtonDelay     = mWindowPosition->selectedId() ? mWindowButtonDelay->value() : 0;
	Preferences::mModalMessages          = mModalMessages->isChecked();
	PrefsTabBase::apply(syncToDisc);
}

void ViewPrefTab::setDefaults()
{
	mDisabledColour->setColor(Preferences::default_disabledColour);
	mArchivedColour->setColor(Preferences::default_archivedColour);
	setTooltip(Preferences::default_tooltipAlarmCount,
	           Preferences::default_showTooltipAlarmTime,
	           Preferences::default_showTooltipTimeToAlarm,
	           Preferences::default_tooltipTimeToPrefix);
	mShowInSystemTray->setChecked(Preferences::default_showInSystemTray);
	mWindowPosition->setButton(Preferences::default_messageButtonDelay ? 1 : 0);
	mWindowButtonDelay->setValue(Preferences::default_messageButtonDelay);
	mModalMessages->setChecked(Preferences::default_modalMessages);
}

void ViewPrefTab::setTooltip(int maxAlarms, bool time, bool timeTo, const QString& prefix)
{
	if (!timeTo)
		time = true;    // ensure that at least one time option is ticked

	// Set the states of the controls without calling signal
	// handlers, since these could change the checkboxes' states.
	mTooltipShowAlarms->blockSignals(true);
	mTooltipShowTime->blockSignals(true);
	mTooltipShowTimeTo->blockSignals(true);

	mTooltipShowAlarms->setChecked(maxAlarms);
	mTooltipMaxAlarms->setChecked(maxAlarms > 0);
	mTooltipMaxAlarmCount->setValue(maxAlarms > 0 ? maxAlarms : 1);
	mTooltipShowTime->setChecked(time);
	mTooltipShowTimeTo->setChecked(timeTo);
	mTooltipTimeToPrefix->setText(prefix);

	mTooltipShowAlarms->blockSignals(false);
	mTooltipShowTime->blockSignals(false);
	mTooltipShowTimeTo->blockSignals(false);

	// Enable/disable controls according to their states
	slotTooltipTimeToToggled(timeTo);
	slotTooltipAlarmsToggled(maxAlarms);
}

void ViewPrefTab::slotTooltipAlarmsToggled(bool on)
{
	mTooltipMaxAlarms->setEnabled(on);
	mTooltipMaxAlarmCount->setEnabled(on && mTooltipMaxAlarms->isChecked());
	mTooltipShowTime->setEnabled(on);
	mTooltipShowTimeTo->setEnabled(on);
	on = on && mTooltipShowTimeTo->isChecked();
	mTooltipTimeToPrefix->setEnabled(on);
	mTooltipTimeToPrefixLabel->setEnabled(on);
}

void ViewPrefTab::slotTooltipMaxToggled(bool on)
{
	mTooltipMaxAlarmCount->setEnabled(on && mTooltipMaxAlarms->isEnabled());
}

void ViewPrefTab::slotTooltipTimeToggled(bool on)
{
	if (!on  &&  !mTooltipShowTimeTo->isChecked())
		mTooltipShowTimeTo->setChecked(true);
}

void ViewPrefTab::slotTooltipTimeToToggled(bool on)
{
	if (!on  &&  !mTooltipShowTime->isChecked())
		mTooltipShowTime->setChecked(true);
	on = on && mTooltipShowTimeTo->isEnabled();
	mTooltipTimeToPrefix->setEnabled(on);
	mTooltipTimeToPrefixLabel->setEnabled(on);
}

void ViewPrefTab::slotWindowPosChanged(int id)
{
	mWindowButtonDelay->setEnabled(id);
	mWindowButtonDelayLabel->setEnabled(id);
}
