/*
 *  preferences.cpp  -  program preference settings
 *  Program:  kalarm
 *  Copyright © 2001-2008 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kalarm.h"

#include <time.h>
#include <unistd.h>

#include <kglobal.h>
#include <kconfig.h>
#include <kstandarddirs.h>
#include <kapplication.h>
#include <kglobalsettings.h>
#include <kmessagebox.h>
#include <kdatetime.h>
#include <ksystemtimezone.h>
#include <kdebug.h>

#include <libkpimidentities/identity.h>
#include <libkpimidentities/identitymanager.h>
#include <libkholidays/kholidays.h>

#include "functions.h"
#include "kamail.h"
#include "messagebox.h"
#include "preferences.moc"


static QString translateXTermPath(KConfig*, const QString& cmdline, bool write);

Preferences* Preferences::mInstance = 0;

// Default config file settings
const QColor                     Preferences::default_defaultBgColour(Qt::red);
const QColor                     Preferences::default_defaultFgColour(Qt::black);
QFont                            Preferences::mDefault_messageFont;    // initialised in constructor
const QTime                      Preferences::default_startOfDay(0, 0);
QBitArray                        Preferences::mDefault_workDays(7);    // initialised in constructor
static unsigned                  defaultWorkDays = 0x1F;   // Mon - Fri
const QTime                      Preferences::default_workDayStart(8, 0);
const QTime                      Preferences::default_workDayEnd(17, 0);
KHolidays*                       Preferences::mDefault_holidays               = 0;
const bool                       Preferences::default_showInSystemTray        = true;
const bool                       Preferences::default_quitWarn                = true;
const bool                       Preferences::default_autoStart               = false;
const bool                       Preferences::default_confirmAlarmDeletion    = true;
const bool                       Preferences::default_askResource             = true;
const bool                       Preferences::default_modalMessages           = true;
const int                        Preferences::default_messageButtonDelay      = 0;     // (seconds)
const int                        Preferences::default_tooltipAlarmCount       = 5;
const bool                       Preferences::default_showTooltipAlarmTime    = true;
const bool                       Preferences::default_showTooltipTimeToAlarm  = true;
const QString                    Preferences::default_tooltipTimeToPrefix     = QString::fromLatin1("+");
const bool                       Preferences::default_emailCopyToKMail        = false;
const bool                       Preferences::default_emailQueuedNotify       = false;
const QColor                     Preferences::default_disabledColour(Qt::lightGray);
const QColor                     Preferences::default_archivedColour(Qt::darkRed);
const int                        Preferences::default_archivedKeepDays        = 7;
const QString                    Preferences::default_defaultSoundFile        = QString::null;
const int                        Preferences::default_defaultSoundVolume      = -1;
const int                        Preferences::default_defaultLateCancel       = 0;
const bool                       Preferences::default_defaultAutoClose        = false;
const bool                       Preferences::default_defaultCopyToKOrganizer = false;
const bool                       Preferences::default_defaultSoundRepeat      = false;
const SoundPicker::Type          Preferences::default_defaultSoundType        = SoundPicker::NONE;
const bool                       Preferences::default_defaultConfirmAck       = false;
const bool                       Preferences::default_defaultCmdScript        = false;
const Preferences::CmdLogType    Preferences::default_defaultCmdLogType       = Preferences::Log_Discard;
const bool                       Preferences::default_defaultEmailBcc         = false;
const QString                    Preferences::default_emailAddress            = QString::null;
const QString                    Preferences::default_emailBccAddress         = QString::null;
const Preferences::MailClient    Preferences::default_emailClient             = KMAIL;
const Preferences::MailFrom      Preferences::default_emailBccFrom            = MAIL_FROM_CONTROL_CENTRE;
const RecurrenceEdit::RepeatType Preferences::default_defaultRecurPeriod      = RecurrenceEdit::NO_RECUR;
const Preferences::Feb29Type     Preferences::default_defaultFeb29Type        = Preferences::Feb29_Mar1;
const TimePeriod::Units          Preferences::default_defaultReminderUnits    = TimePeriod::HOURS_MINUTES;  // must be MINUTES or HOURS_MINUTES
const QString                    Preferences::default_defaultPreAction;
const QString                    Preferences::default_defaultPostAction;
const bool                       Preferences::default_defaultCancelOnPreActionError = false;

Preferences::MailFrom Preferences::default_emailFrom()
{
	return KAMail::identitiesExist() ? MAIL_FROM_KMAIL : MAIL_FROM_CONTROL_CENTRE;
}

// Active config file settings
KTimeZone                  Preferences::mSystemTimeZone;
KTimeZone                  Preferences::mTimeZone;
KHolidays*                 Preferences::mHolidays = 0;
QColor                     Preferences::mDefaultFgColour;
QColor                     Preferences::mDefaultBgColour;
QFont                      Preferences::mMessageFont;
QTime                      Preferences::mStartOfDay;
QBitArray                  Preferences::mWorkDays(7);
QTime                      Preferences::mWorkDayStart;
QTime                      Preferences::mWorkDayEnd;
bool                       Preferences::mShowInSystemTray;
bool                       Preferences::mAutoStart;
Preferences::Feb29Type     Preferences::mDefaultFeb29Type;
bool                       Preferences::mAskResource;
bool                       Preferences::mModalMessages;
int                        Preferences::mMessageButtonDelay;
int                        Preferences::mTooltipAlarmCount;
bool                       Preferences::mShowTooltipAlarmTime;
bool                       Preferences::mShowTooltipTimeToAlarm;
QString                    Preferences::mTooltipTimeToPrefix;
QString                    Preferences::mEmailAddress;
QString                    Preferences::mEmailBccAddress;
Preferences::MailClient    Preferences::mEmailClient;
Preferences::MailFrom      Preferences::mEmailFrom;
Preferences::MailFrom      Preferences::mEmailBccFrom;
bool                       Preferences::mEmailCopyToKMail;
QString                    Preferences::mCmdXTermCommand;
QColor                     Preferences::mDisabledColour;
QColor                     Preferences::mArchivedColour;
int                        Preferences::mArchivedKeepDays;
// Default settings for Edit Alarm dialog
QString                    Preferences::mDefaultSoundFile;
int                        Preferences::mDefaultSoundVolume;
int                        Preferences::mDefaultLateCancel;
bool                       Preferences::mDefaultAutoClose;
bool                       Preferences::mDefaultCopyToKOrganizer;
SoundPicker::Type          Preferences::mDefaultSoundType;
bool                       Preferences::mDefaultSoundRepeat;
bool                       Preferences::mDefaultConfirmAck;
bool                       Preferences::mDefaultEmailBcc;
bool                       Preferences::mDefaultCmdScript;
Preferences::CmdLogType    Preferences::mDefaultCmdLogType;
QString                    Preferences::mDefaultCmdLogFile;
RecurrenceEdit::RepeatType Preferences::mDefaultRecurPeriod;
TimePeriod::Units          Preferences::mDefaultReminderUnits;
QString                    Preferences::mDefaultPreAction;
QString                    Preferences::mDefaultPostAction;
bool                       Preferences::mDefaultCancelOnPreActionError;
// Change tracking
QTime                      Preferences::mOldStartOfDay;
bool                       Preferences::mStartOfDayChanged;


static const QString defaultFeb29RecurType    = QString::fromLatin1("Mar1");
static const QString defaultEmailClient       = QString::fromLatin1("kmail");

// Config file entry names
static const QString GENERAL_SECTION          = QString::fromLatin1("General");
static const QString VERSION_NUM              = QString::fromLatin1("Version");
static const QString TIMEZONE                 = QString::fromLatin1("TimeZone");
static const QString HOLIDAYS                 = QString::fromLatin1("HolidayRegion");
static const QString MESSAGE_FG_COLOUR        = QString::fromLatin1("MessageTextColour");
static const QString MESSAGE_BG_COLOUR        = QString::fromLatin1("MessageBackgroundColour");
static const QString MESSAGE_FONT             = QString::fromLatin1("MessageFont");
static const QString SHOW_IN_SYSTEM_TRAY      = QString::fromLatin1("ShowInSystemTray");
static const QString AUTOSTART                = QString::fromLatin1("AutoStart");
static const QString FEB29_RECUR_TYPE         = QString::fromLatin1("Feb29Recur");
static const QString ASK_RESOURCE             = QString::fromLatin1("AskResource");
static const QString MODAL_MESSAGES           = QString::fromLatin1("ModalMessages");
static const QString MESSAGE_BUTTON_DELAY     = QString::fromLatin1("MessageButtonDelay");
static const QString TOOLTIP_ALARM_COUNT      = QString::fromLatin1("TooltipAlarmCount");
static const QString TOOLTIP_ALARM_TIME       = QString::fromLatin1("ShowTooltipAlarmTime");
static const QString TOOLTIP_TIME_TO_ALARM    = QString::fromLatin1("ShowTooltipTimeToAlarm");
static const QString TOOLTIP_TIME_TO_PREFIX   = QString::fromLatin1("TooltipTimeToPrefix");
static const QString EMAIL_CLIENT             = QString::fromLatin1("EmailClient");
static const QString EMAIL_COPY_TO_KMAIL      = QString::fromLatin1("EmailCopyToKMail");
static const QString EMAIL_FROM               = QString::fromLatin1("EmailFrom");
static const QString EMAIL_BCC_ADDRESS        = QString::fromLatin1("EmailBccAddress");
static const QString CMD_XTERM_COMMAND        = QString::fromLatin1("CmdXTerm");
static const QString START_OF_DAY             = QString::fromLatin1("StartOfDay");
static const QString START_OF_DAY_CHECK       = QString::fromLatin1("Sod");
static const QString WORK_DAYS                = QString::fromLatin1("WorkDays");
static const QString WORK_DAY_START           = QString::fromLatin1("WorkDayStart");
static const QString WORK_DAY_END             = QString::fromLatin1("WorkDayEnd");
static const QString DISABLED_COLOUR          = QString::fromLatin1("DisabledColour");
static const QString ARCHIVED_COLOUR          = QString::fromLatin1("ExpiredColour");
static const QString ARCHIVED_KEEP_DAYS       = QString::fromLatin1("ExpiredKeepDays");

static const QString DEFAULTS_SECTION         = QString::fromLatin1("Defaults");
static const QString DEF_LATE_CANCEL          = QString::fromLatin1("LateCancel");
static const QString DEF_AUTO_CLOSE           = QString::fromLatin1("AutoClose");
static const QString DEF_CONFIRM_ACK          = QString::fromLatin1("ConfirmAck");
static const QString DEF_COPY_TO_KORG         = QString::fromLatin1("CopyKOrg");
static const QString DEF_SOUND_TYPE           = QString::fromLatin1("SoundType");
static const QString DEF_SOUND_FILE           = QString::fromLatin1("SoundFile");
static const QString DEF_SOUND_VOLUME         = QString::fromLatin1("SoundVolume");
static const QString DEF_SOUND_REPEAT         = QString::fromLatin1("SoundRepeat");
static const QString DEF_CMD_SCRIPT           = QString::fromLatin1("CmdScript");
static const QString DEF_CMD_LOG_TYPE         = QString::fromLatin1("CmdLogType");
static const QString DEF_LOG_FILE             = QString::fromLatin1("LogFile");
static const QString DEF_EMAIL_BCC            = QString::fromLatin1("EmailBcc");
static const QString DEF_RECUR_PERIOD         = QString::fromLatin1("RecurPeriod");
static const QString DEF_REMIND_UNITS         = QString::fromLatin1("RemindUnits");
static const QString DEF_PRE_ACTION           = QString::fromLatin1("PreAction");
static const QString DEF_POST_ACTION          = QString::fromLatin1("PostAction");
static const QString DEF_PRE_ACTION_ERRCANCEL = QString::fromLatin1("PreActionErrCancel");

// Config file entry name for temporary use
static const QString TEMP                     = QString::fromLatin1("Temp");

// Values for EmailFrom entry
static const QString FROM_CONTROL_CENTRE      = QString::fromLatin1("@ControlCenter");
static const QString FROM_KMAIL               = QString::fromLatin1("@KMail");

// Config file entry names for notification messages
const QString Preferences::QUIT_WARN              = QString::fromLatin1("QuitWarn");
const QString Preferences::CONFIRM_ALARM_DELETION = QString::fromLatin1("ConfirmAlarmDeletion");
const QString Preferences::EMAIL_QUEUED_NOTIFY    = QString::fromLatin1("EmailQueuedNotify");

static const int SODxor = 0x82451630;
inline int Preferences::startOfDayCheck()
{
	// Combine with a 'random' constant to prevent 'clever' people fiddling the
	// value, and thereby screwing things up.
	return QTime().msecsTo(mStartOfDay) ^ SODxor;
}


void Preferences::initialise()
{
	if (!mInstance)
	{
		// Initialise static variables here to avoid static initialisation
		// sequencing errors.
		mDefault_holidays = new KHolidays(QString::null);
		mDefault_messageFont = QFont(KGlobalSettings::generalFont().family(), 16, QFont::Bold);
		for (int i = 0;  i < 7;  ++i)
			mDefault_workDays.setBit(i, defaultWorkDays & (1 << i));

		mInstance = new Preferences;

		read();

		// Set the default button for the Quit warning message box to Cancel
		MessageBox::setContinueDefault(QUIT_WARN, KMessageBox::Cancel);
		MessageBox::setDefaultShouldBeShownContinue(QUIT_WARN, default_quitWarn);
		MessageBox::setDefaultShouldBeShownContinue(EMAIL_QUEUED_NOTIFY, default_emailQueuedNotify);
		MessageBox::setDefaultShouldBeShownContinue(CONFIRM_ALARM_DELETION, default_confirmAlarmDeletion);
	}
}

void Preferences::setAutoStart(bool start)
{
	mAutoStart = start;
	KConfig* config = KGlobal::config();
	config->setGroup(GENERAL_SECTION);
	config->writeEntry(AUTOSTART, mAutoStart);
	// Don't bother emitting a changed signal here
}

void Preferences::connect(const char* signal, const QObject* receiver, const char* member)
{
	initialise();
	QObject::connect(mInstance, signal, receiver, member);
}

void Preferences::emitStartOfDayChanged()
{
	emit startOfDayChanged(mOldStartOfDay);
}

void Preferences::emitPreferencesChanged()
{
	emit preferencesChanged();
}

/******************************************************************************
* Read preference values from the config file.
*/
void Preferences::read()
{
	QString value;

	initialise();

	KConfig* config = KGlobal::config();
	config->setGroup(GENERAL_SECTION);
	QString timeZone = config->readEntry(TIMEZONE);
	mTimeZone = KTimeZone();
	if (!timeZone.isEmpty())
		mTimeZone = KSystemTimeZones::zone(timeZone);
	if (!mTimeZone.isValid())
		mTimeZone = KSystemTimeZones::local();
	delete mHolidays;
	mHolidays                 = new KHolidays(config->readEntry(HOLIDAYS));
	mDefaultFgColour          = config->readColorEntry(MESSAGE_FG_COLOUR, &default_defaultFgColour);
	mDefaultBgColour          = config->readColorEntry(MESSAGE_BG_COLOUR, &default_defaultBgColour);
	mMessageFont              = config->readFontEntry(MESSAGE_FONT, &mDefault_messageFont);
	mShowInSystemTray         = config->readBoolEntry(SHOW_IN_SYSTEM_TRAY, default_showInSystemTray);
	mAutoStart                = config->readBoolEntry(AUTOSTART, default_autoStart);
	mAskResource              = config->readBoolEntry(ASK_RESOURCE, default_askResource);
	mModalMessages            = config->readBoolEntry(MODAL_MESSAGES, default_modalMessages);
	mMessageButtonDelay       = config->readNumEntry(MESSAGE_BUTTON_DELAY, default_messageButtonDelay);
	if (mMessageButtonDelay > 10)
		mMessageButtonDelay = 10;    // prevent windows being unusable for a long time
	if (mMessageButtonDelay < -1)
		mMessageButtonDelay = -1;
	mTooltipAlarmCount        = static_cast<int>(config->readUnsignedNumEntry(TOOLTIP_ALARM_COUNT, default_tooltipAlarmCount));
	if (mTooltipAlarmCount < 1)
		mTooltipAlarmCount = 1;
	mShowTooltipAlarmTime     = config->readBoolEntry(TOOLTIP_ALARM_TIME, default_showTooltipAlarmTime);
	mShowTooltipTimeToAlarm   = config->readBoolEntry(TOOLTIP_TIME_TO_ALARM, default_showTooltipTimeToAlarm);
	mTooltipTimeToPrefix      = config->readEntry(TOOLTIP_TIME_TO_PREFIX, default_tooltipTimeToPrefix);
	value                     = config->readEntry(EMAIL_CLIENT);  // don't use readPathEntry() here (values are hard-coded)
	mEmailClient              = (value == QString::fromLatin1("sendmail")) ? SENDMAIL
	                          : (value == QString::fromLatin1("kmail"))    ? KMAIL : default_emailClient;
	mEmailCopyToKMail         = config->readBoolEntry(EMAIL_COPY_TO_KMAIL, default_emailCopyToKMail);
	QString from              = config->readEntry(EMAIL_FROM, emailFrom(default_emailFrom(), false, false));
	mEmailFrom                = emailFrom(from);
	QString bccFrom           = config->readEntry(EMAIL_BCC_ADDRESS, emailFrom(default_emailBccFrom, false, true));
	mEmailBccFrom             = emailFrom(bccFrom);
	if (mEmailFrom == MAIL_FROM_CONTROL_CENTRE  ||  mEmailBccFrom == MAIL_FROM_CONTROL_CENTRE)
		mEmailAddress = mEmailBccAddress = KAMail::controlCentreAddress();
	if (mEmailFrom == MAIL_FROM_ADDR)
		mEmailAddress     = from;
	if (mEmailBccFrom == MAIL_FROM_ADDR)
		mEmailBccAddress  = bccFrom;
	mCmdXTermCommand          = translateXTermPath(config, config->readEntry(CMD_XTERM_COMMAND), false);
	QDateTime defTime(QDate(1900,1,1), default_startOfDay);
	mStartOfDay               = config->readDateTimeEntry(START_OF_DAY, &defTime).time();
	mOldStartOfDay.setHMS(0,0,0);
	int sod = config->readNumEntry(START_OF_DAY_CHECK, 0);
	if (sod)
		mOldStartOfDay    = mOldStartOfDay.addMSecs(sod ^ SODxor);
	unsigned days             = config->readUnsignedNumEntry(WORK_DAYS, defaultWorkDays);
	for (int i = 0;  i < 7;  ++i)
		mWorkDays.setBit(i, days & (1 << i));
	defTime.setTime(default_workDayStart);
	mWorkDayStart             = config->readDateTimeEntry(WORK_DAY_START, &defTime).time();
	defTime.setTime(default_workDayEnd);
	mWorkDayEnd               = config->readDateTimeEntry(WORK_DAY_END, &defTime).time();
	mDisabledColour           = config->readColorEntry(DISABLED_COLOUR, &default_disabledColour);
	mArchivedColour           = config->readColorEntry(ARCHIVED_COLOUR, &default_archivedColour);
	mArchivedKeepDays         = config->readNumEntry(ARCHIVED_KEEP_DAYS, default_archivedKeepDays);

	config->setGroup(DEFAULTS_SECTION);
	mDefaultLateCancel        = static_cast<int>(config->readUnsignedNumEntry(DEF_LATE_CANCEL, default_defaultLateCancel));
	mDefaultAutoClose         = config->readBoolEntry(DEF_AUTO_CLOSE, default_defaultAutoClose);
	mDefaultConfirmAck        = config->readBoolEntry(DEF_CONFIRM_ACK, default_defaultConfirmAck);
	mDefaultCopyToKOrganizer  = config->readBoolEntry(DEF_COPY_TO_KORG, default_defaultCopyToKOrganizer);
	value                     = config->readEntry(DEF_SOUND_TYPE);
	mDefaultSoundType         = (value == QString::fromLatin1("None"))  ? SoundPicker::NONE
	                          : (value == QString::fromLatin1("Beep"))  ? SoundPicker::BEEP
	                          : (value == QString::fromLatin1("File"))  ? SoundPicker::PLAY_FILE
	                          : (value == QString::fromLatin1("Speak")) ? SoundPicker::SPEAK : default_defaultSoundType;
	mDefaultSoundVolume       = config->readNumEntry(DEF_SOUND_VOLUME, default_defaultSoundVolume);
	if (mDefaultSoundVolume > 100)
		mDefaultSoundVolume = 100;
#ifdef WITHOUT_ARTS
	mDefaultSoundRepeat       = false;
#else
	mDefaultSoundRepeat       = config->readBoolEntry(DEF_SOUND_REPEAT, default_defaultSoundRepeat);
#endif
	mDefaultSoundFile         = config->readPathEntry(DEF_SOUND_FILE);
	mDefaultCmdScript         = config->readBoolEntry(DEF_CMD_SCRIPT, default_defaultCmdScript);
	value                     = config->readEntry(DEF_CMD_LOG_TYPE);
	mDefaultCmdLogType        = (value == QString::fromLatin1("Discard"))  ? Log_Discard
	                          : (value == QString::fromLatin1("File"))     ? Log_File
	                          : (value == QString::fromLatin1("Terminal")) ? Log_Terminal : default_defaultCmdLogType;
	mDefaultCmdLogFile        = config->readPathEntry(DEF_LOG_FILE);
	mDefaultEmailBcc          = config->readBoolEntry(DEF_EMAIL_BCC, default_defaultEmailBcc);
	int recurPeriod           = config->readNumEntry(DEF_RECUR_PERIOD, default_defaultRecurPeriod);
	mDefaultRecurPeriod       = (recurPeriod < RecurrenceEdit::SUBDAILY || recurPeriod > RecurrenceEdit::ANNUAL)
	                          ? default_defaultRecurPeriod : (RecurrenceEdit::RepeatType)recurPeriod;
	value                     = config->readEntry(DEF_RECUR_PERIOD);
	mDefaultRecurPeriod       = (value == QString::fromLatin1("None"))     ? RecurrenceEdit::NO_RECUR
	                          : (value == QString::fromLatin1("Login"))    ? RecurrenceEdit::AT_LOGIN
	                          : (value == QString::fromLatin1("SubDaily")) ? RecurrenceEdit::SUBDAILY
	                          : (value == QString::fromLatin1("Daily"))    ? RecurrenceEdit::DAILY
	                          : (value == QString::fromLatin1("Weekly"))   ? RecurrenceEdit::WEEKLY
	                          : (value == QString::fromLatin1("Monthly"))  ? RecurrenceEdit::MONTHLY
	                          : (value == QString::fromLatin1("Yearly"))   ? RecurrenceEdit::ANNUAL : default_defaultRecurPeriod;
	value                     = config->readEntry(FEB29_RECUR_TYPE);
	mDefaultFeb29Type         = (value == QString::fromLatin1("Mar1"))  ? Feb29_Mar1
	                          : (value == QString::fromLatin1("Feb28")) ? Feb29_Feb28
	                          : (value == QString::fromLatin1("None"))  ? Feb29_None : default_defaultFeb29Type;
	value                     = config->readEntry(DEF_REMIND_UNITS);
	mDefaultReminderUnits     = (value == QString::fromLatin1("Minutes"))      ? TimePeriod::MINUTES
	                          : (value == QString::fromLatin1("HoursMinutes")) ? TimePeriod::HOURS_MINUTES : default_defaultReminderUnits;
	mDefaultPreAction         = config->readEntry(DEF_PRE_ACTION, default_defaultPreAction);
	mDefaultPostAction        = config->readEntry(DEF_POST_ACTION, default_defaultPostAction);
	mDefaultCancelOnPreActionError = config->readBoolEntry(DEF_PRE_ACTION_ERRCANCEL, default_defaultCancelOnPreActionError);
	mInstance->emitPreferencesChanged();
	mStartOfDayChanged = (mStartOfDay != mOldStartOfDay);
	if (mStartOfDayChanged)
	{
		mInstance->emitStartOfDayChanged();
		mOldStartOfDay = mStartOfDay;
	}
}

/******************************************************************************
* Save preference values to the config file.
*/
void Preferences::save(bool syncToDisc)
{
	QString value;
	KConfig* config = KGlobal::config();
	config->setGroup(GENERAL_SECTION);
	config->writeEntry(VERSION_NUM, KALARM_VERSION);
	config->writeEntry(TIMEZONE, (mTimeZone.isValid() ? mTimeZone.name() : QString::null));
	config->writeEntry(HOLIDAYS, (mHolidays ? mHolidays->location() : QString::null));
	config->writeEntry(MESSAGE_FG_COLOUR, mDefaultFgColour);
	config->writeEntry(MESSAGE_BG_COLOUR, mDefaultBgColour);
	config->writeEntry(MESSAGE_FONT, mMessageFont);
	config->writeEntry(SHOW_IN_SYSTEM_TRAY, mShowInSystemTray);
	config->writeEntry(AUTOSTART, mAutoStart);
	config->writeEntry(ASK_RESOURCE, mAskResource);
	config->writeEntry(MODAL_MESSAGES, mModalMessages);
	config->writeEntry(MESSAGE_BUTTON_DELAY, mMessageButtonDelay);
	config->writeEntry(TOOLTIP_ALARM_COUNT, mTooltipAlarmCount);
	config->writeEntry(TOOLTIP_ALARM_TIME, mShowTooltipAlarmTime);
	config->writeEntry(TOOLTIP_TIME_TO_ALARM, mShowTooltipTimeToAlarm);
	config->writeEntry(TOOLTIP_TIME_TO_PREFIX, mTooltipTimeToPrefix);
	config->writeEntry(EMAIL_CLIENT, (mEmailClient == SENDMAIL ? "sendmail" : "kmail"));
	config->writeEntry(EMAIL_COPY_TO_KMAIL, mEmailCopyToKMail);
	config->writeEntry(EMAIL_FROM, emailFrom(mEmailFrom, true, false));
	config->writeEntry(EMAIL_BCC_ADDRESS, emailFrom(mEmailBccFrom, true, true));
	config->writeEntry(CMD_XTERM_COMMAND, translateXTermPath(config, mCmdXTermCommand, true));
	config->writeEntry(START_OF_DAY, QDateTime(QDate(1900,1,1), mStartOfDay));
	// Start-of-day check value is only written once the start-of-day time has been processed.
	unsigned days = 0;
	unsigned mask = 1;
	for (int i = 0;  i < 7;  mask <<= 1, ++i)
		if (mWorkDays.testBit(i))
			days |= mask;
	config->writeEntry(WORK_DAYS, days);
	config->writeEntry(WORK_DAY_START, QDateTime(QDate(1900,1,1), mWorkDayStart));
	config->writeEntry(WORK_DAY_END, QDateTime(QDate(1900,1,1), mWorkDayEnd));
	config->writeEntry(DISABLED_COLOUR, mDisabledColour);
	config->writeEntry(ARCHIVED_COLOUR, mArchivedColour);
	config->writeEntry(ARCHIVED_KEEP_DAYS, mArchivedKeepDays);

	config->setGroup(DEFAULTS_SECTION);
	config->writeEntry(DEF_LATE_CANCEL, mDefaultLateCancel);
	config->writeEntry(DEF_AUTO_CLOSE, mDefaultAutoClose);
	config->writeEntry(DEF_CONFIRM_ACK, mDefaultConfirmAck);
	config->writeEntry(DEF_COPY_TO_KORG, mDefaultCopyToKOrganizer);
	switch (mDefaultSoundType)
	{
		case SoundPicker::NONE:       value = QString::fromLatin1("None");  break;
		case SoundPicker::BEEP:       value = QString::fromLatin1("Beep");  break;
		case SoundPicker::PLAY_FILE:  value = QString::fromLatin1("File");  break;
		case SoundPicker::SPEAK:      value = QString::fromLatin1("Speak"); break;
		default:                      value = QString::null; break;
	}
	config->writeEntry(DEF_SOUND_TYPE, value);
	config->writePathEntry(DEF_SOUND_FILE, mDefaultSoundFile);
	config->writeEntry(DEF_SOUND_VOLUME, mDefaultSoundVolume);
	config->writeEntry(DEF_SOUND_REPEAT, mDefaultSoundRepeat);
	config->writeEntry(DEF_CMD_SCRIPT, mDefaultCmdScript);
	switch (mDefaultCmdLogType)
	{
		case Log_Discard:   value = QString::fromLatin1("Discard");  break;
		case Log_File:      value = QString::fromLatin1("File");     break;
		case Log_Terminal:  value = QString::fromLatin1("Terminal"); break;
		default:            value = QString::null; break;
	}
	config->writeEntry(DEF_CMD_LOG_TYPE, value);
	config->writePathEntry(DEF_LOG_FILE, mDefaultCmdLogFile);
	config->writeEntry(DEF_EMAIL_BCC, mDefaultEmailBcc);
	switch (mDefaultRecurPeriod)
	{
		case RecurrenceEdit::NO_RECUR:  value = QString::fromLatin1("None");     break;
		case RecurrenceEdit::AT_LOGIN:  value = QString::fromLatin1("Login");    break;
		case RecurrenceEdit::SUBDAILY:  value = QString::fromLatin1("SubDaily"); break;
		case RecurrenceEdit::DAILY:     value = QString::fromLatin1("Daily");    break;
		case RecurrenceEdit::WEEKLY:    value = QString::fromLatin1("Weekly");   break;
		case RecurrenceEdit::MONTHLY:   value = QString::fromLatin1("Monthly");  break;
		case RecurrenceEdit::ANNUAL:    value = QString::fromLatin1("Yearly");   break;
		default:                        value = QString::null; break;
	}
	config->writeEntry(DEF_RECUR_PERIOD, value);
	switch (mDefaultFeb29Type)
	{
		case Feb29_Mar1:  value = QString::fromLatin1("Mar1");  break;
		case Feb29_Feb28: value = QString::fromLatin1("Feb28"); break;
		case Feb29_None:  value = QString::fromLatin1("None");  break;
		default:          value = QString::null; break;
	}
	config->writeEntry(FEB29_RECUR_TYPE, value);
	switch (mDefaultReminderUnits)
	{
		case TimePeriod::MINUTES:       value = QString::fromLatin1("Minutes");      break;
		case TimePeriod::HOURS_MINUTES: value = QString::fromLatin1("HoursMinutes"); break;
		default:                        value = QString::null; break;
	}
	config->writeEntry(DEF_REMIND_UNITS, value);
	config->writeEntry(DEF_PRE_ACTION, mDefaultPreAction);
	config->writeEntry(DEF_PRE_ACTION_ERRCANCEL, mDefaultCancelOnPreActionError);
	config->writeEntry(DEF_POST_ACTION, mDefaultPostAction);

	if (syncToDisc)
		config->sync();
	mInstance->emitPreferencesChanged();
	if (mStartOfDay != mOldStartOfDay)
	{
		mInstance->emitStartOfDayChanged();
		mOldStartOfDay = mStartOfDay;
	}
}

void Preferences::syncToDisc()
{
	KGlobal::config()->sync();
}

void Preferences::updateStartOfDayCheck()
{
	KConfig* config = KGlobal::config();
	config->setGroup(GENERAL_SECTION);
	config->writeEntry(START_OF_DAY_CHECK, startOfDayCheck());
	config->sync();
	mStartOfDayChanged = false;
}

/******************************************************************************
* Get the user's time zone, or if none has been chosen, the system time zone.
* The system time zone is cached, and the cached value will be returned unless
* 'reload' is true, in which case the value is re-read from the system.
*/
KTimeZone Preferences::timeZone(bool reload)
{
	if (reload)
		mSystemTimeZone = KTimeZone();
	if (mTimeZone.isValid())
		return mTimeZone;
	return default_timeZone();
}

KTimeZone Preferences::default_timeZone()
{
	if (!mSystemTimeZone.isValid())
		mSystemTimeZone = KSystemTimeZones::local();
	return mSystemTimeZone;
}

void Preferences::setHolidayRegion(const QString& regionCode)
{
	delete mHolidays;
	mHolidays = new KHolidays(regionCode);
}

QString Preferences::emailFrom(Preferences::MailFrom from, bool useAddress, bool bcc)
{
	switch (from)
	{
		case MAIL_FROM_KMAIL:
			return FROM_KMAIL;
		case MAIL_FROM_CONTROL_CENTRE:
			return FROM_CONTROL_CENTRE;
		case MAIL_FROM_ADDR:
			return useAddress ? (bcc ? mEmailBccAddress : mEmailAddress) : QString::null;
		default:
			return QString::null;
	}
}

Preferences::MailFrom Preferences::emailFrom(const QString& str)
{
	if (str == FROM_KMAIL)
		return MAIL_FROM_KMAIL;
	if (str == FROM_CONTROL_CENTRE)
		return MAIL_FROM_CONTROL_CENTRE;
	return MAIL_FROM_ADDR;
}

/******************************************************************************
* Get user's default 'From' email address.
*/
QString Preferences::emailAddress()
{
	switch (mEmailFrom)
	{
		case MAIL_FROM_KMAIL:
			return KAMail::identityManager()->defaultIdentity().fullEmailAddr();
		case MAIL_FROM_CONTROL_CENTRE:
			return KAMail::controlCentreAddress();
		case MAIL_FROM_ADDR:
			return mEmailAddress;
		default:
			return QString::null;
	}
}

QString Preferences::emailBccAddress()
{
	switch (mEmailBccFrom)
	{
		case MAIL_FROM_CONTROL_CENTRE:
			return KAMail::controlCentreAddress();
		case MAIL_FROM_ADDR:
			return mEmailBccAddress;
		default:
			return QString::null;
	}
}

void Preferences::setEmailAddress(Preferences::MailFrom from, const QString& address)
{
	switch (from)
	{
		case MAIL_FROM_KMAIL:
			break;
		case MAIL_FROM_CONTROL_CENTRE:
			mEmailAddress = KAMail::controlCentreAddress();
			break;
		case MAIL_FROM_ADDR:
			mEmailAddress = address;
			break;
		default:
			return;
	}
	mEmailFrom = from;
}

void Preferences::setEmailBccAddress(bool useControlCentre, const QString& address)
{
	if (useControlCentre)
		mEmailBccAddress = KAMail::controlCentreAddress();
	else
		mEmailBccAddress = address;
	mEmailBccFrom = useControlCentre ? MAIL_FROM_CONTROL_CENTRE : MAIL_FROM_ADDR;
}

/******************************************************************************
* Called to allow or suppress output of the specified message dialog, where the
* dialog has a checkbox to turn notification off.
*/
void Preferences::setNotify(const QString& messageID, bool notify)
{
	MessageBox::saveDontShowAgainContinue(messageID, !notify);
}

/******************************************************************************
* Return whether the specified message dialog is output, where the dialog has
* a checkbox to turn notification off.
* Reply = false if message has been suppressed (by preferences or by selecting
*               "don't ask again")
*       = true in all other cases.
*/
bool Preferences::notifying(const QString& messageID)
{
	return MessageBox::shouldBeShownContinue(messageID);
}

/******************************************************************************
* Translate an X terminal command path to/from config file format.
* Note that only a home directory specification at the start of the path is
* translated, so there's no need to worry about missing out some of the
* executable's path due to quotes etc.
* N.B. Calling KConfig::read/writePathEntry() on the entire command line
*      causes a crash on some systems, so it's necessary to extract the
*      executable path first before processing.
*/
QString translateXTermPath(KConfig* config, const QString& cmdline, bool write)
{
	QString params;
	QString cmd = cmdline;
	if (cmdline.isEmpty())
		return cmdline;
	// Strip any leading quote
	QChar quote = cmdline[0];
	char q = static_cast<char>(quote);
	bool quoted = (q == '"' || q == '\'');
	if (quoted)
		cmd = cmdline.mid(1);
	// Split the command at the first non-escaped space
	for (int i = 0, count = cmd.length();  i < count;  ++i)
	{
		switch (cmd[i].latin1())
		{
			case '\\':
				++i;
				continue;
			case '"':
			case '\'':
				if (cmd[i] != quote)
					continue;
				// fall through to ' '
			case ' ':
				params = cmd.mid(i);
				cmd = cmd.left(i);
				break;
			default:
				continue;
		}
		break;
	}
	// Translate any home directory specification at the start of the
	// executable's path.
	if (write)
	{
		config->writePathEntry(TEMP, cmd);
		cmd = config->readEntry(TEMP);
	}
	else
	{
		config->writeEntry(TEMP, cmd);
		cmd = config->readPathEntry(TEMP);
	}
	config->deleteEntry(TEMP);
	if (quoted)
		return quote + cmd + params;
	else
		return cmd + params;
}
