/*
 *  resourceconfigdialog.cpp  -  KAlarm resource configuration dialog
 *  Program:  kalarm
 *  Copyright © 2006,2008 by David Jarvie <software@astrojar.org.uk>
 *  Based on configdialog.cpp in kdelibs/kresources,
 *  Copyright (c) 2002 Tobias Koenig <tokoe@kde.org>
 *  Copyright (c) 2002 Jan-Pascal van Best <janpascal@vanbest.org>
 *  Copyright (c) 2003 Cornelius Schumacher <schumacher@kde.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qcheckbox.h>

#include <klocale.h>
#include <klineedit.h>
#include <kmessagebox.h>

#include <kresources/factory.h>
#include "alarmresource.h"
#include "resourceconfigdialog.moc"

using namespace KRES;

ResourceConfigDialog::ResourceConfigDialog( QWidget *parent, AlarmResource* resource, const char *name )
  : KDialogBase( parent, name, true, i18n( "Resource Configuration" ),
    Ok|Cancel, Ok, false ), mResource( resource )
{
  Factory *factory = Factory::self(QString::fromLatin1("alarms"));

  QFrame *main = makeMainWidget();

  QVBoxLayout *mainLayout = new QVBoxLayout( main, 0, spacingHint() );

  QGroupBox *generalGroupBox = new QGroupBox( main );
  QGridLayout* grid = new QGridLayout(generalGroupBox, 2, 2, marginHint(), spacingHint());
  grid->addRowSpacing(0, fontMetrics().height() - marginHint() + spacingHint());
  generalGroupBox->setTitle( i18n( "General Settings" ) );

  grid->addWidget(new QLabel( i18n("Resource name", "Name:" ), generalGroupBox ), 1, 0);

  mName = new KLineEdit( generalGroupBox );
  grid->addWidget(mName, 1, 1);

  mReadOnly = new QCheckBox( i18n( "Read-only" ), generalGroupBox );
  grid->addMultiCellWidget(mReadOnly, 2, 2, 0, 1);

  mName->setText( mResource->resourceName() );
  mReadOnly->setChecked( mResource->readOnly() );

  mainLayout->addWidget( generalGroupBox );

  QGroupBox *resourceGroupBox = new QGroupBox( 2, Qt::Horizontal,  main );
  resourceGroupBox->layout()->setSpacing( spacingHint() );
  resourceGroupBox->setTitle( i18n("'%1' Resource Settings")
                              .arg( factory->typeName( resource->type() ) ) );
  mainLayout->addWidget( resourceGroupBox );

  mainLayout->addStretch();

  mConfigWidget = factory->configWidget( resource->type(), resourceGroupBox );
  if ( mConfigWidget ) {
    mConfigWidget->setInEditMode( false );
    mConfigWidget->loadSettings( mResource );
    mConfigWidget->show();
    connect( mConfigWidget, SIGNAL( setReadOnly( bool ) ),
        SLOT( setReadOnly( bool ) ) );
  }

  connect( mName, SIGNAL( textChanged(const QString &)),
      SLOT( slotNameChanged(const QString &)));

  slotNameChanged( mName->text() );
  setMinimumSize( sizeHint() );
}

void ResourceConfigDialog::setInEditMode( bool value )
{
  if ( mConfigWidget )
    mConfigWidget->setInEditMode( value );
}

void ResourceConfigDialog::slotNameChanged( const QString &text)
{
  enableButtonOK( !text.isEmpty() );
}

void ResourceConfigDialog::setReadOnly( bool value )
{
  mReadOnly->setChecked( value );
}

void ResourceConfigDialog::accept()
{
  if ( mName->text().isEmpty() ) {
    KMessageBox::sorry( this, i18n( "Please enter a resource name." ) );
    return;
  }

  mResource->startReconfig();
  mResource->setResourceName( mName->text() );
  mResource->setReadOnly(mReadOnly->isChecked());
  if ( mConfigWidget ) {
    // First save generic information
    // Also save setting of specific resource type
    mConfigWidget->saveSettings( mResource );
  }
  mResource->applyReconfig();

  KDialog::accept();
}
