/*
 *  resourceselector.cpp  -  calendar resource selection widget
 *  Program:  kalarm
 *  Copyright © 2006-2008 by David Jarvie <software@astrojar.org.uk>
 *  Based on KOrganizer's ResourceView class and KAddressBook's ResourceSelection class,
 *  Copyright (C) 2003,2004 Cornelius Schumacher <schumacher@kde.org>
 *  Copyright (C) 2003-2004 Reinhold Kainhofer <reinhold@kainhofer.com>
 *  Copyright (c) 2004 Tobias Koenig <tokoe@kde.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kalarm.h"

#include <qlayout.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qpushbutton.h>
#include <qtimer.h>
#include <qheader.h>
#include <qpainter.h>
#include <qfont.h>
#include <qwhatsthis.h>
#include <qtooltip.h>
#include <qstyle.h>

#include <kaction.h>
#include <kdialog.h>
#include <klistview.h>
#include <klocale.h>
#include <kglobal.h>
#include <kmessagebox.h>
#include <kinputdialog.h>
#include <kpopupmenu.h>
#include <kcolordialog.h>
#include <kdebug.h>

#include <libkcal/resourcecalendar.h>

#include "alarmcalendar.h"
#include "alarmlistview.h"
#include "alarmresources.h"
#include "packedlayout.h"
#include "preferences.h"
#include "resourceconfigdialog.h"
#include "templatelistview.h"
#include "resourceselector.moc"

using namespace KCal;


class ResourceItem : public QCheckListItem
{
    public:
	ResourceItem(AlarmResource*, ResourceSelector*, KListView* parent);
	AlarmResource*    resource()               { return mResource; }
	void              updateActive();

    protected:
	virtual void      stateChange(bool active);
	virtual void      paintCell(QPainter*, const QColorGroup&, int column, int width, int alignment);

    private:
	AlarmResource*    mResource;
	ResourceSelector* mSelector;
	QColor            mLastResourceColour;   // resource colour used to set existing pixmap
	bool              mBlockStateChange;     // prevent stateChange() from doing anything
};

class ResourceListTooltip : public QToolTip
{
	public:
		ResourceListTooltip(QWidget* parent) : QToolTip(parent) { }
		virtual ~ResourceListTooltip()  { remove(parentWidget()); }
	protected:
		virtual void maybeTip(const QPoint&);
};



ResourceSelector::ResourceSelector(AlarmResources* calendar, QWidget* parent, const char* name)
	: QFrame(parent, name),
	  mCalendar(calendar),
	  mContextMenu(0)
{
//	setFrameStyle(Box | Plain);
	QBoxLayout* topLayout = new QVBoxLayout(this, KDialog::spacingHint(), 0);   // use spacingHint for the margin

	QLabel* label = new QLabel(i18n("Resources"), this);
	topLayout->addWidget(label, 0, Qt::AlignHCenter);
	topLayout->addSpacing(KDialog::spacingHint() / 2);

	mAlarmType = new QComboBox(this, "alarmType");
	mAlarmType->insertItem(i18n("Active Alarms"));
	mAlarmType->insertItem(i18n("Archived Alarms"));
	mAlarmType->insertItem(i18n("Alarm Templates"));
	mAlarmType->setFixedHeight(mAlarmType->sizeHint().height());
	QWhatsThis::add(mAlarmType, i18n("Choose which type of data to show alarm resources for"));
	topLayout->addWidget(mAlarmType);
	// No spacing between combo box and listview.

	mListView = new KListView(this);
	mListView->addColumn(QString::null);
	mListView->header()->hide();
	mListView->setResizeMode(QListView::LastColumn);
	mListView->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding);
	mTooltip = new ResourceListTooltip(mListView->viewport());
	connect(mListView, SIGNAL(selectionChanged(QListViewItem*)), SLOT(selectionChanged(QListViewItem*)));
	connect(mListView, SIGNAL(clicked(QListViewItem*)), SLOT(clicked(QListViewItem*)));
	connect(mListView, SIGNAL(doubleClicked(QListViewItem*, const QPoint&, int)), SLOT(editResource()));
	connect(mListView, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
	        SLOT(contextMenuRequested(QListViewItem*, const QPoint&, int)));
	QWhatsThis::add(mListView,
	      i18n("List of available resources of the selected type. The checked state shows whether a resource "
	           "is enabled (checked) or disabled (unchecked). The default resource is shown in bold."));
	topLayout->addWidget(mListView, 1);
	topLayout->addSpacing(KDialog::spacingHint());

	PackedLayout* blayout = new PackedLayout(topLayout, Qt::AlignHCenter, KDialog::spacingHint());
	mAddButton    = new QPushButton(i18n("Add..."), this, "add");
	mEditButton   = new QPushButton(i18n("Edit..."), this, "edit");
	mDeleteButton = new QPushButton(i18n("Remove"), this, "del");
	blayout->add(mAddButton);
	blayout->add(mEditButton);
	blayout->add(mDeleteButton);
	QWhatsThis::add(mEditButton, i18n("Edit the highlighted resource"));
	QWhatsThis::add(mDeleteButton, i18n("Remove the highlighted resource from the list.\n\n"
	                                    "The resource itself is left intact, and may subsequently be reinstated in the list if desired."));
	mEditButton->setDisabled(true);
	mDeleteButton->setDisabled(true);
	connect(mAddButton, SIGNAL(clicked()), SLOT(addResource()));
	connect(mEditButton, SIGNAL(clicked()), SLOT(editResource()));
	connect(mDeleteButton, SIGNAL(clicked()), SLOT(removeResource()));

	connect(mCalendar, SIGNAL(signalResourceModified(AlarmResource*)), SLOT(updateItem(AlarmResource*)));
	connect(mCalendar, SIGNAL(standardResourceChange(AlarmResource::Type)), SLOT(slotStandardChanged(AlarmResource::Type)));
	connect(mCalendar, SIGNAL(resourceStatusChanged(AlarmResource*, AlarmResources::Change)), SLOT(slotStatusChanged(AlarmResource*, AlarmResources::Change)));
	connect(mCalendar, SIGNAL(resourceLoaded(AlarmResource*, bool)), SLOT(slotLoaded(AlarmResource*, bool)));

	connect(mAlarmType, SIGNAL(activated(int)), SLOT(refreshList()));
	QTimer::singleShot(0, this, SLOT(refreshList()));
}

ResourceSelector::~ResourceSelector()
{
	delete mTooltip;   // not deleted automatically when its parent widget is deleted
}

/******************************************************************************
* Called when an alarm type has been selected.
* Refresh the resource list with resources of the selected alarm type, and
* add appropriate whatsThis texts to the list and to the Add button.
*/
void ResourceSelector::refreshList()
{
	mListView->clear();

	QString addTip;
	switch (mAlarmType->currentItem())
	{
		case 0:
			mCurrentAlarmType = AlarmResource::ACTIVE;
			addTip = i18n("Add a new active alarm resource");
			break;
		case 1:
			mCurrentAlarmType = AlarmResource::ARCHIVED;
			addTip = i18n("Add a new archived alarm resource");
			break;
		case 2:
			mCurrentAlarmType = AlarmResource::TEMPLATE;
			addTip = i18n("Add a new alarm template resource");
			break;
	}
	AlarmResourceManager* manager = mCalendar->resourceManager();
	for (AlarmResourceManager::Iterator it = manager->begin();  it != manager->end();  ++it)
	{
		if ((*it)->alarmType() == mCurrentAlarmType)
			addItem(*it);
	}

	QWhatsThis::add(mAddButton, addTip);
	QToolTip::add(mAddButton, addTip);
	selectionChanged(0);
}

/******************************************************************************
* Prompt the user for a new resource to add to the list.
*/
void ResourceSelector::addResource()
{
	AlarmResourceManager* manager = mCalendar->resourceManager();
	QStringList descs = manager->resourceTypeDescriptions();
	bool ok = false;
	QString desc = KInputDialog::getItem(i18n("Resource Configuration"),
	                                     i18n("Select storage type of new resource:"), descs, 0, false, &ok, this);
	if (!ok  ||  descs.isEmpty())
		return;
	QString type = manager->resourceTypeNames()[descs.findIndex(desc)];
	AlarmResource* resource = dynamic_cast<AlarmResource*>(manager->createResource(type));
	if (!resource)
	{
		KMessageBox::error(this, "<qt>" + i18n("Unable to create resource of type '%1'.").arg("<b>" + type + "</b>") + "</qt>");
		return;
	}
	resource->setResourceName(i18n("%1 resource").arg(type));
	resource->setAlarmType(mCurrentAlarmType);
	resource->setActive(false);   // prevent setReadOnly() declaring it as unwritable before we've tried to load it

	ResourceConfigDialog dlg(this, resource, "resourceConfig");
	if (dlg.exec())
	{
		resource->setEnabled(true);
		resource->setTimeSpec(Preferences::timeZone());
		manager->add(resource);
		manager->writeConfig();
		mCalendar->resourceAdded(resource);   // load the resource and connect in-process change signals
	}
	else
	{
		delete resource;
		resource = 0;
	}
}

/******************************************************************************
* Edit the currently selected resource.
*/
void ResourceSelector::editResource()
{
	ResourceItem* item = currentItem();
	if (!item)
		return;
	AlarmResource* resource = item->resource();
	bool readOnly = resource->readOnly();
	ResourceConfigDialog dlg(this, resource, "resourceConfig");
	if (dlg.exec())
	{
		// Act on any changed settings.
		// Read-only is handled automatically by AlarmResource::setReadOnly().
		item->setText(0, resource->resourceName());

		if (!readOnly  &&  resource->readOnly()  &&  resource->standardResource())
		{
			// A standard resource is being made read-only.
			if (resource->alarmType() == AlarmResource::ACTIVE)
			{
				KMessageBox::sorry(this, i18n("You cannot make your default active alarm resource read-only."));
				resource->setReadOnly(false);
			}
			else if (resource->alarmType() == AlarmResource::ARCHIVED  &&  Preferences::archivedKeepDays())
			{
				// Only allow the archived alarms standard resource to be made read-only
				// if we're not saving archived alarms.
				KMessageBox::sorry(this, i18n("You cannot make your default archived alarm resource "
				                              "read-only while expired alarms are configured to be kept."));
				resource->setReadOnly(false);
			}
			else if (KMessageBox::warningContinueCancel(this, i18n("Do you really want to make your default resource read-only?"))
			           == KMessageBox::Cancel)
			{
				resource->setReadOnly(false);
			}
		}
	}
}

/******************************************************************************
* Remove the currently selected resource from the displayed list.
*/
void ResourceSelector::removeResource()
{
	ResourceItem* item = currentItem();
	if (!item)
		return;
	AlarmResource* resource = item->resource();
	bool std = resource->standardResource();
	if (std)
	{
		// It's the standard resource for its type.
		if (resource->alarmType() == AlarmResource::ACTIVE)
		{
			KMessageBox::sorry(this, i18n("You cannot remove your default active alarm resource."));
			return;
		}
		if (resource->alarmType() == AlarmResource::ARCHIVED  &&  Preferences::archivedKeepDays())
		{
			// Only allow the archived alarms standard resource to be removed if
			// we're not saving archived alarms.
			KMessageBox::sorry(this, i18n("You cannot remove your default archived alarm resource "
			                              "while expired alarms are configured to be kept."));
			return;
		}
	}
	QString text = std ? i18n("Do you really want to remove your default resource ('%1') from the list?")
	                   : i18n("Do you really want to remove the resource '%1' from the list?");
	text = "<qt>" + text.arg("<b>" + item->text(0) + "</b>") + "</qt>";
	if (KMessageBox::warningContinueCancel(this, text, "", KStdGuiItem::remove()) == KMessageBox::Cancel)
		return;

	// Remove resource from alarm and resource lists before deleting it, to avoid
	// crashes when display updates occur immediately after it is deleted.
	if (resource->alarmType() == AlarmResource::TEMPLATE)
		TemplateListView::deleteResource(resource);
	else
		AlarmListView::deleteResource(resource);
	mListView->takeItem(item);
	delete item;
	AlarmResourceManager* manager = mCalendar->resourceManager();
	manager->remove(resource);
	manager->writeConfig();
}

/******************************************************************************
* Add the specified resource to the displayed list.
*/
void ResourceSelector::addItem(AlarmResource* resource)
{
	new ResourceItem(resource, this, mListView);
}

/******************************************************************************
* Called when the resource has been updated here or elsewhere, to update the
* active status displayed for the resource item.
*/
void ResourceSelector::updateItem(AlarmResource* resource)
{
	ResourceItem* item = findItem(resource);
	if (item)
		item->updateActive();
}

/******************************************************************************
* Called when the list is clicked. If no item is clicked, deselect any
* currently selected item.
*/
void ResourceSelector::clicked(QListViewItem* item)
{
	if (!item)
		selectionChanged(0);
}

/******************************************************************************
* Called when the current selection changes, to enable/disable the
* Delete and Edit buttons accordingly.
*/
void ResourceSelector::selectionChanged(QListViewItem* item)
{
	if (item)
		mListView->setSelected(item, true);
	else
		mListView->setSelected(mListView->selectedItem(), false);
	bool state = item;
	mDeleteButton->setEnabled(state);
	mEditButton->setEnabled(state);
}

/******************************************************************************
* Initialise the button and context menu actions.
*/
void ResourceSelector::initActions(KActionCollection* actions)
{
	mActionReload      = new KAction(i18n("Reload resource", "Re&load"), "reload", 0, this, SLOT(reloadResource()), actions, "resReload");
	mActionSave        = new KAction(i18n("&Save"), "filesave", 0, this, SLOT(saveResource()), actions, "resSave");
	mActionShowDetails = new KAction(i18n("Show &Details"), "info", 0, this, SLOT(showInfo()), actions, "resDetails");
	mActionSetColour   = new KAction(i18n("Set &Color..."), "colorize", 0, this, SLOT(setColour()), actions, "resSetColour");
	mActionClearColour = new KAction(i18n("Clear C&olor"), 0, 0, this, SLOT(clearColour()), actions, "resClearColour");
	mActionEdit        = new KAction(i18n("&Edit..."), "edit", 0, this, SLOT(editResource()), actions, "resEdit");
	mActionRemove      = new KAction(i18n("&Remove"), "editdelete", 0, this, SLOT(removeResource()), actions, "resRemove");
	mActionSetDefault  = new KToggleAction(QString::null, 0, 0, this, SLOT(setStandard()), actions, "resDefault");
	new KAction(i18n("&Add..."), "filenew", 0, this, SLOT(addResource()), actions, "resAdd");
	new KAction(i18n("Im&port..."), 0, 0, this, SLOT(importCalendar()), actions, "resImport");
}

void ResourceSelector::setContextMenu(KPopupMenu* menu)
{
	mContextMenu = menu;
}

/******************************************************************************
* Display the context menu for the selected resource.
*/
void ResourceSelector::contextMenuRequested(QListViewItem* itm, const QPoint& pos, int)
{
	if (!mContextMenu)
		return;
	if (!itm)
		selectionChanged(0);
	bool active   = false;
	bool writable = false;
	int type = -1;
	AlarmResource* resource = 0;
	ResourceItem* item = static_cast<ResourceItem*>(itm);
	if (item)
	{
		resource = item->resource();
		active   = resource->isEnabled();
		type     = resource->alarmType();
		writable = resource->writable();
	}
	else
	{
		switch (mAlarmType->currentItem())
		{
			case 0:  type = AlarmResource::ACTIVE; break;
			case 1:  type = AlarmResource::ARCHIVED; break;
			case 2:  type = AlarmResource::TEMPLATE; break;
		}
	}
	mActionReload->setEnabled(active);
	mActionSave->setEnabled(active && writable);
	mActionShowDetails->setEnabled(item);
	mActionSetColour->setEnabled(resource);
	mActionClearColour->setEnabled(resource && resource->colour().isValid());
	mActionEdit->setEnabled(item);
	mActionRemove->setEnabled(item);
	QString text;
	switch (type)
	{
		case AlarmResource::ACTIVE:   text = i18n("Use as &Default for Active Alarms");  break;
		case AlarmResource::ARCHIVED: text = i18n("Use as &Default for Archived Alarms");  break;
		case AlarmResource::TEMPLATE: text = i18n("Use as &Default for Alarm Templates");  break;
		default:  break;
	}
	mActionSetDefault->setText(text);
	bool standard = (resource  &&  resource == mCalendar->getStandardResource(static_cast<AlarmResource::Type>(type))  &&  resource->standardResource());
	mActionSetDefault->setChecked(active && writable && standard);
	bool allowChange = (type == AlarmResource::ARCHIVED  &&  !Preferences::archivedKeepDays());
	mActionSetDefault->setEnabled(active && writable && (!standard || allowChange));
	mContextMenu->popup(pos);
}

/******************************************************************************
* Called from the context menu to reload the selected resource.
*/
void ResourceSelector::reloadResource()
{
	ResourceItem* item = currentItem();
	if (item)
		AlarmCalendar::resources()->loadResource(item->resource(), this);
}

/******************************************************************************
* Called from the context menu to save the selected resource.
*/
void ResourceSelector::saveResource()
{
	ResourceItem* item = currentItem();
	if (item)
		item->resource()->save();
}

/******************************************************************************
* Called when the length of time archived alarms are to be stored changes.
* If expired alarms are now to be stored, set any single archived alarm
* resource to be the default.
*/
void ResourceSelector::archiveDaysChanged(int days)
{
	if (days)
	{
		AlarmResources* resources = AlarmResources::instance();
		AlarmResource* std = resources->getStandardResource(AlarmResource::ARCHIVED);
		if (std  &&  !std->standardResource())
			resources->setStandardResource(std);
	}
}

/******************************************************************************
* Called from the context menu to set the selected resource as the default
* for its alarm type. The resource is automatically made active.
*/
void ResourceSelector::setStandard()
{
	ResourceItem* item = currentItem();
	if (item)
	{
		AlarmResource* res = static_cast<ResourceItem*>(item)->resource();
		if (mActionSetDefault->isChecked())
		{
			res->setEnabled(true);
			mCalendar->setStandardResource(res);
		}
		else
			res->setStandardResource(false);
	}
}

/******************************************************************************
* Called when a different resource has been set as the standard resource.
*/
void ResourceSelector::slotStandardChanged(AlarmResource::Type type)
{
	for (QListViewItem* item = mListView->firstChild();  item;  item = item->nextSibling())
	{
		ResourceItem* ritem = static_cast<ResourceItem*>(item);
		AlarmResource* res = ritem->resource();
		if (res->alarmType() == type)
			ritem->repaint();
	}
}

/******************************************************************************
* Called when a resource has completed loading.
* Check in case its status has changed.
*/
void ResourceSelector::slotLoaded(AlarmResource* resource, bool active)
{
	if (active)
	{
		ResourceItem* item = findItem(resource);
		if (item)
			item->repaint();
	}
}

/******************************************************************************
* Called when a resource status has changed, to update the list.
*/
void ResourceSelector::slotStatusChanged(AlarmResource* resource, AlarmResources::Change change)
{
	kdDebug(5950) << "ResourceSelector::slotStatusChanged(" << (change == AlarmResources::Enabled ? "Enabled)" : change == AlarmResources::ReadOnly ? "ReadOnly)" : change == AlarmResources::Location ? "Location)" : change == AlarmResources::Invalidated ? "Invalidated)" : "Colour)") << endl;
	if (change == AlarmResources::Added)
		addItem(resource);
	else
	{
		ResourceItem* item = findItem(resource);
		if (!item)
			return;
		switch (change)
		{
			case AlarmResources::Enabled:
				item->updateActive();
				break;
			case AlarmResources::ReadOnly:
				item->repaint();      // show it in the appropriate font
				break;
			default:
				break;
		}
	}
}

/******************************************************************************
* Called from the context menu to merge alarms from an external calendar into
* the selected resource.
*/
void ResourceSelector::importCalendar()
{
	ResourceItem* item = currentItem();
	AlarmCalendar::importAlarms(this, (item ? item->resource() : 0));
}
 
/******************************************************************************
* Called from the context menu to set a colour for the selected resource.
*/
void ResourceSelector::setColour()
{
	ResourceItem* item = currentItem();
	if (item)
	{
		QColor colour = item->resource()->colour();
		if (!colour.isValid())
			colour = mListView->viewport()->paletteBackgroundColor();
		if (KColorDialog::getColor(colour, QColor(), this) == KColorDialog::Accepted)
			item->resource()->setColour(colour);
	}
}

/******************************************************************************
* Called from the context menu to clear the display colour for the selected
* resource.
*/
void ResourceSelector::clearColour()
{
	ResourceItem* item = currentItem();
	if (item)
		item->resource()->setColour(QColor());
}

/******************************************************************************
* Called from the context menu to display information for the selected resource.
*/
void ResourceSelector::showInfo()
{
	ResourceItem* item = currentItem();
	if (item)
	{
		QString txt = "<qt>" + item->resource()->infoText() + "</qt>";
		KMessageBox::information(this, txt);
	}
}

/******************************************************************************
* Find the list item for the specified calendar.
*/
ResourceItem* ResourceSelector::findItem(AlarmResource* rc)
{
	ResourceItem* ritem = 0;
	for (QListViewItem* item = mListView->firstChild();  item;  item = item->nextSibling())
	{
		ritem = static_cast<ResourceItem*>(item);
		if (ritem->resource() == rc)
			return ritem;
	}
	return 0;
}

/******************************************************************************
* Return the currently selected resource in the list.
*/
ResourceItem* ResourceSelector::currentItem()
{
	return static_cast<ResourceItem*>(mListView->currentItem());
}

void ResourceSelector::resizeEvent(QResizeEvent* re)
{
	emit resized(re->oldSize(), re->size());
}


/*=============================================================================
=  Class: ResourceItem
=============================================================================*/

ResourceItem::ResourceItem(AlarmResource* resource, ResourceSelector* selector, KListView* parent)
	: QCheckListItem(parent, resource->resourceName(), CheckBox),
	  mResource(resource),
	  mSelector(selector),
	  mBlockStateChange(false)
{
	int checkboxsize = parent->style().pixelMetric(QStyle::PM_CheckListButtonSize, parent);
	QPixmap px(checkboxsize, checkboxsize);
	px.fill(parent->viewport()->paletteBackgroundColor());
	setPixmap(0, px);   // reserve space for a resource colour indication icon
	updateActive();
}

/******************************************************************************
* Called when the checkbox changes state.
* Opens or closes the resource.
*/
void ResourceItem::stateChange(bool active)
{
	if (mBlockStateChange)
		return;

	bool refreshList = true;
	bool saveChange = false;
	AlarmResources* resources = AlarmResources::instance();
	if (active)
	{
		// Enable the resource
		mResource->setActive(true);     // enable it now so that load() will work
		saveChange = resources->load(mResource);
		mResource->setActive(false);    // reset so that setEnabled() will work
		refreshList = !saveChange;      // load() emits a signal which refreshes the list
	}
	else
	{
		// Disable the resource
		if (mResource->standardResource())
		{
			// It's the standard resource for its type.
			if (mResource->alarmType() == AlarmResource::ACTIVE)
			{
				KMessageBox::sorry(mSelector, i18n("You cannot disable your default active alarm resource."));
				updateActive();
				return;
			}
			if (mResource->alarmType() == AlarmResource::ARCHIVED  &&  Preferences::archivedKeepDays())
			{
				// Only allow the archived alarms standard resource to be disabled if
				// we're not saving archived alarms.
				KMessageBox::sorry(mSelector, i18n("You cannot disable your default archived alarm resource "
				                                   "while expired alarms are configured to be kept."));
				updateActive();
				return;
			}
			if (KMessageBox::warningContinueCancel(mSelector, i18n("Do you really want to disable your default resource?"))
			           == KMessageBox::Cancel)
			{
				updateActive();
				return;
			}
		}
		saveChange = mResource->saveAndClose();   // close resource after it is saved
	}
	if (saveChange)
		mResource->setEnabled(active);
	updateActive();
}

/******************************************************************************
* Set the checkbox to reflect the status of the resource.
*/
void ResourceItem::updateActive()
{
	mBlockStateChange = true;
	setOn(mResource->isEnabled());
	mBlockStateChange = false;
}

void ResourceItem::paintCell(QPainter* painter, const QColorGroup& cg, int column, int width, int alignment)
{
	const QPixmap* px = pixmap(column);
	if (px  &&  mResource->colour() != mLastResourceColour)
	{
		QPixmap p(px->width(), px->height());
		p.fill(mResource->colour().isValid() ? mResource->colour() : listView()->viewport()->paletteBackgroundColor());
		setPixmap(column, p);
		mLastResourceColour = mResource->colour();
	}
	if (mResource->isEnabled()  &&  AlarmResources::instance()->getStandardResource(AlarmResource::ACTIVE) == mResource)
	{
		QFont font = painter->font();    // show standard resources in bold
		font.setBold(true);
		painter->setFont(font);
	}
	bool readOnly = mResource->readOnly();
	QColor col;
	switch (mResource->alarmType())
	{
		case AlarmResource::ACTIVE:    col = readOnly ? Qt::darkGray : Qt::black;  break;
		case AlarmResource::ARCHIVED:  col = readOnly ? Qt::green : Qt::darkGreen;  break;
		case AlarmResource::TEMPLATE:  col = readOnly ? Qt::blue : Qt::darkBlue;  break;
	}
	QColorGroup cg2 = cg;
	cg2.setColor(QColorGroup::Text, col);
	QCheckListItem::paintCell(painter, cg2, column, width, alignment);
}


/*=============================================================================
=  Class: ResourceListTooltip
=  Displays the full resource name, location and status in a tooltip.
=============================================================================*/

/******************************************************************************
*  Display the full resource name (if not completely displayed), plus
*  location and status.
*/
void ResourceListTooltip::maybeTip(const QPoint& pt)
{
	KListView* listView = (KListView*)parentWidget()->parentWidget();
	ResourceItem* item = (ResourceItem*)listView->itemAt(pt);
	if (!item)
		return;
	QString tipText;
	AlarmResource* resource = item->resource();
	int px = item->pixmap(0) ? item->pixmap(0)->width() + listView->itemMargin() : 0;
	bool showName = item->width(listView->fontMetrics(), listView, 0) + px > listView->viewport()->width();
	QString name;
	if (showName)
		name = resource->resourceName();
	bool inactive = !resource->isActive();
	QString disabled = i18n("Disabled");
	QString readonly = i18n("Read-only");
	if (inactive  &&  resource->readOnly())
		tipText = i18n("%1\n"
		               "%2: '%3'\n"
		               "%4, %5")
		               .arg(name).arg(resource->displayType()).arg(resource->displayLocation()).arg(disabled).arg(readonly);
	else if (inactive  ||  resource->readOnly())
		tipText = i18n("%1\n"
		               "%2: '%3'\n"
		               "%4")
		               .arg(name).arg(resource->displayType()).arg(resource->displayLocation()).arg(inactive ? disabled : readonly);
	else
		tipText = i18n("%1\n"
		               "%2: '%3'")
		               .arg(name).arg(resource->displayType()).arg(resource->displayLocation());
	if (!showName  &&  tipText.startsWith("\n"))
		tipText.remove(0, 1);

	QRect rect = listView->itemRect(item);
	kdDebug(5950) << "ResourceListTooltip::maybeTip(): display\n";
	tip(rect, tipText);
}
