/*
 *  specialactions.cpp  -  widget to specify special alarm actions
 *  Program:  kalarm
 *  Copyright © 2004,2005,2007,2008 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kalarm.h"

#include <qlabel.h>
#include <qgroupbox.h>
#include <qhbox.h>
#include <qlayout.h>
#include <qwhatsthis.h>

#include <klineedit.h>
#include <kapplication.h>
#include <kaboutdata.h>
#include <klocale.h>
#include <kdebug.h>

#include "checkbox.h"
#include "functions.h"
#include "shellprocess.h"
#include "specialactions.moc"


/*=============================================================================
= Class SpecialActionsButton
= Button to display the Special Alarm Actions dialogue.
=============================================================================*/

SpecialActionsButton::SpecialActionsButton(bool enableCancelOnError, QWidget* parent, const char* name)
	: QPushButton(i18n("Special Actions..."), parent, name),
	  mEnableCancel(enableCancelOnError),
	  mReadOnly(false)
{
	setToggleButton(true);
	setOn(false);
	connect(this, SIGNAL(clicked()), SLOT(slotButtonPressed()));
	QWhatsThis::add(this,
	      i18n("Specify actions to execute before and after the alarm is displayed."));
}

/******************************************************************************
*  Set the pre- and post-alarm actions.
*  The button's pressed state is set to reflect whether any actions are set.
*/
void SpecialActionsButton::setActions(const QString& pre, const QString& post, bool cancelOnError)
{
	mPreAction     = pre;
	mPostAction    = post;
	mCancelOnError = cancelOnError;
	setOn(!mPreAction.isEmpty() || !mPostAction.isEmpty());
}

/******************************************************************************
*  Called when the OK button is clicked.
*  Display a font and colour selection dialog and get the selections.
*/
void SpecialActionsButton::slotButtonPressed()
{
	SpecialActionsDlg dlg(mPreAction, mPostAction, i18n("Special Alarm Actions"),
	                      mCancelOnError, mEnableCancel, this, "actionsDlg");
	dlg.setReadOnly(mReadOnly);
	if (dlg.exec() == QDialog::Accepted)
	{
		mPreAction     = dlg.preAction();
		mPostAction    = dlg.postAction();
		mCancelOnError = dlg.cancelOnError();
		emit selected();
	}
	setOn(!mPreAction.isEmpty() || !mPostAction.isEmpty());
}


/*=============================================================================
= Class SpecialActionsDlg
= Pre- and post-alarm actions dialogue.
=============================================================================*/

static const char SPEC_ACT_DIALOG_NAME[] = "SpecialActionsDialog";


SpecialActionsDlg::SpecialActionsDlg(const QString& preAction, const QString& postAction,
                                     const QString& caption, bool cancelOnError, bool enableCancelOnError,
                                     QWidget* parent, const char* name)
	: KDialogBase(parent, name, true, caption, Ok|Cancel, Ok, false)
{
	QWidget* page = new QWidget(this);
	setMainWidget(page);
	QVBoxLayout* layout = new QVBoxLayout(page, 0, spacingHint());

	mActions = new SpecialActions(enableCancelOnError, page);
	mActions->setActions(preAction, postAction, cancelOnError);
	layout->addWidget(mActions);
	layout->addSpacing(KDialog::spacingHint());

	QSize s;
	if (KAlarm::readConfigWindowSize(SPEC_ACT_DIALOG_NAME, s))
		resize(s);
}

/******************************************************************************
*  Called when the OK button is clicked.
*/
void SpecialActionsDlg::slotOk()
{
	if (mActions->isReadOnly())
		reject();
	accept();
}

/******************************************************************************
*  Called when the dialog's size has changed.
*  Records the new size in the config file.
*/
void SpecialActionsDlg::resizeEvent(QResizeEvent* re)
{
	if (isVisible())
		KAlarm::writeConfigWindowSize(SPEC_ACT_DIALOG_NAME, re->size());
	KDialog::resizeEvent(re);
}


/*=============================================================================
= Class SpecialActions
= Pre- and post-alarm actions widget.
=============================================================================*/

SpecialActions::SpecialActions(bool enableCancelOnError, QWidget* parent, const char* name)
	: QWidget(parent, name),
	  mEnableCancel(enableCancelOnError),
	  mReadOnly(false)
{
	QBoxLayout* topLayout = new QVBoxLayout(this, 0, KDialog::spacingHint());

	// Pre-alarm action
	QGroupBox* group = new QGroupBox(i18n("Pre-Alarm Action"), this);
	topLayout->addWidget(group);
	QVBoxLayout* vlayout = new QVBoxLayout(group, KDialog::marginHint(), KDialog::spacingHint());
	vlayout->addSpacing(fontMetrics().height() - KDialog::marginHint() + KDialog::spacingHint());

	QHBox* box = new QHBox(group);
	box->setSpacing(KDialog::spacingHint());
	vlayout->addWidget(box);
	QLabel* label = new QLabel(i18n("Command:"), box);
	mPreAction = new KLineEdit(box);
	label->setBuddy(mPreAction);
	connect(mPreAction, SIGNAL(textChanged(const QString&)), SLOT(slotPreActionChanged(const QString&)));
	QWhatsThis::add(box,
	      i18n("Enter a shell command to execute before the alarm is displayed.\n"
                   "Note that it is executed only when the alarm proper is displayed, not when a reminder or deferred alarm is displayed.\n"
	           "N.B. KAlarm will wait for the command to complete before displaying the alarm."));
	box->setStretchFactor(mPreAction, 1);

	// Cancel if error in pre-alarm action
	mCancelOnError = new CheckBox(i18n("Cancel alarm on error"), group);
	QWhatsThis::add(mCancelOnError,
	      i18n("Cancel the alarm if the pre-alarm command fails, i.e. do not display the alarm or execute any post-alarm action command."));
	vlayout->addWidget(mCancelOnError, 0, Qt::AlignAuto);


	topLayout->addSpacing(KDialog::spacingHint());

	// Post-alarm action
	group = new QGroupBox(i18n("Post-Alarm Action"), this);
	topLayout->addWidget(group);
	vlayout = new QVBoxLayout(group, KDialog::marginHint(), KDialog::spacingHint());
	vlayout->addSpacing(fontMetrics().height() - KDialog::marginHint() + KDialog::spacingHint());

	box = new QHBox(group);
	box->setSpacing(KDialog::spacingHint());
	vlayout->addWidget(box);
	label = new QLabel(i18n("Command:"), box);
	mPostAction = new KLineEdit(box);
	label->setBuddy(mPostAction);
	QWhatsThis::add(box,
	      i18n("Enter a shell command to execute after the alarm window is closed.\n"
	           "Note that it is not executed after closing a reminder window. If you defer "
	           "the alarm, it is not executed until the alarm is finally acknowledged or closed."));
	box->setStretchFactor(mPostAction, 1);

	mCancelOnError->setEnabled(enableCancelOnError);
}

void SpecialActions::setActions(const QString& pre, const QString& post, bool cancelOnError)
{
	mPreAction->setText(pre);
	mPostAction->setText(post);
	mCancelOnError->setChecked(cancelOnError);
}

QString SpecialActions::preAction() const
{
	return mPreAction->text();
}

QString SpecialActions::postAction() const
{
	return mPostAction->text();
}

bool SpecialActions::cancelOnError() const
{
	return mCancelOnError->isChecked();
}

void SpecialActions::setReadOnly(bool ro)
{
	mReadOnly = ro;
	mPreAction->setReadOnly(mReadOnly);
	mPostAction->setReadOnly(mReadOnly);
	mCancelOnError->setReadOnly(mReadOnly);
}

void SpecialActions::slotPreActionChanged(const QString& text)
{
	if (!mEnableCancel)
		mCancelOnError->setEnabled(!text.isEmpty());
}
