/*
 *  templatedlg.cpp  -  dialogue to create, edit and delete alarm templates
 *  Program:  kalarm
 *  Copyright © 2004-2008 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kalarm.h"

#include <qlayout.h>
#include <qpushbutton.h>
#include <qwhatsthis.h>

#include <klocale.h>
#include <kguiitem.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kaccel.h>
#include <kdebug.h>

#include "alarmresources.h"
#include "editdlg.h"
#include "alarmcalendar.h"
#include "alarmresources.h"
#include "functions.h"
#include "newalarmaction.h"
#include "templatelistview.h"
#include "undo.h"
#include "templatedlg.moc"

static const char TMPL_DIALOG_NAME[] = "TemplateDialog";


TemplateDlg* TemplateDlg::mInstance = 0;


TemplateDlg::TemplateDlg(QWidget* parent, const char* name)
	: KDialogBase(KDialogBase::Plain, i18n("Alarm Templates"), Close, Ok, parent, name, false, true)
{
	QWidget* topWidget = plainPage();
	QBoxLayout* topLayout = new QHBoxLayout(topWidget);
	topLayout->setSpacing(spacingHint());

	QBoxLayout* layout = new QVBoxLayout(topLayout);
	mListView = new TemplateListView(true, i18n("The list of alarm templates"), topWidget);
	mListView->setSelectionMode(QListView::Extended);
	mListView->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));
	connect(mListView, SIGNAL(selectionChanged()), SLOT(slotSelectionChanged()));
	connect(mListView, SIGNAL(doubleClicked(QListViewItem*,const QPoint&,int)), SLOT(slotEdit()));
	connect(AlarmResources::instance(), SIGNAL(resourceStatusChanged(AlarmResource*, AlarmResources::Change)),
	                                    SLOT(slotResourceStatusChanged(AlarmResource*, AlarmResources::Change)));
	layout->addWidget(mListView);

	layout = new QVBoxLayout(topLayout);
	QPushButton* button = new QPushButton(i18n("&New"), topWidget);
	mNewAction = new NewAlarmAction(true, i18n("&New"), this);
	button->setPopup(mNewAction->popupMenu());
	connect(mNewAction, SIGNAL(selected(EditAlarmDlg::Type)), SLOT(slotNew(EditAlarmDlg::Type)));
	QWhatsThis::add(button, i18n("Create a new alarm template"));
	layout->addWidget(button);

	mEditButton = new QPushButton(i18n("&Edit..."), topWidget);
	connect(mEditButton, SIGNAL(clicked()), SLOT(slotEdit()));
	QWhatsThis::add(mEditButton, i18n("Edit the currently highlighted alarm template"));
	layout->addWidget(mEditButton);

	mCopyButton = new QPushButton(i18n("Co&py"), topWidget);
	connect(mCopyButton, SIGNAL(clicked()), SLOT(slotCopy()));
	QWhatsThis::add(mCopyButton,
	      i18n("Create a new alarm template based on a copy of the currently highlighted template"));
	layout->addWidget(mCopyButton);

	mDeleteButton = new QPushButton(i18n("&Delete"), topWidget);
	connect(mDeleteButton, SIGNAL(clicked()), SLOT(slotDelete()));
	QWhatsThis::add(mDeleteButton, i18n("Delete the currently highlighted alarm template"));
	layout->addWidget(mDeleteButton);

	KAccel* accel = new KAccel(this);
	accel->insert(KStdAccel::SelectAll, mListView, SLOT(slotSelectAll()));
	accel->insert(KStdAccel::Deselect, mListView, SLOT(slotDeselect()));
	accel->readSettings();

	mListView->refresh();
	slotSelectionChanged();          // enable/disable buttons as appropriate

	QSize s;
	if (KAlarm::readConfigWindowSize(TMPL_DIALOG_NAME, s))
		resize(s);
}

/******************************************************************************
*  Destructor.
*/
TemplateDlg::~TemplateDlg()
{
	mInstance = 0;
}

/******************************************************************************
*  Create an instance, if none already exists.
*/
TemplateDlg* TemplateDlg::create(QWidget* parent, const char* name)
{
	if (mInstance)
		return 0;
	mInstance = new TemplateDlg(parent, name);
	return mInstance;
}

/******************************************************************************
*  Called when the New Template button is clicked to create a new template.
*/
void TemplateDlg::slotNew(EditAlarmDlg::Type type)
{
	KAlarm::editNewTemplate(type, this, mListView);
}

/******************************************************************************
*  Called when the Copy button is clicked to edit a copy of an existing alarm,
*  to add to the list.
*/
void TemplateDlg::slotCopy()
{
	TemplateListViewItem* item = mListView->selectedItem();
	if (item)
		KAlarm::editNewTemplate(item->event(), this, mListView);
}

/******************************************************************************
*  Called when the Modify button is clicked to edit the currently highlighted
*  alarm in the list.
*/
void TemplateDlg::slotEdit()
{
	TemplateListViewItem* item = mListView->selectedItem();
	if (item)
		KAlarm::editTemplate(item->event(), this, mListView);
}

/******************************************************************************
*  Called when the Delete button is clicked to delete the currently highlighted
*  alarms in the list.
*/
void TemplateDlg::slotDelete()
{
	KAEvent::List events = mListView->selectedEvents();
	int n = events.count();
	if (KMessageBox::warningContinueCancel(this, i18n("Do you really want to delete the selected alarm template?",
	                                                  "Do you really want to delete the %n selected alarm templates?", n),
	                                       i18n("Delete Alarm Template", "Delete Alarm Templates", n), KGuiItem(i18n("&Delete"), "editdelete"))
		    != KMessageBox::Continue)
		return;

	QStringList eventIDs;
	Undo::EventList undos;
	AlarmCalendar* resources = AlarmCalendar::resources();
	for (KAEvent::List::Iterator it = events.begin();  it != events.end();  ++it)
	{
		const KAEvent* event = *it;
		eventIDs.append(event->id());
		undos.append(*event, resources->resourceForEvent(event->id()));
	}
	KAlarm::deleteTemplates(eventIDs, this);
	Undo::saveDeletes(undos);
}

/******************************************************************************
* Called when the status of a resource has changed.
* Enable or disable actions appropriately.
*/
void TemplateDlg::slotResourceStatusChanged(AlarmResource* resource, AlarmResources::Change change)
{
	if (change == AlarmResources::Colour)
		mListView->repaint(resource);
}

/******************************************************************************
* Called when the group of items selected changes.
* Enable/disable the buttons depending on whether/how many templates are
* currently highlighted.
*/
void TemplateDlg::slotSelectionChanged()
{
	AlarmCalendar* resources = AlarmCalendar::resources();
	KAEvent::List events = mListView->selectedEvents();
	int count = events.count();
	bool readOnly = false;
	for (KAEvent::List::Iterator it = events.begin();  it != events.end();  ++it)
	{
		const KAEvent* event = *it;
		if (resources->eventReadOnly(event->id()))
		{
			readOnly = true;
			break;
		}
	}
	mEditButton->setEnabled(count == 1);
	mCopyButton->setEnabled(count == 1);
	mDeleteButton->setEnabled(count && !readOnly);
}

/******************************************************************************
*  Called when the dialog's size has changed.
*  Records the new size in the config file.
*/
void TemplateDlg::resizeEvent(QResizeEvent* re)
{
	if (isVisible())
		KAlarm::writeConfigWindowSize(TMPL_DIALOG_NAME, re->size());
	KDialog::resizeEvent(re);
}
