/*  -*- mode: C++; c-file-style: "gnu" -*-

    This file is part of kdepim.
    Copyright (c) 2004 KDEPIM developers

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef EMAIL_H
#define EMAIL_H

#include <qstringlist.h>

#include <kdepimmacros.h>

/** @file */

/** 
    \brief KPIM holds all kinds of functions specific to KDE PIM.

    The KPIM namespace hides away functions, enums, and other things
    that are KDE PIM specific and that we don't want to have polluting
    the global namespace.
*/
namespace KPIM {

/**
    Result type for splitAddress, isValidEmailAddress.
*/
enum EmailParseResult { AddressOk, AddressEmpty, UnexpectedEnd,
                        UnbalancedParens, MissingDomainPart,
                        UnclosedAngleAddr, UnopenedAngleAddr,
                        TooManyAts, UnexpectedComma,
                        TooFewAts, MissingLocalPart,
                        UnbalancedQuote, NoAddressSpec,
                        DisallowedChar, InvalidDisplayName };

// Helper functions
/** Split a comma separated list of email addresses. */
KDE_EXPORT QStringList splitEmailAddrList(const QString& aStr);

/** Translate the enum errorcodes from emailParseResult
    into i18n'd strings that can be used for msg boxes. 
    
    @param errorCode    the errorCode from isValidEmailEmailAddress().

    @return             An i18n ready string for use in msgboxes. 
*/
KDE_EXPORT QString emailParseResultToString( EmailParseResult errorCode );

/** Returns the pure email address (addr-spec in RFC2822) of the given address
    (mailbox in RFC2822).

    @param address  an email address, e.g. "Joe User <joe.user@example.org>"
    @return         the addr-spec of @a address, i.e. joe.user@example.org in the
                      example
*/
KDE_EXPORT QCString getEmailAddress( const QCString & address );

/** This is an overloaded member function, provided for convenience. It behaves
    essentially like the above function.

    Returns the pure email address (addr-spec in RFC2822) of the given address
    (mailbox in RFC2822).

    @param address  an email address, e.g. "Joe User <joe.user@example.org>"
    @return         the addr-spec of @a address, i.e. joe.user@example.org in the
                      example
*/
KDE_EXPORT QString getEmailAddress( const QString & address );

/** Return email address and name from string. Examples:
 * "Stefan Taferner <taferner@example.org>" returns "taferner@example.org"
 * and "Stefan Taferner". "joe@example.com" returns "joe@example.com"
 * and "". Note that this only returns the first address.
 * Also note that the return value is TRUE if both the name and the
 * mail are not empty: this does NOT tell you if mail contains a
 * valid email address or just some rubbish.
 */
KDE_EXPORT bool getNameAndMail(const QString& aStr, QString& name, QString& mail);

/**
 * Compare two email addresses. If matchName is false, it just checks
 * the email address, and returns true if this matches. If matchName
 * is true, both the name and the email must be the same.
 */
KDE_EXPORT bool compareEmail( const QString& email1, const QString& email2,
                   bool matchName );

} // namespace

#endif /* EMAIL_H */

