/*
    This file is part of libkcal.

    Copyright (c) 1998 Preston Brown <pbrown@kde.org>
    Copyright (c) 2001 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include <kdebug.h>

#include "incidence.h"

#include "alarm.h"

using namespace KCal;

Alarm::Alarm(Incidence *parent)
 : mParent(parent),
   mType(Invalid),
   mDescription(""),    // to make operator==() not fail
   mFile(""),           // to make operator==() not fail
   mMailSubject(""),    // to make operator==() not fail
   mAlarmSnoozeTime(5),
   mAlarmRepeatCount(0),
   mEndOffset(false),
   mHasTime(false),
   mAlarmEnabled(false)
{
}

Alarm::~Alarm()
{
}

bool Alarm::operator==( const Alarm& rhs ) const
{
  if ( mType != rhs.mType ||
       mAlarmSnoozeTime != rhs.mAlarmSnoozeTime ||
       mAlarmRepeatCount != rhs.mAlarmRepeatCount ||
       mAlarmEnabled != rhs.mAlarmEnabled ||
       mHasTime != rhs.mHasTime)
    return false;

  if (mHasTime) {
    if (mAlarmTime != rhs.mAlarmTime)
      return false;
  } else {
    if (mOffset != rhs.mOffset ||
        mEndOffset != rhs.mEndOffset)
      return false;
  }

  switch (mType) {
    case Display:
      return mDescription == rhs.mDescription;

    case Email:
      return mDescription == rhs.mDescription &&
             mMailAttachFiles == rhs.mMailAttachFiles &&
             mMailAddresses == rhs.mMailAddresses &&
             mMailSubject == rhs.mMailSubject;

    case Procedure:
      return mFile == rhs.mFile &&
             mDescription == rhs.mDescription;

    case Audio:
      return mFile == rhs.mFile;

    case Invalid:
      break;
  }
  return false;
}

void Alarm::setType(Alarm::Type type)
{
  if (type == mType)
    return;

  switch (type) {
    case Display:
      mDescription = "";
      break;
    case Procedure:
      mFile = mDescription = "";
      break;
    case Audio:
      mFile = "";
      break;
    case Email:
      mMailSubject = mDescription = "";
      mMailAddresses.clear();
      mMailAttachFiles.clear();
      break;
    case Invalid:
      break;
    default:
      return;
  }
  mType = type;
  if ( mParent ) mParent->updated();
}

Alarm::Type Alarm::type() const
{
  return mType;
}

void Alarm::setAudioAlarm(const QString &audioFile)
{
  mType = Audio;
  mFile = audioFile;
  if ( mParent ) mParent->updated();
}

void Alarm::setAudioFile(const QString &audioFile)
{
  if (mType == Audio) {
    mFile = audioFile;
    if ( mParent ) mParent->updated();
  }
}

QString Alarm::audioFile() const
{
  return (mType == Audio) ? mFile : QString::null;
}

void Alarm::setProcedureAlarm(const QString &programFile, const QString &arguments)
{
  mType = Procedure;
  mFile = programFile;
  mDescription = arguments;
  if ( mParent ) mParent->updated();
}

void Alarm::setProgramFile(const QString &programFile)
{
  if (mType == Procedure) {
    mFile = programFile;
    if ( mParent ) mParent->updated();
  }
}

QString Alarm::programFile() const
{
  return (mType == Procedure) ? mFile : QString::null;
}

void Alarm::setProgramArguments(const QString &arguments)
{
  if (mType == Procedure) {
    mDescription = arguments;
    if ( mParent ) mParent->updated();
  }
}

QString Alarm::programArguments() const
{
  return (mType == Procedure) ? mDescription : QString::null;
}

void Alarm::setEmailAlarm(const QString &subject, const QString &text,
                          const QValueList<Person> &addressees, const QStringList &attachments)
{
  mType = Email;
  mMailSubject = subject;
  mDescription = text;
  mMailAddresses = addressees;
  mMailAttachFiles = attachments;
  if ( mParent ) mParent->updated();
}

void Alarm::setMailAddress(const Person &mailAddress)
{
  if (mType == Email) {
    mMailAddresses.clear();
    mMailAddresses += mailAddress;
    if ( mParent ) mParent->updated();
  }
}

void Alarm::setMailAddresses(const QValueList<Person> &mailAddresses)
{
  if (mType == Email) {
    mMailAddresses = mailAddresses;
    if ( mParent ) mParent->updated();
  }
}

void Alarm::addMailAddress(const Person &mailAddress)
{
  if (mType == Email) {
    mMailAddresses += mailAddress;
    if ( mParent ) mParent->updated();
  }
}

QValueList<Person> Alarm::mailAddresses() const
{
  return (mType == Email) ? mMailAddresses : QValueList<Person>();
}

void Alarm::setMailSubject(const QString &mailAlarmSubject)
{
  if (mType == Email) {
    mMailSubject = mailAlarmSubject;
    if ( mParent ) mParent->updated();
  }
}

QString Alarm::mailSubject() const
{
  return (mType == Email) ? mMailSubject : QString::null;
}

void Alarm::setMailAttachment(const QString &mailAttachFile)
{
  if (mType == Email) {
    mMailAttachFiles.clear();
    mMailAttachFiles += mailAttachFile;
    if ( mParent ) mParent->updated();
  }
}

void Alarm::setMailAttachments(const QStringList &mailAttachFiles)
{
  if (mType == Email) {
    mMailAttachFiles = mailAttachFiles;
    if ( mParent ) mParent->updated();
  }
}

void Alarm::addMailAttachment(const QString &mailAttachFile)
{
  if (mType == Email) {
    mMailAttachFiles += mailAttachFile;
    if ( mParent ) mParent->updated();
  }
}

QStringList Alarm::mailAttachments() const
{
  return (mType == Email) ? mMailAttachFiles : QStringList();
}

void Alarm::setMailText(const QString &text)
{
  if (mType == Email) {
    mDescription = text;
    if ( mParent ) mParent->updated();
  }
}

QString Alarm::mailText() const
{
  return (mType == Email) ? mDescription : QString::null;
}

void Alarm::setDisplayAlarm(const QString &text)
{
  mType = Display;
  if ( !text.isNull() )
    mDescription = text;
  if ( mParent ) mParent->updated();
}

void Alarm::setText(const QString &text)
{
  if (mType == Display) {
    mDescription = text;
    if ( mParent ) mParent->updated();
  }
}

QString Alarm::text() const
{
  return (mType == Display) ? mDescription : QString::null;
}

void Alarm::setTime(const KDateTime &alarmTime)
{
  mAlarmTime = alarmTime;
  mHasTime = true;

  if ( mParent ) mParent->updated();
}

KDateTime Alarm::time() const
{
  if ( hasTime() )
    return mAlarmTime;
  else if ( mParent ) 
  {
    if (mEndOffset) {
      return mOffset.end( mParent->dtEnd() );
    } else {
      return mOffset.end( mParent->dtStart() );
    }
  } else return KDateTime();
}

bool Alarm::hasTime() const
{
  return mHasTime;
}

void Alarm::shiftTimes(const KDateTime::Spec &oldSpec, const KDateTime::Spec &newSpec)
{
  mAlarmTime = mAlarmTime.toTimeSpec( oldSpec );
  mAlarmTime.setTimeSpec( newSpec );
  if ( mParent ) mParent->updated();
}

void Alarm::setSnoozeTime(const Duration &alarmSnoozeTime)
{
  if (alarmSnoozeTime.value() > 0) {
    mAlarmSnoozeTime = alarmSnoozeTime;
    if ( mParent ) mParent->updated();
  }
}

Duration Alarm::snoozeTime() const
{
  return mAlarmSnoozeTime;
}

void Alarm::setRepeatCount(int alarmRepeatCount)
{
  mAlarmRepeatCount = alarmRepeatCount;
  if ( mParent ) mParent->updated();
}

int Alarm::repeatCount() const
{
  return mAlarmRepeatCount;
}

Duration Alarm::duration() const
{
  return Duration(mAlarmSnoozeTime.value() * mAlarmRepeatCount, mAlarmSnoozeTime.type());
}

KDateTime Alarm::nextRepetition(const KDateTime& preTime) const
{
  KDateTime at = time();
  if (at > preTime)
    return at;
  if (!mAlarmRepeatCount)
    return KDateTime();   // there isn't an occurrence after the specified time
  int repetition;
  int interval = mAlarmSnoozeTime.value();
  bool daily = mAlarmSnoozeTime.isDaily();
  if ( daily ) {
    int daysTo = at.daysTo( preTime );
    if ( !preTime.isDateOnly() && preTime.time() <= at.time() ) {
      --daysTo;
    }
    repetition = daysTo / interval + 1;
  } else {
    repetition = at.secsTo_long( preTime ) / interval + 1;
  }
  if ( repetition > mAlarmRepeatCount ) {
    // all repetitions have finished before the specified time
    return KDateTime();
  }
  return daily ? at.addDays( repetition * interval )
               : at.addSecs( repetition * interval );
}

KDateTime Alarm::previousRepetition(const KDateTime& afterTime) const
{
  KDateTime at = time();
  if ( at >= afterTime ) {
    // alarm's first/only time is at/after the specified time
    return KDateTime();
  }
  if ( !mAlarmRepeatCount ) {
    return at;
  }
  int repetition;
  int interval = mAlarmSnoozeTime.value();
  bool daily = mAlarmSnoozeTime.isDaily();
  if ( daily ) {
    int daysTo = at.daysTo( afterTime );
    if ( afterTime.isDateOnly() || afterTime.time() <= at.time() ) {
      --daysTo;
    }
    repetition = daysTo / interval;
  } else {
    repetition = ( at.secsTo_long( afterTime ) - 1 ) / interval;
  }
  if ( repetition > mAlarmRepeatCount ) {
    repetition = mAlarmRepeatCount;
  }
  return daily ? at.addDays( repetition * interval )
               : at.addSecs( repetition * interval );
}

KDateTime Alarm::endTime() const
{
  if ( !mAlarmRepeatCount ) {
    return time();
  }
  if ( mAlarmSnoozeTime.isDaily() ) {
    return time().addDays( mAlarmRepeatCount * mAlarmSnoozeTime.asDays() );
  } else {
    return time().addSecs( mAlarmRepeatCount * mAlarmSnoozeTime.asSeconds() );
  }
}

void Alarm::toggleAlarm()
{
  mAlarmEnabled = !mAlarmEnabled;
  if ( mParent ) mParent->updated();
}

void Alarm::setEnabled(bool enable)
{
  mAlarmEnabled = enable;
  if ( mParent ) mParent->updated();
}

bool Alarm::enabled() const
{
  return mAlarmEnabled;
}

void Alarm::setStartOffset( const Duration &offset )
{
  mOffset = offset;
  mEndOffset = false;
  mHasTime = false;
  if ( mParent ) mParent->updated();
}

Duration Alarm::startOffset() const
{
  return (mHasTime || mEndOffset) ? Duration(0) : mOffset;
}

bool Alarm::hasStartOffset() const
{
  return !mHasTime && !mEndOffset;
}

bool Alarm::hasEndOffset() const
{
  return !mHasTime && mEndOffset;
}

void Alarm::setEndOffset( const Duration &offset )
{
  mOffset = offset;
  mEndOffset = true;
  mHasTime = false;
  if ( mParent ) mParent->updated();
}

Duration Alarm::endOffset() const
{
  return (mHasTime || !mEndOffset) ? Duration(0) : mOffset;
}

void Alarm::setParent( Incidence *parent )
{
  mParent = parent;
}

void Alarm::customPropertyUpdated()
{
  if ( mParent ) mParent->updated();
}
