/*
    This file is part of libkcal.

    Copyright (c) 1998 Preston Brown <pbrown@kde.org>
    Copyright (c) 2001,2003,2004 Cornelius Schumacher <schumacher@kde.org>
    Copyright (C) 2003-2004 Reinhold Kainhofer <reinhold@kainhofer.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include <qstring.h>
#include <qptrlist.h>

#include <kdebug.h>
#include <kdatetime.h>
#include <klocale.h>
#include <kmessagebox.h>

#include "vcaldrag.h"
#include "vcalformat.h"
#include "icalformat.h"
#include "exceptions.h"
#include "incidence.h"
#include "filestorage.h"

#include "calendarlocal.h"

using namespace KCal;

CalendarLocal::CalendarLocal( const KDateTime::Spec &timeSpec )
  : Calendar( timeSpec )
{
}

CalendarLocal::CalendarLocal( const QString &timeZoneId )
  : Calendar( timeZoneId ), mEvents( 47 )
{
  init();
}

void CalendarLocal::init()
{
  mDeletedIncidences.setAutoDelete( true );
  mFileName = QString::null;
}


CalendarLocal::~CalendarLocal()
{
  close();
}

bool CalendarLocal::load( const QString &fileName, CalFormat *format )
{
  mFileName = fileName;
  FileStorage storage( this, fileName, format );
  return storage.load();
}

bool CalendarLocal::reload()
{
  const QString filename = mFileName;
  save();
  close();
  mFileName = filename;
  FileStorage storage( this, mFileName );
  return storage.load();
}

bool CalendarLocal::save( const QString &fileName, CalFormat *format )
{
  // Save only if the calendar is either modified, or saved to a
  // different file than it was loaded from
  if ( mFileName != fileName || isModified() ) {
    FileStorage storage( this, fileName, format );
    return storage.save();
  } else {
    return true;
  }
}

void CalendarLocal::close()
{
  setObserversEnabled( false );
  mFileName = QString::null;

  deleteAllEvents();

  mDeletedIncidences.clearAll();
  setModified( false );

  setObserversEnabled( true );
}


bool CalendarLocal::addEvent( Event *event )
{
  insertEvent( event );

  event->registerObserver( this );

  setModified( true );

  notifyIncidenceAdded( event );

  return true;
}

bool CalendarLocal::deleteEvent( Event *event )
{
//  kdDebug(5800) << "CalendarLocal::deleteEvent" << endl;

  if ( mEvents.remove( event->uid() ) ) {
    setModified( true );
    notifyIncidenceDeleted( event );
    mDeletedIncidences.append( event );
    return true;
  } else {
    kdWarning() << "CalendarLocal::deleteEvent(): Event not found." << endl;
    return false;
  }
}

void CalendarLocal::deleteAllEvents()
{
  // kdDebug(5800) << "CalendarLocal::deleteAllEvents" << endl;
  QDictIterator<Event> it( mEvents );
  while( it.current() ) {
    notifyIncidenceDeleted( it.current() );
    ++it;
  }

  mEvents.setAutoDelete( true );
  mEvents.clear();
  mEvents.setAutoDelete( false );
}

Event *CalendarLocal::event( const QString &uid )
{
//  kdDebug(5800) << "CalendarLocal::event(): " << uid << endl;
  return mEvents[ uid ];
}

Alarm::List CalendarLocal::alarmsTo( const KDateTime &to )
{
  return alarms( KDateTime( QDate( 1900, 1, 1 ) ), to );
}

Alarm::List CalendarLocal::alarms( const KDateTime &from, const KDateTime &to )
{
//  kdDebug(5800) << "CalendarLocal::alarms(" << from.toString() << " - "
//                << to.toString() << ")" << endl;

  Alarm::List alarms;

  EventDictIterator it( mEvents );
  for( ; it.current(); ++it ) {
    Event *e = *it;
    if ( e->doesRecur() ) appendRecurringAlarms( alarms, e, from, to );
    else appendAlarms( alarms, e, from, to );
  }


  return alarms;
}

void CalendarLocal::appendAlarms( Alarm::List &alarms, Incidence *incidence,
                                  const KDateTime &from, const KDateTime &to )
{
  KDateTime preTime = from.addSecs(-1);
  Alarm::List::ConstIterator it;
  for( it = incidence->alarms().begin(); it != incidence->alarms().end();
       ++it ) {
    if ( (*it)->enabled() ) {
      KDateTime dt = (*it)->nextRepetition(preTime);
      if ( dt.isValid() && dt <= to ) {
        kdDebug(5800) << "CalendarLocal::appendAlarms() '"
                      << incidence->summary() << "': "
                      << dt.toString() << endl;
        alarms.append( *it );
      }
    }
  }
}

void CalendarLocal::appendRecurringAlarms( Alarm::List &alarms,
                                           Incidence *incidence,
                                           const KDateTime &from,
                                           const KDateTime &to )
{
  KDateTime dt;
  bool endOffsetValid = false;
  Duration endOffset( 0 );
  Duration period( from, to );

  Alarm::List::ConstIterator it;
  for( it = incidence->alarms().begin(); it != incidence->alarms().end();
       ++it ) {
    Alarm *alarm = *it;
    if ( alarm->enabled() ) {
      if ( alarm->hasTime() ) {
        // The alarm time is defined as an absolute date/time
        dt = alarm->nextRepetition( from.addSecs(-1) );
        if ( !dt.isValid() || dt > to )
          continue;
      } else {
        // The alarm time is defined by an offset from the event start or end time.
        // Find the offset from the event start time, which is also used as the
        // offset from the recurrence time.
        Duration offset( 0 );
        if ( alarm->hasStartOffset() ) {
          offset = alarm->startOffset();
        } else if ( alarm->hasEndOffset() ) {
          offset = alarm->endOffset();
          if ( !endOffsetValid ) {
            endOffset = Duration( incidence->dtStart(), incidence->dtEnd() );
            endOffsetValid = true;
          }
        }

        // Find the incidence's earliest alarm
        KDateTime alarmStart =
          offset.end( alarm->hasEndOffset() ? incidence->dtEnd() : incidence->dtStart() );
//        KDateTime alarmStart = incidence->dtStart().addSecs( offset );
        if ( alarmStart > to ) {
          continue;
        }
        KDateTime baseStart = incidence->dtStart();
        if ( from > alarmStart ) {
          alarmStart = from;   // don't look earlier than the earliest alarm
          baseStart = (-offset).end( (-endOffset).end( alarmStart ) );
        }

        // Adjust the 'alarmStart' date/time and find the next recurrence at or after it.
        // Treate the two offsets separately in case one is daily and the other not.
        dt = incidence->recurrence()->getNextDateTime( baseStart.addSecs(-1) );
        if ( !dt.isValid() ||
             ( dt = endOffset.end( offset.end( dt ) ) ) > to ) // adjust 'dt' to get the alarm time
        {
          // The next recurrence is too late.
          if ( !alarm->repeatCount() )
            continue;
          // The alarm has repetitions, so check whether repetitions of previous
          // recurrences fall within the time period.
          bool found = false;
          Duration alarmDuration = alarm->duration();
          for ( KDateTime base = baseStart;
                ( dt = incidence->recurrence()->getPreviousDateTime( base ) ).isValid();
                base = dt ) {
            if ( alarm->duration().end( dt ) < base ) {
              break;  // this recurrence's last repetition is too early, so give up
            }

            // The last repetition of this recurrence is at or after 'alarmStart' time.
            // Check if a repetition occurs between 'alarmStart' and 'to'.
            int snooze = alarm->snoozeTime().value();   // in seconds or days
            if ( alarm->snoozeTime().isDaily() ) {
              Duration toFromDuration( dt, base );
              int toFrom = toFromDuration.asDays();
              if ( alarm->snoozeTime().end( from ) <= to ||
                   toFromDuration.isDaily() && toFrom % snooze == 0 ||
                   ( toFrom / snooze + 1 ) * snooze <= toFrom + period.asDays() )
              {
                found = true;
#ifndef NDEBUG
                // for debug output
                dt = offset.end( dt ).addDays( ( ( toFrom - 1 ) / snooze + 1 ) * snooze );
#endif
                break;
              }
            } else {
              int toFrom = dt.secsTo( base );
              if ( period.asSeconds() >= snooze ||
                   toFrom % snooze == 0 ||
                   ( toFrom / snooze + 1 ) * snooze <= toFrom + period.asSeconds() )
              {
                found = true;
#ifndef NDEBUG
                // for debug output
                dt = offset.end( dt ).addSecs( ( ( toFrom - 1 ) / snooze + 1 ) * snooze );
#endif
                break;
              }
            }
          }
          if ( !found )
            continue;
        }
      }
      kdDebug(5800) << "CalendarLocal::appendAlarms() '" << incidence->summary()
                    << "': " << dt.toString() << endl;
      alarms.append( alarm );
    }
  }
}


void CalendarLocal::incidenceUpdated( IncidenceBase *incidence )
{
  incidence->setSyncStatusSilent( Event::SYNCMOD );
  incidence->setLastModified( KDateTime::currentUtcDateTime() );
  // we should probably update the revision number here,
  // or internally in the Event itself when certain things change.
  // need to verify with ical documentation.

  // The static_cast is ok as the CalendarLocal only observes Incidence objects
  notifyIncidenceChanged( static_cast<Incidence *>( incidence ) );

  setModified( true );
}

void CalendarLocal::insertEvent( Event *event )
{
  QString uid = event->uid();
  if ( mEvents[ uid ] == 0 ) {
    mEvents.insert( uid, event );
  }
#ifndef NDEBUG
  else // if we already have an event with this UID, it has to be the same event,
      // otherwise something's really broken
      Q_ASSERT( mEvents[uid] == event );
#endif
}

Event::List CalendarLocal::rawEventsForDate( const QDate &qd,
                                             EventSortField sortField,
                                             SortDirection sortDirection )
{
//TODO: add KDateTime::Spec parameter?
  Event::List eventList;

  EventDictIterator it( mEvents );
  for( ; it.current(); ++it ) {
    Event *event = *it;

    if ( event->doesRecur() ) {
      if ( event->isMultiDay() ) {
        int extraDays = event->dtStart().date().daysTo( event->dtEnd().date() );
        int i;
        for ( i = 0; i <= extraDays; i++ ) {
          if ( event->recursOn( qd.addDays( -i ), timeSpec() ) ) {
            eventList.append( event );
            break;
          }
        }
      } else {
        if ( event->recursOn( qd, timeSpec() ) )
          eventList.append( event );
      }
    } else {
      if ( event->dtStart().date() <= qd && event->dateEnd() >= qd ) {
        eventList.append( event );
      }
    }
  }

  return sortEvents( &eventList, sortField, sortDirection );
}

Event::List CalendarLocal::rawEvents( const QDate &start, const QDate &end,
                                          bool inclusive )
{
  Event::List eventList;
  KDateTime::Spec ts = timeSpec();
  KDateTime st(start, ts);
  KDateTime nd(end, ts);
  KDateTime yesterStart = st.addDays(-1);

  // Get non-recurring events
  EventDictIterator it( mEvents );
  for( ; it.current(); ++it ) {
    Event *event = *it;

    KDateTime rStart = event->dtStart();
    if (nd < rStart) {
      continue;
    }
    if ( inclusive && rStart < st) {
      continue;
    }

    if ( ! event->doesRecur() ) { // non-recurring events
      KDateTime rEnd = event->dtEnd();
      if (rEnd < st) {
        continue;
      }
      if ( inclusive && nd < rEnd ) {
        continue;
      }
    } else { // recurring events
      switch ( event->recurrence()->duration() ) {
        case -1: // infinite
          if ( inclusive ) {
            continue;
          }
          break;
        case 0: // end date given
        default: // count given
          KDateTime rEnd(event->recurrence()->endDate(), ts);
          if ( ! rEnd.isValid() ) {
            continue;
          }
          if ( rEnd < st ) {
            continue;
          }
          if ( inclusive && nd < rEnd ) {
            continue;
          }
          break;
      } // switch(duration)
    } // if(doesRecur)

    eventList.append( event );
  }

  return eventList;
}

Event::List CalendarLocal::rawEventsForDate( const KDateTime &qdt )
{
  return rawEventsForDate( qdt.date() );
}

Event::List CalendarLocal::rawEvents( EventSortField sortField, SortDirection sortDirection )
{
  Event::List eventList;
  EventDictIterator it( mEvents );
  for( ; it.current(); ++it )
    eventList.append( *it );
  return sortEvents( &eventList, sortField, sortDirection );
}
