/*
    This file is part of libkcal.

    Copyright (c) 2001 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include <qdatetime.h>
#include <qstring.h>
#include <qptrlist.h>
#include <qregexp.h>
#include <qclipboard.h>
#include <qfile.h>
#include <qtextstream.h>

#include <kdebug.h>
#include <klocale.h>

extern "C" {
  #include <ical.h>
  #include <icalss.h>
  #include <icalparser.h>
  #include <icalrestriction.h>
  #include <icalmemory.h>
}

#include "calendar.h"
#include "calendarlocal.h"
#include "icaltimezones.h"
#include "icalformat.h"
#include "icalformatimpl.h"
#include <ksavefile.h>

#include <stdio.h>

#define _ICAL_VERSION "2.0"

using namespace KCal;

ICalFormat::ICalFormat() : mImpl(0)
{
  setImplementation( new ICalFormatImpl( this ) );

  mTimeSpec = KDateTime::UTC;
}

ICalFormat::~ICalFormat()
{
  delete mImpl;
}

void ICalFormat::setImplementation( ICalFormatImpl *impl )
{
  if ( mImpl ) delete mImpl;
  mImpl = impl;
}

#if defined(_AIX) && defined(open)
#undef open
#endif

bool ICalFormat::load( Calendar *calendar, const QString &fileName)
{
  kdDebug(5800) << "ICalFormat::load() " << fileName << endl;

  clearException();

  QFile file( fileName );
  if (!file.open( IO_ReadOnly ) ) {
    kdDebug(5800) << "ICalFormat::load() load error" << endl;
    setException(new ErrorFormat(ErrorFormat::LoadError));
    return false;
  }
  QTextStream ts( &file );
  ts.setEncoding( QTextStream::Latin1 );
  QString text = ts.read();
  file.close();

  if ( text.stripWhiteSpace().isEmpty() ) // empty files are valid
    return true;
  else
    return fromRawString( calendar, text.latin1() );
}


bool ICalFormat::save( Calendar *calendar, const QString &fileName )
{
  kdDebug(5800) << "ICalFormat::save(): " << fileName << endl;

  clearException();

  QString text = toString( calendar );

  if ( text.isNull() ) return false;

  // Write backup file
  KSaveFile::backupFile( fileName );

  KSaveFile file( fileName );
  if ( file.status() != 0 ) {
    kdDebug(5800) << "ICalFormat::save() errno: " << strerror( file.status() )
              << endl;
    setException( new ErrorFormat( ErrorFormat::SaveError,
                  i18n( "Error saving to '%1'." ).arg( fileName ) ) );
    return false;
  }

  // Convert to UTF8 and save
  QCString textUtf8 = text.utf8();
  file.file()->writeBlock( textUtf8.data(), textUtf8.size() - 1 );

  if ( !file.close() ) {
    setException(new ErrorFormat(ErrorFormat::SaveError,
                 i18n("Could not save '%1'").arg(fileName)));
    return false;
  }

  return true;
}

bool ICalFormat::fromString( Calendar *cal, const QString &text )
{
  return fromRawString( cal, text.utf8() );
}

bool ICalFormat::fromRawString( Calendar *cal, const QCString &text )
{
  // Get first VCALENDAR component.
  // TODO: Handle more than one VCALENDAR or non-VCALENDAR top components
  icalcomponent *calendar;

  // Let's defend const correctness until the very gates of hell^Wlibical
  calendar = icalcomponent_new_from_string( const_cast<char*>( (const char*)text ) );
  //  kdDebug(5800) << "Error: " << icalerror_perror() << endl;
  if (!calendar) {
    kdDebug(5800) << "ICalFormat::load() parse error" << endl;
    setException(new ErrorFormat(ErrorFormat::ParseErrorIcal));
    return false;
  }

  bool success = true;

  if (icalcomponent_isa(calendar) == ICAL_XROOT_COMPONENT) {
    icalcomponent *comp;
    for ( comp = icalcomponent_get_first_component(calendar, ICAL_VCALENDAR_COMPONENT);
          comp != 0; comp = icalcomponent_get_next_component(calendar, ICAL_VCALENDAR_COMPONENT) ) {
      // put all objects into their proper places
      if ( !mImpl->populate( cal, comp ) ) {
        kdDebug(5800) << "ICalFormat::load(): Could not populate calendar" << endl;
        if ( !exception() ) {
          setException(new ErrorFormat(ErrorFormat::ParseErrorKcal));
        }
        success = false;
      } else
        mLoadedProductId = mImpl->loadedProductId();
    }
  } else if (icalcomponent_isa(calendar) != ICAL_VCALENDAR_COMPONENT) {
    kdDebug(5800) << "ICalFormat::load(): No VCALENDAR component found" << endl;
    setException(new ErrorFormat(ErrorFormat::NoCalendar));
    success = false;
  } else {
    // put all objects into their proper places
    if ( !mImpl->populate( cal, calendar ) ) {
      kdDebug(5800) << "ICalFormat::load(): Could not populate calendar" << endl;
      if ( !exception() ) {
        setException(new ErrorFormat(ErrorFormat::ParseErrorKcal));
      }
      success = false;
    } else
      mLoadedProductId = mImpl->loadedProductId();
  }

  icalcomponent_free( calendar );
  icalmemory_free_ring();

  return success;
}

Incidence *ICalFormat::fromString( const QString &text )
{
  CalendarLocal cal( mTimeSpec );
  fromString(&cal, text);

  Incidence *ical = 0;
  Event::List elist = cal.events();
  if ( elist.count() > 0 ) {
    ical = elist.first();
  } else {
  }

  return ical ? ical->clone() : 0;
}

QString ICalFormat::toString( Calendar *cal )
{
  icalcomponent *calendar = mImpl->createCalendarComponent(cal);

  icalcomponent *component;

  ICalTimeZones *tzlist = cal->timeZones();   // time zones possibly used in the calendar
  ICalTimeZones tzUsedList;                   // time zones actually used in the calendar

  // todos

  // events
  Event::List events = cal->rawEvents();
  Event::List::ConstIterator it2;
  for( it2 = events.begin(); it2 != events.end(); ++it2 ) {
//    kdDebug(5800) << "ICalFormat::toString() write event "
//                  << (*it2)->uid() << endl;
    component = mImpl->writeEvent( *it2, tzlist, &tzUsedList );
    icalcomponent_add_component( calendar, component );
  }

  // journals

  // time zones
  const ICalTimeZones::ZoneMap zones = tzUsedList.zones();
  for ( ICalTimeZones::ZoneMap::ConstIterator it = zones.begin();  it != zones.end();  ++it) {
    icaltimezone *tz = (*it).icalTimezone();
    if ( !tz ) {
      kdError(5800) << "ICalFormat::toString(): bad time zone" << endl;
    } else {
      component = icalcomponent_new_clone( icaltimezone_get_component( tz ) );
      icalcomponent_add_component( calendar, component );
      icaltimezone_free( tz, 1 );
    }
  }

  QString text = QString::fromUtf8( icalcomponent_as_ical_string( calendar ) );

  icalcomponent_free( calendar );
  icalmemory_free_ring();

  if (!text) {
    setException(new ErrorFormat(ErrorFormat::SaveError,
                 i18n("libical error")));
    return QString::null;
  }

  return text;
}

QString ICalFormat::toICalString( Incidence *incidence )
{
  CalendarLocal cal( mTimeSpec );
  cal.addIncidence( incidence->clone() );
  return toString( &cal );
}

QString ICalFormat::toString( Incidence *incidence )
{
  icalcomponent *component;

  component = mImpl->writeIncidence( incidence );

  QString text = QString::fromUtf8( icalcomponent_as_ical_string( component ) );

  icalcomponent_free( component );

  return text;
}

QString ICalFormat::toString( RecurrenceRule *recurrence )
{
  icalproperty *property;
  property = icalproperty_new_rrule( mImpl->writeRecurrenceRule( recurrence ) );
  QString text = QString::fromUtf8( icalproperty_as_ical_string( property ) );
  icalproperty_free( property );
  return text;
}

bool ICalFormat::fromString( RecurrenceRule * recurrence, const QString& rrule )
{
  if ( !recurrence ) return false;
  bool success = true;
  icalerror_clear_errno();
  struct icalrecurrencetype recur = icalrecurrencetype_from_string( rrule.latin1() );
  if ( icalerrno != ICAL_NO_ERROR ) {
    kdDebug(5800) << "Recurrence parsing error: " << icalerror_strerror( icalerrno ) << endl;
    success = false;
  }

  if ( success ) {
    mImpl->readRecurrence( recur, recurrence );
  }

  return success;
}


void ICalFormat::setTimeSpec( const KDateTime::Spec &timeSpec )
{
  mTimeSpec = timeSpec;
}

KDateTime::Spec ICalFormat::timeSpec() const
{
  return mTimeSpec;
}

QString ICalFormat::timeZoneId() const
{
  KTimeZone tz = mTimeSpec.timeZone();
  return tz.isValid() ? tz.name() : QString();
}
