/*
    This file is part of libkcal.

    Copyright (c) 2001,2004 Cornelius Schumacher <schumacher@kde.org>
    Copyright (C) 2003-2004 Reinhold Kainhofer <reinhold@kainhofer.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include <kglobal.h>
#include <klocale.h>
#include <kdebug.h>

#include "calformat.h"

#include "incidencebase.h"

using namespace KCal;

IncidenceBase::IncidenceBase()
  : mReadOnly( false ), mUpdateGroupLevel(0), mUpdatedPending( false ),
    mFloats( true ), mDuration( 0 ), mHasDuration( false ),
    mPilotId( 0 ), mSyncStatus( SYNCMOD )
{
  setUid( CalFormat::createUniqueId() );

}

IncidenceBase::IncidenceBase(const IncidenceBase &i) :
  CustomProperties( i )
{
  mReadOnly = i.mReadOnly;
  mUpdateGroupLevel = 0;
  mUpdatedPending = false;
  mDtStart = i.mDtStart;
  mDuration = i.mDuration;
  mHasDuration = i.mHasDuration;
  mOrganizer = i.mOrganizer;
  mUid = i.mUid;
  mFloats = i.mFloats;
  mLastModified = i.mLastModified;
  mPilotId = i.mPilotId;
  mSyncStatus = i.mSyncStatus;
  mComments = i.mComments;

}

IncidenceBase::~IncidenceBase()
{
}

IncidenceBase& IncidenceBase::operator=( const IncidenceBase& i )
{
  CustomProperties::operator=( i );
  mReadOnly = i.mReadOnly;
  mDtStart = i.mDtStart;
  mDuration = i.mDuration;
  mHasDuration = i.mHasDuration;
  mOrganizer = i.mOrganizer;
  mUid = i.mUid;
  mFloats = i.mFloats;
  mLastModified = i.mLastModified;
  mPilotId = i.mPilotId;
  mSyncStatus = i.mSyncStatus;
  mComments = i.mComments;

  // The copied object is a new one, so it isn't observed by the observer
  // of the original object.
  mObservers.clear();
  return *this;
}

bool IncidenceBase::operator==( const IncidenceBase& i2 ) const
{


  if ( !CustomProperties::operator==(i2) )
    return false;

  return ( dtStart() == i2.dtStart() &&
           organizer() == i2.organizer() &&
           uid() == i2.uid() &&
           // Don't compare lastModified, otherwise the operator is not
           // of much use. We are not comparing for identity, after all.
           doesFloat() == i2.doesFloat() &&
           duration() == i2.duration() &&
           hasDuration() == i2.hasDuration() &&
           pilotId() == i2.pilotId() &&
           syncStatus() == i2.syncStatus() );
  // no need to compare mObserver
}




void IncidenceBase::setUid(const QString &uid)
{
  mUid = uid;
  updated();
}

QString IncidenceBase::uid() const
{
  return mUid;
}

void IncidenceBase::setLastModified(const KDateTime &lm)
{
  // DON'T! updated() because we call this from
  // Calendar::updateEvent().

  // Convert to UTC and remove milliseconds part.
  KDateTime current = lm.toUtc();
  QTime t = current.time();
  t.setHMS( t.hour(), t.minute(), t.second(), 0 );
  current.setTime( t );

  mLastModified = current;
}

KDateTime IncidenceBase::lastModified() const
{
  return mLastModified;
}

void IncidenceBase::setOrganizer( const Person &o )
{
  // we don't check for readonly here, because it is
  // possible that by setting the organizer we are changing
  // the event's readonly status...
  mOrganizer = o;

  updated();
}

void IncidenceBase::setOrganizer(const QString &o)
{
  QString mail( o );
  if ( mail.startsWith("MAILTO:", false) )
    mail = mail.remove( 0, 7 );
  // split the string into full name plus email.
  Person organizer( mail );
  setOrganizer( organizer );
}

Person IncidenceBase::organizer() const
{
  return mOrganizer;
}

void IncidenceBase::setReadOnly( bool readOnly )
{
  mReadOnly = readOnly;
}

void IncidenceBase::setDtStart(const KDateTime &dtStart)
{
//  if (mReadOnly) return;
  mDtStart = dtStart;
  mFloats = dtStart.isDateOnly();
  updated();
}

KDateTime IncidenceBase::dtStart() const
{
  return mDtStart;
}

QString IncidenceBase::dtStartTimeStr() const
{
  return KGlobal::locale()->formatTime(dtStart().time());
}

QString IncidenceBase::dtStartDateStr(bool shortfmt) const
{
  return KGlobal::locale()->formatDate(dtStart().date(),shortfmt);
}

QString IncidenceBase::dtStartStr() const
{
  return KGlobal::locale()->formatDateTime(dtStart().dateTime());
}


bool IncidenceBase::doesFloat() const
{
  return mFloats;
}

void IncidenceBase::setFloats(bool f)
{
  if ( mReadOnly || f == mFloats ) return;
  mFloats = f;
  updated();
}

void IncidenceBase::shiftTimes(const KDateTime::Spec &oldSpec, const KDateTime::Spec &newSpec)
{
  mDtStart = mDtStart.toTimeSpec( oldSpec );
  mDtStart.setTimeSpec( newSpec );
  updated();
}

void IncidenceBase::addComment(const QString& comment)
{
  mComments += comment;
}

bool IncidenceBase::removeComment( const QString& comment)
{
  bool found = false;
  QStringList::Iterator i;

  for ( i = mComments.begin(); !found && i != mComments.end(); ++i ) {
    if ( (*i) == comment ) {
      found = true;
      mComments.remove(i);
    }
  }

  return found;
}

void IncidenceBase::clearComments()
{
  mComments.clear();
}

QStringList IncidenceBase::comments() const
{
  return mComments;
}


#if 0
#endif


void IncidenceBase::setDuration(int seconds)
{
  mDuration = seconds;
  setHasDuration(true);
  updated();
}

int IncidenceBase::duration() const
{
  return mDuration;
}

void IncidenceBase::setHasDuration(bool hasDuration)
{
  mHasDuration = hasDuration;
}

bool IncidenceBase::hasDuration() const
{
  return mHasDuration;
}

void IncidenceBase::setSyncStatus(int stat)
{
  if (mReadOnly) return;
  if ( mSyncStatus == stat ) return;
  mSyncStatus = stat;
  updatedSilent();
}
void IncidenceBase::setSyncStatusSilent(int stat)
{
  if (mReadOnly) return;
  mSyncStatus = stat;
}

int IncidenceBase::syncStatus() const
{
  return mSyncStatus;
}

void IncidenceBase::setPilotId( unsigned long id )
{
  if (mReadOnly) return;
  if ( mPilotId == id) return;
  mPilotId = id;
  updatedSilent();
}

unsigned long IncidenceBase::pilotId() const
{
  return mPilotId;
}

void IncidenceBase::registerObserver( IncidenceBase::Observer *observer )
{
  if( !mObservers.contains( observer ) ) mObservers.append( observer );
}

void IncidenceBase::unRegisterObserver( IncidenceBase::Observer *observer )
{
  mObservers.remove( observer );
}

void IncidenceBase::updated()
{
  if ( mUpdateGroupLevel ) {
    mUpdatedPending = true;
  } else {
    QPtrListIterator<Observer> it(mObservers);
    while( it.current() ) {
      Observer *o = it.current();
      ++it;
      o->incidenceUpdated( this );
    }
  }
}

void IncidenceBase::startUpdates() 
{ 
  ++mUpdateGroupLevel; 
} 
 
void IncidenceBase::endUpdates() 
{ 
  if ( mUpdateGroupLevel > 0 ) { 
    if ( --mUpdateGroupLevel == 0 && mUpdatedPending ) { 
      mUpdatedPending = false;
      updated(); 
    }
  }
}

void IncidenceBase::customPropertyUpdated()
{
  updated();
}

void IncidenceBase::updatedSilent()
{
  QPtrListIterator<Observer> it(mObservers);
  while( it.current() ) {
    Observer *o = it.current();
    ++it;
    o->incidenceUpdatedSilent( this );
  }
}

