/*
    This file is part of libkcal.

    Copyright (c) 2001-2003 Cornelius Schumacher <schumacher@kde.org>
    Copyright (C) 2003-2004 Reinhold Kainhofer <reinhold@kainhofer.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/
#ifndef KCAL_INCIDENCEBASE_H
#define KCAL_INCIDENCEBASE_H

#include <qstringlist.h>
#include <qvaluelist.h>
#include <qptrlist.h>

#include <kdatetime.h>

#include "customproperties.h"
#include "sortablelist.h"

namespace KCal {

typedef SortableList<QDate> DateList;
typedef SortableList<KDateTime> DateTimeList;
class Event;

/**
  This class provides the base class common to all calendar components.
*/
class LIBKCAL_EXPORT IncidenceBase : public CustomProperties
{
  public:
    /**
      This class provides the interface for a visitor of calendar components. It
      serves as base class for concrete visitors, which implement certain actions on
      calendar components. It allows to add functions, which operate on the concrete
      types of calendar components, without changing the calendar component classes.
    */
    class Visitor
    {
      public:
        /** Destruct Incidence::Visitor */
        virtual ~Visitor() {}

        /**
          Reimplement this function in your concrete subclass of IncidenceBase::Visitor to perform actions
          on an Event object.
        */
        virtual bool visit(Event *) { return false; }

      protected:
        /** Constructor is protected to prevent direct creation of visitor base class. */
        Visitor() {}
    };

    class Observer {
      public:
        virtual ~Observer() {}
        virtual void incidenceUpdated( IncidenceBase * ) = 0;
        virtual void incidenceUpdatedSilent( IncidenceBase * ) {};
    };

    IncidenceBase();
    IncidenceBase( const IncidenceBase & );
    virtual ~IncidenceBase();

    IncidenceBase& operator=( const IncidenceBase &i );
    bool operator==( const IncidenceBase & ) const;

    /**
      Accept IncidenceVisitor. A class taking part in the visitor mechanism has to
      provide this implementation:
      <pre>
        bool accept(Visitor &v) { return v.visit(this); }
      </pre>
    */
    virtual bool accept(Visitor &) { return false; }

    virtual QCString type() const = 0;

    /** Set the unique id for the event */
    void setUid( const QString & );
    /** Return the unique id for the event */
    QString uid() const;

    /** Sets the time the incidence was last modified. It is stored as a UTC date/time. */
    void setLastModified( const KDateTime &lm );
    /** Return the time the incidence was last modified. */
    KDateTime lastModified() const;

    /** sets the organizer for the event */
    void setOrganizer( const Person &o );
    void setOrganizer( const QString &o );
    Person organizer() const;

    /** Set readonly status. */
    virtual void setReadOnly( bool );
    /** Return if the object is read-only. */
    bool isReadOnly() const { return mReadOnly; }

    /**
      Sets the incidence's starting date/time with a KDateTime.
      The incidence's floating status is set according to whether @p dtStart
      is a date/time (not floating) or date-only (floating).
    */
    virtual void setDtStart( const KDateTime &dtStart );
    /** returns an event's starting date/time as a KDateTime. */
    virtual KDateTime dtStart() const;
    /** returns an event's starting time as a string formatted according to the
     users locale settings */
    virtual QString dtStartTimeStr() const;
    /** returns an event's starting date as a string formatted according to the
     users locale settings */
    virtual QString dtStartDateStr( bool shortfmt = true ) const;
    /** returns an event's starting date and time as a string formatted according
     to the users locale settings */
    virtual QString dtStartStr() const;

    virtual void setDuration( int seconds );
    int duration() const;
    void setHasDuration( bool );
    bool hasDuration() const;

    /** Return true or false depending on whether the incidence "floats,"
     * i.e. has a date but no time attached to it. */
    bool doesFloat() const;
    /** Set whether the incidence floats, i.e. has a date but no time attached to it. */
    void setFloats( bool f );
 
    /**
      Shift the times of the incidence so that they appear at the same clock
      time as before but in a new time zone. The shift is done from a viewing
      time zone rather than from the actual incidence time zone.

      For example, shifting an incidence whose start time is 09:00 America/New York,
      using an old viewing time zone (@p oldSpec) of Europe/London, to a new time
      zone (@p newSpec) of Europe/Paris, will result in the time being shifted
      from 14:00 (which is the London time of the incidence start) to 14:00 Paris
      time.

      @param oldSpec the time specification which provides the clock times
      @param newSpec the new time specification
    */
    virtual void shiftTimes(const KDateTime::Spec &oldSpec, const KDateTime::Spec &newSpec);

    //
    // Comments
    //

    /**
     * Add a comment to this incidence.
     *
     * Does not add a linefeed character.  Just appends the text as passed in.
     *
     * @param comment  The comment to add.
     */
    void addComment(const QString& comment);

    /**
     * Remove a comment from the event.
     *
     * Removes first comment whose string is an exact match for the string
     * passed in.
     *
     * @return true if match found, false otherwise.
     */
    bool removeComment( const QString& comment );

    /** Delete all comments associated with this incidence. */
    void clearComments();

    /** Return all comments associated with this incidence.  */
    QStringList comments() const;


    /**
      Pilot synchronization states
    */
    enum { SYNCNONE = 0, SYNCMOD = 1, SYNCDEL = 3 };
    /**
      Set synchronisation satus.
    */
    void setSyncStatus( int status );
    void setSyncStatusSilent( int status );
    /**
      Return synchronisation status.
    */
    int syncStatus() const;

    /**
      Set Pilot Id.
    */
    void setPilotId( unsigned long id );
    /**
      Return Pilot Id.
    */
    unsigned long pilotId() const;

    /**
      Register observer. The observer is notified when the observed object
      changes.
    */
    void registerObserver( Observer * );
    /**
      Unregister observer. It isn't notified anymore about changes.
    */
    void unRegisterObserver( Observer * );
    /**
      Call this to notify the observers after the IncidenceBas object has
      changed.
    */
    void updated();
    void updatedSilent();

    /** 
      Call this when a group of updates is going to be made. This suppresses 
      change notifications until endUpdates() is called, at which point 
      updated() will automatically be called. 
    */ 
    void startUpdates(); 

    /** 
      Call this when a group of updates is complete, to notify observers that 
      the instance has changed. This should be called in conjunction with 
      startUpdates(). 
    */ 
    void endUpdates(); 

  protected:
    /**
      @copydoc
      CustomProperties::customPropertyUpdated()
    */
    virtual void customPropertyUpdated();

    bool mReadOnly;

  private:
    // base components
    KDateTime mDtStart;
    Person mOrganizer;
    QString mUid;
    KDateTime mLastModified;
    QStringList mComments;
    int mUpdateGroupLevel;
    bool mUpdatedPending;

    bool mFloats;

    int mDuration;
    bool mHasDuration;

    // PILOT SYNCHRONIZATION STUFF
    unsigned long mPilotId;                         // unique id for pilot sync
    int mSyncStatus;                      // status (for sync)

    QPtrList<Observer> mObservers;

    class Private;
    Private *d;
};

}

#endif
