/*
    This file is part of the kcal library.

    Copyright (c) 2006 David Jarvie <software@astrojar.org.uk>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef KCAL_SORTEDLIST_H
#define KCAL_SORTEDLIST_H

#include <qvaluelist.h>
#include <qtl.h>

namespace KCal {

template <class T>
void qSortUnique( QValueList<T> &list )
{
  if ( list.count() <= 1 )
    return;
  qHeapSort( list );
  QValueListIterator<T> prev = list.begin();
  for ( QValueListIterator<T> it = prev;  ++it != list.end(); ) {
    if ( *it == *prev ) {
      // Found two equal values. Search for any further equal values and remove
      // them all together for efficiency.
      while ( ++it != list.end()  &&  *it == *prev ) ;
      prev = it = list.erase( ++prev, it );
      if ( it == list.end() )
        break;
    } else {
      prev = it;
    }
  }
}

/**
 * @short A QValueList which can be sorted
 *
 * For a QValueList is capable of being sorted, SortedList provides additional
 * optimized methods which can be used when the list is sorted and has no
 * duplicate entries.
 *
 * Because SortableList has no data members, an object may be referred to
 * interchangeably as either a QValueList or SortableList. Just bear in mind that
 * the results of the SortableList methods are undefined when the list is
 * unsorted or contains duplicate entries.
 *
 * To sort the list and remove duplicate entries, thereby allowing use of
 * other SortableList methods, use sortUnique(). Once sortUnique() has been
 * called, use findSorted(), containsSorted() and removeSorted() in preference
 * to QValueList::indexOf(), QValueList::contains() and QValueList::removeAll(). Use findLE(),
 * findLT(), findGE(), findGT() to find the index to the nearest value in the
 * list which is <=, <, >= or > a given value. To add a value to the list,
 * use insertSorted() in preference to insert(), append(), prepend(),
 * operator<<() or operator+=().
 *
 * @author David Jarvie \<software@astrojar.org.uk\>.
 */
template <class T>
class SortableList : public QValueList<T>
{
  public:
    SortableList() {}
    SortableList( const QValueList<T> &list ) : QValueList<T>( list ) {}   // implicit conversion
    /**
     * Return whether the list contains value @p value. The list must be sorted;
     * if not, the result is undefined.
     * When the list is sorted, use this optimised method in preference to
     * QValueList<T>::contains().
     *
     * @param value value to find
     * @return true if list contains @p value; false otherwise
     */
    bool containsSorted( const T &value ) const  { return findSorted( value ) != QValueList<T>::end(); }
    bool containsSorted( const T &value )  { return findSorted( value ) != QValueList<T>::end(); }
//    bool containsSorted( const T &value ) const
//    {
//        QValueListConstIterator<T> it = findSorted( value );
//        QValueListConstIterator<T> itend = QValueList<T>::end();
//        return it != itend;
//    }
    /**
     * Search the list for the item equal to @p value. The list must be sorted;
     * if not, the result is undefined.
     * When the list is sorted, use this optimised method in preference to
     * QValueList<T>::indexOf().
     *
     * @param value value to find
     * @param start start index for search (default is from beginning)
     * @return index to item in list, or end() if @p value not found in the list
     */
    QValueListConstIterator<T> findSorted( const T &value ) const  { return findSorted(value, QValueList<T>::begin()); }
    QValueListConstIterator<T> findSorted( const T &value, const typename QValueList<T>::const_iterator start ) const;
    QValueListIterator<T> findSorted( const T &value )  { return findSorted(value, QValueList<T>::begin()); }
    QValueListIterator<T> findSorted( const T &value, QValueListIterator<T> start );
    /**
     * Search the list for the last item <= @p value. The list must be sorted;
     * if not, the result is undefined.
     * @param value value to find
     * @param start start index for search (default is from beginning)
     * @return index to item in list, or end() if @p value < first value in the list
     */
    QValueListConstIterator<T> findLE( const T &value ) const  { return findLE(value, QValueList<T>::begin()); }
    QValueListConstIterator<T> findLE( const T &value, const typename QValueList<T>::const_iterator start ) const;
    QValueListIterator<T> findLE( const T &value )  { return findLE(value, QValueList<T>::begin()); }
    QValueListIterator<T> findLE( const T &value, QValueListIterator<T> start );
    /**
     * Search the list for the last item < @p value. The list must be sorted;
     * if not, the result is undefined.
     * @param value value to find
     * @param start start index for search (default is from beginning)
     * @return index to item in list, or end() if @p value <= first value in the list
     */
    QValueListConstIterator<T> findLT( const T &value ) const  { return findLT(value, QValueList<T>::begin()); }
    QValueListConstIterator<T> findLT( const T &value, const typename QValueList<T>::const_iterator start ) const;
    QValueListIterator<T> findLT( const T &value )  { return findLT(value, QValueList<T>::begin()); }
    QValueListIterator<T> findLT( const T &value, QValueListIterator<T> start );
    /**
     * Search the list for the first item >= @p value. The list must be sorted;
     * if not, the result is undefined.
     * @param value value to find
     * @param start start index for search (default is from beginning)
     * @return index to item in list, or end() if @p value > last value in the list
     */
    QValueListConstIterator<T> findGE( const T &value ) const  { return findGE(value, QValueList<T>::begin()); }
    QValueListConstIterator<T> findGE( const T &value, const typename QValueList<T>::const_iterator start ) const;
    QValueListIterator<T> findGE( const T &value )  { return findGE(value, QValueList<T>::begin()); }
    QValueListIterator<T> findGE( const T &value, QValueListIterator<T> start );
    /**
     * Search the list for the first item > @p value. The list must be sorted;
     * if not, the result is undefined.
     * @param value value to find
     * @param start start index for search (default is from beginning)
     * @return index to item in list, or end() if @p value >= last value in the list
     */
    QValueListConstIterator<T> findGT( const T &value ) const  { return findGT(value, QValueList<T>::begin()); }
    QValueListConstIterator<T> findGT( const T &value, const typename QValueList<T>::const_iterator start ) const;
    QValueListIterator<T> findGT( const T &value )  { return findGT(value, QValueList<T>::begin()); }
    QValueListIterator<T> findGT( const T &value, QValueListIterator<T> start );
    /**
     * Insert a value in the list, in correct sorted order. If the same value
     * is already in the list, no change is made.
     *
     * The list must already be sorted before calling this method; otherwise
     * the result is undefined.
     *
     * @param value value to insert
     * @return index to inserted item in list, or to the pre-existing entry
     *         equal to @p value
     */
    QValueListIterator<T> insertSorted( const T &value );
    /**
     * Remove value @p value from the list. The list must be sorted.
     * When the list is sorted, use this optimised method in preference to
     * QValueList<T>::removeAll().
     *
     * @param value value to remove
     * @param start start index for search (default is from beginning)
     * @return index to removed value, or end() if not found
     */
    QValueListIterator<T> removeSorted( const T &value )  { return removeSorted(value, QValueList<T>::begin()); }
    QValueListIterator<T> removeSorted( const T &value, QValueListIterator<T> start = QValueList<T>::begin() );
    /**
     * Sort the list. Any duplicate values are removed.
     */
    void sortUnique()  { qSortUnique( *this ); }
};


template <class T>
QValueListConstIterator<T> SortableList<T>::findSorted( const T &value, const typename QValueList<T>::const_iterator start ) const
{
  // Do a binary search to find the item == value
  QValueListConstIterator<T> i, d;
  QValueListConstIterator<T> st = start;
  QValueListConstIterator<T> nd = QValueList<T>::end();
  while ( nd != st )
  {
    // Start at the midpoint: i = st + ( nd - st - 1 )/2;
    i = st;
    d = nd;
    if (--d != i)
      while (--d != i  &&  ++i != d) ;

    if ( value < *i )
      nd = i;
    else
    {
      st = i;
      ++st;
    }
  }
  return ( st != start && value == *(--st) ) ? st : QValueList<T>::end();
}

template <class T>
QValueListIterator<T> SortableList<T>::findSorted( const T &value, QValueListIterator<T> start )
{
  // Do a binary search to find the item == value
  QValueListIterator<T> i, d;
  QValueListIterator<T> st = start;
  QValueListIterator<T> nd = QValueList<T>::end();
  while ( nd != st )
  {
    // Start at the midpoint: i = st + ( nd - st - 1 )/2;
    i = st;
    d = nd;
    if (--d != i)
      while (--d != i  &&  ++i != d) ;

    if ( value < *i )
      nd = i;
    else
    {
      st = i;
      ++st;
    }
  }
  return ( st != start && value == *(--st) ) ? st : QValueList<T>::end();
}

template <class T>
QValueListConstIterator<T> SortableList<T>::findLE( const T &value, const typename QValueList<T>::const_iterator start ) const
{
  // Do a binary search to find the last item <= value
  QValueListConstIterator<T> i, d;
  QValueListConstIterator<T> st = start;
  QValueListConstIterator<T> nd = QValueList<T>::end();
  while ( nd != st )
  {
    // Start at the midpoint: i = st + ( nd - st - 1 )/2;
    i = st;
    d = nd;
    if (--d != i)
      while (--d != i  &&  ++i != d) ;

    if ( value < *i )
      nd = i;
    else
    {
      st = i;
      ++st;
    }
  }
  return ( st != start) ? --st : QValueList<T>::end();
}

template <class T>
QValueListIterator<T> SortableList<T>::findLE( const T &value, QValueListIterator<T> start )
{
  // Do a binary search to find the last item <= value
  QValueListIterator<T> i, d;
  QValueListIterator<T> st = start;
  QValueListIterator<T> nd = QValueList<T>::end();
  while ( nd != st )
  {
    // Start at the midpoint: i = st + ( nd - st - 1 )/2;
    i = st;
    d = nd;
    if (--d != i)
      while (--d != i  &&  ++i != d) ;

    if ( value < *i )
      nd = i;
    else
    {
      st = i;
      ++st;
    }
  }
  return ( st != start) ? --st : QValueList<T>::end();
}

template <class T>
QValueListConstIterator<T> SortableList<T>::findLT( const T &value, const typename QValueList<T>::const_iterator start ) const
{
  // Do a binary search to find the last item < value
  QValueListConstIterator<T> i, d;
  QValueListConstIterator<T> st = start;
  QValueListConstIterator<T> nd = QValueList<T>::end();
  while ( nd != st )
  {
    // Start at the midpoint: i = st + ( nd - st - 1 )/2;
    i = st;
    d = nd;
    if (--d != i)
      while (--d != i  &&  ++i != d) ;

    if ( value <= *i )
      nd = i;
    else
    {
      st = i;
      ++st;
    }
  }
  return ( st != start) ? --st : QValueList<T>::end();
}

template <class T>
QValueListIterator<T> SortableList<T>::findLT( const T &value, QValueListIterator<T> start )
{
  // Do a binary search to find the last item < value
  QValueListIterator<T> i, d;
  QValueListIterator<T> st = start;
  QValueListIterator<T> nd = QValueList<T>::end();
  while ( nd != st )
  {
    // Start at the midpoint: i = st + ( nd - st - 1 )/2;
    i = st;
    d = nd;
    if (--d != i)
      while (--d != i  &&  ++i != d) ;

    if ( value <= *i )
      nd = i;
    else
    {
      st = i;
      ++st;
    }
  }
  return ( st != start) ? --st : QValueList<T>::end();
}

template <class T>
QValueListConstIterator<T> SortableList<T>::findGE( const T &value, const typename QValueList<T>::const_iterator start ) const
{
  // Do a binary search to find the first item >= value
  QValueListConstIterator<T> i, d;
  QValueListConstIterator<T> st = start;
  QValueListConstIterator<T> nd = QValueList<T>::end();
  while ( nd != st )
  {
    // Start at the midpoint: i = st + ( nd - st - 1 )/2;
    i = st;
    d = nd;
    if (--d != i)
      while (--d != i  &&  ++i != d) ;

    if ( value <= *i )
      nd = i;
    else
    {
      st = i;
      ++st;
    }
  }
  return st;
}

template <class T>
QValueListIterator<T> SortableList<T>::findGE( const T &value, QValueListIterator<T> start )
{
  // Do a binary search to find the first item >= value
  QValueListIterator<T> i, d;
  QValueListIterator<T> st = start;
  QValueListIterator<T> nd = QValueList<T>::end();
  while ( nd != st )
  {
    // Start at the midpoint: i = st + ( nd - st - 1 )/2;
    i = st;
    d = nd;
    if (--d != i)
      while (--d != i  &&  ++i != d) ;

    if ( value <= *i )
      nd = i;
    else
    {
      st = i;
      ++st;
    }
  }
  return st;
}

template <class T>
QValueListConstIterator<T> SortableList<T>::findGT( const T &value, const typename QValueList<T>::const_iterator start ) const
{
  // Do a binary search to find the first item > value
  QValueListConstIterator<T> i, d;
  QValueListConstIterator<T> st = start;
  QValueListConstIterator<T> nd = QValueList<T>::end();
  while ( nd != st )
  {
    // Start at the midpoint: i = st + ( nd - st - 1 )/2;
    i = st;
    d = nd;
    if (--d != i)
      while (--d != i  &&  ++i != d) ;

    if ( value < *i )
      nd = i;
    else
    {
      st = i;
      ++st;
    }
  }
  return st;
}

template <class T>
QValueListIterator<T> SortableList<T>::findGT( const T &value, QValueListIterator<T> start )
{
  // Do a binary search to find the first item > value
  QValueListIterator<T> i, d;
  QValueListIterator<T> st = start;
  QValueListIterator<T> nd = QValueList<T>::end();
  while ( nd != st )
  {
    // Start at the midpoint: i = st + ( nd - st - 1 )/2;
    i = st;
    d = nd;
    if (--d != i)
      while (--d != i  &&  ++i != d) ;

    if ( value < *i )
      nd = i;
    else
    {
      st = i;
      ++st;
    }
  }
  return st;
}

template <class T>
QValueListIterator<T> SortableList<T>::insertSorted( const T &value )
{
  QValueListIterator<T> i = findLE( value );
  if ( i != QValueList<T>::end() ) {
    if ( *i == value )
      return i;
    ++i;
  }
  QValueList<T>::insert( i, value );
  return i;
}

template <class T>
QValueListIterator<T> SortableList<T>::removeSorted( const T &value, QValueListIterator<T> start )
{
  QValueListIterator<T> i = findSorted( value, start );
  if ( i != QValueList<T>::end() )
    QValueList<T>::remove( i );
  return i;
}

} // namespace KCal

#endif
