/*
 *  identitymanager.cpp  -  manages the list of email identities
 *  Program:  kalarm
 *  Copyright (c) 2004,2007 by David Jarvie <software@astrojar.org.uk>
 *  This is a modified version of the KDEPIM file of the same name,
 *  Copyright (c) 2002 Marc Mutz <mutz@kde.org>
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 *
 *  This library is distributed in the hope that it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 *  License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 *  02110-1301, USA.
 */

// config keys:
static const char configKeyDefaultIdentity[] = "Default Identity";

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "identitymanager.moc"

#include "identity.h" // for IdentityList::{export,import}Data

#include <kapplication.h>
#include <klocale.h>
#include <kdebug.h>
#include <kconfig.h>
#include <kuser.h>
#include <dcopclient.h>

#include <qregexp.h>

#include <assert.h>

using namespace KPIM;

static QCString newDCOPObjectName()
{
    static int s_count = 0;
    QCString name( "KPIM::IdentityManager" );
    if ( s_count++ ) {
      name += '-';
      name += QCString().setNum( s_count );
    }
    return name;
}

IdentityManager::IdentityManager( bool readonly, QObject * parent, const char * name )
  : QObject( parent, name ), DCOPObject( newDCOPObjectName() )
{
  mReadOnly = readonly;
  mConfig = new KConfig( "emailidentities", readonly );
  readConfig(mConfig);
  if ( mIdentities.isEmpty() ) {
    kdDebug(5006) << "emailidentities is empty -> convert from kmailrc" << endl;
    // No emailidentities file, or an empty one due to broken conversion (kconf_update bug in kdelibs <= 3.2.2)
    // => convert it, i.e. read settings from kmailrc
    KConfig kmailConf( "kmailrc", true );
    readConfig( &kmailConf );
  }
  // we need at least a default identity:
  if ( mIdentities.isEmpty() )
    kdDebug( 5006 ) << "IdentityManager: No identity found." << endl;

  // The emitter is always called KPIM::IdentityManager even if we are not
  if ( !connectDCOPSignal( 0, "KPIM::IdentityManager", "identitiesChanged(QCString,QCString)",
                           "slotIdentitiesChanged(QCString,QCString)", false ) )
      kdError(5650) << "IdentityManager: connection to identitiesChanged failed" << endl;
}

IdentityManager::~IdentityManager()
{
  delete mConfig;
}

QStringList IdentityManager::identities() const
{
  QStringList result;
  for ( ConstIterator it = mIdentities.begin() ;
	it != mIdentities.end() ; ++it )
    result << (*it).identityName();
  return result;
}

void IdentityManager::readConfig(KConfigBase* config) {
  mIdentities.clear();

  QStringList identities = groupList(config);
  if ( identities.isEmpty() ) return; // nothing to be done...

  KConfigGroup general( config, "General" );
  uint defaultIdentity = general.readUnsignedNumEntry( configKeyDefaultIdentity );
  bool haveDefault = false;

  for ( QStringList::Iterator group = identities.begin() ;
	group != identities.end() ; ++group ) {
    KConfigGroup configGroup( config, *group );
    mIdentities << Identity();
    mIdentities.last().readConfig( &configGroup );
    if ( !haveDefault && mIdentities.last().uoid() == defaultIdentity ) {
      haveDefault = true;
      mIdentities.last().setIsDefault( true );
    }
  }
  if ( !haveDefault ) {
    kdWarning( 5006 ) << "IdentityManager: There was no default identity. Marking first one as default." << endl;
    mIdentities.first().setIsDefault( true );
  }
  qHeapSort( mIdentities );
}

QStringList IdentityManager::groupList(KConfigBase* config) const {
  return config->groupList().grep( QRegExp("^Identity #\\d+$") );
}

IdentityManager::ConstIterator IdentityManager::begin() const {
  return mIdentities.begin();
}

IdentityManager::ConstIterator IdentityManager::end() const {
  return mIdentities.end();
}

const Identity & IdentityManager::identityForUoid( uint uoid ) const {
  for ( ConstIterator it = begin() ; it != end() ; ++it )
    if ( (*it).uoid() == uoid ) return (*it);
  return Identity::null();
}

const Identity & IdentityManager::identityForUoidOrDefault( uint uoid ) const
{
  const Identity & ident = identityForUoid( uoid );
  if ( ident.isNull() )
    return defaultIdentity();
  else
    return ident;
}

const Identity & IdentityManager::defaultIdentity() const {
  for ( ConstIterator it = begin() ; it != end() ; ++it )
    if ( (*it).isDefault() ) return (*it);
  (mIdentities.isEmpty() ? kdFatal( 5006 ) : kdWarning( 5006 ) )
    << "IdentityManager: No default identity found!" << endl;
  return *begin();
}

void KPIM::IdentityManager::slotIdentitiesChanged( QCString appId, QCString objId )
{
  // From standalone kmail to standalone korganizer, the appId will differ
  // From kontact the appId will match, so we need to test the objId
  if ( kapp->dcopClient()->appId() != appId || DCOPObject::objId() != objId ) {
    mConfig->reparseConfiguration();
    readConfig( mConfig );
  }
}
