/*
 *  identitymanager.h  -  manages the list of email identities
 *  Program:  kalarm
 *  Copyright (c) 2004,2007 by David Jarvie <software@astrojar.org.uk>
 *  This is a modified version of the KDEPIM file of the same name,
 *  Copyright (c) 2002 Marc Mutz <mutz@kde.org>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */
#ifndef KPIM_IDENTITYMANAGER_H
#define KPIM_IDENTITYMANAGER_H

#include <qvaluelist.h>
#include <dcopobject.h>

class KConfigBase;
class KConfig;
class QStringList;

namespace KPIM
{

  class Identity;
  /**
   * @short Manages the list of identities.
   * @author Marc Mutz <mutz@kde.org>
   **/
  class IdentityManager : public QObject, public DCOPObject
  {
      Q_OBJECT
      K_DCOP

    public:
      /**
       * Create an identity manager, which loads the emailidentities file
       * to create identities.
       * @param readonly if true, no changes can be made to the identity manager
       * This means in particular that if there is no identity configured,
       * the default identity created here will not be saved.
       * It is assumed that a minimum of one identity is always present.
       */
      IdentityManager( bool readonly = false, QObject * parent=0, const char * name=0 );
      virtual ~IdentityManager();

    public:
      typedef QValueList<Identity>::Iterator Iterator;
      typedef QValueList<Identity>::ConstIterator ConstIterator;

      /** @return the list of identities */
      QStringList identities() const;

      /** @return the identity with Unique Object Identifier (UOID) @p
                  uoid or @ref Identity::null if not found.
       **/
      const Identity & identityForUoid( uint uoid ) const;

      /** Convenience menthod.

          @return the identity with Unique Object Identifier (UOID) @p
                  uoid or the default identity if not found.
      **/
      const Identity & identityForUoidOrDefault( uint uoid ) const;

      /** @return the default identity */
      const Identity & defaultIdentity() const;

      ConstIterator begin() const;
      ConstIterator end() const;

    signals:
      /** Emitted whenever the identity with Unique Object Identifier
          (UOID) @p uoid changed. Useful for more fine-grained change
          notifications than what is possible with the standard @ref
          changed() signal. */
      void changed( uint uoid );
      /** Emitted whenever the identity @p ident changed. Useful for more
          fine-grained change notifications than what is possible with the
          standard @ref changed() signal. */
      void changed( const KPIM::Identity & ident );

    protected:
      /** The list that will be seen by everyone */
      QValueList<Identity> mIdentities;

    k_dcop:
      // Connected to the DCOP signal
      void slotIdentitiesChanged( QCString appId, QCString objId );

    private:
      void writeConfig() const;
      void readConfig(KConfigBase* config);
      QStringList groupList(KConfigBase* config) const;

    private:
      KConfig* mConfig;
      bool mReadOnly;
  };

} // namespace

#endif // KPIM_IDENTITYMANAGER_H
