#warning If resource file deleted and then another file copied to it, alarm list does not update
/*
 *  resourcelocal.cpp  -  KAlarm local calendar resource
 *  Program:  kalarm
 *  Copyright © 2006-2008 by David Jarvie <software@astrojar.org.uk>
 *  Based on resourcelocal.cpp in libkcal (updated to rev 650427),
 *  Copyright (c) 1998 Preston Brown <pbrown@kde.org>
 *  Copyright (c) 2001,2003 Cornelius Schumacher <schumacher@kde.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kalarm.h"

#include <qfileinfo.h>

#include <kstandarddirs.h>
#include <klocale.h>
#include <kdebug.h>

#include "resourcelocal.moc"

using namespace KCal;


KAResourceLocal::KAResourceLocal(const KConfig* config)
	: AlarmResource(config),
	  mFileReadOnly(false)
{
	if (config)
		mURL = KURL(config->readPathEntry("CalendarURL"));
	init();
}

KAResourceLocal::KAResourceLocal(Type type, const QString& fileName)
	: AlarmResource(type),
	  mURL(KURL(fileName)),
	  mFileReadOnly(false)
{
	init();
}

void KAResourceLocal::init()
{
	setType("file");   // set resource type

	//setSavePolicy(SaveDelayed);  // unnecessary for KAlarm, and would override base class setting

	connect(&mDirWatch, SIGNAL(dirty(const QString&)), SLOT(reload()));
	connect(&mDirWatch, SIGNAL(created(const QString&)), SLOT(reload()));
	connect(&mDirWatch, SIGNAL(deleted(const QString&)), SLOT(reload()));
	mDirWatch.addFile(mURL.path());

	enableResource(isActive());
}

KAResourceLocal::~KAResourceLocal()
{
	mDirWatch.stopScan();
	if (isOpen())
		close();
}

void KAResourceLocal::writeConfig(KConfig* config)
{
	config->writePathEntry("CalendarURL", mURL.prettyURL());
	AlarmResource::writeConfig(config);
}

void KAResourceLocal::startReconfig()
{
	mNewURL = mURL;
	AlarmResource::startReconfig();
}

void KAResourceLocal::applyReconfig()
{
	if (mReconfiguring)
	{
		AlarmResource::applyReconfig();
		if (setFileName(mNewURL))
			mReconfiguring = 3;    // indicate that location has changed
		AlarmResource::applyReconfig();
	}
}

bool KAResourceLocal::readOnly() const
{
	return mFileReadOnly || AlarmResource::readOnly();
}

void KAResourceLocal::enableResource(bool enable)
{
	kdDebug(KARES_DEBUG) << "KAResourceLocal::enableResource(" << enable << "): " << mURL.path() << endl;
	if (enable)
	{
		lock(mURL.path());
		mDirWatch.startScan();
	}
	else
	{
		lock(QString::null);
		mDirWatch.stopScan();
	}
}

bool KAResourceLocal::doLoad(bool)
{
	if (!KStandardDirs::exists(mURL.path()))
	{
		kdDebug(KARES_DEBUG) << "KAResourceLocal::doLoad(): File doesn't exist yet." << endl;
		mLoaded = false;
		emit invalidate(this);
		mCalendar.close();
		clearChanges();
		updateCustomEvents(false);   // calendar is now empty
		if (!isActive())
			return false;
		mLoading = true;
		if (!doSave(true))   // save the empty calendar, to create the calendar file
		{
			mLoading = false;
			return false;
		}
		mFileReadOnly = false;
		setCompatibility(KCalendar::Current);
		mLoading = false;
		mLoaded = true;
		setReloaded(true);   // the resource has now been loaded at least once
		emit loaded(this);
		return true;
	}
	return loadFile();
}

void KAResourceLocal::reload()
{
	kdDebug(KARES_DEBUG) << "KAResourceLocal::reload(" << mURL.path() << ")" << endl;
	if (!isOpen())
		return;
	if (mLastModified == readLastModified())
	{
		kdDebug(KARES_DEBUG) << "KAResourceLocal::reload(): file not modified since last read." << endl;
		QFileInfo fi(mURL.path());
		mFileReadOnly = !fi.isWritable();
		return;
	}
	loadFile();
	emit resourceChanged(this);
}

bool KAResourceLocal::loadFile()
{
	kdDebug(KARES_DEBUG) << "KAResourceLocal::loadFile(" << mURL.path() << ")" << endl;
	mLoaded = false;
	emit invalidate(this);
	mCalendar.close();
	clearChanges();
	if (!isActive())
	{
		updateCustomEvents(false);   // calendar is now empty
		return false;
	}
	mLoading = true;
	disableChangeNotification();
	bool success = mCalendar.load(mURL.path());
	enableChangeNotification();
	if (!success)
	{
		mLoading = false;
		updateCustomEvents();
		return false;
	}
	mLastModified = readLastModified();
	QFileInfo fi(mURL.path());
	mFileReadOnly = !fi.isWritable();
	checkCompatibility(fileName());
	mLoading = false;
	updateCustomEvents();
	mLoaded = true;
	setReloaded(true);   // the resource has now been loaded at least once
	emit loaded(this);
	return true;
}

bool KAResourceLocal::doSave(bool)
{
	kdDebug(KARES_DEBUG) << "KAResourceLocal::doSave(" << mURL.path() << ")" <<endl;
	if (mCalIDFunction)
		(*mCalIDFunction)(mCalendar);    // write the application ID into the calendar
	bool success = mCalendar.save(mURL.path());
	clearChanges();
	mLastModified = readLastModified();
	emit resourceSaved(this);
	return success;
}

QDateTime KAResourceLocal::readLastModified()
{
	QFileInfo fi(mURL.path());
	return fi.lastModified();
}

QString KAResourceLocal::fileName() const
{
	return mURL.path();
}

bool KAResourceLocal::setFileName(const KURL& newURL)
{
	if (mReconfiguring == 1)
	{
		mNewURL = newURL;
		return true;
	}
	if (newURL.path() == mURL.path()  ||  !newURL.isLocalFile())
		return false;
	kdDebug(KARES_DEBUG) << "KAResourceLocal::setFileName(" << newURL.path() << ")\n";
	if (isOpen())
		close();
	bool active = isActive();
	if (active)
		enableResource(false);
	mDirWatch.removeFile(mURL.path());
	mURL = newURL;
	mDirWatch.addFile(mURL.path());
	if (active)
		enableResource(true);
	// Trigger loading the new resource, and ensure that the new configuration is saved
	emit locationChanged(this);
	return true;
}

bool KAResourceLocal::setLocation(const QString& fileName, const QString&)
{
	KURL newURL;
	newURL.setPath(fileName);
	return setFileName(newURL);
}

QString KAResourceLocal::displayLocation() const
{
	return mURL.path();
}

QString KAResourceLocal::displayType() const
{
	return i18n("File");
}
