/***************************************************************************
 *   Copyright (C) 2006 by Meni Livne <livne@kde.org>                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "plugin_karpion.h"
#include "karpionconfigdlgimpl.h"

#include <string.h>

#include <qlabel.h>
#include <qtimer.h>

#include <kgenericfactory.h>
#include <khtmlview.h>
#include <khtml_part.h>
#include <kstandarddirs.h>
#include <kprotocolmanager.h>
#include <kconfig.h>
#include <kaction.h>
#include <klocale.h>
#include <kglobal.h>
#include <kparts/mainwindow.h>
#include <ktoolbarlabelaction.h>
#include <kaboutapplication.h>

#include <kdialogbase.h>


#define KARPION_VERSION "0.1.0"

typedef KGenericFactory<KarpionPlugin> KarpionPluginFactory;
K_EXPORT_COMPONENT_FACTORY(libkarpionplugin, KarpionPluginFactory("karpion"))


enum {KarpionXMLUpdateEvent = QEvent::User,
      KarpionCheckURLEvent,
      KarpionCheckCountryEvent };


QCache<KarpionURLData> KarpionPlugin::m_url_cache;
KConfig *KarpionPlugin::m_config;
bool KarpionPlugin::m_config_read = false;
phish_mode_t KarpionPlugin::m_running_mode;
bool KarpionPlugin::m_check_country;
int KarpionPlugin::m_update_interval;
QTimer KarpionPlugin::m_update_timer;


KarpionPlugin::KarpionPlugin(QObject* parent, const char* name,
                             const QStringList&) : Plugin(parent, name),
                                                   m_url_data(NULL),
                                                   m_part(0),
                                                   xml_thread(this),
                                                   url_thread(this),
                                                   country_thread(this)
{
  if (!parent->inherits("KHTMLPart"))
  {
    return;
  }
  
  setInstance(KarpionPluginFactory::instance());
  
  m_part = dynamic_cast<KHTMLPart *>(parent);
  
  m_about_data = new KAboutData("karpion", I18N_NOOP("Konqueror Anti-"
                                "Phishing Extension"), KARPION_VERSION,
                                I18N_NOOP("A Konqueror toolbar extension to "
                                          "prevent phishing."),
                                KAboutData::License_GPL,
                                I18N_NOOP("(c) 2006, Meni Livne"),
                                I18N_NOOP("Part of the Open Phishing "
                                          "Database Project."),
                                "http://opdb.berlios.de/");
  
  m_about_data->addAuthor("Meni Livne", 0, "livne@kde.org");
  m_about_data->addCredit("Boaz Anin", I18N_NOOP("libphish co-author"),
                          "boazanin@gmail.com");
  m_about_data->addCredit("Shahar Karin", I18N_NOOP("libphish co-author"),
                          "shaharka@post.tau.ac.il");
  
  m_aa = new KAboutApplication(m_about_data, 0, 0, false);
  
  m_menu = new KActionMenu(i18n("Anti-Phishing"), "karpion",
                           actionCollection(), "karpion_sitemenu");
  m_menu->setDelayed(false);
  
  m_reportphishing = new KAction(i18n("Report This Site As Phishing..."), 0,
                                 this, SLOT(reportPhishing()),
                                 actionCollection(), "karpion_reportphishing");
  m_addsafelist = new KAction(i18n("Add Site To Safe List..."), 0,
                                   this, SLOT(addToSafeList()),
                                   actionCollection(), "karpion_addsafelist");
  m_configure = new KAction(i18n("Configure Phishing Toolbar..."), "configure",
                            0, this, SLOT(configure()), actionCollection(),
                            "karpion_configure");
  m_about = new KAction(i18n("About Konqueror Phishing Toolbar"), "karpion",
                        0, this, SLOT(about()), actionCollection(),
                        "karpion_about");
  
  m_menu->insert(m_reportphishing);
  m_menu->insert(m_addsafelist);
  m_menu->insert(new KActionSeparator(actionCollection()));
  m_menu->insert(m_configure);
  m_menu->insert(new KActionSeparator(actionCollection()));
  m_menu->insert(m_about);
  
  m_risk = new KToolBarLabelAction("", 0, 0, 0, actionCollection(),
                                   "karpion_risklabel");
  
  m_country = new KToolBarLabelAction("", 0, 0, 0, actionCollection(),
                                      "karpion_countrylabel");
  
  m_siteinfo = new KAction(i18n("Site Information..."), 0, this,
                           SLOT(siteInfo()), actionCollection(),
                           "karpion_siteinfo");
  
  m_url_cache.setAutoDelete(true);
  
  clear();
  
  QTimer::singleShot(0, this, SLOT(delayedSetup()));
}

KarpionPlugin::~KarpionPlugin()
{
  phish_shutdown();
}

void KarpionPlugin::readConfig()
{
  m_config->setGroup("General");
  
  m_check_country = (m_config->readEntry("CheckCountry", "0").toInt() != 0);
  m_update_interval = m_config->readEntry("UpdateInterval", "30").toInt();
  
  phish_runningMode(&m_running_mode);
  
  if (m_running_mode == PHISH_ONLINE_MODE)
    m_update_timer.stop();
  else
  {
    updateXML();
    m_update_timer.start(m_update_interval * 60 * 1000);
  }
}

void KarpionPlugin::delayedSetup()
{
  phish_result_t r;
  
  if (m_part == 0)
    return;
  
  KMainWindow *mainwin = (KMainWindow *)m_part->view()->topLevelWidget();
  
  KToolBar *toolbar = mainwin->toolBar("karpionToolBar");
  
  toolbar->setIconText(KToolBar::IconTextRight);
  
  QString ver = QString("Karpion/%1").arg(KARPION_VERSION);
  r = phish_init(KProtocolManager::defaultUserAgent().ascii(), ver.ascii());
  if (r != PHISH_SUCCESS)
    kdDebug() << "phish_init() result = " << r << endl;
  
  connect(m_part, SIGNAL(docCreated()), this, SLOT(checkCurrentURL()));
  
  connect(&m_update_timer, SIGNAL(timeout()), SLOT(updateXML()));
  
  if (!m_config_read)
  {
    m_config_read = true;
    
    m_config = new KConfig("karpionrc", false, false);
    readConfig();
  }
}

void KarpionPlugin::customEvent(QCustomEvent *ev)
{
  switch(ev->type())
  {
    case KarpionXMLUpdateEvent:
      break;
      
    case KarpionCheckURLEvent:
      m_url_data = new phish_url_data_t;
      *m_url_data = url_thread.urlData();
      m_url_cache.insert(url_thread.url(), new KarpionURLData(m_url_data));
      
      setRiskLevel(m_url_data->risk_level);
      m_siteinfo->setEnabled(true);
      
      if (m_check_country)
        setCountry(m_url_data->country);
      
      break;
      
    case KarpionCheckCountryEvent:
      if (m_safe == 1)
      {
        m_url_data = new phish_url_data_t;
        *m_url_data = country_thread.urlData();
        m_url_cache.insert(country_thread.url(), new KarpionURLData(m_url_data));
      }
      else
      {
        strncpy(m_url_data->country, country_thread.urlData().country, 2);
        m_url_data->country[2] = '\0';
      }
      
      m_siteinfo->setEnabled(true);
      setCountry(m_url_data->country);
      
      break;
      
    default:
      break;
  }
}

void KarpionPlugin::checkCurrentURL()
{
  KarpionURLData *url_d;
  phish_result_t r;
  QString url;
  int safe;
  
  if (m_check_country)
  {
    m_country->setEnabled(true);
  }
  else
  {
    clearCountry();
    m_country->setEnabled(false);
  }
  
  KURL kurl(m_part->url());
  if (!kurl.isValid() || kurl.host().isEmpty())
  {
    clear();
    return;
  }
  
  url = kurl.url();
  
  r = phish_checkSafeList(url.ascii(), &safe);
  m_safe = safe;
  
  if (m_safe)
  {
    if (m_check_country)
    {
      if ((url_d = m_url_cache.find(url)) != 0)
      {
        m_url_data = url_d->data();
        setCountry(m_url_data->country);
        m_siteinfo->setEnabled(true);
      }
      else
      {
        checkCountry(url);
      }
    }
    
    setRiskLevel(PHISH_RISK_NONE);
  }
  else
  {
    if ((url_d = m_url_cache.find(url)) != 0)
    {
      m_url_data = url_d->data();
      setRiskLevel(m_url_data->risk_level);
      m_siteinfo->setEnabled(true);
      
      if (m_check_country)
        setCountry(m_url_data->country);
    }
    else
    {
      if (m_running_mode == PHISH_OFFLINE_MODE)
      {
        m_url_data = new phish_url_data_t;
        r = phish_checkURL(url.ascii(), m_url_data);
        
        m_url_cache.insert(url, new KarpionURLData(m_url_data));
        
        setRiskLevel(m_url_data->risk_level);
        m_siteinfo->setEnabled(true);
        
        if (m_check_country)
          checkCountry(url);
      }
      else
      {
        checkURL(url);
      }
    }
  }
}

void KarpionPlugin::checkURL(const QString& url)
{
  setCheckingRiskLevel();
  m_siteinfo->setEnabled(false);
  
  if (m_check_country)
    setCheckingCountry();
  
  url_thread.setURL(url);
  url_thread.start();
}

void KarpionPlugin::checkCountry(const QString& url)
{
  setCheckingCountry();
  country_thread.setURL(url);
  country_thread.start();
}

void KarpionPlugin::updateXML()
{
  xml_thread.start();
}

void KarpionPlugin::setRiskLevel(phish_risk_t level)
{
  QString text = "<nobr>";
  text += i18n("Risk Level:");
  text += " ";
  
  switch(level)
  {
    case PHISH_RISK_UNKNOWN:
      text += i18n("<b>Unknown</b>");
      break;
    case PHISH_RISK_NONE:
      text += i18n("<font color=darkgreen><b>None</b></font>");
      break;
    case PHISH_RISK_LOW:
      text += i18n("<b>Low</b>");
      break;
    case PHISH_RISK_MEDIUM:
      text += i18n("<font color=orange><b>Medium</b></font>");
      break;
    case PHISH_RISK_HIGH:
      text += i18n("<font color=red><b>High</b></font>");
      break;
  }
  
  text += "</nobr>";
  
  m_risk->setText(text);
}

void KarpionPlugin::setCheckingRiskLevel()
{
  QString risk_text = i18n("Risk Level:");
  QString checking_text = i18n("Checking...");
  m_risk->setText(QString("<nobr>%1 <font color=darkgrey><b>%2</b></font>")
                                      .arg(risk_text).arg(checking_text));
}

void KarpionPlugin::setCountry(const QString& country_code)
{
  if (country_code.length() != 2)
    clearCountry();
  
  QString full_name = KGlobal::locale()->twoAlphaToCountryName(country_code);
  if (!full_name.isEmpty())
  {
    QString flag_file = locate("locale", QString("l10n/%1/flag.png")
        .arg(country_code.lower()));
    
    QString text = "<nobr>";
    text += i18n("Country");
    m_country->setText(text + QString(":&nbsp;<img src=\"%1\"/>&nbsp;%2</nobr>")
                                              .arg(flag_file).arg(full_name));
  }
  else
  {
    clearCountry();
  }
}

void KarpionPlugin::clearCountry()
{
  m_country->setText("<nobr>" + i18n("Country: Unknown") + "</nobr>");
}

void KarpionPlugin::setCheckingCountry()
{
  QString text = "<nobr>";
  text += i18n("Country");
  m_country->setText(text + QString(":&nbsp;%1</nobr>").arg(i18n("Checking...")));
}

void KarpionPlugin::clear()
{
  m_url_data = NULL;
  m_siteinfo->setEnabled(false);
  setRiskLevel(PHISH_RISK_UNKNOWN);
  clearCountry();
}

void KarpionPlugin::reportPhishing()
{
  char *url;
  phish_result_t r;
  
  KURL kurl(m_part->url());
  if (!kurl.isValid() || kurl.host().isEmpty())
    return;
  
  r = phish_getReportingURL(kurl.url(), &url);
  
  if (r != PHISH_SUCCESS)
  {
    // handle error
  }
  
  m_part->browserExtension()->createNewWindow(KURL(url));
  
  free(url);
}

void KarpionPlugin::addToSafeList()
{
  phish_result_t r;
  
  KURL kurl(m_part->url());
  if (!kurl.isValid() || kurl.host().isEmpty())
    return;
  
  r = phish_addToCurrentSafeList(kurl.url());
  
  if (r != PHISH_SUCCESS)
  {
    // handle error
  }
  
  // update display now that site is safe
  checkCurrentURL();
}

void KarpionPlugin::siteInfo()
{
  if (m_url_data == NULL)
    return;
  
  QString text;
  
  KDialogBase *dlg = new KDialogBase(0, 0, true, i18n("Site Information"),
                                     KDialogBase::Close);
  QLabel *label = new QLabel(dlg);
  dlg->setMainWidget(label);
  
  text += QString("<nobr>%1</nobr><br><br>")
                                  .arg(i18n("Site Information For %1")
                                   .arg(m_part->url().url()));
  
  if (m_safe)
  {
    text += i18n("Site is in safe list.<br>");
  }
  else
  {
    QString risk;
    
    switch (m_url_data->risk_level)
    {
      case PHISH_RISK_UNKNOWN:
        risk = i18n("Unknown");
        break;
      case PHISH_RISK_NONE:
        risk = i18n("None");
        break;
      case PHISH_RISK_LOW:
        risk = i18n("Low");
        break;
      case PHISH_RISK_MEDIUM:
        risk = i18n("Medium");
        break;
      case PHISH_RISK_HIGH:
        risk = i18n("High");
        break;
    }
    
    text += i18n("Risk Level: %1<br>").arg(risk);
    
    if (m_url_data->server == -1)
    {
      if (m_url_data->ip == 1 && m_url_data->path == 1)
        text += i18n("Complete address of site is in database.<br>");
      else if (m_url_data->ip == 1)
        text += i18n("IP address of site is in database.<br>");
    }
    else
    {
      if (m_url_data->server == 1 && m_url_data->path == 1)
      {
        text += i18n("Complete address of site is in database.<br>");
        
        if (m_url_data->ip == 1)
          text += i18n("IP address of site is in database.<br>");
      }
      else
      {
        if (m_url_data->server == 1)
          text += i18n("Hostname of site is in database.<br>");
        if (m_url_data->domain == 1)
          text += i18n("Domain name of site is in database.<br>");
        if (m_url_data->path == 1)
          text += i18n("Path of site is in database.<br>");
        if (m_url_data->ip == 1)
          text += i18n("IP address of site is in database.<br>");
      }
    }
    
    if (m_url_data->user_scheme == 1)
      text += i18n("Site address has user@host scheme.<br>");
    
    if (m_url_data->suspicious_host == 1)
      text += i18n("Hostname of site contains suspicious characters.<br>");
    
    if (m_url_data->comments_length > 0)
    {
      text += "<br>";
      text += i18n("Comments:");
      text += "<br>";
      text += m_url_data->comments;
    }
  }
  
  label->setText(text);
  dlg->show();
}

void KarpionPlugin::configure()
{
  KarpionConfigDialog *dlg = new KarpionConfigDialog(m_config);
  
  if (dlg->exec() == QDialog::Accepted)
  {
    readConfig();
  }
  
  delete dlg;
}

void KarpionPlugin::about()
{
  m_aa->show();
}

KarpionURLData::KarpionURLData(phish_url_data_t *url_data)
{
  m_url_data = url_data;
}

KarpionURLData::~KarpionURLData()
{
  phish_deleteURLData(m_url_data);
  delete m_url_data;
}


void KarpionXMLUpdateThread::run()
{
  m_result = phish_downloadDBAsXML();
  QCustomEvent *ev = new QCustomEvent(KarpionXMLUpdateEvent);
  QApplication::postEvent(m_caller, ev);
  exit();
}

void KarpionCheckURLThread::run()
{
  m_result = phish_checkURL(m_url.ascii(), &m_url_data);
  QCustomEvent *ev = new QCustomEvent(KarpionCheckURLEvent);
  QApplication::postEvent(m_caller, ev);
  exit();
}

void KarpionCheckCountryThread::run()
{
  m_result = phish_checkCountry(m_url.ascii(), &m_url_data);
  QCustomEvent *ev = new QCustomEvent(KarpionCheckCountryEvent);
  QApplication::postEvent(m_caller, ev);
  exit();
}


#include "plugin_karpion.moc"

