/***************************************************************************
 *   Copyright (C) 2005 by Roberto Cappuccio and the Kat team              *
 *   Roberto Cappuccio : roberto.cappuccio@gmail.com                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           *
 ***************************************************************************/

#include <kdialog.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kstandarddirs.h>
#include <kurl.h>
#include <klocale.h>
#include <qpushbutton.h>
#include <qtabwidget.h>
#include <qwidget.h>
#include <qheader.h>
#include <qlistview.h>
#include <qlistbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <kinputdialog.h>
#include <kdebug.h>
#include <qdom.h>
#include <qfile.h>
#include <qcstring.h>
#include <qpainter.h>
#include <qvbox.h>
#include <unistd.h>

#include <keditlistbox.h>
#include "languagemanagement.h"
#include "katlanguagemanager.h"

class NGramItem : public QListViewItem
{
public:

    NGramItem( QListView* parent ) : QListViewItem( parent ) {};
    ~NGramItem() {};

    long occurrences;

    int compare( QListViewItem*, int, bool ) const;
    void paintCell( QPainter*, const QColorGroup&, int, int, int );
};

int NGramItem::compare( QListViewItem* other, int column, bool ascending ) const {
    if ( column == 1 )
    {
        if ( occurrences < static_cast<NGramItem*>(other)->occurrences )
            return -1;
        else if ( occurrences > static_cast<NGramItem*>(other)->occurrences )
            return 1;
        else
            return 0;
    }
    else
    {
        return QListViewItem::compare( other, column, ascending );
    }
}

void NGramItem::paintCell( QPainter* p,
                           const QColorGroup& cg,
                           int column,
                           int width,
                           int align )
{
        // Do the standard painting
        QListViewItem::paintCell( p, cg, column, width, align );
        // Draw a box around the Cell
        p->setPen( cg.color( QColorGroup::Mid ) );
        p->drawLine( 0, height() - 1, width - 1, height() - 1 );
        p->lineTo( width - 1, 0 );
}

languageManagement::languageManagement( QWidget* parent, const char* name )
   : QWidget( parent, name )
{
    QGridLayout* DlgLanguageManagementLayout = new QGridLayout( parent, 1, 1, 11, 6, "DlgLanguageManagementLayout" );

    QHBoxLayout* Layout1 = new QHBoxLayout( 0, 0, 6, "Layout1" );
    QSpacerItem* Horizontal_Spacing2 = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
    Layout1->addItem( Horizontal_Spacing2 );

    DlgLanguageManagementLayout->addMultiCellLayout( Layout1, 4, 4, 0, 1 );

    QPushButton* bAddLanguage = new QPushButton( parent, "bAddLanguage" );
    bAddLanguage->setText( i18n( "&Add..." ) );

    DlgLanguageManagementLayout->addWidget( bAddLanguage, 2, 0 );

    bDeleteLanguage = new QPushButton( parent, "bDeleteLanguage" );
    bDeleteLanguage->setText( i18n( "&Delete" ) );

    DlgLanguageManagementLayout->addWidget( bDeleteLanguage, 3, 0 );

    tabWidget = new QTabWidget( parent, "tabWidget" );

    Widget2 = new QWidget( tabWidget, "Widget2" );

    updateGUI( false );

    bDeleteLanguageProfile = new QPushButton( Widget2, "bDeleteLanguageProfile" );
    bDeleteLanguageProfile->setGeometry( QRect( 270, 40, 112, 24 ) );
    bDeleteLanguageProfile->setText( i18n( "&Delete" ) );

    lvLanguageProfile = new QListView( Widget2, "lvLanguageProfile" );
    lvLanguageProfile->addColumn( i18n( "NGram" ) );
    lvLanguageProfile->addColumn( i18n( "Occurrences" ) );
    lvLanguageProfile->setShowSortIndicator( true );
    lvLanguageProfile->setColumnAlignment( 0, Qt::AlignLeft );
    lvLanguageProfile->setColumnAlignment( 1, Qt::AlignRight );
    lvLanguageProfile->setGeometry( QRect( 10, 10, 250, 260 ) );
    lvLanguageProfile->setSorting( 1, false );
    lvLanguageProfile->setResizeMode( QListView::LastColumn );
    lvLanguageProfile->header()->setLabel( 0, i18n( "NGram" ) );
    lvLanguageProfile->header()->setLabel( 1, i18n( "Occurrences" ) );

    QPushButton* bCreateLanguageProfile = new QPushButton( Widget2, "bCreateLanguageProfile" );
    bCreateLanguageProfile->setGeometry( QRect( 270, 10, 112, 24 ) );
    bCreateLanguageProfile->setText( i18n( "&Create..." ) );
    tabWidget->insertTab( Widget2, QString::fromLatin1("") );

    Widget3 = new QWidget( tabWidget, "Widget3" );
    Widget3->setEnabled( false );

    QVBoxLayout* lay = new QVBoxLayout( Widget3 );

    lbExclusions= new KEditListBox(Widget3, 0L, false, KEditListBox::Add|KEditListBox::Remove );
    lay->addWidget( lbExclusions );
    tabWidget->insertTab( Widget3, QString::fromLatin1( "" ) );

    DlgLanguageManagementLayout->addMultiCellWidget( tabWidget, 0, 3, 1, 1 );

    QLabel* tlLanguages = new QLabel( parent, "tlLanguages" );

    DlgLanguageManagementLayout->addWidget( tlLanguages, 0, 0 );

    lbLanguages = new QListBox( parent, "lbLanguages" );
    lbLanguages->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)0,
                (QSizePolicy::SizeType)7, 0, 0, lbLanguages->sizePolicy().hasHeightForWidth() ) );
    lbLanguages->setSelectionMode( QListBox::Single );

    // Global + local schemes
    m_languageFiles = KGlobal::dirs()->findAllResources( "data", "kat/language/*.klp", false, true );

    QStringList deletedLanguageList = KGlobal::dirs()->findAllResources( "data", "kat/language/*.klpd", false, true );
    QStringList deletedFileLanguage;
    QStringList::Iterator end(  deletedLanguageList.end() );
    for ( QStringList::Iterator it = deletedLanguageList.begin(); it != end; ++it )
    {
        KURL file( *it );
        QString tmp =file.filename().mid( 0, file.filename().length()-5 );
        kdDebug()<< "initialise tmp :" << tmp << endl;
        deletedFileLanguage.append( tmp );
    }

    QStringList::Iterator endLang( m_languageFiles.end() );
    for ( QStringList::Iterator it = m_languageFiles.begin(); it != endLang; ++it )
    {
        KURL file( *it );
        QString lname = (file.filename()).mid( 0, (file.filename()).length()-4 );
        //it was removed => don't load it
        if ( deletedFileLanguage.contains( lname ) )
            continue;
        lbLanguages->insertItem( lname );
    }
    lbLanguages->sort();

    DlgLanguageManagementLayout->addWidget( lbLanguages, 1, 0 );
    tabWidget->changeTab( Widget2, i18n( "Language Profile" ) );
    tabWidget->changeTab( Widget3, i18n( "E&xclusion List" ) );
    tlLanguages->setText( i18n( "Languages" ) );

    // signals and slots connections
    connect( bAddLanguage, SIGNAL( clicked() ),
             this, SLOT( slotAddLanguage() ) );
    connect( bDeleteLanguage, SIGNAL( clicked() ),
             this, SLOT( slotDeleteLanguage() ) );
    connect( bCreateLanguageProfile, SIGNAL( clicked() ),
             this, SLOT( slotCreateProfile() ) );
    connect( bDeleteLanguageProfile, SIGNAL( clicked() ),
             this, SLOT( slotDeleteProfile() ) );
    connect( lbExclusions, SIGNAL( added( const QString &) ),
             this, SLOT( slotAddExclusion(const QString&) ) );
    //connect( bModifyExclusion, SIGNAL( clicked() ), this, SLOT( modifyExclusion() ) );
    connect( lbExclusions, SIGNAL( removed( const QString &  ) ),
             this, SLOT( slotDeleteExclusion(const QString&) ) );
    connect( lbLanguages, SIGNAL( selectionChanged( QListBoxItem* ) ),
             this, SLOT( slotSelectLanguage( QListBoxItem* ) ) );
}

languageManagement::~languageManagement()
{
}

void languageManagement::slotAddLanguage()
{
    bool ok;
    QString text = KInputDialog::getText( i18n( "Name of Language" ),
                                          i18n( "Enter the name of the language:" ),
                                          QString::null,
                                          &ok,
                                          this );
    //test if we have already this language
    QString newLanguage = text + ".klp";
    if ( m_languageFiles.contains( newLanguage ) )
    {
        KMessageBox::error( this, i18n( "Sorry this language already exists." ) );
        return;
    }
    if ( ok )
    {
        lbLanguages->insertItem( text );
        lbLanguages->sort();
    }
}

void languageManagement::slotDeleteLanguage()
{
    int result = KMessageBox::warningContinueCancel( this,
                                             i18n( "Do you really want to delete the current language?" ),
                                             i18n( "Delete Language" ), KStdGuiItem::del() );
    if ( result == KMessageBox::Continue )
    {
            QString currentLanguage = lbLanguages->currentText();
            QString profilePath = KGlobal::dirs()->saveLocation("data", "kat/language/") + currentLanguage + ".klpd";
            unlink( QFile::encodeName( KGlobal::dirs()->saveLocation("data", "kat/language/") + currentLanguage + ".klp" ) );
            QFile file( profilePath );
            if( file.open( IO_WriteOnly ) )
            {
                QTextStream ts( &file );
                ts << "deleted language";
                file.close();
            }
            lbLanguages->removeItem( lbLanguages->currentItem() );

            lvLanguageProfile->clear();

            if ( lbLanguages->count() == 0 )
                updateGUI( false );
    }
}

void languageManagement::slotCreateProfile()
{
    KURL fn = KFileDialog::getOpenURL( QString::null,
                                       "*.txt|" +i18n( "Text Files (*.txt)" ),
                                       this,
                                       i18n( "Select Sample Text in Specified Language" ) );

    if ( !fn.isEmpty() )
    {
        QString currentLanguage = lbLanguages->currentText();
        QString sampleFile = fn.path();

        QFile sf( sampleFile );
        if ( !sf.exists() )
            return;

        NGramsList nl = KatLanguageManager::createFingerprintFromFile( sampleFile );
        if ( !nl.isEmpty() )
        {
            lvLanguageProfile->clear();

            // display the language profile
            NGram* n = nl.first();
            for ( int i = 0; i < MAXNGRAMS; i++ )
            {
                NGramItem* lvItem = new NGramItem( lvLanguageProfile );
                lvItem->occurrences = n->occurrences;

                lvItem->setText( 0, n->ngram );
                lvItem->setText( 1, QString::number( n->occurrences ) );

                n = nl.next();
            }

            // save the language profile
            QString profilePath = KGlobal::dirs()->saveLocation( "data", "kat/language/" ) + currentLanguage + ".klp";

            kdDebug()<<" profilePath :"<<profilePath<<endl;
            // delete the xml file if it already exists
            unlink( QFile::encodeName( profilePath ) );
            unlink( QFile::encodeName( KGlobal::dirs()->saveLocation( "data", "kat/language/" ) + currentLanguage + ".klpd" ) );
            // save the xml file
            QDomDocument doc;
            // add the <?xml> line
            QDomProcessingInstruction instr = doc.createProcessingInstruction( "xml", "version=\"1.0\" encoding=\"UTF-8\" ");
            doc.appendChild( instr );

            // add the <!DOCTYPE> line
            QDomImplementation impl;
            QDomDocumentType type = impl.createDocumentType( "klp", 0, "klp.dtd" );
            doc.appendChild(type);

            // create the root
            QDomElement root = doc.createElement( "ngrams" );
            doc.appendChild( root );

            // insert the elements
            NGramsList::Iterator end( nl.end() );
            for ( NGramsList::Iterator it = nl.begin(); it != end; ++it )
            {
                NGram* currentFileNGram = *it;

                QDomElement tag = doc.createElement( "ngram" );
                tag.setAttribute( "value", currentFileNGram->ngram );
                tag.setAttribute( "occurrences", currentFileNGram->occurrences );
                root.appendChild( tag );
            }

            QFile file( profilePath );
            if( file.open( IO_WriteOnly ) )
            {
                QTextStream ts( &file );
                ts << doc.toString();

                file.close();

                KMessageBox::information( this, i18n( "The language profile %1 has been correctly saved" ).arg( profilePath ) );
            }
            else
            {
                KMessageBox::information( this, i18n( "The language profile %1 has NOT been saved" ).arg( profilePath ) );
            }
        }
    }
}

void languageManagement::slotDeleteProfile()
{
    int result = KMessageBox::warningContinueCancel( this,
                                             i18n( "Do you really want to delete the current language profile?" ),
                                             i18n( "Delete Language Profile" ), KStdGuiItem::del() );

    switch (result)
    {
        case KMessageBox::Continue:
            QString currentLanguage = lbLanguages->currentText();
            QString profilePath = KGlobal::dirs()->saveLocation("data", "kat/language/") + currentLanguage + ".klpd";
            unlink( QFile::encodeName( KGlobal::dirs()->saveLocation("data", "kat/language/") + currentLanguage + ".klp" ) );
            QFile file( profilePath );
            if( file.open( IO_WriteOnly ) )
            {
                QTextStream ts( &file );
                ts << "deleted language";
                file.close();
            }
            m_languageFiles.remove( profilePath );
            lvLanguageProfile->clear();
    }
}

void languageManagement::slotAddExclusion( const QString& )
{
}

void languageManagement::slotDeleteExclusion( const QString& )
{
}

void languageManagement::slotSelectLanguage( QListBoxItem* item )
{
    if ( item )
    {
        lvLanguageProfile->clear();
        updateGUI( true );

        // load the klp file
        QString profilePath = locate( "data", "kat/language/" + item->text() + ".klp" );
        QDomDocument doc( profilePath );

        QFile file( profilePath );
        if ( !file.exists() )
            return;
        if ( !file.open( IO_ReadOnly ) )
        {
            kdDebug() << "Impossible to open " << profilePath << endl;
            return;
        }
        QByteArray m_data = file.readAll();

        QString qs;
        if ( !doc.setContent( QString( m_data ).utf8(), &qs ) )
        {
            kdDebug() << "Impossible to set content from " << profilePath << " ERROR: " << qs << endl;
            file.close();
            return;
        }
        file.close();

        // insert all ngrams in lvLanguageProfile
        QDomElement docElem = doc.documentElement();
        QDomNode n = docElem.firstChild();

        while( !n.isNull() )
        {
            QDomElement e = n.toElement(); // try to convert the node to an element.
            if( !e.isNull() )
            {
                NGramItem* lvItem = new NGramItem( lvLanguageProfile );
                lvItem->occurrences = QString( e.attribute( "occurrences" ) ).toLong();

                lvItem->setText( 0, e.attribute( "value" ) );
                lvItem->setText( 1, e.attribute( "occurrences" ) );
            }
            n = n.nextSibling();
        }

    }
    else
    {
        updateGUI( false );
    }
}

void languageManagement::updateGUI( bool _b )
{
    tabWidget->setEnabled( _b );
    Widget2->setEnabled( _b );
    bDeleteLanguage->setEnabled( _b );
}

#include "languagemanagement.moc"
