/***************************************************************************
 *   Copyright (C) 2005 by Roberto Cappuccio and the Kat team              *
 *   Roberto Cappuccio : roberto.cappuccio@gmail.com                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           *
 ***************************************************************************/

#include <kdatastream.h>
#include <kfileitem.h>
#include <qtimer.h>
#include <kdebug.h>
#include <kfilemetainfo.h>
#include <kio/global.h>
#include <kio/jobclasses.h>
#include <kio/previewjob.h>
#include <kio/metainfojob.h>
#include <kparts/componentfactory.h>

#include "katfulltextjob.h"

using namespace KIO;

struct KatFullTextJobPrivate
{
    KFileItemList items; // all the items we got
    KFileItemListIterator* currentItem; // argh! No default constructor
    bool deleteItems; // Delete the KFileItems when done?
    bool succeeded; // if the current item is ok
    const QStringList* enabledPlugins; // plugins to be used
};

KatFullTextJob::KatFullTextJob( const KFileItemList& items,
                                const QStringList* enabledPlugins,
                                bool deleteItems ) : KIO::Job( false )
{
    d = new KatFullTextJobPrivate;
    d->deleteItems = deleteItems;
    d->succeeded = false;
    d->items = items;
    d->currentItem = new KFileItemListIterator( d->items );
    d->enabledPlugins = enabledPlugins;

    d->items.setAutoDelete( deleteItems );

    if ( d->currentItem->isEmpty() )
    {
        emitResult();
        return;
    }

    QTimer::singleShot( 0, this, SLOT( start() ) );
}

KatFullTextJob::~KatFullTextJob()
{
    kdDebug() << " KatFullTextJob::~KatFullTextJob()" << endl;
    delete d->currentItem;
    delete d;
}

void KatFullTextJob::start()
{
    // TODO: implement the management of enabledPlugins (look at KIO::PreviewJob)
    getFullText();
}

void KatFullTextJob::removeItem( const KFileItem* item )
{
    if ( d->currentItem->current() == item )
    {
        subjobs.first()->kill();
        subjobs.removeFirst();
        determineNextFile();
    }

    d->items.remove( d->items.find( item ) );
}

void KatFullTextJob::determineNextFile()
{
    if ( d->currentItem->atLast() )
    {
        emitResult();
        return;
    }

    ++( *d->currentItem );
    d->succeeded = false;

    getFullText();
}

void KatFullTextJob::slotResult( KIO::Job *job )
{
    subjobs.remove( job );
    Q_ASSERT( subjobs.isEmpty() ); // We should have only one job at a time ...

    determineNextFile();
}

void KatFullTextJob::getFullText()
{
    Q_ASSERT( !d->currentItem->isEmpty() );

    KURL URL;
    URL.setProtocol( "fulltext" );
    URL.setPath( d->currentItem->current()->url().path() );

    KIO::TransferJob* job = KIO::get( URL, false, false );
    addSubjob( job );

    connect( job,  SIGNAL( data( KIO::Job*, const QByteArray& ) ),
             this, SLOT( slotFullText( KIO::Job *, const QByteArray& ) ) );

    job->addMetaData( "mimeType", d->currentItem->current()->mimetype() );
}

void KatFullTextJob::slotFullText( KIO::Job* job, const QByteArray &data )
{

    int len = data.size();
    char* cUTF8 = new char[ len + 1 ];
    cUTF8 = (char*)memcpy( cUTF8, data.data(), len );
    cUTF8[ len ]= '\0';

    kdDebug() << "data = " << cUTF8 << endl;

    QString fullText = QString::fromUtf8( cUTF8 );
/*
    kdDebug() << "%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%" << endl;
    kdDebug() << fullText << endl;
    kdDebug() << "%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%" << endl;
*/
    emit gotFullText( job, d->currentItem->current(), fullText );
    d->succeeded = true;
}

QStringList KatFullTextJob::availablePlugins()
{
    QStringList result;
    KTrader::OfferList plugins = KTrader::self()->query( "FulltextExtractor" );
    KTrader::OfferList::ConstIterator end( plugins.end() );
    for ( KTrader::OfferList::ConstIterator it = plugins.begin(); it != end; ++it )
    {
        kdDebug()<<" (*it)->desktopEntryName() :"<<(*it)->desktopEntryName()<<endl;
        result.append( (*it)->name() );
    }
    return result;
}

QStringList KatFullTextJob::supportedMimeTypes()
{
    QStringList result;
    KTrader::OfferList plugins = KTrader::self()->query( "FulltextExtractor" );
    KTrader::OfferList::ConstIterator end( plugins.end() );
    for ( KTrader::OfferList::ConstIterator it = plugins.begin(); it != end; ++it )
        result += (*it)->property( "MimeTypes" ).toStringList();

    return result;
}

KatFullTextJob* fileFullText( const KFileItemList &items )
{
    return new KatFullTextJob( items, NULL, false );
}

KatFullTextJob* fileFullText( const KURL::List &items )
{
    KFileItemList fileItems;
    KURL::List::ConstIterator end( items.end() );
    for ( KURL::List::ConstIterator it = items.begin(); it != end; ++it)
        fileItems.append(new KFileItem( KFileItem::Unknown, KFileItem::Unknown, *it, true) );

    return new KatFullTextJob( fileItems, NULL, true );
}

#include "katfulltextjob.moc"

