/***************************************************************************
 *   Copyright (C) 2005 by Roberto Cappuccio and the KAT Team              *
 *   Roberto Cappuccio : roberto.cappuccio@gmail.com                       *
 *                                                                         *
 *   Portions of the code contained in this file have been taken from      *
 *   CppSqlite3 - Copyright (C) 2004 Rob Groves                            *
 *   Rob Groves : rob.groves@btinternet.com                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <cstdlib>
#include <kdebug.h>
#include <qcstring.h>
#include <qdatastream.h>

#include "qtsqlite3.h"

// Named constant for passing to QtSQLite3Exception when passing it a string
// that cannot be deleted.
static const bool DONT_DELETE_MSG = false;

// Prototypes for SQLite functions not included in SQLite DLL, but copied below
// from SQLite encode.c
int sqlite3_encode_binary( const unsigned char *in, int n, unsigned char *out );
int sqlite3_decode_binary( const unsigned char *in, unsigned char *out );

/*
    QtSQLite3Exception
*/

QtSQLite3Exception::QtSQLite3Exception( const int nErrCode,
                                        char* szErrMess,
                                        bool bDeleteMsg ) : mnErrCode( nErrCode )
{
    mpszErrMess = sqlite3_mprintf( "%s[%d]: %s",
                                   errorCodeAsString( nErrCode ),
                                   nErrCode,
                                   szErrMess ? szErrMess : "" );

    if ( bDeleteMsg && szErrMess )
        sqlite3_free( szErrMess );
}

QtSQLite3Exception::QtSQLite3Exception( const QtSQLite3Exception& e ) : mnErrCode( e.mnErrCode )
{
    mpszErrMess = 0;
    if ( e.mpszErrMess )
        mpszErrMess = sqlite3_mprintf( "%s", e.mpszErrMess );
}

const char* QtSQLite3Exception::errorCodeAsString( int nErrCode )
{
    switch (nErrCode)
    {
        case SQLITE_OK          : return "SQLITE_OK";
        case SQLITE_ERROR       : return "SQLITE_ERROR";
        case SQLITE_INTERNAL    : return "SQLITE_INTERNAL";
        case SQLITE_PERM        : return "SQLITE_PERM";
        case SQLITE_ABORT       : return "SQLITE_ABORT";
        case SQLITE_BUSY        : return "SQLITE_BUSY";
        case SQLITE_LOCKED      : return "SQLITE_LOCKED";
        case SQLITE_NOMEM       : return "SQLITE_NOMEM";
        case SQLITE_READONLY    : return "SQLITE_READONLY";
        case SQLITE_INTERRUPT   : return "SQLITE_INTERRUPT";
        case SQLITE_IOERR       : return "SQLITE_IOERR";
        case SQLITE_CORRUPT     : return "SQLITE_CORRUPT";
        case SQLITE_NOTFOUND    : return "SQLITE_NOTFOUND";
        case SQLITE_FULL        : return "SQLITE_FULL";
        case SQLITE_CANTOPEN    : return "SQLITE_CANTOPEN";
        case SQLITE_PROTOCOL    : return "SQLITE_PROTOCOL";
        case SQLITE_EMPTY       : return "SQLITE_EMPTY";
        case SQLITE_SCHEMA      : return "SQLITE_SCHEMA";
        case SQLITE_TOOBIG      : return "SQLITE_TOOBIG";
        case SQLITE_CONSTRAINT  : return "SQLITE_CONSTRAINT";
        case SQLITE_MISMATCH    : return "SQLITE_MISMATCH";
        case SQLITE_MISUSE      : return "SQLITE_MISUSE";
        case SQLITE_NOLFS       : return "SQLITE_NOLFS";
        case SQLITE_AUTH        : return "SQLITE_AUTH";
        case SQLITE_FORMAT      : return "SQLITE_FORMAT";
        case SQLITE_RANGE       : return "SQLITE_RANGE";
        case SQLITE_ROW         : return "SQLITE_ROW";
        case SQLITE_DONE        : return "SQLITE_DONE";
        case QTSQLITE_ERROR     : return "QTSQLITE_ERROR";
        default                 : return "UNKNOWN_ERROR";
    }
}

QtSQLite3Exception::~QtSQLite3Exception()
{
    if ( mpszErrMess )
    {
        sqlite3_free( mpszErrMess );
        mpszErrMess = 0;
    }
}

/*
    QtSQLite3Buffer
*/

QtSQLite3Buffer::QtSQLite3Buffer()
{
    mpBuf = 0;
}

QtSQLite3Buffer::~QtSQLite3Buffer()
{
    clear();
}

void QtSQLite3Buffer::clear()
{
    if ( mpBuf )
    {
        sqlite3_free( mpBuf );
        mpBuf = 0;
    }

}

const char* QtSQLite3Buffer::format( const char* szFormat, ... )
{
    clear();
    va_list va;
    va_start( va, szFormat );
    mpBuf = sqlite3_vmprintf( szFormat, va );
    va_end( va );
    return mpBuf;
}

/*
    QtSQLite3Binary
*/

QtSQLite3Binary::QtSQLite3Binary() :
                                    mpBuf( 0 ),
                                    mnBinaryLen( 0 ),
                                    mnBufferLen( 0 ),
                                    mnEncodedLen( 0 ),
                                    mbEncoded( false )
{
}

QtSQLite3Binary::~QtSQLite3Binary()
{
    clear();
}


void QtSQLite3Binary::setBinary( const unsigned char* pBuf, int nLen )
{
    mpBuf = allocBuffer( nLen );
    memcpy( mpBuf, pBuf, nLen );
}

void QtSQLite3Binary::setEncoded( const unsigned char* pBuf )
{
    clear();

    mnEncodedLen = strlen( (const char*)pBuf );
    mnBufferLen = mnEncodedLen + 1; // Allow for NULL terminator

    mpBuf = (unsigned char*)malloc( mnBufferLen );

    if ( !mpBuf )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Cannot allocate memory",
                                  DONT_DELETE_MSG );

    memcpy( mpBuf, pBuf, mnBufferLen );
    mbEncoded = true;
}

const unsigned char* QtSQLite3Binary::getEncoded()
{
    if ( !mbEncoded )
    {
        unsigned char* ptmp = (unsigned char*)malloc( mnBinaryLen );
        memcpy( ptmp, mpBuf, mnBinaryLen );
        mnEncodedLen = sqlite3_encode_binary( ptmp, mnBinaryLen, mpBuf );
        free( ptmp );
        mbEncoded = true;
    }

    return mpBuf;
}

const unsigned char* QtSQLite3Binary::getBinary()
{
    if ( mbEncoded )
    {
        // in/out buffers can be the same
        mnBinaryLen = sqlite3_decode_binary( mpBuf, mpBuf );

        if ( mnBinaryLen == -1 )
            throw QtSQLite3Exception( QTSQLITE_ERROR,
                                      "Cannot decode binary",
                                      DONT_DELETE_MSG );
        mbEncoded = false;
    }

    return mpBuf;
}

int QtSQLite3Binary::getBinaryLength()
{
    getBinary();
    return mnBinaryLen;
}

unsigned char* QtSQLite3Binary::allocBuffer( int nLen )
{
    clear();

    // Allow extra space for encoded binary as per comments in
    // SQLite encode.c See bottom of this file for implementation
    // of SQLite functions use 3 instead of 2 just to be sure ;-)
    mnBinaryLen = nLen;
    mnBufferLen = 3 + (257*nLen)/254;

    mpBuf = (unsigned char*)malloc( mnBufferLen );

    if ( !mpBuf )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Cannot allocate memory",
                                  DONT_DELETE_MSG );

    mbEncoded = false;
    return mpBuf;
}

void QtSQLite3Binary::clear()
{
    if ( mpBuf )
    {
        mnBinaryLen = 0;
        mnBufferLen = 0;
        free( mpBuf );
        mpBuf = 0;
    }
}

/*
    QtSQLite3Query
*/

QtSQLite3Query::QtSQLite3Query()
{
    mpVM = 0;
    mbEof = true;
    mnCols = 0;
    mbOwnVM = false;
}

QtSQLite3Query::QtSQLite3Query( const QtSQLite3Query& rQuery )
{
    mpVM = rQuery.mpVM;
    // Only one object can own the VM
    const_cast<QtSQLite3Query&>(rQuery).mpVM = 0;
    mbEof = rQuery.mbEof;
    mnCols = rQuery.mnCols;
    mbOwnVM = rQuery.mbOwnVM;
}

QtSQLite3Query::QtSQLite3Query( sqlite3* pDB,
                                sqlite3_stmt* pVM,
                                bool bEof,
                                bool bOwnVM )
{
    mpDB = pDB;
    mpVM = pVM;
    mbEof = bEof;
    mnCols = sqlite3_column_count( mpVM );
    mbOwnVM = bOwnVM;
}

QtSQLite3Query::~QtSQLite3Query()
{
    try
    {
        finalize();
    }
    catch (...)
    {
    }
}

QtSQLite3Query& QtSQLite3Query::operator = ( const QtSQLite3Query& rQuery )
{
    try
    {
        finalize();
    }
    catch (...)
    {
    }
    mpVM = rQuery.mpVM;
    // Only one object can own the VM
    const_cast<QtSQLite3Query&>(rQuery).mpVM = 0;
    mbEof = rQuery.mbEof;
    mnCols = rQuery.mnCols;
    mbOwnVM = rQuery.mbOwnVM;
    return *this;
}

int QtSQLite3Query::numFields()
{
    checkVM();
    return mnCols;
}

const char* QtSQLite3Query::fieldValue( int nField )
{
    checkVM();

    if ( nField < 0 || nField > mnCols-1 )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Invalid field index requested",
                                  DONT_DELETE_MSG );

    return (const char*)sqlite3_column_text( mpVM, nField );
}

const char* QtSQLite3Query::fieldValue( const QString& qsField )
{
    int nField = fieldIndex( qsField );
    return (const char*)sqlite3_column_text( mpVM, nField );
}

int QtSQLite3Query::getIntField( int nField, int nNullValue )
{
    if ( fieldDataType( nField ) == SQLITE_NULL )
        return nNullValue;
    else
        return sqlite3_column_int( mpVM, nField );
}

int QtSQLite3Query::getIntField( const QString& qsField, int nNullValue )
{
    int nField = fieldIndex( qsField );
    return getIntField( nField, nNullValue );
}

double QtSQLite3Query::getFloatField( int nField, double fNullValue )
{

    if ( fieldDataType( nField ) == SQLITE_NULL )
        return fNullValue;
    else
        return sqlite3_column_double( mpVM, nField );
}

double QtSQLite3Query::getFloatField( const QString& qsField, double fNullValue )
{
    int nField = fieldIndex( qsField );
    return getFloatField( nField, fNullValue );
}

const QString QtSQLite3Query::getStringField( int nField )
{
    if ( fieldDataType( nField ) == SQLITE_NULL )
        return QString::null;
    else
        return QString::fromUtf8( (const char*)sqlite3_column_text( mpVM, nField ) );
}

const QString QtSQLite3Query::getStringField( const QString& qsField )
{
    int nField = fieldIndex( qsField );
    return getStringField( nField );
}

const unsigned char* QtSQLite3Query::getBlobField( int nField, int& nLen )
{
    checkVM();

    if (nField < 0 || nField > mnCols-1)
    {
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Invalid field index requested",
                                  DONT_DELETE_MSG );
    }

    nLen = sqlite3_column_bytes( mpVM, nField );
    return (const unsigned char*)sqlite3_column_blob( mpVM, nField );
}

const unsigned char* QtSQLite3Query::getBlobField( const QString& qsField, int& nLen )
{
    int nField = fieldIndex( qsField );
    return getBlobField( nField, nLen );
}

bool QtSQLite3Query::fieldIsNull( int nField )
{
    return ( fieldDataType( nField ) == SQLITE_NULL );
}

bool QtSQLite3Query::fieldIsNull( const QString& qsField )
{
    int nField = fieldIndex( qsField );
    return ( fieldDataType( nField ) == SQLITE_NULL );
}

int QtSQLite3Query::fieldIndex( const QString& qsField )
{
    checkVM();

    if ( qsField != QString::null )
    {
        for ( int nField = 0; nField < mnCols; nField++ )
        {
            QString qsTemp( sqlite3_column_name( mpVM, nField ) );

            if ( qsField == qsTemp )
                return nField;
        }
    }

    throw QtSQLite3Exception( QTSQLITE_ERROR,
                              "Invalid field name requested",
                              DONT_DELETE_MSG );
}

const char* QtSQLite3Query::fieldName( int nCol )
{
    checkVM();

    if ( nCol < 0 || nCol > mnCols-1 )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Invalid field index requested",
                                  DONT_DELETE_MSG );

    return sqlite3_column_name( mpVM, nCol );
}

const char* QtSQLite3Query::fieldDeclType( int nCol )
{
    checkVM();

    if ( nCol < 0 || nCol > mnCols-1 )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Invalid field index requested",
                                  DONT_DELETE_MSG );

    return sqlite3_column_decltype( mpVM, nCol );
}

int QtSQLite3Query::fieldDataType( int nCol )
{
    checkVM();

    if ( nCol < 0 || nCol > mnCols-1 )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Invalid field index requested",
                                  DONT_DELETE_MSG );

    return sqlite3_column_type( mpVM, nCol );
}

bool QtSQLite3Query::eof()
{
    checkVM();
    return mbEof;
}

void QtSQLite3Query::nextRow()
{
    checkVM();

    int nRet = sqlite3_step( mpVM );

    if ( nRet == SQLITE_DONE )
    {
        // no rows
        mbEof = true;
    }
    else if ( nRet == SQLITE_ROW )
    {
        // more rows, nothing to do
    }
    else
    {
        nRet = sqlite3_finalize( mpVM );
        mpVM = 0;
        const char* szError = sqlite3_errmsg( mpDB );
        throw QtSQLite3Exception( nRet,
                                  (char*)szError,
                                  DONT_DELETE_MSG );
    }
}

void QtSQLite3Query::finalize()
{
    if ( mpVM && mbOwnVM )
    {
        int nRet = sqlite3_finalize( mpVM );
        mpVM = 0;
        if ( nRet != SQLITE_OK )
        {
            const char* szError = sqlite3_errmsg( mpDB );
            throw QtSQLite3Exception( nRet, (char*)szError, DONT_DELETE_MSG );
        }
    }
}

void QtSQLite3Query::checkVM()
{
    if ( mpVM == 0 )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Null Virtual Machine pointer",
                                  DONT_DELETE_MSG );
}

/*
    QtSQLite3Table
*/

QtSQLite3Table::QtSQLite3Table()
{
    mpaszResults = 0;
    mnRows = 0;
    mnCols = 0;
    mnCurrentRow = 0;
}

QtSQLite3Table::QtSQLite3Table( const QtSQLite3Table& rTable )
{
    mpaszResults = rTable.mpaszResults;
    // Only one object can own the results
    const_cast<QtSQLite3Table&>(rTable).mpaszResults = 0;
    mnRows = rTable.mnRows;
    mnCols = rTable.mnCols;
    mnCurrentRow = rTable.mnCurrentRow;
}

QtSQLite3Table::QtSQLite3Table( char** paszResults, int nRows, int nCols )
{
    mpaszResults = paszResults;
    mnRows = nRows;
    mnCols = nCols;
    mnCurrentRow = 0;
}

QtSQLite3Table::~QtSQLite3Table()
{
    try
    {
        finalize();
    }
    catch (...)
    {
    }
}

QtSQLite3Table& QtSQLite3Table::operator = ( const QtSQLite3Table& rTable )
{
    try
    {
        finalize();
    }
    catch (...)
    {
    }
    mpaszResults = rTable.mpaszResults;
    // Only one object can own the results
    const_cast<QtSQLite3Table&>(rTable).mpaszResults = 0;
    mnRows = rTable.mnRows;
    mnCols = rTable.mnCols;
    mnCurrentRow = rTable.mnCurrentRow;
    return *this;
}

void QtSQLite3Table::finalize()
{
    if ( mpaszResults )
    {
        sqlite3_free_table( mpaszResults );
        mpaszResults = 0;
    }
}

int QtSQLite3Table::numFields()
{
    checkResults();
    return mnCols;
}

int QtSQLite3Table::numRows()
{
    checkResults();
    return mnRows;
}

const char* QtSQLite3Table::fieldValue( int nField )
{
    checkResults();

    if (nField < 0 || nField > mnCols-1)
    {
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Invalid field index requested",
                                  DONT_DELETE_MSG );
    }

    int nIndex = ( mnCurrentRow * mnCols ) + mnCols + nField;
    return mpaszResults[ nIndex ];
}

const char* QtSQLite3Table::fieldValue( const QString& qsField )
{
    checkResults();

    if ( qsField != QString::null )
    {
        for ( int nField = 0; nField < mnCols; nField++ )
        {
            if ( qsField == QString( mpaszResults[ nField ] )  )
            {
                int nIndex = ( mnCurrentRow * mnCols ) + mnCols + nField;
                return mpaszResults[ nIndex ];
            }
        }
    }

    throw QtSQLite3Exception( QTSQLITE_ERROR,
                              "Invalid field name requested",
                              DONT_DELETE_MSG );
}

int QtSQLite3Table::getIntField( int nField, int nNullValue )
{
    if ( fieldIsNull( nField ) )
        return nNullValue;
    else
        return atoi( fieldValue( nField ) );
}

int QtSQLite3Table::getIntField( const QString& qsField, int nNullValue )
{
    if ( fieldIsNull( qsField ) )
        return nNullValue;
    else
        return atoi( fieldValue( qsField ) );
}

double QtSQLite3Table::getFloatField( int nField, double fNullValue )
{
    if ( fieldIsNull( nField ) )
        return fNullValue;
    else
        return atof( fieldValue( nField ) );
}

double QtSQLite3Table::getFloatField( const QString& qsField, double fNullValue )
{
    if ( fieldIsNull( qsField ) )
        return fNullValue;
    else
        return atof( fieldValue( qsField ) );
}

const QString QtSQLite3Table::getStringField( int nField )
{
    if ( fieldIsNull( nField ) )
        return QString::null;
    else
        return QString::fromUtf8( fieldValue( nField ) );
}

const QString QtSQLite3Table::getStringField( const QString& qsField )
{
    if ( fieldIsNull( qsField ) )
        return QString::null;
    else
        return QString::fromUtf8( fieldValue( qsField ) );
}

bool QtSQLite3Table::fieldIsNull( int nField )
{
    checkResults();
    return ( fieldValue( nField ) == 0 );
}

bool QtSQLite3Table::fieldIsNull( const QString& qsField )
{
    checkResults();
    return ( fieldValue( qsField ) == 0 );
}

const char* QtSQLite3Table::fieldName( int nCol )
{
    checkResults();

    if ( nCol < 0 || nCol > mnCols-1 )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Invalid field index requested",
                                  DONT_DELETE_MSG );

    return mpaszResults[ nCol ];
}

void QtSQLite3Table::setRow( int nRow )
{
    checkResults();

    if ( nRow < 0 || nRow > mnRows-1 )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Invalid row index requested",
                                  DONT_DELETE_MSG );

    mnCurrentRow = nRow;
}

void QtSQLite3Table::checkResults()
{
    if ( mpaszResults == 0 )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Null Results pointer",
                                  DONT_DELETE_MSG );
}

/*
    QtSQLite3Statement
*/

QtSQLite3Statement::QtSQLite3Statement()
{
    mpDB = 0;
    mpVM = 0;
}

QtSQLite3Statement::QtSQLite3Statement( const QtSQLite3Statement& rStatement )
{
    mpDB = rStatement.mpDB;
    mpVM = rStatement.mpVM;
    // Only one object can own VM
    const_cast<QtSQLite3Statement&>(rStatement).mpVM = 0;
}

QtSQLite3Statement::QtSQLite3Statement( sqlite3* pDB, sqlite3_stmt* pVM )
{
    mpDB = pDB;
    mpVM = pVM;
}

QtSQLite3Statement::~QtSQLite3Statement()
{
    try
    {
        finalize();
    }
    catch (...)
    {
    }
}

QtSQLite3Statement& QtSQLite3Statement::operator = ( const QtSQLite3Statement& rStatement )
{
    mpDB = rStatement.mpDB;
    mpVM = rStatement.mpVM;

    // Only one object can own VM
    const_cast<QtSQLite3Statement&>(rStatement).mpVM = 0;
    return *this;
}

int QtSQLite3Statement::execDML()
{
    checkDB();
    checkVM();

    const char* szError = 0;

    int nRet = sqlite3_step( mpVM );

    if ( nRet == SQLITE_DONE )
    {
        int nRowsChanged = sqlite3_changes( mpDB );
        nRet = sqlite3_reset( mpVM );
        if ( nRet != SQLITE_OK )
        {
            szError = sqlite3_errmsg( mpDB );
            throw QtSQLite3Exception( nRet, (char*)szError, DONT_DELETE_MSG );
        }

        return nRowsChanged;
    }
    else
    {
        nRet = sqlite3_reset( mpVM );
        szError = sqlite3_errmsg( mpDB );
        throw QtSQLite3Exception( nRet, (char*)szError, DONT_DELETE_MSG );
    }
}

QtSQLite3Query QtSQLite3Statement::execQuery()
{
    checkDB();
    checkVM();

    int nRet = sqlite3_step( mpVM );

    if ( nRet == SQLITE_DONE )
    {
        // no rows
        return QtSQLite3Query( mpDB, mpVM, true, false );
    }
    else if ( nRet == SQLITE_ROW )
    {
        // at least 1 row
        return QtSQLite3Query( mpDB, mpVM, false, false );
    }
    else
    {
        nRet = sqlite3_reset( mpVM );
        const char* szError = sqlite3_errmsg( mpDB );
        throw QtSQLite3Exception( nRet, (char*)szError, DONT_DELETE_MSG );
    }
}

void QtSQLite3Statement::bind( int nParam, const QString& qsValue )
{
    checkVM();

    int nRes = sqlite3_bind_text( mpVM, nParam, qsValue.utf8().data(),
                                  qsValue.utf8().length(), SQLITE_TRANSIENT );
    if ( nRes != SQLITE_OK )
        throw QtSQLite3Exception( nRes,
                                  "Error binding string param",
                                  DONT_DELETE_MSG );
}

void QtSQLite3Statement::bind( int nParam, const int nValue )
{
    checkVM();
    int nRes = sqlite3_bind_int( mpVM, nParam, nValue );

    if ( nRes != SQLITE_OK )
        throw QtSQLite3Exception( nRes,
                                  "Error binding int param",
                                  DONT_DELETE_MSG );
}

void QtSQLite3Statement::bind( int nParam, const unsigned int nValue )
{
    checkVM();
    int nRes = sqlite3_bind_int( mpVM, nParam, nValue );

    if ( nRes != SQLITE_OK )
        throw QtSQLite3Exception( nRes,
                                   "Error binding int param",
                                   DONT_DELETE_MSG );
}

void QtSQLite3Statement::bind( int nParam, const long lValue )
{
    checkVM();
    int nRes = sqlite3_bind_int( mpVM, nParam, lValue );

    if ( nRes != SQLITE_OK )
        throw QtSQLite3Exception( nRes,
                                  "Error binding int param",
                                  DONT_DELETE_MSG );
}

void QtSQLite3Statement::bind( int nParam, const unsigned long lValue )
{
    checkVM();
    int nRes = sqlite3_bind_int( mpVM, nParam, lValue );

    if ( nRes != SQLITE_OK )
        throw QtSQLite3Exception( nRes,
                                  "Error binding int param",
                                  DONT_DELETE_MSG );
}

void QtSQLite3Statement::bind( int nParam, const double dValue )
{
    checkVM();
    int nRes = sqlite3_bind_double( mpVM, nParam, dValue );

    if ( nRes != SQLITE_OK )
        throw QtSQLite3Exception( nRes,
                                  "Error binding double param",
                                  DONT_DELETE_MSG );
}

void QtSQLite3Statement::bind( int nParam, const unsigned char* blobValue, int nLen )
{
    checkVM();

    int nRes = sqlite3_bind_blob( mpVM, nParam,
                                  (const void*)blobValue, nLen, SQLITE_TRANSIENT );

    if ( nRes != SQLITE_OK )
        throw QtSQLite3Exception( nRes,
                                  "Error binding blob param",
                                  DONT_DELETE_MSG );
}

void QtSQLite3Statement::bind( int nParam, unsigned char* blobValue, int nLen )
{
    checkVM();

    int nRes = sqlite3_bind_blob( mpVM, nParam,
                                  (const void*)blobValue, nLen, SQLITE_TRANSIENT );

    if ( nRes != SQLITE_OK )
        throw QtSQLite3Exception( nRes,
                                  "Error binding blob param",
                                  DONT_DELETE_MSG );
}

void QtSQLite3Statement::bindNull( int nParam )
{
    checkVM();
    int nRes = sqlite3_bind_null( mpVM, nParam );

    if ( nRes != SQLITE_OK )
        throw QtSQLite3Exception( nRes,
                                  "Error binding NULL param",
                                  DONT_DELETE_MSG );
}

void QtSQLite3Statement::reset()
{
    if ( mpVM )
    {
        int nRet = sqlite3_reset( mpVM );
        if ( nRet != SQLITE_OK )
        {
            const char* szError = sqlite3_errmsg( mpDB );
            throw QtSQLite3Exception( nRet, (char*)szError, DONT_DELETE_MSG );
        }
    }
}

void QtSQLite3Statement::finalize()
{
    if ( mpVM )
    {
        int nRet = sqlite3_finalize( mpVM );
        mpVM = 0;

        if ( nRet != SQLITE_OK )
        {
            const char* szError = sqlite3_errmsg( mpDB );
            throw QtSQLite3Exception( nRet, (char*)szError, DONT_DELETE_MSG );
        }
    }
}

void QtSQLite3Statement::checkDB()
{
    if ( mpDB == 0 )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Database not open",
                                  DONT_DELETE_MSG );
}

void QtSQLite3Statement::checkVM()
{
    if ( mpVM == 0 )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Null Virtual Machine pointer",
                                  DONT_DELETE_MSG );
}

/*
    QtSQLite3DB
*/

QtSQLite3DB::QtSQLite3DB()
{
    mpDB = 0;
    mnBusyTimeoutMs = 60000; // 60 seconds
}

QtSQLite3DB::QtSQLite3DB( const QtSQLite3DB& db )
{
    mpDB = db.mpDB;
    mnBusyTimeoutMs = 60000; // 60 seconds
}

QtSQLite3DB::~QtSQLite3DB()
{
    kdDebug() << "QtSQLite3DB::~QtSQLite3DB()" << endl;
    close();
}

QtSQLite3DB& QtSQLite3DB::operator = ( const QtSQLite3DB& db )
{
    mpDB = db.mpDB;
    mnBusyTimeoutMs = 60000; // 60 seconds
    return *this;
}

void QtSQLite3DB::open( const QString& qsFile )
{
    int nRet = sqlite3_open( qsFile, &mpDB );
    if ( nRet != SQLITE_OK )
    {
        const char* szError = sqlite3_errmsg( mpDB );
        throw QtSQLite3Exception( nRet, (char*)szError, DONT_DELETE_MSG );
    }

    setBusyTimeout( mnBusyTimeoutMs );
}

void QtSQLite3DB::close()
{
    if ( mpDB )
    {
        sqlite3_close( mpDB );
        mpDB = 0;
    }
}

QtSQLite3Statement QtSQLite3DB::compileStatement( const QString& qsSQL )
{
    checkDB();

    sqlite3_stmt* pVM = compile( qsSQL );
    return QtSQLite3Statement( mpDB, pVM );
}

bool QtSQLite3DB::tableExists( const QString& qsTable )
{
    QString qsSQL = "select count(*) from sqlite_master where type = 'table' and name = '" + qsTable + "'";

    int nRet = execScalar( qsSQL );
    return ( nRet > 0 );
}

int QtSQLite3DB::execDML( const QString& qsSQL )
{
    checkDB();

    char* szError = 0;

    int nRet = sqlite3_exec( mpDB, qsSQL.utf8().data(), 0, 0, &szError );

    if ( nRet == SQLITE_OK )
        return sqlite3_changes( mpDB );
    else
        throw QtSQLite3Exception( nRet, szError );
}

QtSQLite3Query QtSQLite3DB::execQuery( const QString& qsSQL )
{
    checkDB();

    sqlite3_stmt* pVM = compile( qsSQL );

    int nRet = sqlite3_step( pVM );
    if ( nRet == SQLITE_DONE )
    {
        // no rows
        return QtSQLite3Query( mpDB, pVM, true );
    }
    else if ( nRet == SQLITE_ROW )
    {
        // at least 1 row
        return QtSQLite3Query( mpDB, pVM, false );
    }
    else
    {
        nRet = sqlite3_finalize( pVM );
        const char* szError = sqlite3_errmsg( mpDB );
        throw QtSQLite3Exception( nRet, (char*)szError, DONT_DELETE_MSG );
    }
}

int QtSQLite3DB::execScalar( const QString& qsSQL )
{
    QtSQLite3Query q = execQuery( qsSQL );

    if ( q.eof() || q.numFields() < 1 )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Invalid scalar query",
                                  DONT_DELETE_MSG );

    return atoi( q.fieldValue( 0 ) );
}

QtSQLite3Table QtSQLite3DB::getTable( const QString& qsSQL )
{
    checkDB();

    char* szError = 0 ;
    char** paszResults = 0;
    int nRet;
    int nRows( 0 );
    int nCols( 0 );

    nRet = sqlite3_get_table( mpDB, qsSQL.utf8().data(), &paszResults, &nRows, &nCols, &szError );

    if ( nRet == SQLITE_OK )
        return QtSQLite3Table( paszResults, nRows, nCols );
    else
        throw QtSQLite3Exception( nRet, szError );
}

sqlite_int64 QtSQLite3DB::lastRowId()
{
    return sqlite3_last_insert_rowid( mpDB );
}

void QtSQLite3DB::setBusyTimeout( int nMillisecs )
{
    mnBusyTimeoutMs = nMillisecs;
    sqlite3_busy_timeout( mpDB, mnBusyTimeoutMs );
}

void QtSQLite3DB::checkDB()
{
    if ( !mpDB )
        throw QtSQLite3Exception( QTSQLITE_ERROR,
                                  "Database not open",
                                  DONT_DELETE_MSG );
}

sqlite3_stmt* QtSQLite3DB::compile( const QString& qsSQL )
{
    checkDB();

    char* szError = 0;
    const char* szTail = 0;
    sqlite3_stmt* pVM;

    int nRet = sqlite3_prepare( mpDB, qsSQL.utf8().data(), qsSQL.utf8().length(), &pVM, &szTail );

    if ( nRet != SQLITE_OK )
        throw QtSQLite3Exception( nRet, szError );

    return pVM;
}

////////////////////////////////////////////////////////////////////////////////
// SQLite encode.c reproduced here, containing implementation notes and source
// for sqlite3_encode_binary() and sqlite3_decode_binary()
////////////////////////////////////////////////////////////////////////////////

/*
** This file contains helper routines used to translate binary data into
** a null-terminated string (suitable for use in SQLite) and back again.
** These are convenience routines for use by people who want to store binary
** data in an SQLite database.  The code in this file is not used by any other
** part of the SQLite library.
**
** How This Encoder Works
**
** The output is allowed to contain any character except 0x27 (') and
** 0x00.  This is accomplished by using an escape character to encode
** 0x27 and 0x00 as a two-byte sequence.  The escape character is always
** 0x01.  An 0x00 is encoded as the two byte sequence 0x01 0x01.  The
** 0x27 character is encoded as the two byte sequence 0x01 0x03.  Finally,
** the escape character itself is encoded as the two-character sequence
** 0x01 0x02.
**
** To summarize, the encoder works by using an escape sequences as follows:
**
**       0x00  ->  0x01 0x01
**       0x01  ->  0x01 0x02
**       0x27  ->  0x01 0x03
**
** If that were all the encoder did, it would work, but in certain cases
** it could double the size of the encoded string.  For example, to
** encode a string of 100 0x27 characters would require 100 instances of
** the 0x01 0x03 escape sequence resulting in a 200-character output.
** We would prefer to keep the size of the encoded string smaller than
** this.
**
** To minimize the encoding size, we first add a fixed offset value to each
** byte in the sequence.  The addition is modulo 256.  (That is to say, if
** the sum of the original character value and the offset exceeds 256, then
** the higher order bits are truncated.)  The offset is chosen to minimize
** the number of characters in the string that need to be escaped.  For
** example, in the case above where the string was composed of 100 0x27
** characters, the offset might be 0x01.  Each of the 0x27 characters would
** then be converted into an 0x28 character which would not need to be
** escaped at all and so the 100 character input string would be converted
** into just 100 characters of output.  Actually 101 characters of output -
** we have to record the offset used as the first byte in the sequence so
** that the string can be decoded.  Since the offset value is stored as
** part of the output string and the output string is not allowed to contain
** characters 0x00 or 0x27, the offset cannot be 0x00 or 0x27.
**
** Here, then, are the encoding steps:
**
**     (1)   Choose an offset value and make it the first character of
**           output.
**
**     (2)   Copy each input character into the output buffer, one by
**           one, adding the offset value as you copy.
**
**     (3)   If the value of an input character plus offset is 0x00, replace
**           that one character by the two-character sequence 0x01 0x01.
**           If the sum is 0x01, replace it with 0x01 0x02.  If the sum
**           is 0x27, replace it with 0x01 0x03.
**
**     (4)   Put a 0x00 terminator at the end of the output.
**
** Decoding is obvious:
**
**     (5)   Copy encoded characters except the first into the decode
**           buffer.  Set the first encoded character aside for use as
**           the offset in step 7 below.
**
**     (6)   Convert each 0x01 0x01 sequence into a single character 0x00.
**           Convert 0x01 0x02 into 0x01.  Convert 0x01 0x03 into 0x27.
**
**     (7)   Subtract the offset value that was the first character of
**           the encoded buffer from all characters in the output buffer.
**
** The only tricky part is step (1) - how to compute an offset value to
** minimize the size of the output buffer.  This is accomplished by testing
** all offset values and picking the one that results in the fewest number
** of escapes.  To do that, we first scan the entire input and count the
** number of occurances of each character value in the input.  Suppose
** the number of 0x00 characters is N(0), the number of occurances of 0x01
** is N(1), and so forth up to the number of occurances of 0xff is N(255).
** An offset of 0 is not allowed so we don't have to test it.  The number
** of escapes required for an offset of 1 is N(1)+N(2)+N(40).  The number
** of escapes required for an offset of 2 is N(2)+N(3)+N(41).  And so forth.
** In this way we find the offset that gives the minimum number of escapes,
** and thus minimizes the length of the output string.
**
** Encode a binary buffer "in" of size n bytes so that it contains
** no instances of characters '\'' or '\000'.  The output is
** null-terminated and can be used as a string value in an INSERT
** or UPDATE statement.  Use sqlite3_decode_binary() to convert the
** string back into its original binary.
**
** The result is written into a preallocated output buffer "out".
** "out" must be able to hold at least 2 +(257*n)/254 bytes.
** In other words, the output will be expanded by as much as 3
** bytes for every 254 bytes of input plus 2 bytes of fixed overhead.
** (This is approximately 2 + 1.0118*n or about a 1.2% size increase.)
**
** The return value is the number of characters in the encoded
** string, excluding the "\000" terminator.
*/

int sqlite3_encode_binary( const unsigned char *in, int n, unsigned char *out ) 
{
    int i, j, e, m;
    int cnt[ 256 ];
    if( n<=0 ){
        out[0] = 'x';
        out[1] = 0;
        return 1;
    }

    memset( cnt, 0, sizeof( cnt ) );
    for( i=n-1; i >= 0; i-- )
    {
        cnt[ in[ i ] ]++;
    }
    m = n;
    for( i = 1; i < 256; i++ )
    {
        int sum;
        if( i=='\'' ) continue;
        sum = cnt[ i ] + cnt[ (i+1) & 0xff ] + cnt[ (i+'\'') & 0xff ];
        if( sum < m ){
            m = sum;
            e = i;
            if( m == 0 ) break;
        }
    }
    out[0] = e;
    j = 1;
    for( i = 0; i < n; i++ ) 
    {
        int c = ( in[ i ] - e ) & 0xff;
        if ( c == 0 )
        {
            out[ j++ ] = 1;
            out[ j++ ] = 1;
        }
        else if( c == 1 )
        {
            out[ j++ ] = 1;
            out[ j++ ] = 2;
        }
        else if( c == '\'' )
        {
            out[ j++ ] = 1;
            out[ j++ ] = 3;
        }
        else
        {
            out[ j++ ] = c;
        }
    }
    out[ j ] = 0;
    return j;
}

/*
** Decode the string "in" into binary data and write it into "out".
** This routine reverses the encoding created by sqlite3_encode_binary().
** The output will always be a few bytes less than the input.  The number
** of bytes of output is returned.  If the input is not a well-formed
** encoding, -1 is returned.
**
** The "in" and "out" parameters may point to the same buffer in order
** to decode a string in place.
*/
int sqlite3_decode_binary( const unsigned char* in, unsigned char* out )
{
    int i, c, e;
    e = *(in++);
    i = 0;
    while( ( c = *(in++) ) != 0 )
    {
        if ( c == 1 )
        {
            c = *(in++);
            if ( c == 1 ){
                c = 0;
            }
            else if( c == 2 )
            {
                c = 1;
            }
            else if( c == 3 )
            {
                c = '\'';
            }
            else
            {
                return -1;
            }
        }
        out[ i++ ] = (c + e) & 0xff;
    }
    return i;
}
