/***************************************************************************
 *   Copyright (C) 2006 by Stefano Salvador                                *
 *   stefano.salvador@gmail.com                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <iostream>

#include <kapplication.h>
#include <kcmdlineargs.h>
#include <klocale.h>
#include <kurl.h>
#include <kaboutdata.h>
#include <kfilterdev.h>
#include <kio/netaccess.h>
#include <ktempfile.h>
#include <kdebug.h>

#include <exportXML.h>
#include <katalogfs.h>
#include <katalogio.h>
#include <kataloguds.h>
#include <kataloginfo.h>

#define DOC_TYPE "KDEKatalog"
#define ROOT "KATALOG"
#define CATALOG "CATALOG"
#define ITEM "ITEM"
#define META "META"
#define GROUP "GROUP"

Katalog_ExportXML::Katalog_ExportXML(){
  m_kfs = new KatalogFS();
  m_kio = new KatalogIO();
  m_kinfo = new KatalogInfo();
}

Katalog_ExportXML::~Katalog_ExportXML(){
  delete m_kfs;
  delete m_kio;
  delete m_kinfo;
}

int Katalog_ExportXML::exportData(KURL & source, KURL & destination)
{
  // 0. Try to open destination file
  bool write_to_stdout = false;
  QString tmpName;
  QIODevice *dev;
  if(destination.isEmpty()){
    write_to_stdout = true;
  }
  else {
    if(!destination.isValid())
      return WRITE_ERROR;

    if(!destination.isLocalFile())
    {
      KTempFile tmpFile;
      tmpName = tmpFile.name();
      dev = KFilterDev::deviceForFile( tmpName, "application/x-gzip" );
    }
    else
      dev = KFilterDev::deviceForFile( destination.path(), "application/x-gzip" );

    if(!dev->open( IO_WriteOnly ))
      return WRITE_ERROR;
  }

  // 1. Create XML Document
  QDomImplementation implementation;
  m_document = implementation.createDocument(
      QString::null,
      ROOT, 
      implementation.createDocumentType(DOC_TYPE, QString::null, QString::null));
  QDomElement root_element = m_document.firstChild().toElement();
  // 2. Create Katalog Nodes
  QStringList path = QStringList::split("/", source.path(), false);
  int len = path.size();
  long item_time;
  // 2.0 First check that we are not trying to export mimetype tree
  if(len > 0 && path[0] == "mimetypes")
    return EXPORT_MIME_NOT_SUPPORTED;

  if(len == 0 || len == 1){
    // 2.1 We have to export all catalogs !!!
    if(len == 0)
      path.append("katalogs");
    KatalogUDSEntryList entries = m_kfs->list(path);
    KatalogUDSEntryListConstIterator it;
    QString name;
    for ( it = entries.begin(); it != entries.end(); ++it ){
      KatalogUDSEntry::ConstIterator entit = (*it).begin();
      for( ; entit != (*it).end(); ++entit ){
        if ( (*entit).m_uds == UDS_NAME )
          name = (*entit).m_str;
        else if ( (*entit).m_uds == UDS_MODIFICATION_TIME )
          item_time = (*entit).m_long;
      }
      QDomElement el = m_document.createElement(CATALOG);
      el.setAttribute("name", name);
      el.setAttribute("type", E_DIR);
      el.setAttribute("mimetype", "inode/directory");
      el.setAttribute("time", item_time);
      KatalogInfo kInfo;
      QStringList path_k(path);
      path_k.append( name );
      QString sourceUrl = kInfo.sourceURL(path_k).url();
      el.setAttribute("mount", sourceUrl);
      QStringList path_temp(path);
      path_temp.append(name);
      exportNode(path_temp, el);
      root_element.appendChild(el);
    }
  }
  else if(len == 2){
    // 2.2 We have to export one catalog
    KatalogUDSEntry entry = m_kfs->stat(path);
    QString name;
    KatalogUDSEntry::ConstIterator entit = entry.begin();
    for( ; entit != entry.end(); ++entit ){
      if ( (*entit).m_uds == UDS_NAME )
        name = (*entit).m_str;
      else if ( (*entit).m_uds == UDS_MODIFICATION_TIME )
        item_time = (*entit).m_long;
    }
    QDomElement el = m_document.createElement(CATALOG);
    el.setAttribute("name", name);
    el.setAttribute("type", E_DIR);
    el.setAttribute("mimetype", "inode/directory");
    el.setAttribute("time", item_time);
    KatalogInfo kInfo;
    QString sourceUrl = kInfo.sourceURL(path).url();
    el.setAttribute("mount", sourceUrl);
    exportNode(path, el);
    root_element.appendChild(el);
  }
  else if(len > 2){
    // 2.3 We have to export a portion of a catalog
    QDomElement el = m_document.createElement(CATALOG);
    el.setAttribute("name", "Partial Katalog");
    el.setAttribute("type", E_DIR);
    el.setAttribute("mimetype", "inode/directory");
    QDateTime date(QDate::currentDate(), QTime::currentTime());
    item_time = date.toTime_t();
    el.setAttribute("time", item_time);
    KatalogInfo kInfo;
    QString sourceUrl = kInfo.sourceURL(path).url();
    el.setAttribute("mount", sourceUrl);
    exportNode(path, el);
    root_element.appendChild(el);
  }
  // 3. Save File
  if(write_to_stdout){
    std::cout << m_document.toString(2).utf8();
  }
  else{
    QTextStream ds( dev );
    ds << m_document.toString(2).utf8();
    dev->close();
    delete dev;

    if(!destination.isLocalFile())
      NetAccess::upload( tmpName, destination, 0 );
  }

  return SUCCESS;
}

int Katalog_ExportXML::exportNode(QStringList path, QDomElement &parent_element)
{
  KatalogUDSEntryList *entries = new KatalogUDSEntryList(m_kfs->list(path));
  QString path_s = path.join("/");
  UDSEntryList *ue = (UDSEntryList *)entries;
  UDSEntryListConstIterator it;
  QString name, mime;
  long size, type, time, permissions;
  for ( it = ue->begin(); it != ue->end(); ++it ){
    UDSEntry::ConstIterator entit = (*it).begin();
    for( ; entit != (*it).end(); ++entit ){
      if ( (*entit).m_uds == UDS_NAME )
        name = (*entit).m_str;
      else if ( (*entit).m_uds == UDS_FILE_TYPE )
        type = (*entit).m_long;
      else if ( (*entit).m_uds == UDS_MODIFICATION_TIME )
        time = (*entit).m_long;
      else if ( (*entit).m_uds == UDS_ACCESS )
        permissions = (*entit).m_long;
      else if ( (*entit).m_uds == UDS_SIZE )
        size = (*entit).m_long;
    }

    QStringList path_temp(path);
    path_temp.append(name);
    mime = m_kinfo->mimetype(path_temp);

    QDomElement childNode = m_document.createElement(ITEM);
    childNode.setAttribute("name", name);
    if(type == S_IFDIR)
      childNode.setAttribute("type", E_DIR);
    else
      childNode.setAttribute("type", E_FILE);
    childNode.setAttribute("mimetype", mime); 
    childNode.setAttribute("time", time);
    childNode.setAttribute("size", size);
    childNode.setAttribute("permissions", permissions);

    if(type == S_IFDIR){
      QStringList path_temp(path);
      path_temp.append(name);
      exportNode(path_temp, childNode);
    }

    QStringList path_mi(path);
    path_mi.append( name );
    exportMetaInfo(path_mi, childNode);

    parent_element.appendChild(childNode);

  }
  delete entries;
  return SUCCESS;
}

void Katalog_ExportXML::exportMetaInfo(QStringList path, QDomElement &element)
{
  KatalogInfoList data = m_kinfo->readInfo(path);
  KatalogInfoList::iterator it;
  QString groupName;
  QDomElement metaEl = m_document.createElement(META);
  element.appendChild(metaEl);
  QDomElement groupEl;
  for ( it = data.begin(); it != data.end(); ++it ) {
    KatalogInfoItem info = (*it);
    if( info[0].toString() == "Katalog")
      continue;
    if( groupName != info[0].toString() ) {
      groupName = info[0].toString();
      groupEl = m_document.createElement(GROUP);
      groupEl.setAttribute("name", groupName);
      metaEl.appendChild(groupEl);
    }
    QDomElement itemEl = m_document.createElement(ITEM);
    itemEl.setAttribute("key", info[1].toString());
    switch(info[3].type())
    {
      case QVariant::Int :
        itemEl.setAttribute("int_value", info[3].asInt());
        break;
      case QVariant::Bool :
        itemEl.setAttribute("bool_value", info[3].asBool());
        break;
      case QVariant::Date :
        itemEl.setAttribute("date_value", info[3].asDate().toString());
        break;
      case QVariant::DateTime :
        itemEl.setAttribute("datetime_value", info[3].asDateTime().toTime_t());
        break;
      case QVariant::Double :
        itemEl.setAttribute("double_value", info[3].asDouble());
        break;
      case QVariant::LongLong :
        itemEl.setAttribute("long_value", (long int)info[3].asLongLong());
        break;
      case QVariant::String :
      default :
        itemEl.setAttribute("string_value", info[3].asString());
        break;
    }
    itemEl.setAttribute("unit", info[4].toString());
    groupEl.appendChild(itemEl);
  }
}

static const char description[] =
    I18N_NOOP("Export katalog infos to gzipped XML file");

static const char version[] = "0.4";

static KCmdLineOptions options[] =
{
    { "+[Source URL]", I18N_NOOP( "Catalog or Folder item inside katalog (a URL starting with katalog:/)." ), 0 },
    { "+[Destination URL]", I18N_NOOP( "URL to a file you have the permission to write." ), 0 },
    KCmdLineLastOption
};

int main(int argc, char **argv)
{
  KAboutData about("katalogXMLexport", I18N_NOOP("Katalog XML Export"), version, description,
                    KAboutData::License_GPL, "(C) 2006 Stefano Salvador", 0, 0, "stefano.salvador@gmail.com");
  KCmdLineArgs::init(argc, argv, &about);
  KCmdLineArgs::addCmdLineOptions( options );
  KApplication app;
  KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
  if (args->count() < 0)
    KCmdLineArgs::usage(i18n("You have to specify source URL (katalog:/) and optinaly destination URL, if ommited the file will be written to standard output!"));

  KURL source(args->arg(0));
  KURL destination;
  if(args->count() > 1)
    destination = args->arg(1);

  Katalog_ExportXML k_exp;

  return k_exp.exportData(source, destination);

  args->clear();
}
