/***************************************************************************
 *   Copyright (C) 2006 by Stefano Salvador                                *
 *   stefano.salvador@gmail.com                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <kapplication.h>
#include <kcmdlineargs.h>
#include <klocale.h>
#include <kurl.h>
#include <kdebug.h>
#include <kaboutdata.h>
#include <kfilterdev.h>
#include <kio/netaccess.h>

#include <importXML.h>

#define DOC_TYPE "KDEKatalog"
#define ROOT "KATALOG"
#define CATALOG "CATALOG"
#define ITEM "ITEM"
#define META "META"
#define GROUP "GROUP"

int Katalog_ImportXML::openDocument(KURL & url)
{
  QString tmpfile;
  QIODevice * dev = 0L;
  if( !url.isEmpty() && NetAccess::download( url, tmpfile, 0 ) )
  {
    dev = KFilterDev::deviceForFile( tmpfile, "application/x-gzip");
    if(!dev->open( IO_ReadOnly ))
      return UNREADABLE_URL;
  }

  QDomDocument document;
  int Line;
  int Col;
  QString str;
  if ( !dev ||
       !document.setContent( dev, true, &str, &Line, &Col ) ||
       document.doctype().name().compare(DOC_TYPE))
  {
    return NOT_KATALOG;
  }
  dev->close();

  QDomNode node = document.firstChild();

  if(node.isNull() || node.nodeName() != ROOT)
    return UNCOMPLETE;

  m_rootElement = node.toElement();

  return SUCCESS;
}

int Katalog_ImportXML::importData()
{
  QDomNode node = m_rootElement.firstChild();
  while ( !node.isNull() )
  {
    if(!node.isElement())
      continue;
    // Read catalog infos
    QString name = node.toElement().attribute("name");
    QString catalogurl = node.toElement().attribute("mount");
    QDateTime date;
    date.setTime_t(node.toElement().attribute("time").toInt());
    // Add the catalog
    m_current_catalogid = addCatalog(name, catalogurl, date);
    kdDebug() << m_current_catalogid << endl;

    m_current_fileid = getNextFileId();

    initInput();

    // Recurse the tree
    QDomNode childNode = node.firstChild();
    while ( !childNode.isNull() )
    {
      if(childNode.nodeName().compare(ITEM) == 0)
      {
        QDomElement el = childNode.toElement();
        importNode(el, -1, catalogurl);
        childNode = childNode.nextSibling();
      }
    }
    // Go to the next
    node = node.nextSibling();
  }

  doneInput();

  return 0;
}

int Katalog_ImportXML::importNode(QDomElement & node, int parent_id, QString catalogurl)
{
  // Read node info
  catalogurl = catalogurl + "/" + node.attribute("name");

  FileInfoDB finfo;
  finfo.parentid = parent_id;
  finfo.fileid = m_current_fileid;
  finfo.catalogid = m_current_catalogid;
  finfo.fullname = catalogurl;
  finfo.filename = node.toElement().attribute("name");
  finfo.filetype = node.attribute( "mimetype" );
  finfo.filesize = node.attribute( "size" ).toInt();
  finfo.modificationdate = node.attribute( "time" ).toInt();
  finfo.lastaccessdate = finfo.modificationdate;
  finfo.username = node.attribute( "user" );
  finfo.groupname = node.attribute( "group" );
  finfo.permissions = node.attribute( "permissions" ).toInt();

  // Add the new node
  int item_id = addEntry(finfo);
  m_current_fileid = item_id + 1;

  // Recurse the tree
  QDomNode childNode = node.firstChild();
  while ( !childNode.isNull() )
  {
    QDomElement el = childNode.toElement();
    if(childNode.nodeName().compare(ITEM) == 0)
      importNode(el, item_id, catalogurl);
    else if(childNode.nodeName().compare(META) == 0)
      importMetaInfo(el, item_id);
    childNode = childNode.nextSibling();
  }
  return item_id;
}

void Katalog_ImportXML::importMetaInfo(QDomElement & node, int item_id)
{
  MetaInfoDB minfo;
  QDomNode groupNode = node.firstChild();
  minfo.fileid = item_id;

  while ( !groupNode.isNull() ) {
    if(!groupNode.isElement() || groupNode.nodeName() != "GROUP") {
      groupNode = groupNode.nextSibling();
      continue;
    }

    minfo.groupname = groupNode.toElement().attribute("name");

    QDomNode itemNode = groupNode.firstChild();

    while ( !itemNode.isNull() )
    {
      if(!itemNode.isElement() || itemNode.nodeName() != "ITEM")
      {
        itemNode = itemNode.nextSibling();
        continue;
      }

      QDomElement itemEl = itemNode.toElement();
      minfo.field = itemEl.attribute("key");
      if(itemEl.hasAttribute("int_value")) {
        minfo.type = "int";
        minfo.value = itemEl.attribute("int_value");
      }
      else if(itemEl.hasAttribute("bool_value")) {
        minfo.type = "bool";
        minfo.value = itemEl.attribute("bool_value");
      }
      else if(itemEl.hasAttribute("date_value"))
      {
        minfo.type = "date";
        minfo.value = itemEl.attribute("date_value");
      }
      else if(itemEl.hasAttribute("datetime_value"))
      {
        minfo.type = "datetime";
        minfo.value = itemEl.attribute("date_value");
      }
      else if(itemEl.hasAttribute("double_value"))
      {
        minfo.type = "datetime";
        minfo.value = itemEl.attribute("date_value");
      }
      else if(itemEl.hasAttribute("long_value"))
      {
        minfo.type = "long";
        minfo.value = itemEl.attribute("long_value");
      }
      else if(itemEl.hasAttribute("string_value"))
      {
        minfo.type = "string";
        minfo.value = itemEl.attribute("string_value");
      }
      minfo.unit = itemEl.attribute("unit").toInt();

      addInfo(minfo);

      itemNode = itemNode.nextSibling();
    }
    groupNode = groupNode.nextSibling();
  }

  return;
}

/** The main application */

static const char description[] =
    I18N_NOOP("Import katalog infos from gzipped XML file");

static const char version[] = "0.4";

static KCmdLineOptions options[] =
{
    { "+[URL]", I18N_NOOP( "The URL pointing to a valid gzipped XML Katalog file)." ), 0 },
    KCmdLineLastOption
};

int main(int argc, char **argv)
{
  KAboutData about("katalogXMLimport", I18N_NOOP("Katalog XML Import"), version, description,
                    KAboutData::License_GPL, "(C) 2006 Stefano Salvador", 0, 0, "stefano.salvador@gmail.com");
  KCmdLineArgs::init(argc, argv, &about);
  KCmdLineArgs::addCmdLineOptions( options );
  KApplication app;
  KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
  if (args->count() == 0)
    KCmdLineArgs::usage(i18n("No URL specified!"));

  KURL url(args->arg(0));

  if (!url.isValid())
    KCmdLineArgs::usage(i18n("URL Not found!"));

  args->clear();

  Katalog_ImportXML import;
  int res = import.openDocument(url);
  if(res == Katalog_ImportXML::SUCCESS)
    return import.importData();
  else
    return res;
}

