/***************************************************************************
 *   Copyright (C) 2004 by Stefano Salvador                                *
 *   stefano@diamante                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include <qapplication.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qtooltip.h>
#include <qthread.h>

#include <kurlrequester.h>
#include <klineedit.h>
#include <kpassivepopup.h>
#include <kmessagebox.h>
#include <dcopclient.h>
#include <kdebug.h>
#include <klocale.h>
#include <kiconloader.h>
#include <ksystemtray.h>
#include <kprogress.h>
#include <kstringhandler.h>

#include <katalogscan.h>
#include <kataloginfo.h>

#include "katalogservicedlg.h"

class JobThread : public QThread {

public:
  JobThread(KatalogServiceDlg *dlg) {
    th_dlg = dlg;
  }

  virtual void run() {
    th_kataloginfo = new KatalogInfo();
    th_dlg->m_tot_items = th_kataloginfo->countFolders( th_dlg->m_url );
    delete th_kataloginfo;
    QCustomEvent *event = new QCustomEvent(QEvent::User);
    QApplication::postEvent(th_dlg, event);
    return;
  }

  void stopJobs() {
    if(th_kataloginfo)
      th_kataloginfo->stopJobs();
  }

private:
  KatalogServiceDlg *th_dlg;
  KatalogInfo *th_kataloginfo;
};

KatalogServiceDlg::KatalogServiceDlg(QWidget* parent, const char* name, bool modal, WFlags fl) :
    KDialogServiceDlgBase(parent,name, modal,fl)
{
  this->setIcon( KGlobal::iconLoader()->loadIcon( "katalog", KIcon::Small ) );
  m_currentStatus = IDLE;
  KatalogIFace iface;
  if(!iface.isReady()){
    KMessageBox::error(this, i18n("No storage found.\nPlease verify your settings"), i18n("General error"));
    QCustomEvent *event = new QCustomEvent(FATAL_ERROR);
    QApplication::postEvent(this, event);
    return;
  }
  else{
    m_base_string = QString(i18n("Now Scanning: %1"));
    m_base_len = infoLabel->fontMetrics().width(m_base_string, m_base_string.length());
    m_sysTray = new KSystemTray(this, "SysTray icon for Katalog");
    m_sysTray->setCaption("Katalog");
    m_sysTray->setPixmap(m_sysTray->loadIcon("katalog"));
    m_sysTray->show();
    // fill the catalog combo
    KatalogInfo kataloginfo;
    QStringList list = kataloginfo.getCatalogsList();
    list.prepend("");
    editName->insertStringList(list);
  }
}

KatalogServiceDlg::~KatalogServiceDlg()
{
  delete m_sysTray;
  delete m_katalogscan;
}

void KatalogServiceDlg::reject()
{
  QDialog::reject();
}

void KatalogServiceDlg::accept()
{
  if(m_currentStatus == IDLE){
    if(editName->currentText().isEmpty()) {
      KMessageBox::error(this, i18n("You have to specify a catalog name"), i18n("Empty catalog name"));
      return;
    }

    infoLabel->setText(i18n("Reading folder structure, please wait ..."));

    m_currentStatus = COUNT_FOLDERS;

    disableUI();

    m_jobThread = new JobThread(this);
    m_jobThread->start();
  }
  else if(m_currentStatus == COUNT_FOLDERS){
    if(m_jobThread)
      m_jobThread->stopJobs();
    infoLabel->setText(i18n("Interupted"));
    restoreUI();
    m_currentStatus = IDLE;
  }
  else if(m_currentStatus == SCANNING){
    if(m_katalogscan)
      m_katalogscan->stopJobs();
    infoLabel->setText(i18n("Interupted"));
    restoreUI();
    m_currentStatus = IDLE;
  }
}

void KatalogServiceDlg::finished(QString url)
{
  if(url.compare(m_url.url()) == 0) {
    infoLabel->setText(i18n("Finished"));
    QToolTip::add( m_sysTray, QString(i18n("Catalog <I>%1</I> complete !")).arg(editName->currentText()) );
    KPassivePopup *popup = new KPassivePopup(m_sysTray);
    popup->setTimeout(0);
    popup->setView( "Katalog", QString(i18n("Catalog <I>%1</I> complete !")).arg(editName->currentText()) );
    popup->show();
    restoreUI();
    m_currentStatus = IDLE;
  }
}

void KatalogServiceDlg::currentFolder(QString name)
{
  if(m_currentStatus == SCANNING)
    infoLabel->setText(
      m_base_string.arg(KStringHandler::cPixelSqueeze(name, infoLabel->fontMetrics(), infoLabel->width() - m_base_len))
    );
}

void KatalogServiceDlg::newItem()
{
  if(m_currentStatus == SCANNING){
    progressBar->setProgress(++m_part_items);
    QToolTip::add( m_sysTray, i18n("Katalog %1 - %2").arg(editName->currentText()).arg(progressBar->progressString()) );
  }
}

void KatalogServiceDlg::slotClose()
{
  QDialog::accept();
}

void KatalogServiceDlg::setURL(KURL url)
{
  if(url.isLocalFile())
    url.setProtocol("file");
  m_url = url;
}

void KatalogServiceDlg::disableUI()
{
  editName->setEnabled(false);
  checkCompress->setEnabled(false);
  checkMeta->setEnabled(false);
  buttonCancel->setEnabled(false);
  buttonOk->setText(i18n("&Stop"));
  progressBar->setEnabled(true);
}

void KatalogServiceDlg::restoreUI()
{
  buttonOk->setText(i18n("&Start"));
  editName->setEnabled(true);
  checkCompress->setEnabled(true);
  checkMeta->setEnabled(true);
  buttonCancel->setEnabled(true);
  progressBar->setEnabled(false);
  progressBar->setProgress(0);
  progressBar->setTotalSteps(0);
}

void KatalogServiceDlg::customEvent( QCustomEvent * e )
{
  switch( e->type()) {
    case TOT_ITEMS:
      if( m_currentStatus != COUNT_FOLDERS)
        return;
        m_currentStatus = SCANNING;
        m_part_items = 0;
        m_katalogscan = new KatalogScan();
        m_katalogscan->start(m_url, editName->currentText(), checkCompress->isChecked(), checkMeta->isChecked() );
        progressBar->setTotalSteps(m_tot_items);
        disableUI();
        hide();
        connect( m_katalogscan, SIGNAL( currentFolder(QString) ), this, SLOT(currentFolder(QString)) );
        connect( m_katalogscan, SIGNAL( newItem() ), this, SLOT(newItem()) );
        connect( m_katalogscan, SIGNAL( finished(QString) ), this, SLOT(finished(QString)) );
      break;
    case FATAL_ERROR:
      done(1);
      break;
    default:
      break;
  }


}
#include "katalogservicedlg.moc"
