/***************************************************************************
 *   Copyright (C) 2004-2006 by Stefano Salvador                           *
 *   stefano.salvador@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qapplication.h>
#include <qsqldatabase.h>
#include <qsqldriver.h>
#include <qdatatable.h>

#include <kdebug.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <kconfig.h>
#include <kurl.h>

#include "katalogiface.h"

#define DRIVER       "QSQLITE3"   /* see the Qt SQL documentation for a list of available drivers */
#define DATABASE     "katalog/katalog.db" /* the name of your database */
#define USER         ""           /* user name with appropriate rights */
#define PASSWORD     ""           /* password for USER */
#define HOST         ""           /* host on which the database is running */

KatalogIFace::KatalogIFace()
{
  m_isOpen = false;
}

KatalogIFace::~KatalogIFace()
{
}

bool KatalogIFace::openDB()
{
  // avoid multiple connections in the same process
  if( ! QSqlDatabase::database() ){
    kdDebug() << "KatalogIO::openDB - trying to open DB" << endl;
    // Read configuration
    KConfig config("katalogrc", true);

    config.setGroup("Database");

    QString db_driver = config.readEntry( "db_driver", DRIVER );
    QString db_name = config.readEntry( "db_name", DATABASE );
    QString db_user = config.readEntry( "db_user", USER );
    QString db_pass = config.readEntry( "db_pass", PASSWORD );
    QString db_host = config.readEntry( "db_host", HOST );

    // Locate the sqlite3 file
    if(db_driver == "QSQLITE3"){
      KURL db_url = KGlobal::dirs()->localkdedir() + "share/apps/" + db_name;
      KStandardDirs::makeDir( db_url.directory() );
      db_name = db_url.path();
    }

    kdDebug() << "KatalogIO::openDB - DB connection: " << db_driver << "://" << db_user << ":" << db_pass << db_host << "/" << db_name << endl;

    QSqlDatabase *db = QSqlDatabase::addDatabase( db_driver );
    db->setDatabaseName( db_name );
    db->setUserName( db_user );
    db->setPassword( db_pass );
    db->setHostName( db_host );

    return db->open() && createDB();
  }


  return true;
}

bool KatalogIFace::createDB()
{
  kdDebug() << "KatalogIO::createDB - starting " << endl;

  QSqlDatabase *db = QSqlDatabase::database();
  // This follows Kat DB schema
  QStringList tables = db->tables();
  if (!tables.contains("catalogs"))
    db->exec("CREATE TABLE catalogs("
                       "catalogid integer primary key,"
                       "autoupdate integer,"
                       "name varchar(500),"
                       "description varchar(500),"
                       "path varchar(1000),"
                       "notes varchar(2000),"
                       "author varchar(100),"
                       "version integer,"
                       "thumbnailsize integer,"
                       "useexclusionlist integer,"
                       "creationdate varchar(12),"
                       "lastupdatedate varchar(12),"
                       "metadata integer,"
                       "files integer,"
                       "folders integer,"
                       "fulltexts integer,"
                       "thumbnails integer,"
                       "words integer,"
                       "filesize integer );");

  if (!tables.contains("files")){
    db->exec("CREATE TABLE files("
                       "fileid integer primary key,"
                       "catalogid integer,"
                       "fullname varchar(5000),"
                       "filename varchar(300),"
                       "parentid integer,"
                       "filetype varchar(100),"
                       "filesize integer,"
                       "statuschangedate integer,"
                       "modificationdate integer,"
                       "lastaccessdate integer,"
                       "lastupdatedate integer,"
                       "username varchar(100),"
                       "groupname varchar(100),"
                       "permissions integer,"
                       "mode integer,"
                       "language varchar(100) );");
    db->exec("CREATE INDEX files_catalogid on files (catalogid);");
    db->exec("CREATE INDEX files_filename on files (filename);");
    db->exec("CREATE UNIQUE INDEX files_fullname on files (catalogid, fullname);");
    db->exec("CREATE INDEX files_parentid on files (parentid);");
  }

  if (!tables.contains("fulltexts"))
    db->exec("CREATE TABLE fulltexts( fileid integer primary key, fulltextdata blob, fulltextdatalength integer );");

  if (!tables.contains("indexerstore"))
    db->exec("CREATE TABLE indexerstore( catalogid integer, key varchar(5000), data integer);");

  if (!tables.contains("metadata")){
    db->exec("CREATE TABLE metadata( "
    "fileid integer, groupname varchar(100), field varchar(100), type varchar(100), "
    "value varchar(5000), unit integer );");
    db->exec("CREATE INDEX metadata_field on metadata (field);");
    db->exec("CREATE INDEX metadata_fileid on metadata (fileid);");
    db->exec("CREATE INDEX metadata_type on metadata (type);");
    db->exec("CREATE INDEX metadata_value on metadata (value);");
  }

  if (!tables.contains("thumbnails"))
    db->exec("CREATE TABLE thumbnails( fileid integer primary key, thumbnaildata blob, thumbnaildatalength integer );");

  if (!tables.contains("wordfile")){
    db->exec("CREATE TABLE wordfile( wordid integer, fileid integer, occurrences integer );");
    db->exec("CREATE INDEX wordfile_fileid on wordfile (fileid);");
    db->exec("CREATE INDEX wordfile_wordid on wordfile (wordid);");
  }

  if (!tables.contains("words")){
    db->exec("CREATE TABLE words( word varchar(100), wordid integer primary key);");
    db->exec("CREATE UNIQUE INDEX words_word on words (word);");
    db->exec("CREATE UNIQUE INDEX words_wordid on words (wordid);");
  }

  kdDebug() << "KatalogIO::createDB - exiting " << endl;

  return true; // TODO Add more controls
}

KatalogNode KatalogIFace::findNode(QStringList& path) const
{
  // Init the node
  KatalogNode knode;
  knode.catalogid = -1;
  knode.fileid = -1;

  // If the path is the root, return now
  if(path.empty())
    return knode;

  // Get the first node of the path: the catalog
  QString catalog = path.first();
  path.pop_front();
  QSqlQuery q(QString("SELECT catalogid from catalogs WHERE catalogs.name='%1'").arg(catalog));

  if(!q.first())
    return knode;

  knode.catalogid = q.value(0).toInt();

  // If the path is only the catalog, return now
  if(path.empty())
    return knode;

  // Browse throught the path
  QStringList::Iterator it = path.begin();
  QStringList::Iterator end = path.end();
  for (  ; it != end; ++it ) {
    QSqlQuery q2(QString("SELECT fileid from files WHERE catalogid='%1' AND filename='%2' AND parentid='%3'").arg(knode.catalogid).arg(*it).arg(knode.fileid));
    if(!q2.first()){
      knode.fileid = -1;
      return knode;
    }
    knode.fileid = q2.value(0).toInt();
  }

  return knode;
}

int KatalogIFace::addCatalog(QString name, QString baseUrl, QDateTime date)
{
  openDB();

  int current_catalogid = 0;

  QSqlQuery catalog_q("SELECT catalogid, name FROM catalogs WHERE name='"+name+"'");
  if(catalog_q.first()){
    current_catalogid = catalog_q.value(0).toInt();
    return current_catalogid;
  }
  else{
    QSqlQuery catalog_id_q("SELECT MAX(catalogid) FROM catalogs");
    if(catalog_id_q.first())
      current_catalogid = catalog_id_q.value(0).toInt() + 1;
    else
      current_catalogid = 1;
  }
  QSqlQuery query;
  query.prepare( "INSERT INTO catalogs ("
      "catalogid, autoupdate, name, description, path, notes, author, version, thumbnailsize, "
      "useexclusionlist, creationdate, lastupdatedate, metadata, files, folders, fulltexts, "
      "words, filesize) "
      "VALUES ("
      ":catalogid, :autoupdate, :name, :description, :path, :notes, :author, :version, :thumbnailsize, "
      ":useexclusionlist, :creationdate, :lastupdatedate, :metadata, :files, :folders, :fulltexts, "
      ":words, :filesize) ");
  query.bindValue(":catalogid", current_catalogid);
  query.bindValue(":autoupdate", 0);
  query.bindValue(":name", name);
  query.bindValue(":description", "");
  query.bindValue(":path", baseUrl);
  query.bindValue(":notes", 0);
  query.bindValue(":author", 0);
  query.bindValue(":version", 0);
  query.bindValue(":thumbnailsize", 0);
  query.bindValue(":useexclusionlist", 0);
  query.bindValue(":creationdate", date.toTime_t());
  query.bindValue(":lastupdatedate", date.toTime_t());
  query.bindValue(":metadata", 0);
  query.bindValue(":files", 0);
  query.bindValue(":folders", 0);
  query.bindValue(":fulltexts", 0);
  query.bindValue(":words", 0);
  query.bindValue(":filesize", 0);
  query.exec();

  return current_catalogid;
}
