/***************************************************************************
 *   Copyright (C) 2004-2006 by Stefano Salvador                           *
 *   stefano.salvador@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qdir.h>
#include <qdatetime.h>
#include <qfileinfo.h>
#include <qsqldatabase.h>
#include <qsqldriver.h>
#include <qsqlquery.h>
#include <qvariant.h>

#include <kurl.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmimetype.h>
#include <klocale.h>
#include <kglobal.h>
#include <kdebug.h>

#include "katalogdefs.h"
#include "kataloginfo.h"

int KatalogInfo::countFolders(const KURL& mount)
{
  if(mount.protocol() == "file"){
    m_stop = false;
    int totFolders = 0;
    _countFolders(mount.path(), totFolders);
    return totFolders;
  }
  return -1;
}

void KatalogInfo::_countFolders(const QString& path, int &totFolders)
{
  if(m_stop)
    return;
  QDir dir(path);
  const QFileInfoList *entryList = dir.entryInfoList(QDir::NoSymLinks | QDir::Readable | QDir::Hidden | QDir::Dirs);
  for ( QFileInfoList::Iterator it = entryList->begin(); it != entryList->end(); ++it ) {
    if((*it)->isDir()){
      if((*it)->fileName() == "." || (*it)->fileName() == "..")
        continue;
      ++totFolders;
      _countFolders((*it)->absFilePath(), totFolders);
    }
  }
}

void KatalogInfo::stopJobs()
{
  m_stop = true;
}

int KatalogInfo::size(const QStringList& path)
{
  QVariant info = getInfo("filesize", path);
  return info.toInt();
}

QString KatalogInfo::prettySize(const QStringList& path)
{
  long l_size = size(path);
  if(l_size > 1073741824)
    return QString::number(l_size >> 30) + " GB";
  else if (l_size > 1048576)
    return QString::number(l_size >> 20) + " MB";
  else if (l_size > 1024)
    return QString::number(l_size >> 10) + " KB";
  else
    return QString::number(l_size) + " B";
}

uint KatalogInfo::dateTime(const QStringList& path)
{
  QVariant info = getInfo("modificationdate", path);
  return info.toUInt();
}

QString KatalogInfo::prettyDateTime(const QStringList& path)
{
  QDateTime dt;
  uint time = dateTime(path);
  dt.setTime_t(time);
  return dt.toString( KGlobal::locale()->formatDateTime( dt ) );
}

QString KatalogInfo::mimetype(const QStringList& path)
{
  QVariant info = getInfo("filetype", path);
  return info.toString();
}

QString KatalogInfo::icon(const QStringList& path)
{
  QVariant info = getInfo("filetype", path);
  KMimeType::Ptr ptr = KMimeType::mimeType(info.toString());
  QString icon = ptr ? ptr->icon(QString::null, true) : QString::null;
  return KGlobal::iconLoader()->iconPath(icon, 0);
}

QVariant KatalogInfo::getInfo(const QString& info, const QStringList& path)
{
  int len = path.count();
  if(len == 0){
    QVariant variant;
    return variant;
  }
  QStringList local_path(path);
  QString type = local_path.first();
  local_path.pop_front();
  if(type = "catalogs"){
    openDB();
    KatalogNode node = findNode(local_path);
    if(node.catalogid == -1 || node.fileid == -1){
      QVariant variant;
      return variant;
    }
    else {
      QSqlQuery query(QString("SELECT %1 FROM files WHERE catalogid='%2' AND fileid='%3'").arg(info).arg(node.catalogid).arg(node.fileid));
      query.first();
      QVariant variant = query.value(0);
      return variant;
    }
  }
  else if(type == "mimetypes"){
    int len = path.count();
    if(len < 3){
      QVariant variant;
      return variant;
    }
    else {
      QVariant variant;
      return variant;
    }
  }
  else {
    QVariant variant;
    return variant;
  }
}

KURL KatalogInfo::sourceURL(const QStringList& path)
{
  int len = path.count();
  if(len == 0){
    KURL url;
    return url;
  }
  QStringList local_path(path);
  QString type = local_path.first();
  local_path.pop_front();
  if(type = "catalogs"){
    openDB();
    KatalogNode node = findNode(local_path);
    if(node.catalogid == -1){
      KURL url;
      return url;
    }
    else if(node.fileid == -1){
      QSqlQuery query(QString("SELECT path FROM catalogs WHERE catalogid='%1'").arg(node.catalogid));
      query.first();
      KURL url(query.value(0).toString());
      return url;
    }
    else {
      openDB();
      QSqlQuery query(QString("SELECT fullname FROM files WHERE catalogid='%1' AND fileid='%2'").arg(node.catalogid).arg(node.fileid));
      query.first();
      KURL url(query.value(0).toString());
      return url;
    }
  }
  else if(type == "mimetypes"){
    int len = path.count();
    if(len < 3){
      KURL url;
      return url;
    }
    else {
      openDB();
      QSqlQuery query(QString("SELECT fullname FROM files WHERE filetype = '%1' AND filename='%2'").arg(path[0] + '/' + path[1]).arg(path[2]));
      if ( query.isActive() && query.first() ) {
        KURL url(query.value(0).toString());
        return url;
      }
    }
  }
  KURL url;
  return url;
}

KatalogInfoList KatalogInfo::readInfo(const QStringList& path)
{
  KatalogInfoList infoList;
  if(path.size() == 0){
    // No info for the root
    return infoList;
  }
  QStringList local_path(path);
  QString type = local_path.first();
  local_path.pop_front();
  if(type = "catalogs"){
    openDB();
    QString catalog = local_path.first();
    KatalogNode node = findNode(local_path);
    if(node.catalogid == -1){
      // This is the root or the path does not exist, No info !!!
      return infoList;
    }
    else if(node.fileid == -1){
      // Info for catalogs
      KatalogInfoItem info;
      info.append(QVariant("Katalog"));
      info.append(QVariant(i18n("Total Items")));
      info.append(QVariant("int"));
      info.append(QVariant(totalItems(catalog)));
      infoList.append(info);
      return infoList;
    }
    else {
      // Info for files or folder
      QSqlQuery query(QString("SELECT groupname, field, type, value, unit FROM metadata WHERE fileid='%1' ORDER BY groupname;").arg(node.fileid));
      QSqlQuery query2(QString("SELECT filetype FROM files WHERE fileid='%1';").arg(node.fileid));
      query2.first();
      KatalogInfoItem infoK;
      infoK.append(QVariant("Katalog"));
      infoK.append(QVariant("Mimetype"));
      infoK.append(QVariant("string"));
      infoK.append(query2.value(0));
      infoK.append(QVariant(0));
      infoList.append(infoK);
      if ( query.isActive() ) {
        while ( query.next() ) {
          KatalogInfoItem info;
          info.append(query.value(0));
          info.append(query.value(1));
          info.append(query.value(2));
          info.append(query.value(3));
          info.append(query.value(4));
          infoList.append(info);
        }
      }
      return infoList;
    }
  }
  else if(type = "mimetypes"){
    // No info for mimetypes
    return infoList;
  }
  else{
    // Undefined case, we can arrive here, isn't it?
    return infoList;
  }
  return infoList;
}

QStringList KatalogInfo::getCatalogsList()
{
  openDB();
  QStringList list;
  QSqlQuery query("SELECT name FROM catalogs");
  if ( query.isActive() ) {
    while ( query.next() ) {
      list.append( query.value(0).toString() );
    }
  }
  return list;
}

int KatalogInfo::totalItems(const QString& catalog)
{
  openDB();
  QSqlQuery query(QString("SELECT COUNT(*) FROM files, catalogs WHERE files.catalogid=catalogs.catalogid AND catalogs.name='%1'").arg(catalog));
  int tot = 0;
  if(query.first())
    tot = query.value(0).toInt();

  return tot;
}

int KatalogInfo::totalCatalogs()
{
  openDB();
  QString select("SELECT COUNT(*) FROM catalogs");
  QSqlQuery query(select);
  int tot = 0;
  if(query.isValid() && query.first())
    tot = query.value(0).toInt();

  return tot;
}
