/***************************************************************************
 *   Copyright (C) 2004-2006 by Stefano Salvador                           *
 *   stefano.salvador@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qdir.h>
#include <qsqlquery.h>
#include <qsqldriver.h>
#include <qsqldatabase.h>
#include <qapplication.h>
#include <qeventloop.h>
#include <qfileinfo.h>

#include <kio/job.h>
#include <kfileitem.h>
#include <kservicetype.h>
#include <kfilemetainfo.h>
#include <kdebug.h>

#include "katalogscan.h"

using namespace KIO;

int KatalogScan::start(const KURL& mount, QString name, bool exploreArchives, bool getMetaInfo)
{
  openDB();

  m_exploreArchives = exploreArchives;
  m_getMetaInfo = getMetaInfo;

  if(name.isNull() || name.isEmpty())
    return -1;

  m_currentUrl = mount;

  m_current_catalogid = addCatalog(name, mount.url());

  QSqlQuery files_id_q(QString("SELECT fileid FROM files ORDER BY fileid DESC"));
  if(files_id_q.first())
    m_current_fileid = files_id_q.value(0).toInt() + 1;
  else
    m_current_fileid = 1;

  m_current_job = listRecursive( mount, false );
  KURL u = m_current_job->url();
  m_jobs.append( u.url() );

  connect( m_current_job, SIGNAL( entries( KIO::Job *, const KIO::UDSEntryList & ) ),
           this, SLOT( slotEntries( KIO::Job *, const KIO::UDSEntryList & ) ) );
  connect( m_current_job, SIGNAL( result( KIO::Job * ) ),
           this, SLOT( slotResult( KIO::Job * ) ) );
  connect( m_current_job, SIGNAL( redirection( KIO::Job *, const KURL& ) ),
           this, SLOT( slotRedirection( KIO::Job *, const KURL& ) ) );

  return SUCCESS; // TODO add more controls
}

void KatalogScan::stopJobs()
{
  if(m_current_job)
    m_current_job->kill();
}

void KatalogScan::slotEntries(Job *job, const UDSEntryList &entries)
{
  KURL url = static_cast<ListJob *>(job)->url();

  if(m_jobs.find(url.url()) == m_jobs.end() )
    return;

  QSqlDatabase *db = QSqlDatabase::database();
  bool hasTransactions = db->driver()->hasFeature(QSqlDriver::Transactions);
  if(hasTransactions)
    db->transaction();

  QSqlQuery query, metaquery;
  query.prepare( "INSERT INTO files ("
    "fileid, catalogid, fullname, filename, parentid, filetype, filesize, statuschangedate,"
    "modificationdate, lastaccessdate, lastupdatedate, username, groupname, permissions, mode, language) "
    "VALUES ("
    ":fileid, :catalogid, :fullname, :filename, :parentid, :filetype, :filesize, :statuschangedate,"
    ":modificationdate, :lastaccessdate, :lastupdatedate, :username, :groupname, :permissions, :mode, :language) " );
  metaquery.prepare("INSERT INTO metadata (fileid, groupname, field, type, value, unit) VALUES (:fileid, :groupname, :field, :type, :value, :unit)");

  UDSEntryListConstIterator it = entries.begin();
  UDSEntryListConstIterator end = entries.end();

  for ( ; it != end; ++it )
  {
    QApplication::eventLoop()->processEvents(QEventLoop::ExcludeSocketNotifiers);
    QString name;

    UDSEntry::ConstIterator entit = (*it).begin();
    for( ; entit != (*it).end(); ++entit )
      if ( (*entit).m_uds == UDS_NAME )
      {
        name = (*entit).m_str;
        break;
      }


    if ( name.isEmpty() || name == "." || name == "..")
      continue;

    KFileItem item( *it, url, false, true );

    // I need three strings:
    // 1. name of the file
    // 2. full path with filename and protocol of the file (file:/path/to/filename or zip:/path/to/filename.zip/path/inside/filename)
    // 3. full path with filename and without protocol to store fileid (/path/to/filename)
    // 4. full path without filename and protocol to retrieve parentid (/path/to/)

    // 1. Find filename: is the last token in the string we receive from listjob
    QStringList namePath = QStringList::split("/", name, false);
    QString filename = namePath.last();
    // 2. Build full url
    QString fullname = url.url(1) + name;
    // 3. Build full path to store fileid
    QString fullpath = url.path(1) + namePath.join("/");
    // 4. Build full path to find parentid
    namePath.pop_back();
    QString strippedpath;
    if(namePath.size() == 0)
      strippedpath = url.path(-1);
    else
      strippedpath = url.path(1) + namePath.join("/");

    int current_fileid = m_current_fileid;
    query.bindValue( ":fileid", m_current_fileid++ );
    query.bindValue( ":catalogid", m_current_catalogid );
    query.bindValue( ":fullname", fullname );
    query.bindValue( ":filename", filename );
    KatalogIds::Iterator it;
    it = m_fileids.find(strippedpath);
    emit currentFolder(fullname);
    //QCustomEvent *event = new QCustomEvent(KatalogScan::CURRENT_FOLDER);
    //event->setData( &fullname );
    //QApplication::postEvent(m_parent, event);
    query.bindValue( ":parentid", it == m_fileids.end() ? -1 : it.data() );
    query.bindValue( ":filetype", item.mimetype() );
    query.bindValue( ":filesize", item.size() );
    query.bindValue( ":statuschangedate", 0 );
    query.bindValue( ":modificationdate", (uint) item.time( UDS_MODIFICATION_TIME ) );
    query.bindValue( ":lastaccessdate", (uint) item.time( UDS_ACCESS_TIME ) );
    query.bindValue( ":lastupdatedate", 0 );
    query.bindValue( ":username", item.user() );
    query.bindValue( ":groupname", item.group() );
    query.bindValue( ":permissions", item.permissions() );
    query.bindValue( ":mode", 0 ); // TODO ???
    query.bindValue( ":language", 0 ); // TODO ???
    query.exec();

    if(m_getMetaInfo)
    {
      KFileMetaInfo minfo = item.metaInfo();

      if(minfo.isValid() && !minfo.isEmpty())
      {
        QStringList groupKeyList = minfo.supportedGroups();
        for ( QStringList::Iterator it = groupKeyList.begin(); it != groupKeyList.end(); ++it )
        {
          KFileMetaInfoGroup group = minfo.group(*it);
          if(!group.isEmpty())
          {
            QStringList itemKeyList = group.supportedKeys();
            for ( QStringList::Iterator it = itemKeyList.begin(); it != itemKeyList.end(); ++it )
            {
              KFileMetaInfoItem minfoitem = minfo.item(*it);
              QVariant v = minfoitem.value();
              if(!v.isValid() || v.isNull())
                continue;
              metaquery.bindValue( ":fileid", current_fileid );
              metaquery.bindValue( ":groupname", group.name() );
              metaquery.bindValue( ":field", minfoitem.translatedKey() );
              metaquery.bindValue( ":value", v );
              switch(v.type())
              {
                case QVariant::Int :
                  metaquery.bindValue( ":type", "int" );
                  break;
                case QVariant::Bool :
                  metaquery.bindValue( ":type", "bool" );
                  break;
                case QVariant::Date :
                  metaquery.bindValue( ":type", "date" );
                  break;
                case QVariant::DateTime :
                  metaquery.bindValue( ":type", "datetime" );
                  break;
                case QVariant::Double :
                  metaquery.bindValue( ":type", "double" );
                  break;
                case QVariant::LongLong :
                  metaquery.bindValue( ":type", "unsignedlonglong" );
                  break;
                case QVariant::Size : {
                  metaquery.bindValue( ":type", "size" );
                  QSize size = v.toSize();
                  metaquery.bindValue( ":value", QString("%1x%2").arg(size.width()).arg(size.height()) );
                  break;
                }
                case QVariant::String :
                default :
                  QString str = v.asString();
                  metaquery.bindValue( ":type", "string" );
                  break;
              }
              metaquery.bindValue( ":unit", minfoitem.unit() );
              metaquery.exec();
            }
          }
        }
      }
    }

    if( m_exploreArchives && url.isLocalFile() )
    {
      KURL subUrl;
      KServiceType::Ptr ptr = KServiceType::serviceType( item.mimetype() );
      if ( ptr )
      {
        const QString protocol = ptr->property("X-KDE-LocalProtocol").toString();
        if( !protocol.isEmpty() && item.mimetype() != "application/x-katalog" )
        {
          subUrl.setProtocol(protocol);
          subUrl.setPath( item.url().path());
          m_fileids[fullpath] = current_fileid;
          m_jobs.append(subUrl.url());
        }
      }
    }

    if(item.isDir()){
      m_fileids[fullpath] = current_fileid;
      emit newItem();
    }
  }

  if(hasTransactions)
    db->commit();
}

void KatalogScan::slotResult(Job *job)
{
  KURL url = static_cast<ListJob *>(job)->url();
  m_jobs.remove(url.url());

  if(m_jobs.isEmpty()){
    m_fileids.clear();
    emit finished(m_currentUrl.url());
  }
  else
  {
    KURL subUrl(*(m_jobs.begin()));
    m_current_job = listRecursive( subUrl, false );
    connect( m_current_job, SIGNAL( entries( KIO::Job *, const KIO::UDSEntryList & ) ),
             this, SLOT( slotEntries( KIO::Job *, const KIO::UDSEntryList & ) ) );
    connect( m_current_job, SIGNAL( result( KIO::Job * ) ),
             this, SLOT( slotResult( KIO::Job * ) ) );
    connect( m_current_job, SIGNAL( redirection( KIO::Job *, const KURL& ) ),
             this, SLOT( slotRedirection( KIO::Job *, const KURL& ) ) );
  }
}

void KatalogScan::slotRedirection(Job *job, const KURL& newUrl)
{
  KURL url = static_cast<ListJob *>(job)->url();

  KURL u(newUrl);

  m_jobs.remove(url.url());
  m_jobs.append(newUrl.url());
}

#include "katalogscan.moc"
