/**
 * @file
 * Worker thread implementation.
 *
 * Kisa provides spell checking as you type and displays the result in a small
 * window on your Desktop.
 *
 * Copyright (c) 2007 by Pete Black <theblackpeter@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * @author Pete Black <theblackpeter@gmail.com>
 */

#include "kisathread.h"


KisaThread::KisaThread(QObject* parent) : QThread(parent){
  isDone = false;

  // get the library instance
  kisaLib = KisaLib::instance();

  qDebug() << "kisathread: Thread created";
}


KisaThread::~KisaThread() {
  if(isRunning()) {
    qDebug() << "kisathread: Stopping thread";
    stop();
  }

  qDebug() << "kisathread: Thread Destroyed";
}


void KisaThread::run() {
  qDebug() << "kisathread: Runnung event listening thread...";

  KisaEvent* kisaEvent;

  // this is the main event loop
  while(!isDone) {
    // block until a new event is available in the X even queue
    kisaEvent = kisaLib->nextEvent();

    // KeyPress events
    if(kisaEvent->type() == KisaEvent::KeyPress) {
      // make a copy for later use, there is no parent object so no worries
      // about it being deleted from elsewhere
      lastKeyEvent = *kisaEvent;

      // key may be a null character ('\0')
      const QChar key = kisaEvent->key();

      emit gotNewCharacter(key);
    }

    // MapNotify and CreateNotify events
    // if window was created or made "visible", we want notification...
    else if(kisaEvent->type() == KisaEvent::WindowActivate)
      kisaLib->addNotification(kisaEvent->window(), KisaEvent::WindowActivate);

    // VisibilityNotify events
    // change in the applications window state
    else if(kisaEvent->type() == KisaEvent::WindowStateChange)
      emit visibilityStateChange(kisaEvent->state());

    delete kisaEvent;
  }
}


void KisaThread::startToggle() {
  if(isRunning())
    stop();

  else {
    // this is the flag that stops the event loop above
    isDone = false;

    qDebug() << "kisathread: Subscribing for event notification from all visible windows...";

    // re-initialize all subscriptions to the X server
    kisaLib->reAddNotification();

    qDebug() << "kisathread: Total number of windows:" << kisaLib->total();
    qDebug() << "kisathread: Starting thread";

    start();
  }
}


bool KisaThread::stop() {
  qDebug() << "kisathread: Stopping thread";

  if(!isRunning()) {
    qDebug() << "kisathread: Thread is not running, can't stop";
    return false;
  }

  // this is the flag that stops the event loop above
  isDone = true;

  // forcefully break any pending XNextEvent calls
  kisaLib->sendBogusXEvent();

  wait();

  // remove all subscriptions from the X server
  kisaLib->removeAllNotifications();

  qDebug() << "kisathread: Thread stopped";

  return true;
}


void KisaThread::replaceInClient(const QString& misspelled,
                                 const QString& suggestion) {
  // make sure that we have a valid event
  if(!lastKeyEvent.isNone())
    kisaLib->replaceInClient(lastKeyEvent.window(), misspelled, suggestion);
}
