/**
 * @file
 * Main window frame implementation.
 *
 * Kisa provides spell checking as you type and displays the result in a small
 * window on your Desktop.
 *
 * Copyright (c) 2007 by Pete Black <theblackpeter@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * @author Pete Black <theblackpeter@gmail.com>
 */

#include <QtDebug>
#include <QMenu>

#include "kisawidget.h"


KisaWidget::KisaWidget(QWidget* parent, Qt::WFlags fl) : QWidget(parent, fl),
                       Ui::KisaWidget() {
  setupUi(this);

  setWindowTitle("Kisa");

  // add a little more flavor to the default context menu for the lineEdit
  createLineEditContextMenu();

  qDebug() << "kisawidget: KisaWidget created";
}


KisaWidget::~KisaWidget() {
  qDebug() << "kisawidget: KisaWidget destroyed";
}


void KisaWidget::createLineEditContextMenu() {
  // get the default menu
  QMenu* menu = lineEdit->createStandardContextMenu();

  // create a map containing the action names and corresponding icons that we
  // want to add to the current default actions
  QMap<QString, QIcon> actionIconsMap;
  actionIconsMap["Undo"] = QIcon("icons:undo.png");
  actionIconsMap["Redo"] = QIcon("icons:redo.png");
  actionIconsMap["Cut"] = QIcon("icons:editcut.png");
  actionIconsMap["Copy"] = QIcon("icons:editcopy.png");
  actionIconsMap["Paste"] = QIcon("icons:editpaste.png");
  actionIconsMap["Delete"] = QIcon("icons:editclear.png");
  actionIconsMap["Select"] = QIcon("icons:edit-select-all.png");

  // add an icon to the corresponding action, previous order is unimportant
  QMapIterator<QString, QIcon> iconMapIterator(actionIconsMap);
  foreach(QAction* action, menu->actions()) {
    // skip separators
    if(action->isSeparator())
      continue;

    // clean up the string, remove shortcut keys and "&" signs
    QString actionText = action->text().simplified();
    actionText = actionText.split(" ").at(0);
    actionText = actionText.remove("&");

    // now finally add the icon it if we have a match between the action names
    if(actionIconsMap.contains(actionText))
      action->setIcon(actionIconsMap.find(actionText).value());
  }

  delete menu;
}


bool KisaWidget::isPartiallyObscured() {
  switch(visibilityState) {
    case VisibilityUnobscured:
      return false;

    case VisibilityPartiallyObscured:
      return true;

    case VisibilityFullyObscured:
      return true;

    default:
      ;
  }

  return false;
}


bool KisaWidget::isFullyObscured() {
  switch(visibilityState) {
    case VisibilityUnobscured:
      return false;

    case VisibilityPartiallyObscured:
      return false;

    case VisibilityFullyObscured:
      return true;

    default:
      ;
  }

  return false;
}


void KisaWidget::updateVisibilityState(const int& state) {
  visibilityState = state;
}


void KisaWidget::toggleLayoutDirection() {
  if(lineEdit->layoutDirection() == Qt::LeftToRight)
    lineEdit->setLayoutDirection(Qt::RightToLeft);

  else
    lineEdit->setLayoutDirection(Qt::LeftToRight);
}
