/**
 * @file
 * Main window frame header file.
 *
 * Kisa provides spell checking as you type and displays the result in a small
 * window on your Desktop.
 *
 * Copyright (c) 2007 by Pete Black <theblackpeter@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * @author Pete Black <theblackpeter@gmail.com>
 */

#ifndef KISAWIDGET_H
#define KISAWIDGET_H

#include "ui_kisawidget.h"

class QWidget;

/**
 * Unobscured visibility state for the window frame. Visibility notify state
 * used in X11/X.h
 */
#ifndef VisibilityUnobscured
#define VisibilityUnobscured 0
#endif


/**
 * Partially obscured visibility state for the window frame. Visibility notify
 * state used in X11/X.h
 */
#ifndef VisibilityPartiallyObscured
#define VisibilityPartiallyObscured 1
#endif


/**
 * Fully obscured visibility state for the window frame. Visibility notify state
 * used in X11/X.h
 */
#ifndef VisibilityFullyObscured
#define VisibilityFullyObscured 2
#endif


/**
 * @class KisaWidget
 * The KisaWidget class provides a main window frame for the application. It is
 * able to keep track of its visibility states continuously.
 *
 * A custom context menu is also provided for the text field, that uses system
 * icons for the default actions and access to the custom command.
 *
 * @image html kisa_dictionary_popup.png "Kisa main widget with a pop-up showing the recently used dictionaries"
 *
 * @author Pete Black <theblackpeter@gmail.com>
 * @see Kisa
 */
class KisaWidget : public QWidget, public Ui::KisaWidget {
  Q_OBJECT

  public:
    /**
     * Constructs a widget which is a child of @p parent, with widget
     * window flags set to @p fl. There is no parent object as this is the main
     * window widget.
     *
     * To customize the frame, use a value composed from the bitwise OR of any
     * of the window flags. Default flags are:
     * "Qt::Tool | Qt::WindowStaysOnTopHint".
     *
     * @param parent pointer to the the parent widget, must be 0
     * @param fl set of window flags conjoined using the bitwise logical
     * manipulator OR, "|"
     */
    KisaWidget(QWidget* parent = NULL, Qt::WFlags fl = 0);


    /**
     * Destructor.
     */
    ~KisaWidget();

    /**
     * Checks the visibility state of the widget. If the widget is partially
     * obscured by any other windows on the desktop @c true is returned,
     * otherwise @c false.
     *
     * @see isFullyObscured()
     *
     * @return @c true if the widget window is partially obscured, @c false
     * otherwise
     */
    bool isPartiallyObscured();


    /**
     * Checks the visibility state of the widget. If the widget is fully
     * obscured by any other windows on the desktop, @c true is returned,
     * otherwise @c false.
     *
     * @note if the option "Stay on top" is selected then this method will
     * always return @c false.
     *
     * @see isPartiallyObscured()
     * @return @c true if the widget window is fully obscured, @c false
     * otherwise
     */
    bool isFullyObscured();


  public slots:
    /**
     * Updates the visibility state of the widget window to that given. The
     * possible X11 visibility* states are:
     *  @li VisibilityUnobscured
     *  @li VisibilityPartiallyObscured
     *  @li VisibilityFullyObscured
     *
     * @param state the new state
     */
    void updateVisibilityState(const int& state);

    /**
     * Convenience slot to toggle the input direction for the text field by
     * calling QWidget::setLayoutDirection() with the opposite direction
     * (Qt::LeftToRight or Qt::RightToLeft).
     */
    void toggleLayoutDirection();


  private:
    /**
     * Creates a custom context menu.
     *
     * Adds system icons to the corresponding
     * menu actions and the custom command action at the bottom.
     *
     * The action icons are selected from the applications resource path, that
     * is set by KisaLib::setIconsSearchPaths(). This is usually the current
     * Desktop icons used, but they can also be the system default or none.
     *
     * see contextMenuEvent()
     */
    void createLineEditContextMenu();


    /**
     * Holds the main window widget visibility state. The possible X11
     * visibility states are:
     *  @li VisibilityUnobscured
     *  @li VisibilityPartiallyObscured
     *  @li VisibilityFullyObscured
     */
    int visibilityState;
};

#endif
