/**
 * @file
 * Library header file.
 *
 * Kisa provides spell checking as you type and displays the result in a small
 * window on your Desktop.
 *
 * Copyright (c) 2007 by Pete Black <theblackpeter@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * @author Pete Black <theblackpeter@gmail.com>
 */

#ifndef KISALIB_H
#define KISALIB_H

#include <QApplication>
#include <QStringList>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <aspell.h>

#include "kisaevent.h"

/**
 * @class KisaLib
 * The KisaLib class provides access to library resources that interface
 * with external libraries, such as X11 and Aspell. All external requests are
 * made from here and then translated back into internal data structures.
 * These are then used by the main application and worker threads.
 *
 * Even though the @em Singleton design pattern is used, make sure that the
 * library is proparly initialized before making any global calls. This
 * is done by acquiring an instance via instance() and then calling
 * initialize(), like so:
 * @code
 *   KisaLib* kisaLib = KisaLib::instance();
 *   kisaLib->initialize();
 * @endcode
 *
 * After doing this, the library is fully initialized and calls can be
 * made from any location that has knowledge about KisaLib. For example, to
 * get a list of suggestions if a word is misspelled, one could do something
 * like this:
 *
 * @code
 *  #include "kisalib.h"
 *  ...
 *  QString myWord;
 *  QStringList suggestions;
 *  ...
 *  if(KisaLib::isMisspelled(myWord))
 *    suggestions = KisaLib::getSuggestions();
 *  ...
 * @endcode
 *
 * When subscribing to X11 events that originate from other windows on the
 * Desktop, these event masks are used:
 *  @li @c KeyPressMask
 *  @li @c SubstructureNotifyMask
 *  @li @c VisibilityChangeMask
 *
 * @c NoEventMask is used to remove notification subscriptions.
 *
 * The above listed event masks generate the following X11 events that are of
 * interest:
 *  @li @c KeyPress, for whenever a key is pressed in any window
 *  @li @c VisibilityNotify, for whenever our @em own application becomes
 *      obstructed or unobstructed
 *  @li @c CreateNotify, for whenever a new window is created
 *  @li @c MapNotify, for whenever a window is made visible
 *  @li @c ConfigureNotify, for whenever a window is reconfigured, for example
 *      when the user clicks in it
 *
 * These are translated to the following KisaEvent events:
 *  @li @c KeyPress to KisaEvent::KeyPress
 *  @li @c VisibilityNotify, @c CreateNotify and @c MapNotify to
 *      KisaEvent::WindowActivate
 *
 * When using the library for subscribing to X11 events, use
 * addNotification(const Window&, KisaEvent::Type type), which uses the
 * KisaEvent type rather then changeNotification(const Window&, EventMask),
 * which uses the X11 event mask. The latter is intended for internal use.
 *
 * @code
 *  ...
 *  KisaEvent* kisaevent = KisaLib::nextEvent();
 *  addNotification(kisaevent->window(), KisaEvent::KeyPress);
 *  ...
 * @endcode
 *
 * @author Pete Black <theblackpeter@gmail.com>
 * @see Kisa
 * @see KisaThread
 * @see KisaEvent
 * @see x11fix.h
 */
class KisaLib {

  public:
    /**
     * Destructor. Reverts back to the original X11 error handler, closes the
     * connection to the X11 server and destroys any dynamically created
     * objects.
     */
    ~KisaLib();


    /**
     * Accesses the current instance of the library or creates a new one if not
     * instantiated before. Classic @em Singleton design pattern approach.
     *
     * No restrictions are posed on @p displayString, as it's simply passed on
     * to the constructor if and when a new instance is created. If an instance
     * already exists @p displayString is ignored.
     *
     * @param displayString the X11 display, or ":0" if none is given
     * @return the current instance
     * @see KisaLib(QString)
     */
    static KisaLib* instance(const QString& displayString = QString());


    /**
     * Initializes the library so that it is ready for use. If the library has
     * not been instantiated before, by calling instance(), it will be done
     * here.
     *
     * Parameters are just passed along and not manage here.
     * See the referencing functions for more detailes on the parameters.
     *
     * @note The library can be used first after this method is called.
     *
     * @param iconspath the path to the icon pixmaps
     * @param flagspath the path to the flags icon pixmaps
     * @param language the default language to use, ISO 639 and ISO 3166 format
     * @param dictDir the main word list location
     * @param aspellPrefix the Aspell prefix directory
     * @param extraDicts the extra dictionaries to use
     */
    void initialize(const QString& iconspath = QString(),
                    const QString& flagspath = QString(),
                    const QString& language = QString(),
                    const QString& dictDir = QString(),
                    const QString& aspellPrefix = QString(),
                    const QString& extraDicts = QString());


    /**
     * Access method that gets current X11 display used.
     *
     * @return the current X11 server display used
     */
    Display* display() { return m_display; }


    /**
     * Gets the X11 window id of the root window (desktop window) used.
     *
     * @return the X11 root window
     */
    Window rootWindow() { return m_rootWindow; }


    /**
     *
     * @return the total number of windows we are subscribing X11 events from
     */
    int total() { return m_total; }


    /**
     * Gets the window name associated with the X11 window ID @p window. If the
     * window isn't available or doesn't have a name an empty string, @c "", is
     * returned.
     *
     * @param window the X11 window id
     * @return the window name or an empty string, @c ""
     */
    QString getWindowName(const Window& window);


    /**
     * Changes the notification subscription for the window with id
     * @p window and all its visible children using the supplied event mask.
     * If no event mask is given, KeyPressMask and StructureNotifyMask are
     * used. SubstructureNotifyMask, as apposed to StructureNotifyMask, is used
     * because we want information about the window that originated the event
     * and not its parent.
     *
     * Events will be reported relative to window @p window.
     *
     * Only subscription to windows that qualify will be made. That is, windows
     * that are actual application windows, are visible and not in the ignore
     * list. This is decided by isWindowDecent().
     *
     * @warning This is a recursive function that evaluates every window on the
     * X11 server starting from @p window. Calling this function too often, with
     * the root window for instance, may reduce performance significantly.
     *
     * @param window the X11 window id used as a starting point
     * @param eventMask specifies a disjunction of X11 event masks. That is,
     * a bit field containing the types of events we want notification for.
     * Append masks using the bitwise logical manipulator OR, "|".
     *
     * @see isWindowDecent()
     * @see changeNotification()
     * @see addNotification()
     * @see removeNotification()
     * @see Xlib::XSelectInput()
     */
    void changeDeepNotification(const Window& window,
                                const EventMask& eventMask = KeyPressMask
                                    | SubstructureNotifyMask);


    /**
     * Changes the notification subscription only for the window, and none of
     * its children, using the suplied event mask.
     *
     * Only subscription to windows that qualify will be made That is, windows
     * that are actual application windows, are visible and not in the ignore
     * list. This is decided by isWindowDecent().
     *
     * @param window the X11 window id used as a starting point
     * @param eventMask specifies a disjuntion of X11 event masks. That is, a
     * bit field containing the types of events we want notification for.
     * Append masks using the bitwise logical manipulator or, "|".
     *
     * @see ignoreList()
     * @see isWindowDecent()
     * @see changeNotification()
     * @see addNotification()
     * @see removeNotification()
     * @see changeDeepNotification()
     * @see Xlib::XSelectInput()
     */
    void changeNotification(const Window& window, EventMask eventMask);


    /**
     * Adds notification subscription for the window with id @p window, using
     * the suplied event type @p type. If no event type is given KisaEvent::None
     * is used.
     *
     * Only subscription to windows that qualify will be made, that is, windows
     * that are actual application windows, are visible and not in the ignore
     * list. This is decided by isWindowDecent().
     *
     * @warning this function uses changeDeepNotification() and as such verifies
     * that the window is not the root window in order to avoid latency issues.
     * If that's the case no notification will be made. If you are planning to
     * add notification to a single window use changeNotification() instead.
     * This is intended for outside use, whenever a new window is mapped (made
     * "visible") and there is no knowledge about the event mask.
     *
     * @param window the X11 window id used as a starting point
     * @param type the event type or if none is given KisaEvent::None is used
     *
     * @see ignoreList()
     * @see isWindowDecent()
     * @see changeNotification()
     * @see changeDeepNotification()
     * @see addNotification()
     * @see removeNotification()
     */
    void addNotification(const Window& window,
                          KisaEvent::Type type = KisaEvent::None);


    /**
     * Convenience function to call changeDeepNotification() with the
     * @c rootWindow and default event masks as parameters.
     *
     * @see rootWindow()
     * @see changeDeepNotification()
     */
    void reAddNotification();


    /**
     * Removes notification subscription for the given window.
     *
     * @param window the X11 window id to remove notification subscription for
     *
     * @see changeNotification()
     * @see addNotification()
     */
    void removeNotification(const Window& window);


    /**
     * Removes all previous made notification subscriptions, for all windows.
     *
     * @see changeDeepNotification()
     */
    void removeAllNotifications();


    /**
     * Calls XNextEvent() to get the first event from the event queue and
     * returns the KisaEvent equivalent. The function block and will wait for
     * the next event before returning. Use sendBogusXEvent() to unblock.
     * If the event type is not supported an event pointer of type
     * KisaEvent::None will be returned.
     *
     * These XEvents with corresponding KisaEvents even are implemented
     * @li @c KeyPress @c KisaEvent::KeyPress
     * @li @c CreateNotify @c KisaEvent::WindowActivate
     * @li @c VisibilityNotify @c KisaEvent::WindowStateChange
     * @li any other XEvent @c KisaEvent::None
     *
     * @return a pointer to the KisaEvent
     * @see sendBogusXEvent()
     * @see Xlib::XNextEvent()
     */
    KisaEvent* nextEvent();


    /**
     * Converts an XKeyEvent keycode to the equivalent character. If the
     * conversion fails an empty null character is return, '\\0'
     *
     * @param xKey the XKeyEvent structure containing the keycode
     * @return the corresponding character or null character, '\\0'
     * @see Xlib::XLookupString()
     */
    QChar lookupKey(XKeyEvent& xKey);


    /**
     * Sends a bogus XEvent to the X11 server in order to break any blocking
     * XNextEvent call in nextEvent().
     *
     * @see nextEvent()
     */
    void sendBogusXEvent();


    /**
     * Sets the value of the ignore list member variable. The ignore list is
     * used to match any window names to ignore when subscribing for
     * notification.
     *
     * @param regExpList an regular expression formatted string of window names
     * @see ignoreList()
     * @see changeNotification()
     * @see addNotification()
     */
    void setIgnoreList(const QString& regExpList) {
      m_ignoreList = regExpList;
    }


    /**
     * Quires the spell checker to with the given word. If the word is not in
     * the spell checker's word list @c true is returned, otherwise @c false.
     *
     * @param word the word to checker
     * @return @c true if the word is not found in the current word list,
     * @c false otherwise
     */
    bool isMisspelled(const QString& word);


    /**
     * Quires the spell checker to get a list of suggestions for the word.
     * Usually the word is misspelled, that is not located in the word list and
     * we want a set of alternatives.
     *
     * @param word the word to get suggestions for
     */
    QStringList getSuggestions(const QString& word);


    /**
     * Gets all the known dictionaries that where located when the spell checker
     * was initialized. These are in a user friendly format, perfectible
     * readable natural language.
     *
     * @return the dictionaries in natural language format
     * @see dictionaryCodes()
     */
    QStringList dictionaries() {
      return m_dictionaries;
    }


    /**
     * Gets all the known dictionary codes that where located when the spell
     * checker was initialized. These are in language code, up to 5 characters
     * in specified by the ISO 639 and ISO 3166 standards. The code is on the
     * form @c language[_COUNTRY].
     *
     * @note Some dictionaries only have the language code associated with them.
     *
     * @return the dictionaries in language/country code format
     * @see dictionaries()
     * @see currentDictionaryCode()
     */
    QStringList dictionaryCodes() {
      return m_dictionaryCodes;
    }


    /**
     * Gets all the Qt locale best guesses of the dictionaries language/country
     * codes. These codes are 5 characters long specified by the ISO 639 and ISO
     * 3166 standards. The code is on the form @c language_COUNTRY.
     *
     * @note Qt's locale system is used in order to make a best match for the
     * country given the language. This might not always be accurate as there
     * are languages that are spoken in several countries.
     *
     * @return the best guesses of the dictionaries in language/country code
     * format
     * @see dictionaries()
     * @see currentDictionaryCodeGuess()
     */
    QStringList dictionaryCodesGuessed() {
      return m_dictionaryCodesGuessed;
    }


    /**
     * Gets the index of the current dictionary in the dictionaries list.
     *
     * @return the current index of the dictionary used
     * @see currentDictionary()
     */
    int currentDictionaryIndex() {
      return m_currentDictionaryIndex;
    }


    /**
     * Gets the current dictionary used by the spell checker, that is the name
     * of the dictionary in natural language.
     *
     * For instance, the name of the dictionary given by the code @c en_US will
     * be translated to English (United States).
     *
     * If there is only a language code available the dictionary variant
     * (common) will be attached to the name. For instance, the name of the
     * dictionary given by the code @c en will be translated to English
     * (common).
     *
     * @note Some dictionaries only have the language code associated with them.
     *
     * @return the current dictionary
     * @see currentDictionaryCode()
     */
    QString currentDictionary() {
      return m_dictionaries.at(m_currentDictionaryIndex);
    }


    /**
     * Gets the current dictionary's true language/country code as represented
     * when the spell checker was initialized. These codes are up to 5
     * characters specified by the ISO 639 and ISO 3166 standards. The code is
     * on the form @c language[_COUNTRY].
     *
     * For example, an American English dictionary will have the code: @c en_US
     *
     * @note Some dictionaries are only given by language code. For instance the
     * dictionary given by the code @c en. No attempt will be made to guess the
     * appropriate country.
     *
     * @return the current dictionary's code
     * @see currentDictionaryCodeGuess()
     */
    QString currentDictionaryCode() {
      return m_dictionaryCodes.at(m_currentDictionaryIndex);
    }


    /**
     * Gets the Qt locale best guess of the current dictionary language/country
     * code. These codes are 5 characters long specified by the ISO 639 and ISO
     * 3166 standards. The code is on the form @c language_COUNTRY.
     *
     * For example, an English dictionary, given by the language code @c en will
     * be matched to the code: @c en_US.
     *
     * @note Qt's locale system will be used in order to make a best match for
     * the country given the language. This might not always be accurate as
     * there are languages that are spoken in several countries.
     *
     * @return the best guess of the current dictionary's code
     * @see currentDictionaryCode()
     */
    QString currentDictionaryCodeGuess() {
      return m_dictionaryCodesGuessed.at(m_currentDictionaryIndex);
    }


    /**
     * Initializes the spell checker and sets the dictionary to that of the
     * specified language. If no language is give the default system locale
     * language will be used.
     *
     * @param language the language to use, or if none is given the system
     * default locale language will be used
     * @param dictDir the main word list location
     * @param aspellPrefix the Aspell prefix directory
     * @param extraDicts the extra dictionaries to use
     */
    void initializeAspell(QString language = QString(),
                          const QString& dictDir = QString(),
                          const QString& aspellPrefix = QString(),
                          const QString& extraDicts = QString());


    /**
     *  Updates the current dictionary to that of the given index.
     *
     * @param index the index of the new dictionary
     * @see updateDictionary(const QString&)
     */
    bool updateDictionary(const int& index);


    /**
     * Updates the current dictionary to that of the given language.
     *
     * The language can be either the full ISO 639 and ISO 3166 code, or just
     * the two character language code. The code format is
     * @c language[_COUNTRY].
     *
     * @param language the new language of the dictionary
     * @see updateDictionary(const int&)
     */
    bool updateDictionary(const QString& language);


    /**
     * Gets the current keyboard layout locale code in ISO 639 and ISO 3166
     * format.
     * For example, a default American English keyboard layout (sometimes
     * donate by the code "C") will return the locale code:
     * @c en_US.
     *
     * @warning This method may produces incorrect results on some systems.
     *
     * @bug kxkb stops updating the X11 keyboard layout whenever switching back
     * to and already seen layout.
     * This can be verifed this using @c setxkbmap @c -print.
     *
     * @bug On some systems QApplication::keyboardInputLocale() does not always
     * return the right layout. For instance, it may be needed to change the
     * layout twice using @c setxkbmap @c -layout @c \<layout\>  before
     * QApplication::keyboardInputLocale() returns the right layout.
     * This can be verified this using @c setxkbmap @c -print.
     *
     * @return the current keyboard layout locale code
     */
    QString getKeyboardInputLayout();


    /**
     * Returns a string containing the location of the current country flag
     * pixmap, given by the two character ISO 3166 country code. If none is
     * found an empty string is returned.
     *
     * @param countryCode the two character country code in ISO 3166 format
     * @return the location of the pixmap, or an empty string, ""
     */
    QString getCountryFlagURL(QString countryCode);


    /**
     * Replaces the last occurrence of @p misspelled with @p suggestion in the
     * client window with id @p window. This is done by first sending equal
     * number of backspace, '\\b' characters as the string @p misspelled has
     * and then sending the string @p suggestion, character by character.
     *
     * @warning Be careful when doing this, as we can't be sure that we are
     * replacing the right word in the right window. There is no way of knowing
     * what exactly is going on in any arbitrary window on the desktop.
     * Unexpected behavior may occur.
     *
     * @param window the X11 window id of the client application
     * @param misspelled the word to replace, that is the misspelled word
     * @param suggestion the word new replacement word, that is the suggestion
     * selected by the user
     */
    void replaceInClient(const Window& window, const QString& misspelled,
                          const QString& suggestion);


    /**
     * Sets icons resource path, that is, where to look for icon pixmaps.
     *
     * If a faulty or no path is given, then the current KDE user icon theme an
     * d default path will be used (/usr/share/icons/default.kde).
     *
     * Currently, no GNOME support is available, so pass the icon location as a
     * command line argument.
     *
     * @param path the path to the icon pixmaps, or default values will be used
     * if none is given
     */
    void setIconsSearchPaths(QString path = QString());


    /**
     * Sets the flag icons resource path. If @p path is not given then
     * @c "/usr/share/locale/l10n" is tried (standard in many distros).
     *
     * When all else fails, then the last path of the default
     * KDE locale path (given by "kde-config --path locale") will be
     * used as a prefix.
     *
     * @param path the path to the flag icon pixmaps, or the default value will
     * be used if none is given
     */
    void setFlagsSearchPaths(QString path = QString());


    /**
     * Verifies if the window is visible or not. That is, if the window is
     * mapped and is viewable.
     *
     * @param window the X11 window id to checker
     * @return @c true if the window is visible, @c false otherwise
     * @see isWindowDecent()
     */
    bool isWindowVisible(const Window& window);


    /**
     * Checks if the given window is usable for the application. That is, if the
     * window id is valid, has usable attributes, is mapped, viewable and so on.
     *
     * @param window the X11 window id to checker
     * @return @c true if the window is usable, @c false otherwise
     * @see isVisible()
     */
    bool isWindowDecent(const Window& window);


    /**
     * Convenience method to print notification information whenever a
     * subscription change was made.
     *
     * @param window the X11 window id that the change was made to
     * @param eventMask the new event mask used to change notification
     * subscription
     */
    void printNotificationOutput(const Window& window,
                                  const EventMask& eventMask);

    /**
     * Checks whether the given window is actually our own application window.
     *
     * @param window the X11 window id to check
     * @return @c true if so, @c false otherwise
     */
    bool isOurOwnWindow(const Window& window);


    /**
     * Adds the word @p word to the personal word list.
     *
     * @param word the word to add
     */
    void addToPersonalWordList(const QString& word);


    /**
     * Adds the word @p word to the session list. Subsequent occurrences of this
     * word will be ignored by the spell checker for the reminder of the session.
     *
     * @param word the word to add
     */
    void addToSessionWordList(const QString& word);


  protected:
    /**
     * Constructor. Creates a Singleton instance of the library and connects to
     * the X11 server display given by @p displayString. If a faulty or no
     * display is given the display in the environment varaiable $DISPLAY will
     * be used. Failing that, the default display, ":0", will by used.
     *
     * @param displayString the X11 server display to connect to, if non is
     * given then the display specified in $DISPLAY will be used or ":0" if that
     * fails
     * @see instance()
     */
    KisaLib(QString displayString = QString());


  private:
    /**
     * Overrides the default X11 error handler, so that X11 error messages can
     * be parsed to a more suitable format.
     *
     * Also, there is no suitable way to verify if a window exists or not
     * (remember events occur asynchronously), so many of the X11 functions
     * used will generate "BadWindow" errors whenever using a window that is
     * not defined.
     *
     * @param d pointer to the current display
     * @param xe pointer to error event struct
     * @return see X11/Xlib documentation
     */
    static int setErrorHandler(Display *d, XErrorEvent *xe);


    /**
     * Convenience method to print error output from the spell checker.
     */
    void printAspellErrorAndExit(QString);


    /**
     * Pointer to the current instance of the library.
     *
     * @see instance()
     */
    static KisaLib* m_instance;


    /**
     * Pointer to the Apsell spell cheker object.
     */
    AspellSpeller* m_aspellSpeller;


    /**
     * Specifies the default connection to the X11 server acquired from the
     * $DISPLAY environment variable.
     */
    Display* m_display;


    /**
     * Specifies the default root window, also known as the desktop window. All
     * windows created by the window manager are children of this window.
     */
    Window m_rootWindow;


    /**
     * The total number of windows we are subscribing X11 notification events
     * from.
     */
    int m_total;


    /**
     * Specifies a regular expression formatted string that contains a
     * disjuntion of window names that should be ignored when subscribing for
     * notification events
     *
     * Matching is made on any of the strings in the ignoreList.
     * The stirng should be in the format of "(word1|word2|...|wordN)".
     */
    QString m_ignoreList;


    /**
     * Holds all the known dictionaries that where located when the spell
     * checker was initialized. These are in a user friendly format, a
     * perfectible readable natural language.
     *
     * For example, the name of the dictionary given by the code @c en_US will
     * have the name "English (United States)".
     *
     * If there is only a language code available the dictionary variant
     * (common) will be attached to the name. For instance, the name of the
     * dictionary given by the code @c en will be translated to "English
     * (common)".
     *
     * @see dictionaries()
     */
    QStringList m_dictionaries;


    /**
     * Holds the known dictionary codes that where located when the spell
     * checker was initialized. These are in language code, up to 5 characters
     * in specified by the ISO 639 and ISO 3166 standards. The code is on the
     * form @c language[_COUNTRY].
     *
     * For example, an American English dictionary will have the code @c en_US
     *
     * @note Some dictionaries only have the language code associated with them.
     *
     * @see dictionaryCodes()
     */
    QStringList m_dictionaryCodes;


    /**
     * Holds the Qt locale best guess of the current dictionary language/country
     * code. These codes are 5 characters long specified by the ISO 639 and ISO
     * 3166 standards. The code is on the form @c language_COUNTRY.
     *
     * For example, an English dictionary, given by the language code @c en will
     * be matched to the code: @c en_US.
     *
     * @note Qt's locale system will be used in order to make a best match for
     * the country given the language. This might not always be accurate as
     * there are languages that are spoken in several countries.
     *
     * @see dictionaryCodesGuessed()
     */
    QStringList m_dictionaryCodesGuessed;


    /**
     * Holds the current index of the dictionary in use.
     *
     * @see currentDictionaryIndex()
     */
    int m_currentDictionaryIndex;


    /**
     * Previous X11 error handler.
     *
     * @see setErrorHandler()
     */
    static XErrorHandler m_oldXErrorHandler;


    /**
     * The possible X11 visibility states.
     *
     * @note @c LASTEvent is defined in X11/X.h
     */
    QString m_eventTypes[LASTEvent];


    /**
     * The possible X11 visibility states.
     */
    QString m_visibilityStates[3];
};
#endif
