/*  KRunning - DigitalClock.cpp
 *  Copyright (C) 2002 by Andreas Scherf
 *  Begin Mon Mai 6 2002 write to scherfa@web.de if you found a bug.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "DigitalClock.h"

// DEBUG
#include <iostream>

#include <qvgroupbox.h>
#include <qlabel.h>
#include <qlcdnumber.h>
#include <qdatetime.h>
//
// Constructs a DigitalClock widget with a parent and a name.
//

DigitalClock::DigitalClock (QWidget * parent, const char *name) : QWidget (parent, name)
{

    vboxLayout = new QVBoxLayout (this, 0, 4);
    hboxLayout = new QHBoxLayout (0, 0, 6, "hboxLayout");

    label = new QLabel (this);
    //label->setText();
    vboxLayout->addWidget (label);

    lcdNumber = new QLCDNumber (this);
    lcdNumber->setSegmentStyle (QLCDNumber::Filled);
    lcdNumber->setNumDigits (8);

    lcdNumber->setLineWidth (2);	// set frame line width
    vboxLayout->addWidget (lcdNumber);

    sepLine = new QFrame (this);
    sepLine->setFrameStyle (QFrame::HLine | QFrame::Sunken);
    vboxLayout->addWidget (sepLine);

    spacer = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
    hboxLayout->addItem( spacer );
    message = new QLabel (this);

    message->setPaletteForegroundColor(QColor(255,0,0));
    /*v*/
    hboxLayout->addWidget (message);

    hboxLayout->addItem( spacer );
    vboxLayout->addLayout(hboxLayout);

    showStartTimer = -1;		// not showing starttime
	normalTimer = -1;

    m_bShowStartTimeOnly = false;
    m_bShowCurrentTime = false;
    m_sTimeFormat=QString("hh:mm:ss");
}


DigitalClock::~DigitalClock ()
{
    delete label;
    delete lcdNumber;
    delete sepLine;
    delete message;
	delete spacer;
	delete vboxLayout;
	delete hboxLayout;
}

void DigitalClock::start () 
{
    showTime ();			// display the current time
	if (normalTimer != -1)
	{
		killTimer(normalTimer);
	}
	
	if (m_sTimeFormat=="hh:mm:ss.zzz") 
	{
        lcdNumber->setNumDigits (12);
        normalTimer=startTimer(1);
    }
	else 
	{
        lcdNumber->setNumDigits (8);
        normalTimer = startTimer (500);
    }	// 1/2 second timer events        500
}



//
// Handles timer events for the digital clock widget.
// There are two different timers; one timer for updating the clock
// and another one for switching back from date mode to time mode.
//

void DigitalClock::timerEvent (QTimerEvent * e) {
    if (e->timerId () == showStartTimer)	// stop showing date
        stopStartTime ();
    else if (showStartTimer == -1 && !m_bShowStartTimeOnly) {	// not showing date
        showTime ();
    } else {
        showStartTime ();
    }
}

//
// Enters date mode when the left mouse button is pressed.
//

void DigitalClock::mousePressEvent (QMouseEvent * e) 
{
    if (e->button () == QMouseEvent::LeftButton)	// left button pressed
        showStartTime ();
}




/** getTimeDiff - Main Function which calculates the Time Difference from Start to Now time.
 ** @author Andreas Scherf
 ** @version 0.5
 **/
QTime DigitalClock::getTimeDiff (const QTime & t) 
{
    QTime           oResultTime;

    oResultTime = oResultTime.addMSecs (m_oStarttime.msecsTo(t));
	
    if (oResultTime.isValid ()) 
	{
        return oResultTime;
    } 
	else 
	{
        return QTime::currentTime ();
    }
}


//
// Shows the current date in the internal lcd widget.
// Fires a timer to stop showing the date.
//

void DigitalClock::showStartTime () 
{
	// already showing date
    if (showStartTimer != -1)  return;

    lcdNumber->display (m_oStarttime.toString(m_sTimeFormat));	// sets the LCD number/text
    showStartTimer = startTimer (2000);	// keep this state for 2 secs
}

//
// Stops showing the date.
//

void DigitalClock::stopStartTime ()
{
    killTimer (showStartTimer);
    showStartTimer = -1;
    showTime ();
}

//
// Shows the current time in the internal lcd widget.
//

void DigitalClock::showTime () 
{
    QTime oTime;
    if (m_bShowCurrentTime) 
	{
    	oTime = QTime::currentTime();
    } 
	else 
	{
    	oTime = getTimeDiff (QTime::currentTime ());//.left(5);
    }
    lcdNumber->display (oTime.toString (m_sTimeFormat));	// set LCD number/text
}

QString DigitalClock::getCurrentRunningTime () 
{
    return (getTimeDiff (QTime::currentTime ()).toString (m_sTimeFormat));
}




void DigitalClock::setShowCTime (bool flag) {
    m_bShowCurrentTime=flag;
}

void DigitalClock::setStartTime (const QString & time) {
    m_oStarttime = QTime::fromString (time, Qt::TextDate);
}

void DigitalClock::setMessageText (const QString & text) {
    message->setText (text);
}

void DigitalClock::setMessageText (const int &number) {
    message->setNum (number);
}

void DigitalClock::setMessageFont (const QFont & font) {
    message->setFont( font );
}

void DigitalClock::setLabelText (const QString & text) {
    label->setText (text);
}

void DigitalClock::setDisplayTime(const QString &dtime) {
    lcdNumber->display(dtime);
}
