/*  KRunning - krunning.cpp
 *  Copyright (C) 2002 by Andreas Scherf
 *  Begin Mon Mai  6 11:16:42 CEST 2002 write to scherfa@web.de if you found a bug.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
// include files for QT
#include <qdir.h>

#include <qpainter.h>

// include files for KDE
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <klocale.h>
#include <kconfig.h>
#include <kstdaction.h>
#include <kpopupmenu.h>
#include <kprinter.h>

// application specific includes
#include "krunning.h"
#include "krunningview.h"
#include "krunningdoc.h"
#include "DigitalClock.h"
#include "Preferences.h"
#include "Time.h"

#include <iostream>

using namespace std;

#define ID_STATUS_MSG 1

// Global Database Connection
KRunningUseCase *g_pMainUseCase;
// Global Preferences
Preferences * prefs;

KRunningApp::KRunningApp(QWidget* , const char* name):KMainWindow(0, name)
{
	KGlobal::dirs()->addResourceType("pictures", KStandardDirs::kde_default("data") + "krunning/pictures/");
	config=kapp->config();

	//     KStandardDirs kdehome;

	///////////////////////////////////////////////////////////////////
	// call inits to invoke all other construction parts
	initStatusBar();
	initActions();
	readOptions();

	// Init the Global Preferences
	prefs= new Preferences();
	prefs->init();

	// Now the database Connection must be initiated
	g_pMainUseCase = new KRunningUseCase();

	initDocument();

	if (!g_pMainUseCase->isConnected())
	{
		slotConnectDB();
	}
	
	
	
	if (g_pMainUseCase->isConnected())
	{
		initView();

		toolBar("mainToolBar")->setEnabled(true);	
		view->setEnabled(true);
		connectDB->setEnabled(false);
		disconnectDB->setEnabled(true);
		mergeDB->setEnabled(true);
		setCaption(g_pMainUseCase->getConnection()->getDatabaseName(),false);
		// Now could i fill up events
		initEventSelector();
	}
	else
	{
		//view->setEnabled(false);
		connectDB->setEnabled(true);
		disconnectDB->setEnabled(false);
		mergeDB->setEnabled(false);
	}


	// disable actions at startup
	fileSave->setEnabled(false);
	fileSaveAs->setEnabled(false);
	filePrint->setEnabled(true);
}

KRunningApp::~KRunningApp()
{
	if (g_pMainUseCase)
		delete(g_pMainUseCase);
	if (prefs)
		delete (prefs);
	this->close();
	exit(0);
}



void KRunningApp::initActions()
{

	connectDB = new KAction(i18n("&Connect"),QIconSet(BarIcon("connect_creating")),0,this,SLOT(slotConnectDB()),actionCollection(),"connect");
	disconnectDB = new KAction(i18n("&Disconnect"),QIconSet(BarIcon("connect_no")),0,this,SLOT(slotDisconnectDB()),actionCollection(),"disconnect");
	//------------
	mergeDB = new KAction(i18n("Merge"),QIconSet(BarIcon("message_reply")),0,this,SLOT(slotMergeDB()),actionCollection(),"merge");
	//------------
	importFile = new KActionMenu(i18n("&Import File"),QIconSet(BarIcon("fileimport")), this,"importFile");
	fileImportTimes = new KAction(i18n("Times"),QIconSet(BarIcon("clock")),0,this,SLOT(slotImportTimes()),actionCollection());
	fileImportTeams = new KAction(i18n("Teams"),QIconSet(BarIcon("kuser")),0,this,SLOT(slotImportTeams()),actionCollection());
	importFile->insert(fileImportTimes);
	importFile->insert(fileImportTeams);

	exportFile = new KActionMenu(i18n("&Export File"),QIconSet(BarIcon("fileexport")), this,"exportFile");
	fileExportHTML = new KAction(i18n("HTML"),QIconSet(BarIcon("html")),0,this, SLOT(slotExportHTML()), actionCollection());
	fileExportTimes = new KAction(i18n("Times"),QIconSet(BarIcon("clock")),0,this,SLOT(slotExportTimes()),actionCollection());
	fileExportTeams = new KAction(i18n("Teams"),QIconSet(BarIcon("kuser")),0,this,SLOT(slotExportTeams()),actionCollection());
	exportFile->insert(fileExportHTML);
	exportFile->insert(fileExportTeams);

	exportFile->insert(fileExportTimes);
	//-----------
	filePrint = KStdAction::print(this, SLOT(slotFilePrint()), actionCollection());
	fileQuit = KStdAction::quit(this, SLOT(slotFileQuit()), actionCollection());

	fileSave = KStdAction::save(this, SLOT(slotFileSave()), actionCollection());
	fileSaveAs = KStdAction::saveAs(this, SLOT(slotFileSaveAs()), actionCollection());

	viewToolBar = KStdAction::showToolbar(this, SLOT(slotViewToolBar()), actionCollection());
	viewStatusBar = KStdAction::showStatusbar(this, SLOT(slotViewStatusBar()), actionCollection());
	//-------------
	preferencesWindow = KStdAction::preferences(this,SLOT(slotPreferencesWindow()),actionCollection());


	editTables = new KAction(i18n("&Edit Tables"),0,0,this,SLOT(slotEditTables()),actionCollection(),"edit_tables");

	filePrint ->setStatusText(i18n("Prints out the actual document"));
	fileQuit->setStatusText(i18n("Quits the application"));

	viewToolBar->setStatusText(i18n("Enables/disables the toolbar"));
	viewStatusBar->setStatusText(i18n("Enables/disables the statusbar"));
	mergeDB->setStatusText(i18n("Merges all database tables with the current ones"));

	KPopupMenu *      fileMenu = new KPopupMenu( this );
	KPopupMenu* settings = new KPopupMenu( this );
	KPopupMenu* help = helpMenu();
	connectDB->plug(fileMenu);
	disconnectDB->plug(fileMenu);
	fileMenu->insertSeparator();
	mergeDB->plug(fileMenu);
	fileMenu->insertSeparator();
	importFile->plug(fileMenu);
	fileMenu->insertSeparator();
	exportFile->plug(fileMenu);
	fileMenu->insertSeparator();
	filePrint->plug(fileMenu);
	fileMenu->insertSeparator();
	fileQuit->plug(fileMenu);
	viewToolBar->plug(settings);
	viewStatusBar ->plug(settings);
	settings->insertSeparator();
	preferencesWindow->plug(settings);
	// Insert to the MenuBar
	menuBar()->insertItem( i18n("&File"), fileMenu );
	menuBar()->insertItem( i18n("&Settings"), settings );
	menuBar()->insertItem( i18n("&Help"), help );
	//
	toolBar()->insertButton(QIconSet(BarIcon("fileprint")).pixmap(),-1,SIGNAL(clicked()),this,SLOT(slotFilePrint()),true,i18n("Prints out the current view"));

	// use the absolute path to your krunningui.rc file for testing purpose in createGUI();
	// createGUI();
}


void KRunningApp::initEventSelector()
{
	toolBar("mainToolBar")->insertSeparator();
	toolBar("mainToolBar")->insertCombo (g_pMainUseCase->getConnection()->getEventNameList(), 0,false,SIGNAL(activated(const QString&)),this,SLOT(mainEventActivated(const QString&)));
	g_pMainUseCase->setCurrentEvent(toolBar("mainToolBar")->getCombo(0)->currentText()); // To ensure that the selected event is stored.
	QToolTip::add (toolBar("mainToolBar")->getCombo(0),i18n("Choose the event which you are actually working on."));
	toolBar("mainToolBar")->setItemAutoSized(1,true);
}

void KRunningApp::mainEventActivated( const QString& newEvent )
{
	g_pMainUseCase->setCurrentEvent(newEvent);
	view->globalEventChanged(); 
}


void KRunningApp::slotDisconnectDB()
{
	if (g_pMainUseCase->isConnected())
	{
		g_pMainUseCase->getConnection()->disconnect();
		view->setEnabled(false);
		toolBar("mainToolBar")->getCombo(0)->clear();
		toolBar("mainToolBar")->setEnabled(false);
		connectDB->setEnabled(true);
		disconnectDB->setEnabled(false);
		mergeDB->setEnabled(false);
		view->dbDisconnected();
	}
}

void KRunningApp::slotConnectDB()
{
	bool bQuit = false;
	m_pCreateTables = new CreateTables();
	if (m_pCreateTables->exec()==KDialog::Accepted)
	{
		if (!g_pMainUseCase->getConnection()->testTables())
		{
			if (  KMessageBox::questionYesNo(this, i18n("KRunning couldn`t find the database tables.\n Should they be created ?"),i18n("Question ?"), KStdGuiItem::yes(), KStdGuiItem::no(),"tt",1) == KMessageBox::Yes )
			{
				g_pMainUseCase->getConnection()->createEmptyTables();
			}
		}
	}
	else
	{
		bQuit=true;
	}
	delete (m_pCreateTables);
	m_pCreateTables=NULL;
	if (bQuit)
	{
		exit(0);
	}
}


void KRunningApp::initStatusBar()
{
	///////////////////////////////////////////////////////////////////
	// STATUSBAR
	// TODO: add your own items you need for displaying current application status.
	statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG);
}

void KRunningApp::initDocument()
{
	doc = new KRunningDoc(this);
	doc->newDocument();
}

void KRunningApp::initView()
{
	view = new KRunningView(this);
	//
	doc->addView(view);
	setCentralWidget(view);
	setCaption(g_pMainUseCase->getConnection()->getDatabaseName(),false);
}

void KRunningApp::openDocumentFile(const KURL& url)
{
	slotStatusMsg(i18n("Opening file..."));

	doc->openDocument( url);
	slotStatusMsg(i18n("Ready."));
}


KRunningDoc *KRunningApp::getDocument() const
{
	return doc;
}

void KRunningApp::saveOptions()
{
	config->setGroup("General Options");
	config->writeEntry("Geometry", size());
	config->writeEntry("Show Toolbar", viewToolBar->isChecked());
	config->writeEntry("Show Statusbar",viewStatusBar->isChecked());
	config->writeEntry("ToolBarPos", (int) toolBar("mainToolBar")->barPos());
	config->sync();
}


void KRunningApp::readOptions()
{
	config->setGroup("General Options");
	// bar status settings
	bool bViewToolbar = config->readBoolEntry("Show Toolbar", true);
	viewToolBar->setChecked(bViewToolbar);
	slotViewToolBar();

	bool bViewStatusbar = config->readBoolEntry("Show Statusbar", true);
	viewStatusBar->setChecked(bViewStatusbar);
	slotViewStatusBar();

	// bar position settings
	KToolBar::BarPosition toolBarPos;
	toolBarPos=(KToolBar::BarPosition) config->readNumEntry("ToolBarPos", KToolBar::Top);
	toolBar("mainToolBar")->setBarPos(toolBarPos);
	// initialize the recent file list
	QSize size=config->readSizeEntry("Geometry");
	if(!size.isEmpty())
	{
		resize(size);
	}
}

void KRunningApp::saveProperties(KConfig *_cfg)
{
	if(doc->URL().fileName()!=i18n("Untitled") && !doc->isModified())
	{
		// saving to tempfile not necessary
	}
	else
	{
		KURL url=doc->URL();
		_cfg->writeEntry("filename", url.url());
		_cfg->writeEntry("modified", doc->isModified());
		QString tempname = kapp->tempSaveName(url.url());
		QString tempurl= KURL::encode_string(tempname);
		KURL _url(tempurl);
		doc->saveDocument(_url);
	}
}


void KRunningApp::readProperties(KConfig* _cfg)
{
	QString filename = _cfg->readEntry("filename", "");
	KURL url(filename);
	bool modified = _cfg->readBoolEntry("modified", false);
	if(modified)
	{
		bool canRecover;
		QString tempname = kapp->checkRecoverFile(filename, canRecover);
		KURL _url(tempname);

		if(canRecover)
		{
			doc->openDocument(_url);
			doc->setModified();
			setCaption(_url.fileName(),true);
			QFile::remove(tempname);
		}
	}
	else
	{
		if(!filename.isEmpty())
		{
			doc->openDocument(url);
			setCaption(url.fileName(),false);
		}
	}
}

bool KRunningApp::queryClose()
{
	return doc->saveModified();
}

bool KRunningApp::queryExit()
{
	saveOptions();
	return true;
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////


void KRunningApp::slotFileOpenRecent(const KURL& url)
{
	slotStatusMsg(i18n("Opening file..."));

	if(!doc->saveModified())
	{
		// here saving wasn't successful
	}
	else
	{
		doc->openDocument(url);
		setCaption(url.fileName(), false);
	}

	slotStatusMsg(i18n("Ready."));
}

void KRunningApp::slotFileSave()
{
	slotStatusMsg(i18n("Saving file..."));

	doc->saveDocument(doc->URL());

	slotStatusMsg(i18n("Ready."));
}

void KRunningApp::slotFileSaveAs()
{
	slotStatusMsg(i18n("Saving file with a new filename..."));

	KURL url=KFileDialog::getSaveURL(QDir::currentDirPath(),
	                                 i18n("*|All files"), this, i18n("Save as..."));
	if(!url.isEmpty())
	{
		doc->saveDocument(url);
		setCaption(url.fileName(),doc->isModified());
	}

	slotStatusMsg(i18n("Ready."));
}

void KRunningApp::slotExportHTML()
{
	view->exportFile("HTML");
}

void KRunningApp::slotMergeDB()
{

	cerr << "connected : " << g_pMainUseCase->isConnected() << endl;

}

void KRunningApp::slotExportTimes()
{
	KURL url=KFileDialog::getOpenURL(i18n("times.lzt"),
	                                 i18n("*.lzt *.LZT *.lst *.LST|Timcounter files\n*|All files"), this, i18n("Export times..."));
	if (!url.isValid()) return;
	timeExportDialog*dlg=new timeExportDialog(this,"",url);
	if (dlg->exec()==KDialog::Accepted)
	{
		// Show warning if not connected properly
	}
}

void KRunningApp::slotImportTimes()
{
	//
	KURL url=KFileDialog::getOpenURL(QDir::currentDirPath(),
	                                 i18n("*.lzt *.LZT *.lst *.LST|Timcounter files\n*|All files"), this, i18n("Import times..."));
	if (!url.isValid()) return;
	timeImportDialog *dlg=new timeImportDialog(this,"",url);
	if (dlg->exec()==KDialog::Accepted)
	{
		// Show warning if not connected properly
	}
}

void KRunningApp::slotExportTeams()
{
	DBConnection::TeamList teamlist;
	KURL url=KFileDialog::getOpenURL(QDir::currentDirPath(),
	                                 i18n("*.csv *.CSV|CSV Files\n*|All files"), this, i18n("Export teams..."));
	if (!url.isValid()) return;

	teamlist=g_pMainUseCase->getConnection()->getTeamList();
	DBConnection::TeamList::iterator itteamlist;
	QFile file( url.directory()+'/'+url.filename()  );
	if ( file.open( IO_WriteOnly ) )
	{
		QTextStream stream( &file );
		for ( itteamlist = teamlist.begin(); itteamlist != teamlist.end(); ++itteamlist )
			stream << (*itteamlist).getNumber() << prefs->getSeperator() << prefs->getStringIndent() << (*itteamlist).getName() << prefs->getStringIndent() << "\n";
		file.close();
	}
}

void KRunningApp::slotImportTeams()
{
	DBConnection::TeamList teamlist;
	//KRunningTeamObject *KRunningTeamObject;
	QStringList lines;

	KURL url=KFileDialog::getOpenURL(QDir::currentDirPath(),
	                                 i18n("*.csv *.CSV|CSV Files\n*|All files"), this, i18n("Import teams..."));
	if (!url.isValid()) return;

	teamlist=g_pMainUseCase->getConnection()->getTeamList();
	if (!teamlist.isEmpty())
	{
		// Fragen ob anh�gen oder berschreiben oder abbrechen
	}

	// Datei einlesen

	QFile file( url.directory()+'/'+url.fileName()  );
	if ( file.open( IO_ReadOnly ) )
	{
		QTextStream stream( &file );
		QString line;
		while ( !stream.eof() )
		{
			line = stream.readLine(); // line of text excluding '\n'
			lines += line;
		}
		file.close();
	}
	// Stringliste in DB schreiben ...
	QStringList::iterator it;
	for (it=lines.begin();it!=lines.end();it++)
	{
		if ((*it).find('#')==-1 )
		{
			KRunningTeamObject *pTeamObject= new KRunningTeamObject;
			pTeamObject->setName((*it).section( prefs->getSeperator(), -1).remove(prefs->getStringIndent()));
			pTeamObject->save();
			delete(pTeamObject);
		}
	}
}


void KRunningApp::slotFileClose()
{
	slotStatusMsg(i18n("Closing file..."));

	close();

	slotStatusMsg(i18n("Ready."));
}

void KRunningApp::slotFilePrint()
{
	slotStatusMsg(i18n("Printing..."));

	KPrinter printer;

	// do some printer initialization
	printer.setFullPage(false);
	printer.setCreator(QString ("KRunning Version %1").arg(VERSION));

	if (printer.setup(this))
	{
		view->print(&printer);
	}

	slotStatusMsg(i18n("Ready."));
}

void KRunningApp::slotFileQuit()
{
	slotStatusMsg(i18n("Exiting..."));
	saveOptions();
	if (g_pMainUseCase)
		delete (g_pMainUseCase);
	this->close();
	exit(1);
}

void KRunningApp::slotViewToolBar()
{
	slotStatusMsg(i18n("Toggling toolbar..."));
	if(!viewToolBar->isChecked())
	{
		toolBar("mainToolBar")->hide();
	}
	else
	{
		toolBar("mainToolBar")->show();
	}
	slotStatusMsg(i18n("Ready."));
}

void KRunningApp::slotViewStatusBar()
{
	slotStatusMsg(i18n("Toggle the statusbar..."));
	///////////////////////////////////////////////////////////////////
	//turn Statusbar on or off
	if(!viewStatusBar->isChecked())
	{
		statusBar()->hide();
	}
	else
	{
		statusBar()->show();
	}

	slotStatusMsg(i18n("Ready."));
}

void KRunningApp::slotPreferencesWindow()
{
	PreferencesDialog    *prefDlg = new PreferencesDialog (this,"Preferences");
	if (prefDlg->exec () == QDialog::Accepted)
	{}
	delete          (prefDlg);
}

void KRunningApp::slotEditTables()
{
}

void KRunningApp::slotStatusMsg(const QString &text)
{
	///////////////////////////////////////////////////////////////////
	// change status message permanently
	statusBar()->clear();
	statusBar()->changeItem(text, ID_STATUS_MSG);
}

//----------------------
/** timeImportDialog
@version 0.1
*/

timeImportDialog::timeImportDialog(QWidget *parent, const char *name,const KURL & url) :
		KDialogBase( Plain, i18n("Time Import Dialog" ),  Ok|Cancel, Ok, parent,  name,
		             true,                     // modal
		             true),                             // separator
		filepath(url)

{
	QFrame *page=plainPage();
	QGridLayout * grid = new QGridLayout(page,2,1,0,5);
	label = new QLabel(page,"Label");
	label->setText(i18n("Please select one event for the times from")+QString("\n%1").arg(url.fileName()));
	grid->addWidget(label,0,0);
	eventComboBox = new QComboBox(page,"Combo");
	eventComboBox->insertStringList(g_pMainUseCase->getConnection()->getEventNameList());
	grid->addWidget(eventComboBox,1,0);
	connect(this, SIGNAL(okClicked()), SLOT(ok()));
}

timeImportDialog::~timeImportDialog()
{
	destroy();
}

void timeImportDialog::ok()
{
	KRunningEventObject *pEvent = g_pMainUseCase->getConnection()->getEventByName(eventComboBox->currentText());
	int event = pEvent->getNumber();
	delete pEvent;
	pEvent=NULL;
	QStringList lines;

	QFile file( filepath.directory()+'/'+filepath.filename()  );
	if ( file.open( IO_ReadOnly ) )
	{
		QTextStream stream( &file );
		QString line;
		while ( !stream.eof() )
		{
			line = stream.readLine(); // line of text excluding '\n'
			lines += line;
		}
		file.close();
	}
	QStringList::iterator it;
	for (it=lines.begin();it!=lines.end();it++)
	{
		if ((*it).find('#')==-1 && QTime::fromString((*it).section( ' ', -1)).isValid())
		{
			KRunningTimeObject *to= new KRunningTimeObject;
			to->setEvent(event);
			to->setTime(QTime::fromString((*it).section( ' ', -1)));
			to->save();
			delete(to);
		}
	}
}


//----------
timeExportDialog::timeExportDialog(QWidget *parent, const char *name,const KURL & url) :
		KDialogBase( Plain, i18n("Time Export Dialog" ),  Ok|Cancel, Ok, parent,  name,
		             true,                     // modal
		             true),                             // separator
		filepath(url)

{
	QFrame *page=plainPage();
	QGridLayout * grid = new QGridLayout(page,2,1,0,5);
	label = new QLabel(page,"Label");
	label->setText(i18n("Please select one event which you want to extract the times to")+QString("\n%1").arg(url.fileName()));
	grid->addWidget(label,0,0);
	eventComboBox = new QComboBox(page,"Combo");
	eventComboBox->insertStringList(g_pMainUseCase->getConnection()->getEventNameList());
	grid->addWidget(eventComboBox,1,0);
	connect(this, SIGNAL(okClicked()), SLOT(ok()));
}

timeExportDialog::~timeExportDialog()
{
	destroy();
}

void timeExportDialog::ok()
{
	KRunningEventObject *pEvent = g_pMainUseCase->getConnection()->getEventByName(eventComboBox->currentText());
	int event = pEvent->getNumber();
	delete pEvent;

	DBConnection::TimeList timelist;
	// Get all List from the database ...
	timelist=g_pMainUseCase->getConnection()->getTimeListByEvent(event);
	// Get the Iterator
	DBConnection::TimeList::iterator ittimelist;

	QFile file( filepath.directory()+'/'+filepath.filename()  );
	if ( file.open( IO_WriteOnly ) )
	{
		QTextStream stream( &file );
		stream << QString("# KRunning %1 TimeExport File\n").arg(VERSION);
		stream << "# Event: " << g_pMainUseCase->getCurrentEvent()->getName() << "\n";
		for ( ittimelist = timelist.begin(); ittimelist != timelist.end(); ++ittimelist )
			stream << (*ittimelist).getNumber() << " " << (*ittimelist).getTimeString("hh:mm:ss.zzz") << "\n";
		file.close();
	}
}

