/***************************************************************************
                          preferences.cpp  -  description
                             -------------------
    begin                : Mon Aug 28 2000
    copyright            : (C) 2000 by Jrgen Vigna
    email                : jug@lyx.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cstdio>

#include <qframe.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qspinbox.h>
#include <qcombobox.h>
#include <qcheckbox.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qlistview.h>
#include <qtoolbutton.h>
#include <qdir.h>
#include <qfileinfo.h>

#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>

#include "preferences.h"
#include "prefstate.h"
#include "filterdlg.h"
#include "dialogs/formfax.h"
#include "dialogs/formspecial.h"
#include "dialogs/formscanner.h"
#include "dialogs/formpersonal.h"
#include "dialogs/formphonebook.h"
#include "dialogs/formmisc.h"
#include "dialogs/formfilters.h"
#include "helpers.h"

using std::vector;

extern QString debugFlag;

Preferences::Preferences(QWidget *parent, char *name, bool modal)
	:KDialogBase(IconList, i18n("Configure"), Help|Default|Apply|Ok|Cancel,
	             Ok, parent, name, modal, true)
{
	setHelp( "ksendfax/index.html", QString::null);

	addFormFax();
	addFormSpecial();
	addFormPersonal();
	addFormScanner();
	addFormPhonebook();
	addFormMisc();
	addFormFilters();

	Helpers::readConfig(pState);
	setState(pState);
}

Preferences::~Preferences(void)
{
}

void Preferences::addFormFax()
{
	QFrame * page = addPage(i18n("Fax"), i18n("Fax commands"), 
							BarIcon("ksendfax", KIcon::SizeMedium));
	QVBoxLayout *topLayout = new QVBoxLayout(page, 0, spacingHint());
	ffax = new FormFax(page, 0, false);
	topLayout->addWidget(ffax);
}


void Preferences::addFormSpecial()
{
	QFrame * page = addPage(i18n("Special"), i18n("Special Convert options"), 
							BarIcon("ksendfax", KIcon::SizeMedium));
	QVBoxLayout *topLayout = new QVBoxLayout(page, 0, spacingHint());
	fspecial = new FormSpecial(page, 0, false);
	topLayout->addWidget(fspecial);
	/// connections
	connect(fspecial->listExcludeConvert, SIGNAL(doubleClicked(QListBoxItem*)),
			this, SLOT(special_exclude_slotChange()));
	connect(fspecial->buttonAdd, SIGNAL(clicked()), this,
	        SLOT(special_exclude_slotAdd()));
	connect(fspecial->buttonChange, SIGNAL(clicked()), this,
	        SLOT(special_exclude_slotChange()));
	connect(fspecial->buttonRemove, SIGNAL(clicked()), this,
	        SLOT(special_exclude_slotRemove()));

	connect(fspecial->listAlwaysConvert, SIGNAL(doubleClicked(QListBoxItem*)),
			this, SLOT(special_always_slotChange()));
	connect(fspecial->buttonAdd2, SIGNAL(clicked()), this,
	        SLOT(special_always_slotAdd()));
	connect(fspecial->buttonChange2, SIGNAL(clicked()), this,
	        SLOT(special_always_slotChange()));
	connect(fspecial->buttonRemove2, SIGNAL(clicked()), this,
	        SLOT(special_always_slotRemove()));
}


void Preferences::addFormScanner()
{
	QFrame * page = addPage(i18n("Scanner"), i18n("Scanner commands"), 
							BarIcon("scandoc", KIcon::SizeMedium));
	QVBoxLayout *topLayout = new QVBoxLayout(page, 0, spacingHint());
	fscanner = new FormScanner(page, 0, false);
	topLayout->addWidget(fscanner);
}


void Preferences::addFormPersonal()
{
	QFrame * page = addPage(i18n("Personal"), i18n("Personal data"), 
							BarIcon("personal", KIcon::SizeMedium));
	QVBoxLayout *topLayout = new QVBoxLayout(page, 0, spacingHint());
	fpersonal = new FormPersonal(page, 0, false);
	topLayout->addWidget(fpersonal);
}


void Preferences::addFormPhonebook()
{
	QFrame * page = addPage(i18n("Phonebook"), i18n("Phonebook commands"), 
							BarIcon("phonebook", KIcon::SizeMedium));
	QVBoxLayout *topLayout = new QVBoxLayout(page, 0, spacingHint());
	fphonebook = new FormPhonebook(page, 0, false);
	topLayout->addWidget(fphonebook);
}


void Preferences::addFormMisc()
{
	QFrame * page = addPage(i18n("Misc"), i18n("Misc commands"), 
							BarIcon("miscopts", KIcon::SizeMedium));
	QVBoxLayout *topLayout = new QVBoxLayout(page, 0, spacingHint());
	fmisc = new FormMisc(page, 0, false);
	topLayout->addWidget(fmisc);
}


void Preferences::addFormFilters()
{
	QFrame * page = addPage(i18n("Filters"), i18n("Filters to Postscipt"), 
							BarIcon("filteropts", KIcon::SizeMedium));
	QVBoxLayout *topLayout = new QVBoxLayout(page, 0, spacingHint());
	ffilters = new FormFilters(page, 0, false);
	topLayout->addWidget(ffilters);
	ffilters->lv_filters->setSorting(-1);
	/// connections
	connect(ffilters->lv_filters, SIGNAL(doubleClicked(QListViewItem*)), this,
	        SLOT(filters_slotChange()));
	connect(ffilters->buttonAdd, SIGNAL(clicked()), this,
	        SLOT(filters_slotAdd()));
	connect(ffilters->buttonChange, SIGNAL(clicked()), this,
	        SLOT(filters_slotChange()));
	connect(ffilters->buttonRemove, SIGNAL(clicked()), this,
	        SLOT(filters_slotRemove()));
	connect(ffilters->buttonMoveUp, SIGNAL(clicked()), this,
	        SLOT(filters_slotUp()));
	connect(ffilters->buttonMoveDown, SIGNAL(clicked()), this,
	        SLOT(filters_slotDown()));
}


void Preferences::slotOk(void)
{
	slotApply();
	accept();
}


void Preferences::slotApply(void)
{
	///
	// fax part
	///
	pState.fax.sendcmd = ffax->InputFaxCommand->text();
	pState.fax.showcmd = ffax->InputShowCommand->text();
	pState.fax.removecmd = ffax->InputRemoveCommand->text();
	pState.fax.updatesecs = ffax->spinUpdateSeconds->value();
	pState.fax.useps = ffax->checkUsePostscript->isOn();
	if (ffax->comboCoverPage->currentItem())
		pState.fax.coverfile = ffax->comboCoverPage->currentText();
	else
		pState.fax.coverfile = QString();
	///
	// special part
	///
	pState.special.excludeps = fspecial->checkExcludeConvert->isOn();
	pState.special.alwaysps = fspecial->checkAlwaysConvert->isOn();
	pState.special.exlist.clear();
	QListBoxItem * box = fspecial->listExcludeConvert->firstItem();
	while(box) {
		pState.special.exlist.append(box->text());
		box = box->next();
	}
	pState.special.allist.clear();
	box = fspecial->listAlwaysConvert->firstItem();
	while(box) {
		pState.special.allist.append(box->text());
		box = box->next();
	}
	///
	// personal part
	///
	pState.personal.name       = fpersonal->InputName->text();
	pState.personal.enterprise = fpersonal->InputEnterprise->text();
	pState.personal.phone      = fpersonal->InputPhone->text();
	///
	// scanner part
	///
	pState.scanner.type = (PrefScannerState::ScannerImageType)fscanner->comboScanType->currentItem();
	pState.scanner.cmd = fscanner->InputScanCmd->text();
	pState.scanner.use_external = fscanner->checkUseExtProg->isOn();
	///
	// phonebook part
	///
	pState.phonebook.file = fphonebook->InputPhonebook->text();
	pState.phonebook.type = (PrefPhonebookState::PhonebookType)fphonebook->comboPBType->currentItem();
	///
	// misc part
	///
	pState.misc.logdir     = fmisc->InputLogdir->text();
	pState.misc.previewcmd = fmisc->InputPreviewCmd->text();
	pState.misc.datefmt    = fmisc->InputDateFormat->text();
	pState.misc.convertcmd = fmisc->InputConvertCmd->text();
	///
	// filters part
	///
	pState.filters.filters.clear();
	QListViewItem *item = ffilters->lv_filters->firstChild();
	PrefFiltersState::Filters data;
	while (item) {
		data.mimetype = item->text(0);
		data.mimecmd = item->text(1);
		pState.filters.filters.push_back(data);
		item = item->nextSibling();
	}
	
	if (Helpers::writeConfig(pState))
		emit PBChanged();
}


void Preferences::slotCancel(void)
{
	setState(pState);
	reject();
}


void Preferences::slotDefault(void)
{
	//
	// The constructors store the default settings.
	//
	switch(activePageIndex()) {
		
    case page_fax:
		setPrefFax(PrefFaxState());
		break;

	case page_scanner:
		setPrefScanner(PrefScannerState());
		break;
		
    case page_personal:
		setPrefPersonal(PrefPersonalState());
		break;
		
    case page_phonebook:
		setPrefPhonebook(PrefPhonebookState());
		break;

	case page_misc:
		setPrefMisc(PrefMiscState());
		break;

	}
}


void Preferences::slotHelp(void)
{
	KDialogBase::slotHelp();
}


void Preferences::setPrefFax(const PrefFaxState & fax)
{
	ffax->InputFaxCommand->setText(fax.sendcmd);
	ffax->InputShowCommand->setText(fax.showcmd);
	ffax->InputRemoveCommand->setText(fax.removecmd);
	ffax->spinUpdateSeconds->setValue(fax.updatesecs);
	ffax->checkUsePostscript->setChecked(fax.useps);
	ffax->comboCoverPage->setCurrentItem(0);
	if (!fax.coverfile.isEmpty()) {
		for(int i=1; i < ffax->comboCoverPage->count(); ++i) {
			if (ffax->comboCoverPage->text(i) == fax.coverfile) {
				ffax->comboCoverPage->setCurrentItem(i);
				break;
			}
		}
	}
}


void Preferences::setPrefSpecial(const PrefSpecialState & special)
{
	fspecial->checkExcludeConvert->setChecked(special.excludeps);
	fspecial->checkAlwaysConvert->setChecked(special.alwaysps);
	QStringList::ConstIterator it;

	fspecial->listExcludeConvert->clear();
	for(it = special.exlist.begin(); it != special.exlist.end(); ++it) {
		fspecial->listExcludeConvert->insertItem((*it));
	}

	fspecial->listAlwaysConvert->clear();
	for(it = special.allist.begin(); it != special.allist.end(); ++it) {
		fspecial->listAlwaysConvert->insertItem((*it));
	}
}


void Preferences::setPrefScanner(const PrefScannerState & scanner)
{
	fscanner->comboScanType->setCurrentItem(scanner.type);
	fscanner->InputScanCmd->setText(scanner.cmd);
#ifdef HAVE_SANE
	fscanner->checkUseExtProg->setChecked(scanner.use_external);
#else
	fscanner->checkUseExtProg->setChecked(true);
	fscanner->checkUseExtProg->setDisabled(true);
#endif
}


void Preferences::setPrefPersonal(const PrefPersonalState & personal)
{
	fpersonal->InputName->setText(personal.name);
	fpersonal->InputEnterprise->setText(personal.enterprise);
	fpersonal->InputPhone->setText(personal.phone);
}


void Preferences::setPrefPhonebook(const PrefPhonebookState & phonebook)
{
	fphonebook->InputPhonebook->setText(phonebook.file);
#ifdef USE_KAB
	fphonebook->comboPBType->setCurrentItem(phonebook.type);
#else
	fphonebook->comboPBType->setCurrentItem(PrefPhonebookState::ksendfax);
	fphonebook->comboPBType->setEnabled(false);
#endif
}


void Preferences::setPrefMisc(const PrefMiscState & misc)
{
	fmisc->InputLogdir->setText(misc.logdir);
	fmisc->InputPreviewCmd->setText(misc.previewcmd);
	fmisc->InputDateFormat->setText(misc.datefmt);
	fmisc->InputConvertCmd->setText(misc.convertcmd);
}


void Preferences::setPrefFilters(const PrefFiltersState & filters)
{
	ffilters->lv_filters->clear();
	vector<PrefFiltersState::Filters>::const_iterator it;
	QListViewItem * after = 0;
	for(it = filters.filters.begin(); it != filters.filters.end(); ++it) {
		after = new QListViewItem(ffilters->lv_filters, after, (*it).mimetype, (*it).mimecmd);
	}
}


void Preferences::setState(const PrefState & state)
{
	setPrefFax(state.fax);
	setPrefSpecial(state.special);
	setPrefScanner(state.scanner);
	setPrefPersonal(state.personal);
	setPrefPhonebook(state.phonebook);
	setPrefMisc(state.misc);
	setPrefFilters(state.filters);
}


void Preferences::restoreState()
{
	///
	// fill the fax-cover combobox with files in the cover directory
	///
	ffax->comboCoverPage->clear();
	ffax->comboCoverPage->insertItem(i18n("Use cover feature of fax package"));
	QDir coverdir(locate("data","ksendfax/coverpages/"));
	coverdir.setFilter(QDir::Files);
	const QFileInfoList *list = coverdir.entryInfoList();
	if (list) {
		for(QFileInfoListIterator it(*list); it.current(); ++it) {
			ffax->comboCoverPage->insertItem(it.current()->filePath());
			if (!debugFlag.isEmpty())
				fprintf(stderr, "%s\n", it.current()->filePath().data());
		}
	}
	setState(pState);
}

///
// FormFilters specific functions!
///


void Preferences::filters_slotAdd()
{
	QString	mime;
	QString cmd;
	if (FilterDlg::doIt(this, &mime, &cmd))
		if (!mime.isEmpty() && !cmd.isEmpty())
			new QListViewItem(ffilters->lv_filters, ffilters->lv_filters->currentItem(), mime, cmd);
		else
			KMessageBox::error(this, i18n("Empty parameters."));
}

void Preferences::filters_slotRemove()
{
	QListViewItem	*item = ffilters->lv_filters->currentItem();
	if (item)
		delete item;
}

void Preferences::filters_slotChange()
{
	QListViewItem	*item = ffilters->lv_filters->currentItem();
	if (item)
	{
		QString	mime(item->text(0)), cmd(item->text(1));
		if (FilterDlg::doIt(this, &mime, &cmd))
		{
			item->setText(0, mime);
			item->setText(1, cmd);
		}
	}
}

void Preferences::filters_slotUp()
{
	QListViewItem	*item = ffilters->lv_filters->currentItem();
	if (item && item->itemAbove())
	{
		QString t0 = item->text(0);
		QString t1 = item->text(1);
		item->setText(0, item->itemAbove()->text(0));
		item->setText(1, item->itemAbove()->text(1));
		item->itemAbove()->setText(0, t0);
		item->itemAbove()->setText(1, t1);
		ffilters->lv_filters->setCurrentItem(item->itemAbove());
	}
}

void Preferences::filters_slotDown()
{
	QListViewItem	*item = ffilters->lv_filters->currentItem();
	if (item && item->itemBelow())
	{
		QString t0 = item->text(0);
		QString t1 = item->text(1);
		item->setText(0, item->itemBelow()->text(0));
		item->setText(1, item->itemBelow()->text(1));
		item->itemBelow()->setText(0, t0);
		item->itemBelow()->setText(1, t1);
		ffilters->lv_filters->setCurrentItem(item->itemBelow());
	}
}


///
// FormSpecial specific functions!
///
///
//  always part
///


void Preferences::special_exclude_slotAdd()
{
	QString	mime;
	if (FilterDlg::doIt(this, &mime))
		if (!mime.isEmpty())
			fspecial->listExcludeConvert->insertItem(mime);
		else
			KMessageBox::error(this, i18n("Empty parameters."));
}

void Preferences::special_exclude_slotRemove()
{
	fspecial->listExcludeConvert->removeItem(fspecial->listExcludeConvert->currentItem());
}

void Preferences::special_exclude_slotChange()
{
	int cur = fspecial->listExcludeConvert->currentItem();
	
	if (cur < 0)
		return;
	QString	mime(fspecial->listExcludeConvert->text(cur));
	if (FilterDlg::doIt(this, &mime)) {
		fspecial->listExcludeConvert->changeItem(mime, cur);
	}
}

///
//  always part
///


void Preferences::special_always_slotAdd()
{
	QString	mime;
	if (FilterDlg::doIt(this, &mime))
		if (!mime.isEmpty())
			fspecial->listAlwaysConvert->insertItem(mime);
		else
			KMessageBox::error(this, i18n("Empty parameters."));
}

void Preferences::special_always_slotRemove()
{
	fspecial->listAlwaysConvert->removeItem(fspecial->listAlwaysConvert->currentItem());
}

void Preferences::special_always_slotChange()
{
	int cur = fspecial->listAlwaysConvert->currentItem();
	
	if (cur < 0)
		return;
	QString	mime(fspecial->listAlwaysConvert->text(cur));
	if (FilterDlg::doIt(this, &mime)) {
		fspecial->listAlwaysConvert->changeItem(mime, cur);
	}
}
