#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#ifdef HAVE_WINCONFIG_H
# include "MSWin_config.h"
#endif

#include "cfg.h"

#include <stdio.h>
#ifdef  HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <string.h>
#include <stdlib.h>
#define _GNU_SOURCE
#include <getopt.h>

#include <time.h>
#include <curl/curl.h>
#include <curl/mprintf.h>
#include "larepubblica.h"

static void aggiungi_data(time_t * day);

extern Configurazione config;

static void usage() __attribute__ ((noreturn));

static char *program_name;
static int show_help;
static int show_version;

/* getopt long options equivalences.  */
static struct option long_options[] = {
    {"chiave-tel", required_argument, NULL, 'k'},
    {"numero-giorni", required_argument, NULL, 'n'},
    {"pdf", no_argument, NULL, 'p'},
    {"html", no_argument, NULL, 'h'},
    {"testo", no_argument, NULL, 't'},
    {"lista-edizioni", no_argument, NULL, 'l'},
    {"senza-articoli", no_argument, NULL, 's'},
    {"senza-immagini", no_argument, NULL, 'j'},
    {"version", no_argument, NULL, 'v'},
    {"help", no_argument, &show_help, 1},
#ifdef WIN32
    {"nogui", no_argument, NULL, 'x'},
#endif
    {0, 0, 0, 0}
};

static void usage(int status)
{
    if (status != 0) {
	fprintf(stderr, "Uso: %s [OPZIONE] [GIORNO] [MESE] [ANNO]...\n",
		program_name);
	fprintf(stderr, "Prova `%s --help' per maggiori informazioni.\n",
		program_name);
    } else {
	printf("Uso: %s [OPZIONE]\n", program_name);
	printf("\
Strumento per il download delle edizioni di \"La Repubblica\"\n\
\n\
  -l, --lista-edizioni      Visualizza le edizioni scaricate\n\
  -k, --chiave-tel NUMERO   Per la chiave d'accesso al servizio valida 24 ore\n\
\n\
[ -p, --pdf                 Download della versione PDF del quotidiano]\n\
  -h, --html                Download della versione HTML con grafica (default)\n\
[ -t, --testo               Download della versione solo testo]\n\
  -s, --senza-articoli      Download solo del materiale accessibile senza abbonamento\n\
  -j, --senza-immagini      Evita il download delle immagini dell'edizione (HTML+JPG)\n\
\n\
  -n, --numero-giorni       Numero arretrati a partire dalla data corrente\n\
                            o dalla data specificata (se ve n' pi di una, l'ultima)\n");
#ifdef WIN32
	printf("\
      --nogui\n");
#endif
	printf("\n\
      --help       display this help and exit\n\
  -v, --version    output version information and exit\n\
\n\
Scritto da Francesco Cosoleto, email a <%s>.\n", PACKAGE_BUGREPORT);
    }
    exit(status);
}

int gnu_getopt(int argc, char **argv)
{
    int opt;
    extern char *optarg;
    extern int optind;

#ifdef WIN32
    if (argc == 0)
	return 0;
#endif

    program_name = argv[0];
    if (program_name && strrchr(program_name, '/'))
	program_name = strrchr(program_name, '/') + 1;

    while ((opt =
	    getopt_long(argc, argv, "vk:n:hptsjlx", long_options,
			NULL)) != EOF)
	switch (opt) {
	case 0:
	    break;
	case 'v':
	    show_version = 1;
	    break;
	case 'k':
	    if (optarg)
		config.key = optarg;
	    config.tipo_accesso = CHIAVE_TELEFONICA;
	    break;
	case 'n':
	    config.numero_giorni = atoi(optarg);
	    if (config.numero_giorni < 0)
		error(1, "Numero giorni non valido");
	case 'h':
	    break;
	case 'p':
	    config.formato_edizione = PDF;
	    error(1, "Opzione -%c non disponibile al momento", opt);
	    break;
	case 't':
	    config.formato_edizione = TESTO;
	    //error(1, "Opzione -%c non disponibile al momento", opt);
	    break;
	case 's':
	    config.senza_articoli = 1;
	    break;
	case 'j':
	    config.senza_immagini = 1;
	    break;
	case 'l':
	    load_config();
	    msg("Directory: %s\n",
		(*config.savepath) ? config.savepath : "corrente");
	    lista_edizioni_disponibili();
	    stampa_edizioni_disponibili();
	    unload_config();
	    exit(0);
	    break;
	case 'x':
	    break;
	default:
	    usage(2);
	    break;
	}

    if (show_version) {
	printf("%s %s\n", PACKAGE_NAME, VERSION);
	exit(0);
    }

    if (show_help)
	usage(0);

    if (optind == argc) {
	if (config.formato_edizione == DEFAULT)
	    dl_edizioni_slist =
		curl_slist_append(dl_edizioni_slist,
				  "Edizione_Giornaliera/");
	else
	    aggiungi_data(0);
    }

    int index = optind;
    static struct tm *mytm;
    static time_t today;
    static time_t save_time;

    today = time(NULL);
    mytm = localtime(&today);

    int step;

    for (index = optind; index < argc;) {
	if (argc - index >= 3)
	    step = 3;
	else
	    step = argc - index;

	mytm->tm_mday = atoi(argv[index++]);

	if (step >= 2)
	    mytm->tm_mon = atoi(argv[index++]) - 1;

	if (step >= 3)
	    mytm->tm_year = atoi(argv[index++]) - 1900;

	if (mytm->tm_mday <= 0 || mytm->tm_mon < 0 || mytm->tm_mon > 11
	    || mytm->tm_year < 0)
	    error(1, "Inserita una data non valida");

	save_time = mktime(mytm);
	if (save_time == -1)
	    error(1, "Inserita una data non valida");

	aggiungi_data(&save_time);
    }

    if (config.numero_giorni > 0) {

	time_t inizio = mktime(mytm);

	inizio -= 86400 * (config.numero_giorni);

	while (config.numero_giorni--) {
	    aggiungi_data(&inizio);
	    inizio += 86400;
	}
    }
    return 0;
}

static void aggiungi_data(time_t * day)
{

    struct tm *mytm;
    time_t today;

    if (!day) {
	today = time(NULL);
	day = &today;
    }

    mytm = localtime(day);

    dl_edizioni_slist = curl_slist_append(dl_edizioni_slist,
					  componi_stringa_edizione(mytm->
								   tm_mday,
								   mytm->
								   tm_mon,
								   1900 +
								   mytm->
								   tm_year));
}
