/* ----------------------------------------------------------------------------
   pfortune - generate a random fortune
   Copyright (C) 2001-2007  Mark A Lindner

   This file is part of misctools.
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, see
   <http://www.gnu.org/licenses/>.
   ----------------------------------------------------------------------------
*/

/* --- Feature Test Switches --- */

#include "config.h"

/* --- System Headers --- */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <signal.h>
#include <syslog.h>
#include <dirent.h>
#include <string.h>

#include <cbase/cbase.h>

/* --- Local Headers --- */

/* --- Macros --- */

#define HEADER "pfortune v" VERSION " - Mark Lindner"
#define USAGE "[-hiln] [-t <template>] [-d <dbdir>] [-p <pipe>]\n\t" \
    "<file> [ <file> ... ]"

#define FORTUNE_SUFFIX ".fdb"
#define FORTUNE_TOKEN "@FORTUNE@"
#define FORTUNE_DIR "databases"

static const char *libdir = DATA_DIR;
static char *template = NULL;
static char **dbnames = NULL;
static size_t dbcount = 0;
static char **userdbnames = NULL;
static size_t userdbcount = 0;
static c_hashtable_t *filemap;

/* --- Functions --- */

/*
 */

static c_bool_t scan_dir(char *dir)
{
  DIR *dp;
  struct dirent *de;
  char *buf, *name;
  int l, dlen = strlen(dir) + 1;

  C_debug_printf("scan_dir: %s\n", dir);
  
  if(!(dp = opendir(dir)))
    return(FALSE);

  while((de = readdir(dp)))
  {
    if(C_string_endswith(de->d_name, FORTUNE_SUFFIX))
    {
      buf = C_newstr(strlen(de->d_name) + dlen + 1);
      sprintf(buf, "%s/%s", dir, de->d_name);

      l = strlen(de->d_name) - 4;
      *(de->d_name + l) = NUL;
      name = C_string_dup(de->d_name);

      if(! C_hashtable_restore(filemap, name))
        C_hashtable_store(filemap, name, buf);
      else
      {
        C_free(name);
        C_free(buf);
      }
    }
  }
  
  closedir(dp);

  return(TRUE);
}

/*
 */

static void select_fortune(FILE *fp)
{
  int fi;
  const char *text;
  const char *path = NULL;
  c_fortune_db_t *db;

  if(userdbcount > 0)
  {
    fi = C_random(userdbcount);
    path = C_hashtable_restore(filemap, userdbnames[fi]);
  }
  else
  {
    fi = C_random(dbcount);
    path = C_hashtable_restore(filemap, dbnames[fi]);
  }

  if(! path)
  {
    C_error_printf("No databases available.\n");
    return;
  }

  C_debug_printf("selected db: %s\n", path);

  db = C_fortune_opendb(path);
  if(! db)
  {
    C_error_printf("Unable to open database: %s\n", path);
    return;
  }
  
  text = C_fortune_select(db);

  fputs(text, fp);
  C_free(text);

  C_fortune_closedb(db);
}

/*
 */

static void print_fortune(FILE *stream)
{
  FILE *fp;
  
  if(template)
  {
    char buf[256];

    if(!(fp = fopen(template, "r")))
    {
      C_error_printf("Unable to read template file: %s\n", template);
      exit(EXIT_FAILURE);
    }

    while(C_io_gets(fp, buf, sizeof(buf), '\n') != EOF)
    {
      if(!strcmp(buf, FORTUNE_TOKEN))
        select_fortune(stream);
      else
      {
        fputs(buf, stream);
        fputc('\n', stream);
      }
    }

    fclose(fp);
  }
  else
    select_fortune(stream);
}

/*
 */

int main(int argc, char **argv)
{
  int ch;
  c_bool_t errflag = FALSE, indexflag = FALSE, nlflag = FALSE,
    listflag = FALSE, dirflag = FALSE;
  char *fifo = NULL;
  struct stat stbuf;
  pid_t pid;
  int i;
  char **pp;
  FILE *fp;
  c_vector_t *vec;
  
  C_error_init(*argv);
  C_random_seed();

  filemap = C_hashtable_create(26);

  while((ch = getopt(argc, argv, "hlnp:d:t:i")) != EOF)
    switch((char)ch)
    {
      case 'd':
        if(! scan_dir(optarg))
        {
          C_error_printf("Directory not found: %s\n", optarg);
          exit(EXIT_FAILURE);
        }
        dirflag = TRUE;
        break;

      case 'h':
        C_error_printf("%s\n", HEADER);
        C_error_usage(USAGE);
        exit(EXIT_SUCCESS);

      case 'l':
        listflag = TRUE;
        break;

      case 'p':
        fifo = C_string_dup(optarg);
        break;

      case 't':
        template = C_string_dup(optarg);
        break;

      case 'i':
        indexflag = TRUE;
        break;

      case 'n':
        nlflag = TRUE;
        break;

      default:
        errflag = TRUE;
    }
  
  if(errflag)
  {
    C_error_usage(USAGE);
    exit(EXIT_FAILURE);
  }

  /* if no directories specified on command line, use the default one */

  if(!dirflag && (strlen(libdir) > 0))
  {
    char *p = C_newstr(strlen(libdir) + strlen(FORTUNE_DIR) + 2);

    sprintf(p, "%s/%s", libdir, FORTUNE_DIR);
    scan_dir(p);
    C_free(p);
  }

  dbnames = C_hashtable_keys(filemap, &dbcount);
  C_string_sortvec(dbnames, dbcount);

  /* list? */
  
  if(listflag)
  {
    for(pp = dbnames; *pp; pp++)
      puts(*pp);
    
    exit(EXIT_SUCCESS);
  }
  
  /* index? */
  
  else if(indexflag)
  {
    for(i = optind; i < argc; i++)
    {
      C_error_printf("Reindexing: %s\n", argv[i]);
      if(! C_fortune_indexdb(argv[i]))
        C_error_printf("  Failed.");
    }

    exit(EXIT_SUCCESS);
  }

  /* remember user-specified dbs */

  vec = C_vector_start(20);
  for(i = optind; i < argc; i++)
  {
    if(! C_hashtable_restore(filemap, argv[i]))
      C_error_printf("No such database: %s\n", argv[i]);
    else
      C_vector_store(vec, argv[i]);
  }
  userdbnames = C_vector_end(vec, &userdbcount);

  /* daemon mode? */
  
  if(fifo)
  {
    /* create the FIFO, if we need to */
    
    if(access(fifo, F_OK))
    {
      if(mkfifo(fifo, (S_IWUSR | S_IRUSR | S_IRGRP | S_IROTH)))
      {
        C_error_printf("Unable to create pipe: %s\n", fifo);
        exit(EXIT_FAILURE);
      }
    }
    else
    {
      stat(fifo, &stbuf);
      if(!S_ISFIFO(stbuf.st_mode))
      {
        C_error_printf("%s exists, but it is not a pipe.\n", fifo);
        exit(EXIT_FAILURE);
      }
      
      if(access(fifo, (R_OK | W_OK)))
      {
        C_error_printf("%s exists, but it is not readable and/or writable.\n",
                       fifo);
        exit(EXIT_FAILURE);
      }
    }
        
    /* spawn as a daemon */
    
    pid = fork();
    if(pid < 0)
    {
      C_error_printf("Unable to fork.\n");
      exit(EXIT_FAILURE);
    }
    else if(pid > 0)
    {
      C_error_printf("Daemon forked (pid = %d)\n", pid);
      exit(EXIT_SUCCESS);
    }
    
    /* child code follows */

    signal(SIGPIPE, SIG_IGN);
    
#ifndef DEBUG
    setsid();
    close(STDIN_FILENO);
    close(STDOUT_FILENO);
    close(STDERR_FILENO);

    signal(SIGHUP, SIG_IGN);
#endif

    /* open syslog */

    openlog("pfortune", (LOG_PID | LOG_CONS), LOG_DAEMON);

    for(;;)
    {
      /* open the FIFO for writing */

      if(!(fp = fopen(fifo, "w")))
      {
        syslog(LOG_ERR, "Unable to open FIFO %s for writing; exiting.", fifo);
        exit(EXIT_FAILURE);
      }

      /* write out a fortune */
      
      print_fortune(fp);
      fclose(fp);
      
      sleep(1);
    }
  }

  /* default mode */
  
  print_fortune(stdout);

  if(nlflag)
    putchar('\n');
  
  return(EXIT_SUCCESS);
}

/* end of source file */
