/* ----------------------------------------------------------------------------
   watcherd - file watcher daemon
   Copyright (C) 2000-2007  Mark A Lindner

   This file is part of misctools.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, see
   <http://www.gnu.org/licenses/>.
   ----------------------------------------------------------------------------
*/

/* --- Feature Test Switches --- */

#include "config.h"

/* --- System Headers --- */

#include <stdio.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <signal.h>

#include <cbase/cbase.h>

/* --- Local Headers --- */

/* --- Macros --- */

#define HEADER "watcherd v" VERSION " - Mark Lindner"
#define USAGE "[-i <interval>] [-h] <file> <cmd> [<arg> ...]"
#define DFL_INTERVAL 60

/* --- File Scope Variables --- */

c_bool_t watch(char *, char **);

static off_t old_size = 0;
static struct stat buf;
static int interval = DFL_INTERVAL;

/* --- Functions --- */

int main(int argc, char **argv)
{
  pid_t pid;
  c_bool_t errflag = FALSE;
  extern char *optarg;
  extern int optind;
  int ch, n;

  C_error_init(*argv);
  
  while((ch = getopt(argc, argv, "hi:")) != EOF)
    switch((char)ch)
    {
      case 'h':
        C_error_printf("%s\n", HEADER);
        C_error_usage(USAGE);
        exit(EXIT_SUCCESS);

      case 'i':
        n = atoi(optarg);
        if(n < 1)
        {
          C_error_printf("Bad interval value: %d\n", n);
          errflag = TRUE;
        }
        else
          interval = n;
        break;

      default:
        errflag = TRUE;
    }

  n = (argc - optind);

  if(errflag || (n < 2))
  {
    C_error_usage(USAGE);
    exit(EXIT_FAILURE);
  }

  if(stat(argv[optind], &buf) != 0)
  {
    C_error_printf("Unable to stat file: %s\n", argv[optind]);
    exit(EXIT_FAILURE);
  }
  
  old_size = buf.st_size;
  
#ifdef DEBUG

  watch(argv[optind], &argv[++optind]);

#else
  
  pid = fork();
  if(!pid)
  {
    setsid();
    close(STDIN_FILENO);
    close(STDOUT_FILENO);
    close(STDERR_FILENO);
    signal(SIGINT, SIG_IGN);
    watch(argv[optind], &argv[optind + 1]);
    exit(EXIT_FAILURE);
  }
  
  C_error_printf("daemon forked: PID=%ld)\n", pid);

#endif

  exit(EXIT_SUCCESS);
}

/*
 */

c_bool_t watch(char *path, char **cmd)
{
  for(;;)
  {
    sleep(interval);
    
    if(stat(path, &buf) != 0)
    {
      C_debug_printf("stat failed: %s\n", path);
      return(FALSE);
    }

    if(buf.st_size > old_size)
    {
      C_debug_printf("size is now: %d\n", buf.st_size);
      C_exec_run(cmd, -1, -1, TRUE);
    }

    old_size = buf.st_size;    
  }

  return(TRUE);
}

/* end of source file */
