/***************************************************************************
 *   Copyright (C) 2005 by Dmitry Nezhevenko                               *
 *   dion@rcom.zp.ua                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "p2kproc.h"

#include <time.h>

#include <qthread.h>
#include <qmutex.h>
#include <qapplication.h>
// #include <qptrlist.h>

#include <usb.h>
#include <string.h>

#include "custmsg.h"

// Intefaces for Motorola P2k phones
#define INTERFACE_ACCESSORIES 0x05
#define INTERFACE_DATA_LOGGING_MCU 0x06
#define INTERFACE_TEST_COMMAND 0x08

// Transfer direction
#define DIR_IN 0
#define DIR_OUT 1

// #define MY_DEBUG
// #define DEBUG_FN

#ifdef MY_DEBUG
#define FUNC(val)						\
	char FUNC_NAME[]=val;				\
	printf("=== [P2k API: %s]\n", val);
#else
#define FUNC(val)						\
	char FUNC_NAME[]=val;				
#endif
// 	printf("Launch: %s\n",FUNC_NAME);

#define RAISE(...)								\
{ 												\
	qDebug("(E_%s: %s)", FUNC_NAME, __VA_ARGS__);	\
	return(-1);									\
}

/*myDev devlst[3]={{0x00, 0x00, "NONE"},
	{DEF_PHONE_AT_VENDOR, DEF_PHONE_AT_PRODUCT, "AT"},
	{DEF_PHONE_P2K_VENDOR, DEF_PHONE_P2K_PRODUCT, "P2K"}
}
*/


P2kProc::P2kProc(QObject * o)
		: parent(o)
{
	drv_initUsb();
	phoneMode=-1;
	phoneHandle=0;
	phone=0;
	state=CONNECT_NONE;
	freeID.wordId=0;
	isFirstRead=0;
	lastControlStatus=0;

	devlst[0].vendor=0x00;
	devlst[0].product=0x00;
	strcpy(devlst[0].name,"NONE");

	devlst[1].vendor=DEF_PHONE_AT_VENDOR;
	devlst[1].product=DEF_PHONE_AT_PRODUCT;
	strcpy(devlst[1].name,"AT");

	devlst[2].vendor=DEF_PHONE_P2K_VENDOR;
	devlst[2].product=DEF_PHONE_P2K_PRODUCT;
	strcpy(devlst[2].name,"P2K");
}


void P2kProc::setATconfig (unsigned int vendor, unsigned int product)
{
	devlst[1].vendor=vendor;
	devlst[1].product=product;
}

void P2kProc::setP2kconfig (unsigned int vendor, unsigned int product)
{
	devlst[2].vendor=vendor;
	devlst[2].product=product;
}

void P2kProc::setDevice(char * cmd)
{
	strcpy(ACMdev,cmd);
}

void P2kProc::stopThread()
{
	isRunning=0;
}

void P2kProc::postMessage(int style, const QString &msg)
{
	msg_P2kLog * m = new msg_P2kLog(style, msg);
	QApplication::postEvent(parent, m);
}

void P2kProc::postProgress(int pg)
{
	QApplication::postEvent(parent, new msg_ParamInfo(PARAM_PROGRESS, QString("%1").arg(pg)));
}

int P2kProc::isBusy()
{
	if (task)
		postMessage(MSGSTYLE_ERROR, "Phone is busy. Please try later");
	return(task);
}

// ===================== Thread main loop =======================

void P2kProc::run()
{
	isRunning=1;
	task=0;
	while (isRunning)
	{
		doUpdateMode();
		if (task)
		{
			switch (task)
			{
				case TASK_CONNECT: doConnect(); break;
				case TASK_DISCONNECT: doDisconnect(); break;
				case TASK_REBOOT: doReboot(); break;
				case TASK_SUSPEND: doSuspend(); break;
				case TASK_GETINFO: doGetInfo(); break;
				case TASK_FILELIST: doGetFileList(); break;
				case TASK_DOWNLOADFILES: doDownloadFiles(); break;
				case TASK_UPLOADFILES: doUploadFiles(); break;
				case TASK_DELETEFILES: doDeleteFiles(); break;
				case TASK_CHANGEATTR: doChangeAttr(); break;
				case TASK_READSEEM: doReadSeem(); break;
				case TASK_WRITESEEM: doWriteSeem(); break;
				case TASK_BACKUP: doBackup(); break;
				default: postMessage(MSGSTYLE_ERROR, QString("Unknown task ignored: %1").arg(task));
			}
			QApplication::postEvent(parent, new msg_ParamInfo(PARAM_COMPLETE, QString("%1").arg(task)));
			task=0;
		}
		else msleep(100);
	}
}

// ======================  Action Implementation

void P2kProc::setState(int st)
{
	if (state == st) return;
	state=st;
	switch (state)
	{
		case CONNECT_NONE:
		{
			postMessage(MSGSTYLE_INFO, "Phone disconnected");
			phone=0;
			phoneHandle=0;
			break;
		}
		case CONNECT_P2K: postMessage(MSGSTYLE_INFO, "Phone connected as P2K"); break;
		default:;
	}

	msg_StatusChanged * msg = new msg_StatusChanged(state);
	QApplication::postEvent(parent, msg);
}

int P2kProc::getState()
{
	return state;
}

void P2kProc::doUpdateMode()
{
	QMutexLocker lock ( &mtxInUse );
	int pm;

	pm=drv_findPhone();

	if (pm!=phoneMode)
	{
		msg_ModeChanged * msg = new msg_ModeChanged(pm);
		QApplication::postEvent(parent, msg);
	}

	if ((phoneMode==-1) && (pm==PHONE_NONE))
	{
		postMessage(MSGSTYLE_INFO,"Phone is unpluged. Please connect it");
		phoneMode=0;
	}
	if (pm!=phoneMode)
	{
		if (pm!=state)
			setState(CONNECT_NONE);
		qDebug("New mode: %d", pm);
		if (pm==0)
			postMessage(MSGSTYLE_INFO, QString("Phone is unpluged"));
		else
			postMessage(MSGSTYLE_INFO, QString("Phone pluged as %1").arg(devlst[pm].name));
		phoneMode=pm;
	}
}

void P2kProc::doConnect()
{
	qDebug("P2kProc::doConnect()");
	postMessage(MSGSTYLE_NONE, "Try to connect");
	if (drv_connect()>0)
	{
		doUpdateMode();
		setState(CONNECT_P2K);
		//doGetInfo();
	}
	else
		postMessage(MSGSTYLE_ERROR, "Unable to connect");
}

void P2kProc::doDisconnect()
{
	postMessage(MSGSTYLE_NONE, "Try to disconnect");
	if (drv_closePhone()>0)
		setState(CONNECT_NONE);
	else
		postMessage(MSGSTYLE_ERROR, "Unable to disconnect");
}

void P2kProc::doReboot()
{
	postMessage(MSGSTYLE_NONE, "Try to reboot");
	if (drv_reboot()>0)
	{
		msg_StatusChanged * msg = new msg_StatusChanged(CONNECT_NONE);
		QApplication::postEvent(parent, msg);
		postMessage(MSGSTYLE_NONE, "Phone rebooted");
	}
	else
		postMessage(MSGSTYLE_ERROR, "Unable to reboot");
}

void P2kProc::doSuspend()
{
	postMessage(MSGSTYLE_NONE, "Try to suspend");
	if (drv_suspend()>0)
	{
		msg_StatusChanged * msg = new msg_StatusChanged(CONNECT_NONE);
		QApplication::postEvent(parent, msg);
		postMessage(MSGSTYLE_NONE, "Phone suspended");
	}
	else
		postMessage(MSGSTYLE_ERROR, "Unable to suspend");
}

void P2kProc::doGetInfo()
{
	doGetPhoneModel();
	doGetDriveName();
	doGetFileCount();
	doGetFreeSpace();
	if (memCnt == 1) doGetFileList();
}

void P2kProc::doGetFileCount()
{
	int i;
	i=drv_fileCount();
	if (i<0)
	{
		postMessage(MSGSTYLE_ERROR, "Unable to get file count");
		return;
	}
	QApplication::postEvent(parent, new msg_ParamInfo(PARAM_FILECOUNT, QString("%1").arg(i)));
}

void P2kProc::doGetDriveName()
{
	int i;
	char s[256];
	i=drv_getDriveName((unsigned char *) s);
	if (i<0)
	{
		postMessage(MSGSTYLE_ERROR, "Unable to get drive name");
		return;
	}
	QApplication::postEvent(parent, new msg_ParamInfo(PARAM_DRIVENAME, QString("%1").arg(s)));
}

void P2kProc::doGetFreeSpace()
{
	long i;
	char s[256];

	if (drv_getDriveName((unsigned char *) s)<0)
	{
		postMessage(MSGSTYLE_ERROR, "Unable to get drive name");
		return;
	}
	i=drv_freeSpace((unsigned char *) s);
	if (i<0)
	{
		postMessage(MSGSTYLE_ERROR, "Unable to get free space");
		return;
	}
	QApplication::postEvent(parent, new msg_ParamInfo(PARAM_FREESPACE, QString("%1").arg(i)));
}

void P2kProc::doGetPhoneModel()
{
	int i;
	char s[256];
	i=drv_getPhoneModel((unsigned char *) s);
	if (i<0)
	{
		postMessage(MSGSTYLE_ERROR, "Unable to get phone model");
		return;
	}
	QApplication::postEvent(parent, new msg_ParamInfo(PARAM_PHONEMODEL, QString("%1").arg(s)));
}

void P2kProc::doGetFileList()
{
	postMessage(MSGSTYLE_NONE, "Getting file list");
	int i;
	i=drv_fileList();
	if (i<0)
	{
		postMessage(MSGSTYLE_ERROR, "Unable to get file list");
		return;
	}
	else
		postMessage(MSGSTYLE_NONE, "Complete");

	// 	postMessage(MSGSTYLE_NONE, "AAA");
}

void P2kProc::doDownloadFiles()
{
	postMessage(MSGSTYLE_NONE, QString("Downloading %1 file(s)").arg(memCnt));
	if (drv_downloadFiles(memLst, memCnt, memDir)<0)
	{
		postMessage(MSGSTYLE_ERROR, "Unable to download files");
	}
	else
		postMessage(MSGSTYLE_NONE, "Downloading complete");
	free(memLst);
}

void P2kProc::doUploadFiles()
{
	postMessage(MSGSTYLE_NONE, QString("Uploading %1 file(s)").arg(memCnt));
	if (drv_uploadFiles(memLst, memCnt, memDir, memDir2)<0)
	{
		postMessage(MSGSTYLE_ERROR, "Unable to upload files");
	}
	else
		postMessage(MSGSTYLE_NONE, "Uploading complete");
	free(memLst);
}

void P2kProc::doDeleteFiles()
{
	postMessage(MSGSTYLE_NONE, QString("Deleting %1 file(s)").arg(memCnt));
	if (drv_deleteFiles(memLst, memCnt)<0)
	{
		postMessage(MSGSTYLE_ERROR, "Unable to delete files");
	}
	else
		postMessage(MSGSTYLE_NONE, "Deleting complete");
	free(memLst);
}

void P2kProc::doChangeAttr()
{
	postMessage(MSGSTYLE_NONE, QString("Changing attributes  for %1 file(s)").arg(memCnt));
	if (drv_changeAttr(memLst, memCnt)<0)
	{
		postMessage(MSGSTYLE_ERROR, "Unable to change attributes");
	}
	else
		postMessage(MSGSTYLE_NONE, "Complete");
	free(memLst);
}

void P2kProc::doReadSeem()
{
	//FIXME: doReadSeem
	QString sX=QString("%1").arg(memX,0,16).rightJustify(4,'0');
	QString sY=QString("%1").arg(memY,0,16).rightJustify(4,'0');
	postMessage(MSGSTYLE_NONE, QString("Reading seem %1_%2.seem").arg(sX).arg(sY));
	int t=*memSize;
	*memSize=drv_read_seem(memX, memY, memBuf, t);
	if (*memSize<0)
	{
		postMessage(MSGSTYLE_ERROR, "Unable to read seem");
	}
	else
	{
		postMessage(MSGSTYLE_NONE, "Complete");
		QApplication::postEvent(parent, new msg_ParamInfo(PARAM_SEEMREADY, "1"));
	}
}

void P2kProc::doWriteSeem()
{
	QString sX=QString("%1").arg(memX,0,16).rightJustify(4,'0');
	QString sY=QString("%1").arg(memY,0,16).rightJustify(4,'0');
	postMessage(MSGSTYLE_NONE, QString("Writing seem %1_%2.seem").arg(sX).arg(sY));
	if (drv_write_seem(memX, memY, memBuf, memCnt)<0)
	{
		postMessage(MSGSTYLE_ERROR, "Unable to write seem");
	}
	else
	{
		postMessage(MSGSTYLE_NONE, "Complete");
		QApplication::postEvent(parent, new msg_ParamInfo(PARAM_SEEMREADY, "2"));
	}

}

void P2kProc::doBackup()
{
	QString sX1=QString("%1").arg(memX,0,16).rightJustify(4,'0');
	QString sY1=QString("%1").arg(memY,0,16).rightJustify(4,'0');
	QString sX2=QString("%1").arg(memX1,0,16).rightJustify(4,'0');
	QString sY2=QString("%1").arg(memY1,0,16).rightJustify(4,'0');

	postMessage(MSGSTYLE_NONE, QString("Doing backup of seem %1_%2.seem to %3_%4.seem").arg(sX1).arg(sY1).arg(sX2).arg(sY2));

	if (drv_backup(memX, memY, memX1, memY1, (unsigned char*)memDir) <0)
	{
		postMessage(MSGSTYLE_ERROR, "Unable to backup seem");
	}
	else
	{
		postMessage(MSGSTYLE_NONE, "Backup complete");
		// 		QApplication::postEvent(parent, new msg_ParamInfo(PARAM_SEEMREADY, "2"));
	}
}

// ======================= Public methods ============================

int P2kProc::phGetDevList(devInfo * lst, int cnt)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return 0;
	
	if (state != CONNECT_NONE) doDisconnect();

	usb_find_busses();
	usb_find_devices();
	
	struct usb_bus *bus;
	struct usb_device *dev;
	usb_dev_handle *udev;
	
	int idx=0;
	unsigned int devVendor;
	unsigned int devProduct;
	char devManufacturerStr[200];
	char devProductStr[200];
	int ret;
	
	for (bus = usb_busses; bus; bus = bus->next)
	{
    	for (dev = bus->devices; dev; dev = dev->next)
		{
			devVendor=dev->descriptor.idVendor;
			devProduct=dev->descriptor.idProduct;
			devManufacturerStr[0]=0;
			devProductStr[0]=0;
			
			udev = usb_open(dev);
			if (udev)
			{
				if (dev->descriptor.iManufacturer)
					ret = usb_get_string_simple(udev, dev->descriptor.iManufacturer, 
						devManufacturerStr, sizeof(devManufacturerStr));
  
				if (dev->descriptor.iProduct)
					ret = usb_get_string_simple(udev, dev->descriptor.iProduct,
						devProductStr, sizeof(devProductStr));				
   				usb_close (udev);
			}
			
			lst[idx].vendor=devVendor;
			lst[idx].product=devProduct;
			strcpy(lst[idx].manufacturerStr, devManufacturerStr);
			strcpy(lst[idx].productStr, devProductStr);
			if (++idx > cnt-1)
			{
				qDebug("phGetDevList: Please, increase cnt!!!");
				return idx;
			}
		}
	}
	
	return idx;

	//task=TASK_CONNECT;
	//
}

void P2kProc::phConnect()
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	task=TASK_CONNECT;
}

void P2kProc::phDisconnect()
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	task=TASK_DISCONNECT;
}

void P2kProc::phReboot()
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	task=TASK_REBOOT;
}

void P2kProc::phSuspend()
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	task=TASK_SUSPEND;
}

void P2kProc::phGetInfo(int what)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	memCnt=what;
	task=TASK_GETINFO;
}

void P2kProc::phGetFileList()
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	task=TASK_FILELIST;
}

void P2kProc::phDownloadFiles(P2kFile * lst, int cnt, const char * dir)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	memLst=(P2kFile *) malloc(sizeof(P2kFile)*cnt);
	memcpy(memLst,lst,sizeof(P2kFile)*cnt);
	memCnt=cnt;
	strcpy(memDir, dir);
	task=TASK_DOWNLOADFILES;
}

void P2kProc::phUploadFiles(P2kFile * lst, int cnt, const char * dirLocal, const char * dirPhone)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	memLst=(P2kFile *) malloc(sizeof(P2kFile)*cnt);
	memcpy(memLst,lst,sizeof(P2kFile)*cnt);
	memCnt=cnt;
	strcpy(memDir, dirLocal);
	strcpy(memDir2, dirPhone);
	task=TASK_UPLOADFILES;
}

void P2kProc::phDeleteFiles(P2kFile * lst, int cnt)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;

	memLst=(P2kFile *) malloc(sizeof(P2kFile)*cnt);
	memcpy(memLst,lst,sizeof(P2kFile)*cnt);
	memCnt=cnt;
	task=TASK_DELETEFILES;
}

void P2kProc::phChangeAttr(P2kFile * lst, int cnt)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;

	memLst=(P2kFile *) malloc(sizeof(P2kFile)*cnt);
	memcpy(memLst,lst,sizeof(P2kFile)*cnt);
	memCnt=cnt;
	task=TASK_CHANGEATTR;
}

void P2kProc::phReadSeem(int x, int y, unsigned char * buf, int * size)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;

	memBuf=buf;
	memSize=size;
	memX=x;
	memY=y;
	task=TASK_READSEEM;
}

void P2kProc::phWriteSeem(int x, int y, unsigned char * buf, int size)
{
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;

	memBuf=buf;
	memCnt=size;
	memX=x;
	memY=y;
	task=TASK_WRITESEEM;

}

void P2kProc::phBackup(int x1, int y1, int x2, int y2, unsigned char * dir)
{
	//FIXME:ph
	QMutexLocker locker ( &mtxInUse );
	if (isBusy()) return;
	memX=x1;
	memY=y1;
	memX1=x2;
	memY1=y2;
	strcpy((char *)memDir, (char *)dir);
	task=TASK_BACKUP;
}

// ======================= libUsb interface ===========================

// Initialize libUsb
void P2kProc::drv_initUsb()
{
	usb_init();
}

// Detect current phone state
struct usb_device * P2kProc::drv_findDevice(int vendor, int product)
{
	struct usb_bus *busses;
	struct usb_bus *bus;

	usb_find_busses();
	usb_find_devices();

	busses = usb_get_busses();

	for (bus = busses; bus; bus = bus->next)
	{
		struct usb_device *dev;

		for (dev = bus->devices; dev; dev = dev->next)
			if ((dev->descriptor.idVendor==vendor) && (dev->descriptor.idProduct==product))
				return(dev);

	}
	return(0);
}

// Return Current phone state
int P2kProc::drv_findPhone()
{
	for (int i=1; i<=2; i++)
	{
		phone=(drv_findDevice(devlst[i].vendor, devlst[i].product));
		if (phone)  return(i);
	}
	return(PHONE_NONE);
}

void P2kProc::drv_switchP2K(char * st)
{
	char cmd[1024];
	sprintf(cmd,"echo \"AT+MODE=8\" > %s", (st == 0) ? ACMdev : st);
	system(cmd);
}

// Open phone
int P2kProc::drv_openPhone()
{
	FUNC("openPhone");

	phoneHandle=0;
	phoneHandle = usb_open(phone);
	if (phoneHandle==0) RAISE("Unable to open phone\n");
	if (usb_set_configuration(phoneHandle,1)) RAISE("Unable to set configuration");
	if (usb_claim_interface(phoneHandle, INTERFACE_TEST_COMMAND)) RAISE("Unable to claim the interface");
	return(1);
}

// Close phone handle
int P2kProc::drv_closePhone()
{
	FUNC("closePhone");
	if (phoneHandle<=0) return(-1);
	if (usb_close(phoneHandle)) RAISE("Unable to close phone");
	return(1);
}

// Connect to phone.
int P2kProc::drv_connect()
{
	FUNC("drv_connect");
	int ph=drv_findPhone();
	if (ph==PHONE_NONE) RAISE("no phone")
		if (ph==PHONE_AT) drv_switchP2K();

	int t;
	t=time(NULL);

	while ((time(NULL)-t<5) && (ph!=PHONE_P2K))
	{
		usb_find_devices();
		ph=drv_findPhone();
		usleep(10000);
	}
	if (ph!=PHONE_P2K) return(-1);
	return(drv_openPhone());
}

// ====================== Debug methods ==============================

// Out buffer content to standart output
void P2kProc::showArr(unsigned char *bytes, int size)
{
#define BYTES_IN_LINE 16

	int i;
	if (size==0) return;

	printf("   -------");
	for (i=0; i<16; i++)
	{
		printf("%3X",i);
	}
	printf("\n");

	for (i=0; i<size; i++)
	{
		if (i % 16 ==0)
		{
			if (i!=0) printf("\n");
			printf("   %06x: ",i);
		}
		printf("%02x ",bytes[i]);
	}
	printf("\n");
}


// ====================== Main USB IO ================================

int P2kProc::get_cmd_size(unsigned char * packetCount)
{
	mWord cnt;
	mWord size;
	int bufsize;
	cnt.id[0]=packetCount[1]; cnt.id[1]=packetCount[0];
	size.id[0]=packetCount[3]; size.id[1]=packetCount[2];
	bufsize=2*cnt.wordId+size.wordId+4;
	return(bufsize);
}

// Send control message to device, but with debug loging and error handling
// Return !=0 if error
int P2kProc::sendControl(int dir, usb_dev_handle *dev, int requesttype, int request, int value,
                         int index, char *bytes, int size, int timeout)
{
	FUNC("sendControl");
	if (dev==0) RAISE("no connection");
	int ret;
#ifdef MY_DEBUG
	if (dir==DIR_IN) printf("<==== "); else printf("====> ");
	printf("Control Message\n");
	printf("   requesttype=0x%02x, request=0x%02x, value=0x%02x, index=0x%02x, size=0x%02x (%04d), timeout=%04d\n", requesttype, request, value, index, size, size, timeout);

	if (dir==DIR_OUT) showArr((unsigned char *)bytes, size);
#endif
	ret=usb_control_msg(dev, requesttype, request, value, index, bytes, size, timeout);
	lastControlStatus=ret;
#ifdef MY_DEBUG
	printf("   result=%04d",ret);
#endif
	if (ret<0) printf(" Error:[%s]\n",usb_strerror());
#ifdef MY_DEBUG
	else printf("\n");
	if ((dir==DIR_IN) && (ret>0)) showArr((unsigned char *)bytes,ret);
	printf("\n");
#endif
	return(ret);
}

// Add header and set data to phone via sendControl
int P2kProc::outData(unsigned char * data, unsigned int size)
{
	/*41 02 00 00 08 00 x1 x1		- Setup packet
	  x2 x2 x3 x3 x4 x4 00 00		- data
	x1 -    
	x2 - ID -  .  p2kman     0.
	x3 - .  restart - 22
	x4 -    .*/

	// Set Packet ID
	freeID.wordId++;
	freeID.wordId&=0xFFF;
	// 	printf("<<<< ID=%04X\n\n\n", freeID.wordId);
	data[0]=freeID.id[1];
	data[1]=freeID.id[0];

	return(sendControl(DIR_OUT, phoneHandle, 0x41, 0x02, 0x00, 0x08, (char*) data, size,1000));
}

// Request Packet Count
int P2kProc::inpSize(unsigned char * cmd, int size)
{
	/*  -      -:
	1 00 00 00 08 00 08 00 */
	int t;
	int ret;

	t=time(NULL);

	while (time(NULL)-t<5)
	{
		ret = sendControl(DIR_IN, phoneHandle, 0xc1, 0x00, 0x00, 0x08, (char *)cmd, size, 1000);
		if ((ret<0) || (ret>2)) break;
#ifdef MY_DEBUG
		else printf("Error answer. try again\n");
#endif
		usleep(1000);
	}
	return(ret);

}

// Get data from phone and perform error handling
int P2kProc::inpData(unsigned char * data, unsigned int count, unsigned int size)
{
	FUNC("inpData");
	/*   :
	1 01 x1 x1 08 00 x2 x2
	x1 - - ,     .
	x2 -  . 

	 :
	  0  1  2  3  4  5   6  7  8  9  0  1  2  3  4  5  6  6  7
	    01 00 00 00 00 87  80 03 80 4a 00 7f 00 00 10
	 01 00 00 00 00 09 80  04  80  4a 00 01 00 00 08
	 x1 ?? ?? ?? x2 x2 [x3 x3 x4 x4 x5 x5 ?? ?? yy yy yy yy yy] ...
	x1 - -  .
	x2 -   (   +8)
	x3 - PacketID or 8000h    PacketID      !
	x4 -  ... ( 1 ) 60 - ok,   , 80 - ok,   ,  - .
	x5 -   
	yy -  
	[] -   1 .    -  .
	*/

	int ret=sendControl(DIR_IN , phoneHandle, 0xc1, 0x01, 0x01, 0x08, (char*)data, size, 1000);
	if (ret<0) return (ret);

	int err;
	mWord x2;
	x2.id[1]=data[4]; x2.id[0]=data[5];
	err=0;
	if (err+=(data[0]!=count)) RAISE("E001");
	if (err+=(x2.wordId!=size-6)) RAISE("E002");
	if (err) return(-999); else return(ret);
}

int P2kProc::check_packet_header(unsigned char * buf, int bufsize, int adr, char needdata, char checky)
{
	FUNC("CPH");

	mWord packId;
	mWord packSize;
	int myid;
	unsigned char packType;
	unsigned char iserr;

	packId.id[0]=buf[adr+1];
	packId.id[1]=buf[adr];
	myid= packId.wordId & 0xFFF;

	packType=buf[adr+2];
	packSize.id[0]=buf[adr+5];
	packSize.id[1]=buf[adr+4];
	iserr=buf[adr+8];

	if (bufsize<0) RAISE("E00");

	if (myid != freeID.wordId) RAISE("E01");
	if (needdata)
	{
		if (packType!=0x80) RAISE("E02a");
	}
	else
	{
		if (packType!=0x60) RAISE("E02b");
	}
	if (checky && iserr) RAISE("E03");

	/*	if ((myid != freeID.wordId) || (packType!=0x80) || (iserr))
		{
			printf("(E_CPH_01)");
			return(-1);
		}
	*/
	return(packSize.wordId);
}

void P2kProc::swapLong(mLong * nm)
{
	mLong t;
	t.longId=nm->longId;
	nm->id[0]=t.id[3];
	nm->id[1]=t.id[2];
	nm->id[2]=t.id[1];
	nm->id[3]=t.id[0];
	// 	printf(" -- Swaplong: Was=%08X  Now=%08X -- \n",t.longId, nm->longId);s
}

void swapWord(mWord * nm)
{
	mWord t;
	t.wordId=nm->wordId;
	nm->id[0]=t.id[1];
	nm->id[1]=t.id[2];
}

// ==================== P2k Mode methods

// Reboot phone
int P2kProc::drv_reboot()
{
	FUNC("drv_reboot");
	//This is reboot p2k command
	unsigned char cmd1[]={0xFF, 0xFF, 0x00, 0x22, 0x00, 0x00, 0x00, 0x00};
	unsigned char cmd2[0x0e];
	if (outData(cmd1, sizeof(cmd1))<0) RAISE("E001");
	if (inpSize(packetCount, sizeof(packetCount))<0) RAISE("E002");
	if (inpData(cmd2, 1, sizeof(cmd2))<0) RAISE("E003");
	return(1);
}

// Suspend phone
int P2kProc::drv_suspend()
{
	FUNC("drv_suspend");
	unsigned char cmd1[]={0xFF, 0xFF, 0x00, 0x36, 0x00, 0x01, 0x00, 0x00,0x00};
	unsigned char * tmp;
	int sz;
	if (outData(cmd1, sizeof(cmd1))<0) RAISE("E01");
	if (inpSize(packetCount, sizeof(packetCount))<0) RAISE("E02");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E03");
	free(tmp);
	return(1);
}

// Get drive information
int P2kProc::drv_getDriveName(unsigned char * buf)
{
	FUNC("getDriveName");

	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0A};
	// 	unsigned char tmp[1024];
	unsigned char * tmp;
	int sz;
	int buf_ps;
	int ps;
	// 0x00 0x4A -  

	outData(cmd, sizeof(cmd));
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E001");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E002");
	if (!check_packet_header(tmp, sz, 6)) RAISE("E003");
	buf_ps=0;
	ps=15;
	while (tmp[ps]!=0)
	{
		buf[buf_ps++]=tmp[ps];
		ps+=2;
	}
	buf[buf_ps++]=0;
	// 	printf("%02X",tmp[ps]);

	free(tmp);
	return(1);
}

int P2kProc::drv_getPhoneModel(unsigned char * buf)
{
	FUNC("getPhoneName");

	unsigned char cmd[0x10]={0xFF, 0xFF, 0x00, 0x20, 0x00, 0x08, 0x00, 0x00, 0x01, 0x17, 0x00, 0x01, 0x00, 0x00, 0x00};
	// 00 02 00 20 00 08 00 00 01 17 00 01 00 00 00 00
	// 	unsigned char tmp[1024];
	unsigned char * tmp;
	int sz;
	int buf_ps;
	int ps;
	// 0x00 0x4A -  

	outData(cmd, sizeof(cmd));
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E001");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E002");
	if (!check_packet_header(tmp, sz, 6)) RAISE("E003");
	buf_ps=0;
	ps=0x10;
	while (tmp[ps]!=0)
	{
		buf[buf_ps++]=tmp[ps];
		ps+=2;
	}
	buf[buf_ps++]=0;
	// 	printf("%02X",tmp[ps]);

	free(tmp);
	return(1);
}

int P2kProc::drv_freeSpace(unsigned char * dev)
{
	FUNC("freeSpace");

	unsigned char cmd[0x208] ={0xFF, 0xFF, 0x00, 0x4A, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0B, 0x00};
	unsigned char * tmp;
	int i;
	int ps;
	int sz;
	int size=strlen((char*)dev);

	mLong lng;

	ps=13;
	for (i=0; i< size; i++)
	{
		cmd[ps++]=dev[i];
		cmd[ps++]=0;
	}
	cmd[ps++]=0;
	cmd[ps++]=0;

	if (outData(cmd, sizeof(cmd))<=0) RAISE("E001");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E002");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E003");
	if (!check_packet_header(tmp, sz, 6)) RAISE("E004");

	lng.id[3]=tmp[14]; lng.id[2]=tmp[15]; lng.id[1]=tmp[16]; lng.id[0]=tmp[17];
	return(lng.longId);

	free(tmp);
}

int P2kProc::drv_fileCount()
{
	FUNC("fileCount");

	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07};
	unsigned char * tmp;
	int sz;
	mWord t;

	if (outData(cmd, sizeof(cmd))<=0) RAISE("E001");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E002");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);

	if (inpData(tmp, 0x01, sz)<0) RAISE("E003");

	// 	This package has incorrect yy. So we can not check it.
	// 	if (check_packet_header(tmp,sz,6)<0) RAISE("E004");
	t.id[1]=tmp[14]; t.id[0]=tmp[15];
	free(tmp);
	return(t.wordId);
}

int P2kProc::drv_fileList()
{
	FUNC("fileList");

	struct filerec
	{
		char name[0x104];
		char b1;
		unsigned char attr;
		char b2;
		unsigned char owner;
		mLong size;
	};

	struct filerec * rc;

	int fCnt;
	unsigned char cmd[] = {0xFF, 0xFF, 0x00, 0x4A, 0x00, 0x04, 0x00, 0x00, 0x00,  0x00, 0x00, 0x08};
	unsigned char * tmp;
	int sz;
	int sz1;
	mWord rCnt;
	int inRec;
	int i;
	int curidx=0;
#ifdef DEBUG_FN
	int tmpC;
#endif


	fCnt=drv_fileCount();
	if (fCnt<0) RAISE("E000");
	sz=-1;

	msg_FileList * msg = new msg_FileList(fCnt);

	while (curidx < fCnt)
	{
		if (outData(cmd, sizeof(cmd))<=0) RAISE ("E001");
		// 		usleep(100000);
		if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E002");
		sz1=get_cmd_size(packetCount);
		// 	printf("!!!!! sz1=%d\n",sz1);
		if (sz1>sz)
		{
			if (sz!=-1) free(tmp);
			sz=sz1;
			tmp=(unsigned char *) malloc (sz);
		}

		if (inpData(tmp, 0x01, sz1)<0) RAISE ("E003");
		if (check_packet_header(tmp,sz,6)<0) RAISE("E004");

		rCnt.id[0]=tmp[0x0f]; rCnt.id[1]=tmp[0x10];

		//printf("Files in rec:%d\n",rCnt.wordId);
		// 		inRec=(sz-0x11) / sizeof(filerec);
		// 		if (inRec!=rCnt.wordId)
		// 			printf("File count in record do not mach: inRec %4d, detected %4d\n", rCnt.wordId, inRec);

		inRec=rCnt.wordId;

		rc=(struct filerec *) (tmp+0x12);
		for (i=0; i<inRec; i++)
		{
			swapLong(&(rc[i].size));
			if ((rc[i].b1!=0) || (rc[i].b2!=0)) printf("!!!!!!!!!!");

			curidx++;
			// 			printf("[%06d] attr=0x%02X, owner=0x%02X, size=%08d, name=[%s]\n",curidx,rc[i].attr, rc[i].owner, rc[i].size, rc[i].name);

#ifdef DEBUG_FN
			tmpC=0;
			printf("[FILE %5d (%02x %02x)] %s [ ", curidx,rc[i].b1, rc[i].b2, rc[i].name);
			while (rc[i].name[tmpC]!=0)
			{
				printf("%02x ",rc[i].name[tmpC++]);
			}
			printf("]\n");
			if (curidx==250)
				showArr(tmp, sz);

#endif

			msg->addItem(curidx, QString(rc[i].name), rc[i].size.longId , rc[i].owner, rc[i].attr);
			postProgress(100*(curidx+1)/fCnt);
		}
	}
	postProgress(-1);
	QApplication::postEvent(parent, msg);
	// 	printf("%d=%x",sizeof(filerec), sizeof(filerec));

	free(tmp);
	return(1);
}


// ====================== FSAC Methods =========================

int P2kProc::FSAC_Delete(char * fname)
{
	FUNC("FSAC_Delete")
	//                        0     1     2     3     4     5     6     7    8       9     A     B    C     D     E     F
	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05};
	unsigned char * buf;
	unsigned char * tmp;
	int bufsize;
	int sz;
	mWord t;

	bufsize=sizeof(cmd)+strlen(fname);
	buf=(unsigned char *) malloc (bufsize);
	memcpy(buf,cmd,sizeof(cmd));
	memcpy(buf+sizeof(cmd), fname, strlen(fname));
	t.wordId=bufsize-8;
	buf[0x04]=t.id[1];
	buf[0x05]=t.id[0];

	if (outData(buf, bufsize)<=0) RAISE ("E001");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E002");
	sz=get_cmd_size(packetCount);

	tmp=(unsigned char*) malloc (sz);

	if (inpData(tmp, 0x01, sz)<0) RAISE ("E003");
	if (check_packet_header(tmp,sz,6,0,0)<0) RAISE("E004");

	free(tmp);
	free(buf);
	return(1);

}

int P2kProc::FSAC_Open(char * fname, unsigned char attr)
{
	FUNC("FSAC_Open");


	//   00 7B 00 4A 00 13 00 00 00 00 00 00 00 00 00 04   .{.J............
	//   2F 61 2F 6F 69 63 71 2E 63 66 67

	/*x2 x2 x3 x3 x4 x4 00 00		- data
	x1 -    
	x2 - ID -  .  p2kman     0.
	x3 - .  restart - 22
	x4 -    .*/

	// 						 0xFF, 0xFF, 0x00, 0x4A, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08
	// 					    0    1     2     3     4     5     6     7       8     9     A     B       C     D     E     F
	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0xAA, 0xAA, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0xAA};
	int bufsize;
	unsigned char * buf;
	unsigned char * tmp;
	int sz;
	mWord tWord;

	bufsize=sizeof(cmd)+strlen(fname);
	buf=(unsigned char *)malloc(bufsize);
	memcpy(buf,cmd, sizeof(cmd));
	memcpy(buf+sizeof(cmd),fname,strlen(fname));

	tWord.wordId=bufsize-8;
	buf[0x04]=tWord.id[1];
	buf[0x05]=tWord.id[0];
	buf[0x0F]=attr;

	if (outData(buf, bufsize)<=0) RAISE ("E001");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E002");
	sz=get_cmd_size(packetCount);

	tmp=(unsigned char*) malloc (sz);

	if (inpData(tmp, 0x01, sz)<0) RAISE ("E003");
	if (check_packet_header(tmp,sz,6,0,0)<0) RAISE("E004");

	free(tmp);
	free(buf);
	return(1);
}

int P2kProc::FSAC_Close()
{
	FUNC("FSAC_Close");

	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04};
	unsigned char * tmp;
	int sz;

	if (outData(cmd, sizeof(cmd))<0) RAISE("E01");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E02");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E03");
	if (check_packet_header(tmp,sz,6,0,0)<0) RAISE("E04");
	free(tmp);
	return(1);
}

int P2kProc::FSAC_Seek(unsigned long offset, char dir)
{
	FUNC("FSAC_Seek");

	// 	4  - offset  seek, 5 -   - 0 -  , 1 -   , 2 -  .

	//  					    0    1     2     3     4     5     6     7    8     9     A     B       C     D     E     F
	unsigned char cmd[]={0x00, 0x7C, 0x00, 0x4A, 0x00, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xAA, 0xAA, 0xAA, 0xAA, 0x00 };
	unsigned char * tmp;
	int sz;

	mLong t;
	t.longId=offset;
	swapLong(&t);

	memcpy(cmd+0x0C,&t,4);
	cmd[0x10]=dir;

	if (outData(cmd, sizeof(cmd))<0) RAISE("E01");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E02");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E03");
	if (check_packet_header(tmp,sz,6,0,0)<0) RAISE("E04");
	free(tmp);
	return(1);
}

int P2kProc::FSAC_Read(unsigned char * buf, unsigned short size)
{
	FUNC("FSAC_Read");
	//                        0     1     2     3     4     5     6     7    8       9     A     B    C     D     E     F
	unsigned char cmd[]={0xFF, 0xFF, 0x00, 0x4A, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0xAA, 0xAA};
	//00 7D 00 4A 00 08 00 00 00 00 00 01 00 00 00 0C
	mWord t;
	int sz;
	unsigned char * tmp;
	int ret;

	if (size>0x400) RAISE("E00");
	t.wordId=size;


	cmd[0x0e]=t.id[1];
	cmd[0x0f]=t.id[0];
	if (outData(cmd, sizeof(cmd))<0) RAISE("E01");
	// 	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E02");
	sz=get_cmd_size(packetCount);
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)!=size+0xE) RAISE("E03");
	if ((ret=check_packet_header(tmp,sz,6,1,0))!=size) RAISE("E04");

	memcpy(buf, tmp+0x0E, size);

	free(tmp);
	return(1);
}

int P2kProc::FSAC_Write(unsigned char * tbuf, int size)
{
	FUNC("FSAC_Write");
	//	                        0     1     2     3     4     5     6     7    8       9     A     B    C     D     E     F
	unsigned char cmd[] ={0xFF, 0xFF, 0x00, 0x4A, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xAA, 0xAA};
	unsigned char * buf;
	unsigned char * tmp;
	int bufsize;
	mWord t;
	int sz;

	if (size>0x400) RAISE("E00");

	bufsize=size+sizeof(cmd);
	buf=0;
	buf = (unsigned char *) malloc (bufsize);
	memcpy(buf, cmd, sizeof(cmd));
	memcpy(buf+0x10, tbuf, size);

	t.wordId=size+8;
	buf[0x04]=t.id[1];
	buf[0x05]=t.id[0];

	t.wordId=size;
	buf[0x0E]=t.id[1];
	buf[0x0F]=t.id[0];

	if (outData(buf, bufsize)<0) RAISE("E01");
	//	usleep(100000);
	if (inpSize(packetCount, sizeof(packetCount))!=4) RAISE("E02");
	sz=get_cmd_size(packetCount);
	tmp=0;
	tmp=(unsigned char *)malloc(sz);
	if (inpData(tmp , 0x01 , sz)<0) RAISE("E03");
	if (check_packet_header(tmp,sz,6,0,0)<0) RAISE("E04");

	free(buf);
	free(tmp);

	return(1);
}

// =============================================================

int P2kProc::drv_downloadOneFile(char * fn, char attr, unsigned long fsize, unsigned char * tbuf,
                                 unsigned long curSize, unsigned long allSize)
{
	FUNC("downloadOneFile");

#define READ_BUF_SIZE 0x400

	unsigned char * buf;
	unsigned long numreaded;
	unsigned short rec_size;


	// 	FSAC_Close();

	// 	return(1);
	if (FSAC_Open(fn,attr)<0) RAISE("E001");
	if (FSAC_Seek(0x0,0)<0)
	{
		FSAC_Close();
		RAISE("E002");
	}
	numreaded=0; buf=tbuf;
	while (numreaded<fsize)
	{
		if (fsize-numreaded >=READ_BUF_SIZE) rec_size=READ_BUF_SIZE; else
		{
			rec_size=fsize-numreaded;
			// 			printf("Last record.\n");
		}

		// 		printf("!:numread=%d, fsize=%d, rec_size=%d\n",numreaded,fsize,rec_size);

		if (FSAC_Read(buf,rec_size)<0)
		{
			FSAC_Close();
			// 			RAISE("E003");
		}
		buf+=rec_size;
		numreaded+=rec_size;

		postProgress(100*(curSize+numreaded)/allSize);
	}

	if (FSAC_Close()<0) RAISE("E004");
	return(1);
}

int P2kProc::drv_downloadFiles(P2kFile * lst, int cnt, char * dir)
{
	FUNC("drv_downloadFiles");
	unsigned char * buf;
	unsigned long bufsize;
	unsigned long totalSize;
	unsigned long currentSize;
	int i;
	FILE * fp;
	char fn[1024];
	char * ts;

	if (cnt==0) return(1);

	postProgress(0);
	totalSize=0; currentSize=0;
	for (i=0; i<cnt; i++)
		totalSize+=lst[i].size;

	for (i=0; i< cnt; i++)
	{
		bufsize=lst[i].size;
		buf=(unsigned char *) malloc (bufsize);
		postMessage(MSGSTYLE_NONE, QString("Downloading: %1 (%2 bytes)").arg(QString::fromLocal8Bit(lst[i].name)).arg(lst[i].size));
		if (drv_downloadOneFile(lst[i].name, lst[i].attr, lst[i].size, buf, currentSize, totalSize)<0) RAISE("E01");
		currentSize+=lst[i].size;
		if (cnt==1)
			strcpy(fn,dir);  //dir is already file name
		else
		{
			ts=strrchr(lst[i].name, '/')+1;
			strcpy(fn,dir);
			strcat(fn,ts);
		}
		fp=fopen(fn,"w");
		if (fp==NULL) RAISE("E02");
		if (fwrite(buf, 1, bufsize, fp)!=bufsize) RAISE("E03");
		fclose(fp);
		free(buf);
	}
	postProgress(-1);
	return(1);
}

int P2kProc::drv_uploadOneFile(char * phoneFn, unsigned char attr, unsigned long fsize, unsigned char * tbuf,
                               unsigned long curSize, unsigned long allSize)
{
	FUNC("writeFile");
#define WRITE_BUF_SIZE 0x400

	unsigned char *buf;
	// 	unsigned long bufpos;
	unsigned long numread;
	unsigned long totalread=0;

	if (FSAC_Open(phoneFn,attr)<0) RAISE("E02");

	buf=tbuf;
	// 	bufpos=0;
	numread=WRITE_BUF_SIZE;

	while (numread == WRITE_BUF_SIZE)
	{
		if (totalread+WRITE_BUF_SIZE<fsize)
			numread=WRITE_BUF_SIZE;
		else
			numread=fsize-totalread;
		if (!numread) continue;
		if (FSAC_Write(buf, numread)<0) RAISE("E02");
		buf=buf+numread;
		totalread+=numread;
		postProgress(100*(curSize+totalread)/allSize);
	}

	if (FSAC_Close()<0) RAISE("E03");
	return(1);
}

int P2kProc::drv_uploadFiles(P2kFile * lst, int cnt, const char * dirLocal, const char * dirPhone)
{
	FUNC("drv_uploadFiles");
	if (cnt==0) return(1);

	unsigned char * buf;
	unsigned long bufsize=0;
	unsigned long totalSize=0;
	unsigned long currentSize=0;
	int i;
	FILE * fp;
	char s[1024];
	char s1[1024];


	postProgress(0);

	for  (i=0; i<cnt;i++)
	{
		strcpy(s,dirLocal);
		strcat(s,lst[i].name);
		if ((fp=fopen(s, "r"))==NULL) RAISE("E01");
		fseek(fp, 0L, SEEK_END);
		lst[i].size=ftell(fp);
		totalSize+=lst[i].size;
		fclose(fp);
	}

	for (int i=0; i<cnt; i++)
	{
		bufsize=lst[i].size;
		buf=(unsigned char *) malloc (bufsize);
		strcpy(s,dirLocal);
		strcat(s,lst[i].name);

		strcpy(s1,dirPhone);
		strcat(s1,lst[i].name);

		fp=fopen(s,"r");
		if (fp==NULL) RAISE("E02");
		if (fread(buf, 1, bufsize, fp)!=bufsize) RAISE("E03");
		fclose(fp);

		postMessage(MSGSTYLE_NONE, QString("Uploading: %1 (%2 bytes)").arg(QString::fromLocal8Bit(lst[i].name)).arg(lst[i].size));
		if (drv_uploadOneFile( s1, lst[i].attr, lst[i].size, buf, currentSize, totalSize)<0) RAISE("E01");
		currentSize+=lst[i].size;
		free(buf);
	}

	postProgress(-1);
	doGetFreeSpace();
	doGetFileCount();
	return(1);
}

int P2kProc::drv_deleteFiles(P2kFile * lst, int cnt)
{
	FUNC("drv_deleteFiles");
	if (cnt==0) return 1;
	postProgress(0);
	for (int i=0; i<cnt; i++)
	{
		postMessage(MSGSTYLE_NONE, QString("Deleting file: %1").arg(QString::fromLocal8Bit(lst[i].name)));
		if (FSAC_Delete(lst[i].name)<0) RAISE("E01");
		postProgress(100*(i+1)/cnt);
	}
	postProgress(-1);
	doGetFreeSpace();
	doGetFileCount();
	return 1;
}

int P2kProc::drv_changeAttr(P2kFile * lst, int cnt)
{
	FUNC("drv_changeAttr");
	if (cnt==0) return 1;
	postProgress(0);
	for (int i=0; i<cnt; i++)
	{
		postMessage(MSGSTYLE_NONE, QString("Changing attributes for: %1").arg(QString::fromLocal8Bit(lst[i].name)));
		if (FSAC_Open(lst[i].name, lst[i].attr)<0) RAISE("E001");
		if (FSAC_Close()<0) RAISE("E002");
		postProgress(100*(i+1)/cnt);
	}
	postProgress(-1);

	return 1;
}

int P2kProc::drv_read_seem(int x, int y, unsigned char * seemBuf, int seemBufSize)
{
	unsigned char cmd1[0x10]={0xFF, 0xFF, 0x00, 0x20, 0x00, 0x08, 0x00, 0x00, 0xAA, 0xAA, 0xBB, 0xBB, 0x00, 0x00, 0x00, 0x00};\
	// 	unsigned char * mem;
	mWord xxxx;
	mWord yyyy;
	mWord cnt;
	mWord size;
	mWord packId;
	int bufsize;
	unsigned char * buf;
	int adr;
	// 	int dadr;
	unsigned short myid;
	mWord packSize;
	unsigned char packType;
	int seemPos;
	int j;
	unsigned char iserr;

	xxxx.wordId=x;	cmd1[8]=xxxx.id[1]; cmd1[9]=xxxx.id[0];
	yyyy.wordId=y;	cmd1[10]=yyyy.id[1]; cmd1[11]=yyyy.id[0];
	outData(cmd1,sizeof(cmd1));
	// 	usleep(100000);
	// 	sleep(1);
	if (inpSize(packetCount, sizeof(packetCount))<=2)
	{
		printf("(E003)");
		return(-1);
	}
	// 	sleep(1);
	cnt.id[0]=packetCount[1]; cnt.id[1]=packetCount[0];
	size.id[0]=packetCount[3]; size.id[1]=packetCount[2];
	bufsize=2*cnt.wordId+size.wordId+4;
	buf = (unsigned char *) malloc (bufsize);
	if (inpData(buf, cnt.wordId, bufsize)!=bufsize)
	{
		printf("(E004)");
		return(-1);
	}

	adr=6;
	seemPos=0;

	for (int i=0; i<cnt.wordId; i++)
	{
		packId.id[0]=buf[adr+1];
		packId.id[1]=buf[adr];
		myid= packId.wordId & 0xFFF;
		packType=buf[adr+2];
		packSize.id[0]=buf[adr+5];
		packSize.id[1]=buf[adr+4];
		iserr=buf[adr+8];

		if ((myid != freeID.wordId) || (packType!=0x80) || (iserr))
		{
			if (packType==0x60) adr+=7;	else adr+=8+packSize.wordId;
			// 			printf("Packet Error. myid=0x%04X, needID=0x%04X, packID=0x%04X packType=0x%02X, yy=0x%02X\n",myid, freeID.wordId, packId, packType, iserr);
			printf("(E01)");
			return(-1);
		}
		for (j=0; j<packSize.wordId-1; j++)
		{
			if (seemPos>=seemBufSize)
			{
				printf("(E_RS_01_%04d_%04d\n",seemBufSize,packSize.wordId-1);
				return(-1);
			}
			seemBuf[seemPos++]=buf[adr+9+j];
		}
		break;
	}
	return(seemPos);
}

int P2kProc::drv_write_seem(int x, int y, unsigned char * seemBuf, int seemSize)
{
#define MAX_SEEM_SIZE 102400

	unsigned char * temp_seem;
	unsigned char * temp_packet;

	temp_packet=(unsigned char *) malloc (MAX_SEEM_SIZE);
	temp_seem=temp_packet+16;

	int actualSize;
	int ret;

	// Check seem size by re-reading it from phone

	// 	if (!isFirstRead)
	// 	{
	actualSize=drv_read_seem(x,y, temp_seem, MAX_SEEM_SIZE-16);
	actualSize=seemSize;

	if (actualSize != seemSize)
	{
		printf("(E_WS_01_%04d)",actualSize);
		free(temp_packet);
		return(-1);
	}

	mWord wrd;

	memcpy(temp_seem,seemBuf,seemSize);

	wrd.wordId=seemSize+8;

	//	 0	1  2  3  4  5  6  7  8  9 10 11 12 13 14 15
	//  00 05 00 2F 00 88 00 00 02 06 00 01 00 00 00 80
	//	FF FF 00 2F AA AA 00 00 XX XX YY YY 00 00 ZZ ZZ


	/*	FF FF - ID . AUTO
		00 2F -  seem
		AA AA -  
		XX XX - Seem XXXX
		YY YY - Seem YYYY
		ZZ ZZ - Seem size
	*/

	temp_packet[2]=0x00; temp_packet[3]=0x2F;
	wrd.wordId=seemSize+8; temp_packet[4]=wrd.id[1]; temp_packet[5]=wrd.id[0];
	wrd.wordId=x; temp_packet[8]=wrd.id[1]; temp_packet[9]=wrd.id[0];
	wrd.wordId=y; temp_packet[10]=wrd.id[1]; temp_packet[11]=wrd.id[0];
	wrd.wordId=seemSize; temp_packet[14]=wrd.id[1]; temp_packet[15]=wrd.id[0];

	ret=outData(temp_packet,seemSize+16);
	if (ret!=seemSize+16)
	{
		printf("(E_WS_02)");
		return(-1);
	}

	// 	usleep(100000);

	if (inpSize(packetCount, sizeof(packetCount))<=2)
	{
		printf("(E_WS_03)");
		return(-1);
	}

	mWord pcnt;
	mWord psize;
	int bufsize;


	pcnt.id[0]=packetCount[1]; pcnt.id[1]=packetCount[0];
	psize.id[0]=packetCount[3]; psize.id[1]=packetCount[2];
	bufsize=2*pcnt.wordId+psize.wordId+4;

	if (inpData(temp_packet, pcnt.wordId, bufsize)!=bufsize)
	{
		printf("(E_WS_04)");
		return(-1);
	}

	// 	unsigned char * buf, int bufsize, int adr, char needdata=1, char checky=1
	ret=check_packet_header(temp_packet, pcnt.wordId, 6);
	if (ret<0)
	{
		printf("(E_WS_05)");
		return(-1);
	}
	free(temp_packet);
	return(1);
}

int P2kProc::drv_backup(int x1, int y1, int x2, int y2, unsigned char * dir)
{

	qDebug("drv_backup");

	int x;
	int y;
	unsigned char buf[102400];
	int ret;
	FILE * fp;
	char s[256];
	// 	int idx=0;
	int all=(x2-x1);
	if (all==0) all=1;
	x=x1;

	postProgress(0);


	while (x<=x2)
	{
		if(x==x1) y=y1; else y=0x01;

		while (1)
		{
			ret=drv_read_seem(x,y,buf,sizeof(buf));

			if (ret<0)
			{
				if (lastControlStatus>=0) break;
				if (lastControlStatus==-110)
				{
					postMessage(MSGSTYLE_NONE, "Phone reboot detected. Try to restore (Timeout:30 sec)");
					setState(CONNECT_NONE);
					sleep(2);
					int tm=time(NULL);
					int ok=0;
					while (time(NULL)-tm<30)
					{
						doUpdateMode();
						if (drv_connect()>0)
						{
							ok=1;
							break;
						}
						else
							sleep(1);
					}
					doUpdateMode();
					if (!ok)
					{
						postMessage(MSGSTYLE_ERROR, "Unable to restore connection");
						return(-1);
					}
					else
					{
						postMessage(MSGSTYLE_NONE,"Connection restored");
						setState(CONNECT_P2K);
					}

					x=x-11; if (x<x1) x=x1-1;
					break;
				}

				postMessage(MSGSTYLE_NONE, QString("Unknown Error. USB IO code=%1").arg(lastControlStatus));
				return(-1);

			}

			sprintf(s,"%s%04x_%04x.seem",dir,x,y);
			fp=fopen(s,"w+");
			if (fp==NULL)
			{
				postMessage(MSGSTYLE_NONE, QString("Unable to open file [%1]").arg(s));
				return(-1);
			}

			fwrite(buf,1, ret, fp);
			fclose(fp); fp=NULL;
			y++;

			if ((x==x2) && (y==y2)) break;
			if (y>0xff) break;
		}
		x++;
		postProgress(100*(x-x1)/all);
	}
	postProgress(-1);
	return(1);
}
//
