/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		modified 2005 ... 2008 by Harald Foerster		   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include <string.h>
#include <unistd.h>

#ifdef HAVE_FNMATCH
#include <fnmatch.h>
#else
#include "fnmatch.h"
#endif

#include "gui.h"
#include "apassign.h"
#include "charconv.h"
#include "gprop.h"

/*
#include <gtk/gtk.h>
#include "gui-gtk.h"
#include "nls.h"
#include "nscache.h"
*/


enum
{
	kColumnMime,
	kColumnFile,
	kColumnURL,
#ifndef CONFIG_GTKCLIST_GTKCTREE
	kColumnRowData,
#endif
	kNumberOfColumns
};


#ifdef CONFIG_GTKCLIST_GTKCTREE
#define ApassignListNew(list) \
	GuiListInit((list), kNumberOfColumns)
#else
#define ApassignListNew(list) \
{ \
	(list)->slist = \
		gtk_list_store_new (kNumberOfColumns, G_TYPE_STRING, \
			G_TYPE_STRING, G_TYPE_STRING, G_TYPE_POINTER); \
	GuiListInit((list), kColumnRowData); \
}
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define ApassignListUnselectRow(list) {;}
#else
#define ApassignListUnselectRow(list) \
	gtk_tree_selection_unselect_all((list)->selection)
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define ApassignListAppend(list, str, data) \
{ \
	gint row; \
	row = gtk_clist_append((list)->clist, (str)); \
	gtk_clist_set_row_data((list)->clist, row, (data)); \
}
#else
#define ApassignListAppend(list, str, data) \
{ \
	GtkTreeIter iter; \
	gtk_list_store_append((list)->slist, &iter); \
	gtk_list_store_set((list)->slist, &iter, \
		kColumnMime, (str)[0], kColumnFile, (str)[1], \
		kColumnURL, (str)[2],  kColumnRowData, (data), -1); \
}
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define ApassignListSetAll(list, str, data) \
{ \
	gtk_clist_set_text((list)->clist, (list)->row, kColumnMime, (str)[0]); \
	gtk_clist_set_text((list)->clist, (list)->row, kColumnFile, (str)[1]); \
	gtk_clist_set_text((list)->clist, (list)->row, kColumnURL, (str)[2]); \
	gtk_clist_set_row_data((list)->clist, (list)->row, (data)); \
}
#else
#define ApassignListSetAll(list, str, data) \
{ \
	gtk_list_store_set((list)->slist, &((list)->iter), \
		kColumnMime, (str)[0], kColumnFile, (str)[1], \
		kColumnURL, (str)[2],  kColumnRowData, (data), -1); \
}
#endif


typedef struct
{
	GuiListStore	list;
	gboolean	okay;
	apassign_t	*rowdata;
	GtkCombo	*combo;
	GtkEntry	*mentry;
	GtkEntry	*aentry;
	GtkEntry	*uentry;
	GtkWidget	*window;

} ApassignDialog;

static ApassignDialog *apassign_dialog = NULL;
static GSList *apassign_data = NULL;

static apassign_t default_viewers[] = {
	{"text/*", "gless %f", "gnome-moz-remote %u"},
	{"text/*", "xterm -e less %f", "xterm -e lynx %u"},
	{"text/*", "xterm -e bash -c 'gzip -cd \\'%f\\' | less'", "xterm -e lynx %u"},
	{"text/html", "mozilla %f", "mozilla %u"},
	{"text/html", "mozilla -remote openFile(%f)", "mozilla -remote openURL(%u)"},
	{"text/html", "netscape %f", "netscape %u"},
	{"text/html", "netscape -remote openFile(%f)", "netscape -remote openURL(%u)"},
	{"text/html", "firefox %f", "firefox %u"},
	{"text/html", "firefox -remote openFile(%f)", "firefox -remote openURL(%u)"},
	{"text/html", "seamonkey %f", "seamonkey %u"},
	{"text/html", "seamonkey -remote openFile(%f)", "seamonkey -remote openURL(%u)"},
	{"image/*", "gqview %f", "xterm -e lynx %u"},
	{"image/*", "display -immutable %f", "xterm -e lynx %u"},
	{"image/*", "ee %f", "netscape %u"},
	{"*", "gnome-moz-remote %f", "gnome-moz-remote %u"},
	{"*", "xterm -e lynx %f", "xterm -e lynx %u"},
	{NULL, NULL, NULL}
};


GSList *apassign_get_matching_viewers(const gchar *mimetype)
{
	GSList *rv  = NULL;
	GSList *ptr = apassign_data;

	while (ptr)
	{
		apassign_t *ve = ptr->data;

		if ( ve && ve->mimetype &&
			(mimetype == NULL ?
				(ve->mimetype[0] == '*' &&
				ve->mimetype[1] == '\0') :
				fnmatch(ve->mimetype, mimetype, 0) == 0) ) 
		{
			rv = g_slist_append(rv, ve);
		}

		ptr = ptr->next;
	}

	return rv;
}

static void apassign_data_append(apassign_t *ad, GSList **list)
{
	*list = g_slist_append(*list, ad);
}

static void apassign_data_free(apassign_t *ad)
{
	if (ad)
	{
		g_free(ad->mimetype);
		g_free(ad->args_file);
		g_free(ad->args_url);

		g_free(ad);
	}
}

static gboolean apassign_get_entry(GtkEntry *entry, gchar *str[6])
{
	gchar *p;

	p = (gchar *) gtk_entry_get_text(entry);

	if (p && *p == '\0')
	{
		p = NULL;
	}

	str[0] = p;
	str[3] = CHARCONV_UTF8_TO_LOCALE_DUP(p);

	if (str[0] && str[3] == NULL)
	{
		return FALSE;
	}

	return TRUE;
}

static gint apassign_get_strings(ApassignDialog *apdlg, gchar *str[6])
{
	/* str[0] => UF8 --- str[3] => C-Locale */
	if (apassign_get_entry(apdlg->mentry, (gchar **) &str[0]) && str[3])
	{
		/* str[1] => UF8 --- str[4] => C-Locale */
		if (apassign_get_entry(apdlg->aentry, (gchar **) &str[1]))
		{
			/* str[2] => UF8 --- str[5] => C-Locale */
			if (apassign_get_entry(apdlg->uentry, (gchar **) &str[2]))
			{
				if (str[4] != str[5])
				{
					return TRUE;
				}

				g_free(str[5]);
			}

			g_free(str[4]);
		}

		g_free(str[3]);
	}

	/* (str[3] == NULL) || (str[4] == NULL && str[5] == NULL) */

	return FALSE;
}

static gint apassign_string_compare(const gchar *s1, const gchar *s2)
{
	if (s1 == s2)
	{
		return 0;
	}

	if (s1 == NULL || s2 == NULL)
	{
		return 1;
	}

	return strcmp(s1, s2);
}

static gint apassign_find_data(apassign_t *ad, const gchar *p[3])
{
	gint rv;

	rv = apassign_string_compare(ad->mimetype, p[0]);

	if (rv) return rv;

	rv = apassign_string_compare(ad->args_file, p[1]);

	if (rv) return rv;

	return apassign_string_compare(ad->args_url, p[2]);
}

static void apassign_dialog_destroy(GtkWidget *w, ApassignDialog *apdlg)
{
	if (apdlg->okay == FALSE)
	{
		guitl_list_foreach_row_data(&(apdlg->list),
				(GFunc) apassign_data_free, NULL);
	}

	combo_mime_slist = g_slist_remove(combo_mime_slist, apdlg->combo);
	g_free(apdlg);
	apassign_dialog = NULL;
}

static void apassign_dialog_okay(GtkWidget *w, ApassignDialog *apdlg)
{
	g_slist_foreach(apassign_data, (GFunc) apassign_data_free, NULL);

	g_slist_free(apassign_data);
	apassign_data = NULL;

	guitl_list_foreach_row_data(&(apdlg->list),
			(GFunc) apassign_data_append, &apassign_data);

	apdlg->okay = TRUE;

	gtk_widget_destroy(apdlg->window);
}

static void apassign_dialog_append(GtkWidget *w, ApassignDialog *apdlg)
{
	gchar *p[6];

	if (apassign_get_strings(apdlg, (gchar **) p))
	{
		if (guitl_list_compare_row_data(&(apdlg->list), (GCompareFunc)
				apassign_find_data, (const gchar **) &p[3]) == NULL)

		{
			apassign_t *ad;

			ad = g_malloc(sizeof(apassign_t));

			ad->mimetype  = p[3];
			ad->args_file = p[4];
			ad->args_url  = p[5];

			ApassignListAppend(&(apdlg->list), (gchar **) p, ad);

			if (apdlg->rowdata == NULL)
			{
				gtk_entry_set_text(apdlg->mentry, "");
				gtk_entry_set_text(apdlg->aentry, "");
				gtk_entry_set_text(apdlg->uentry, "");
			}

			else
			{
				GuiListUnselectRow(&(apdlg->list));
			}

			return;
		}

		g_free(p[3]);
		g_free(p[4]);
		g_free(p[5]);
	}

	gdk_beep();
}

static void apassign_dialog_modify(GtkWidget *w, ApassignDialog *apdlg)
{
	gchar *p[6];

	if (apdlg->rowdata && apassign_get_strings(apdlg, (gchar **) p))
	{
		if (guitl_list_compare_row_data(&(apdlg->list), (GCompareFunc)
				apassign_find_data, (const gchar **) &p[3]) == NULL)
		{
			apassign_t *ad;

			apassign_data_free(apdlg->rowdata);

			ad = g_malloc(sizeof(apassign_t));

			ad->mimetype  = p[3];
			ad->args_file = p[4];
			ad->args_url  = p[5];

			ApassignListSetAll(&(apdlg->list), p, ad);
			GuiListUnselectRow(&(apdlg->list));

			return;
		}

		g_free(p[3]);
		g_free(p[4]);
		g_free(p[5]);
	}

	gdk_beep();
}

static void apassign_dialog_delete(GtkWidget *w, ApassignDialog *apdlg)
{
	if (apdlg->rowdata == NULL)
	{
		gdk_beep();
		return;
	}

	apassign_data_free(apdlg->rowdata);
	GuiListRemove(&(apdlg->list));
}


#ifdef CONFIG_GTKCLIST_GTKCTREE

static void apassign_dialog_row_selected(GtkCList* clist, gint row, gint col,
			GdkEventButton *e, ApassignDialog *apdlg)
{
	gchar *p;

	apdlg->list.row = row;
	apdlg->rowdata = gtk_clist_get_row_data(apdlg->list.clist, row);

	p = "";
	gtk_clist_get_text(clist, row, kColumnMime, &p);
	gtk_entry_set_text(apdlg->mentry, p);

	p = "";
	gtk_clist_get_text(clist, row, kColumnFile, &p);
	gtk_entry_set_text(apdlg->aentry, p);

	p = "";
	gtk_clist_get_text(clist, row, kColumnURL, &p);
	gtk_entry_set_text(apdlg->uentry, p);
}

static void apassign_dialog_row_unselected(GtkCList* clist, gint row, gint col,
			GdkEventButton *e, ApassignDialog *apdlg)
{
	apdlg->list.row = -1;
	apdlg->rowdata  = NULL;

	gtk_entry_set_text(apdlg->mentry, "");
	gtk_entry_set_text(apdlg->aentry, "");
	gtk_entry_set_text(apdlg->uentry, "");
}

#else /* ! CONFIG_GTKCLIST_GTKCTREE */

static void apassign_dialog_row_selected(GtkTreeSelection *selection, ApassignDialog *apdlg)
{
	GtkTreeIter 	iter;
	GtkTreeModel	*model;
	gchar		*p[3];

	p[0] = p[1] = p[2] = NULL;
	apdlg->rowdata = NULL;

	if (gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		apdlg->list.iter = iter;

		gtk_tree_model_get(model, &iter, kColumnMime, &p[0],
				kColumnFile, &p[1], kColumnURL, &p[2],
					kColumnRowData, &(apdlg->rowdata), -1);
	}

	gtk_entry_set_text(apdlg->mentry, (p[0] == NULL ? "" : p[0]));
	gtk_entry_set_text(apdlg->aentry, (p[1] == NULL ? "" : p[1]));
	gtk_entry_set_text(apdlg->uentry, (p[2] == NULL ? "" : p[2]));

	g_free(p[0]);
	g_free(p[1]);
	g_free(p[2]);
}

#endif /* #ifdef CONFIG_GTKCLIST_GTKCTREE .. #else */

static void apassign_clist_fill(apassign_t *orig, GuiListStore *list)
{
	gchar    	*p[3];
	apassign_t	*ad;

	p[0] = CHARCONV_UTF8_FROM_LOCALE(orig->mimetype);

	if (p[0] == NULL)
	{
		return;
	}

	ad = g_malloc(sizeof(apassign_t));

	ad->mimetype = g_strdup(orig->mimetype);

	ad->args_file = g_strdup(orig->args_file);
	p[1] = CHARCONV_UTF8_FROM_LOCALE(ad->args_file);

	ad->args_url = g_strdup(orig->args_url);
	p[2] = CHARCONV_UTF8_FROM_LOCALE(ad->args_url);

	if (p[1] == p[2])
	{
		/* both NULL */
		g_free(ad);
	}

	else
	{
		ApassignListAppend(list, (gchar **) p, ad);
		CHARCONV_UTF8_FREE(p[1], ad->args_file);
		CHARCONV_UTF8_FREE(p[2], ad->args_url);
	}

	CHARCONV_UTF8_FREE(p[0], orig->mimetype);
}

static void apassign_build_column(GuiListStore *list, gint colnr, gint height, gchar *title)
{
	gchar *str;

	GuiListColumnTextNew(list, colnr, height, GTK_JUSTIFY_LEFT);
	str = CHARCONV_UTF8_GETTEXT(title);
	GuiListSetColumnTitle(list, colnr, str);
	CHARCONV_UTF8_DESTROY(str);
	GuiListSetColumnAutoResize(list, colnr, TRUE);
}

void apassign_dialog_open(void)
{
	static const gchar wtitle[] =
		gettext_nop("MozCache: assign applications to MIME types");

	gint		height;
	GtkBox		*vbox;
	GtkCombo	*combo;
	GtkTable	*table;
	GtkWidget	*button, *frame, *bbox, *swin, *widget, *window;

	ApassignDialog *apdlg = apassign_dialog;

	if (apdlg)
	{
		GuiWindowPresent(apdlg->window);
		return;
	}

	apdlg = g_malloc(sizeof(ApassignDialog));
	apassign_dialog = apdlg;

	apdlg->okay = FALSE;
	apdlg->rowdata = NULL;

	widget = gtk_vbox_new(FALSE, 2);
	window = guitl_window_new_gettext(wtitle, widget, 4);
	apdlg->window = window;

	GuiSignalConnect(window, "destroy", apassign_dialog_destroy, apdlg);

	swin = gtk_scrolled_window_new(NULL, NULL);
	gtk_container_add(GTK_CONTAINER(widget), swin);
	GuiWidgetSetSizeRequest(swin, -1, 180);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (swin),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	ApassignListNew(&(apdlg->list));
	GuiListContainerAdd(&(apdlg->list), swin);
	GuiListSetSelectionMode(&(apdlg->list), GTK_SELECTION_SINGLE);

	height = guitl_list_calculate_row_height(&(apdlg->list));
	GuiListSetRowHeight(&(apdlg->list), height);

	apassign_build_column(&(apdlg->list), kColumnMime, height, gettext_nop("MIME type"));
	apassign_build_column(&(apdlg->list), kColumnFile, height, gettext_nop("File viewer"));
	apassign_build_column(&(apdlg->list), kColumnURL,  height, gettext_nop("URL viewer"));

	g_slist_foreach(apassign_data,
			(GFunc) apassign_clist_fill, &(apdlg->list));

	GuiListColumnTitlesShow(&(apdlg->list));
	GuiListColumnTitlesPassive(&(apdlg->list));
	GuiListSetReorderable(&(apdlg->list));

	GuiListSignalConnectSelectedRow(&(apdlg->list),
				apassign_dialog_row_selected, apdlg);
	GuiListSignalConnectUnselectedRow(&(apdlg->list),
				apassign_dialog_row_unselected, apdlg);

	vbox   = GTK_BOX(widget);
	widget = gtk_table_new(2, 3, FALSE);
	frame  = guitl_frame_new_add_child_gettext(NULL, widget);
	gtk_box_pack_start(vbox, frame, FALSE, FALSE, 0);

	table = GTK_TABLE(widget);

	combo = guitl_tab_add_combo(table,
		gettext_nop("MIME type: "), mime_types_available, 0, 0);
	apdlg->combo = combo;
	apdlg->mentry = GTK_ENTRY(combo->entry);
	combo_mime_slist = g_slist_prepend(combo_mime_slist, combo);

	apdlg->aentry = guitl_tab_add_entry(table,
		gettext_nop("File viewer application: "), 0, 1);
	GuiSignalConnect(apdlg->aentry, "activate", apassign_dialog_append, apdlg);

	apdlg->uentry = guitl_tab_add_entry(table,
		gettext_nop("URL viewer application: "), 0, 2);
	GuiSignalConnect(apdlg->uentry, "activate", apassign_dialog_append, apdlg);

	frame = gtk_frame_new(NULL);
	gtk_box_pack_start(vbox, frame, FALSE, FALSE, 0);

	bbox = guitl_hbutton_box_new(frame, 2, GTK_BUTTONBOX_SPREAD);

	button = guitl_button_box_add_icon_button(bbox, kIconButtonAppend);
	GuiSignalConnect(button, "clicked", apassign_dialog_append, apdlg);

	button = guitl_button_box_add_icon_button(bbox, kIconButtonModify);
	GuiSignalConnect(button, "clicked", apassign_dialog_modify, apdlg);

	button = guitl_button_box_add_icon_button(bbox, kIconButtonDelete);
	GuiSignalConnect(button, "clicked", apassign_dialog_delete, apdlg);

	bbox = guitl_hbutton_box_new(GTK_WIDGET(vbox), 8, GTK_BUTTONBOX_END);

	button = guitl_button_box_add_icon_button(bbox, kIconButtonCancel);
	GuiSignalConnectSwapped(button, "clicked", gtk_widget_destroy, apdlg->window);

	button = guitl_button_box_add_icon_button(bbox, kIconButtonOkay);
	GuiSignalConnect(button, "clicked", apassign_dialog_okay, apdlg);
	gtk_widget_grab_default(button);

	gtk_widget_show_all(apdlg->window);
	ApassignListUnselectRow(&(apdlg->list));
}

static gchar *apassign_check_program(const gchar *exec, gchar **path)
{
	gchar *rv = NULL;

	if (exec)
	{
		const static gchar xterm_str[] = "xterm -e ";

		gchar *prog = (gchar *) exec;

		if (strncmp(prog, xterm_str, sizeof(xterm_str) - 1) == 0)
		{
			prog += sizeof(xterm_str) - 1;
		}

		prog = g_strndup(prog, strcspn(prog, " "));

		if (prog && prog[0] != '\0')
		{
			gchar *str;

			while ( rv == NULL && (str = *(path++)) != NULL )
			{
				gchar *ap;

				ap = g_strconcat(str, "/", prog, NULL);

				if (access(ap, X_OK) == 0)
				{
					rv = g_strdup(exec);
				}

				g_free(ap);
			}
		}

		g_free(prog);
	}

	return rv;
}

void apassign_set_default_viewers(void)
{
	if (apassign_data == NULL)
	{
		gchar **vec;
		const apassign_t *def_viewer;

		const gchar *p = g_getenv("PATH");

		if (p == NULL)
		{
			return;
		}

		vec = g_strsplit(p, ":", 100);

		if (vec == NULL)
		{
			return;
		}

		for (def_viewer = default_viewers; def_viewer->mimetype != NULL; def_viewer++)
		{
			gchar *args_file, *args_url;

			args_file = apassign_check_program(def_viewer->args_file, vec);
			args_url  = apassign_check_program(def_viewer->args_url,  vec);

			if (args_file || args_url)
			{
				apassign_t *na = g_malloc(sizeof(apassign_t));

				na->mimetype  = g_strdup(def_viewer->mimetype);
				na->args_file = args_file;
				na->args_url  = args_url;

				apassign_data = g_slist_append(apassign_data, na);
			}
		}

		g_strfreev(vec);
	}
}

static void apassign_dump(apassign_t *ad, gpointer data)
{
	static const gchar escape_chars[] = "\\\"";

	struct
	{
		FILE		*fd;
		const gchar	*id;

	} *s = data;

	gint	len;
	gchar	*mt, *fv, *uv;
	gchar	sbuf[3 * 1024];

	mt  = sbuf;
	len = tl_str_escape_chars(mt, sizeof(sbuf),
					ad->mimetype,  escape_chars);
	fv  = &mt[len + 1];
	len = tl_str_escape_chars(fv, &sbuf[sizeof(sbuf)] - fv,
					ad->args_file, escape_chars);
	uv  = &fv[len + 1];
	tl_str_escape_chars(uv, &sbuf[sizeof(sbuf)] - uv,
					ad->args_url, escape_chars);

	fprintf(s->fd, "%s \"%s\" \"%s\" \"%s\"\n", s->id, mt, fv, uv);
}

void apassign_save_data(FILE *fd, const gchar *id)
{
	struct
	{
		FILE		*fd;
		const gchar	*id;
	} s;

	s.fd = fd;
	s.id = id;

	g_slist_foreach(apassign_data, (GFunc) apassign_dump, &s);
}

static gchar *apassign_next_str(gchar *str)
{
	str = tl_get_1qstr(str);

	if (str == NULL || *str == '\0')
	{
		return NULL;
	}

	return g_strdup(str);
}

void apassign_parse_str(gchar *str)
{
	str = apassign_next_str(str);

	if (str)
	{
		apassign_t *ad;

		ad = g_malloc(sizeof(apassign_t));
		apassign_data = g_slist_append(apassign_data, ad);

		ad->mimetype  = str;
		g_strdown(str);
		ad->args_file = apassign_next_str(NULL);
		ad->args_url  = apassign_next_str(NULL);
	}
}

/* EOF */
