/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		created and modified 2006,2007 by Harald Foerster	   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

/*
	> The following source includes modified parts of the Bekeley Database.
	> Everything which were not neccessary for reading and deleting entries
	> from the Netscape cache index file were removed.
	> Harald
*/

/*
 * Copyright (c) 1990, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Margo Seltzer.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifdef CONFIG_BERKELEY_DB

#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

#include "berkeley.h"
#include "bswap32.h"
#include "file.h"
#include "nls.h"
#include "nscache.h"

/*
#include <sys/types.h>
#include <glib.h>
*/


#define	OLDHASHVERSION	1
#define	HASHVERSION	2
#define	HASHMAGIC	0x061561
#define CHARKEY		"%$sniglet^&"

/* number of bit maps and spare points */
#define NCACHED		32

/* Hash Table Information */

typedef struct hashhdr
{					/* Disk resident portion */
	gint32		magic;		/* Magic NO for hash tables */
	gint32		version;	/* Version ID */
	guint32		lorder;		/* Byte Order */
	gint32		bsize;		/* Bucket/Page Size */
	gint32		bshift;		/* Bucket shift */
	gint32		dsize;		/* Directory Size */
	gint32		ssize;		/* Segment Size */
	gint32		sshift;		/* Segment shift */
	gint32		ovfl_point;	/* Where overflow pages are being 
					 * allocated */
	gint32		last_freed;	/* Last overflow page freed */
	gint32		max_bucket;	/* ID of Maximum bucket in use */
	gint32		high_mask;	/* Mask to modulo into entire table */
	gint32		low_mask;	/* Mask to modulo into lower half of 
					 * table */
	gint32		ffactor;	/* Fill factor */
	gint32		nkeys;		/* Number of keys in hash table */
	gint32		hdrpages;	/* Size of table header */
	gint32		h_charkey;	/* value of hash(CHARKEY) */
	gint32		spares[NCACHED];/* spare pages for overflow */
	guint16		bitmaps[NCACHED]; /* address of overflow page 
					   * bitmaps */
} HASHHDR;

typedef struct htab
{					/* Memory resident data structure */
	HASHHDR 	hdr;		/* Header */
	int		fd;		/* File descriptor */
	char            *pagebuf;	/* Allocated page */
	char		*cpage;		/* Current page */
	int		cbucket;	/* Current bucket */
	int		cndx;		/* Index of next item on cpage */

} HTAB;

/*
	---------------------
	 Overflow management
	---------------------

	Overflow page numbers are allocated per split point.
	At each doubling of the table, we can allocate extra
	pages. So, an overflow page number has the top 5 bits
	indicate which split point and the lower 11 bits indicate
	which page at that split point is indicated (pages within
	split points are numberered starting with 1).
*/

#define SPLITSHIFT	11
#define SPLITMASK	0x7FF
#define SPLITNUM(N)	(((guint32)(N)) >> SPLITSHIFT)
#define OPAGENUM(N)	((N) & SPLITMASK)


#define OVFLPAGE	0
#define PARTIAL_KEY	1
#define FULL_KEY	2
#define FULL_KEY_DATA	3
#define	REAL_KEY	4

#define FREESPACE(ptr)	((ptr)[(ptr)[0]+1])
#define	OFFSET(ptr)	((ptr)[(ptr)[0]+2])

#ifdef UINT32_IS_PDP_ENDIAN
#define BYTEORDER_DIFFERS(byteorder)		\
		((byteorder) != G_BYTE_ORDER &&	\
			(byteorder) != G_LITTLE_ENDIAN)
#else
#define BYTEORDER_DIFFERS(byteorder)		\
		((byteorder) != G_BYTE_ORDER)
#endif



#ifndef UINT32_IS_BIG_ENDIAN

static void _bdb_swap_header(HTAB *htab)
{
	HASHHDR	*hdrp;
	int	i;

	hdrp = &(htab->hdr);

	hdrp->magic	= UINT32_FROM_BIG_ENDIAN(hdrp->magic);
	hdrp->version	= UINT32_FROM_BIG_ENDIAN(hdrp->version);
	hdrp->lorder	= UINT32_FROM_BIG_ENDIAN(hdrp->lorder);
	hdrp->bsize	= UINT32_FROM_BIG_ENDIAN(hdrp->bsize);
	hdrp->bshift	= UINT32_FROM_BIG_ENDIAN(hdrp->bshift);
	hdrp->max_bucket = UINT32_FROM_BIG_ENDIAN(hdrp->max_bucket);
	hdrp->hdrpages	= UINT32_FROM_BIG_ENDIAN(hdrp->hdrpages);
	hdrp->h_charkey	= UINT32_FROM_BIG_ENDIAN(hdrp->h_charkey);

	for (i = 0; i < NCACHED; i++)
	{
		hdrp->spares[i]  = UINT32_FROM_BIG_ENDIAN(hdrp->spares[i]);
	}

} /* static void _bdb_swap_header(HTAB *htab) */

#endif /* #ifndef UINT32_IS_BIG_ENDIAN */

static guint32 _bdb_hash4(const void *keyarg, size_t len)
{
	/* Hash function from Chris Torek */

#define HASH4a(h, key)	((h) << 5) - (h) + *(key++)
#define HASH4b(h, key)	((h) << 5) + (h) + *(key++)
#define HASH4(h, key)	HASH4b((h), (key))

	guint32		h;
	size_t		loop;
	const guint8	*key;

	h	= 0;
	key	= (const guint8 *) keyarg;

	if (len > 0)
	{
		loop = (len + 8 - 1) >> 3;

		switch (len & (8 - 1))
		{
			case 0:

			do
			{
					h = HASH4(h, key);
					/* FALLTHROUGH */
				case 7:
					h = HASH4(h, key);
					/* FALLTHROUGH */
				case 6:
					h = HASH4(h, key);
					/* FALLTHROUGH */
				case 5:
					h = HASH4(h, key);
					/* FALLTHROUGH */
				case 4:
					h = HASH4(h, key);
					/* FALLTHROUGH */
				case 3:
					h = HASH4(h, key);
					/* FALLTHROUGH */
				case 2:
					h = HASH4(h, key);
					/* FALLTHROUGH */
				case 1:
					h = HASH4(h, key);
			}
			while (--loop);
		}
	}

	return h;

} /* static guint32 _bdb_hash4(const void *keyarg, size_t len) */

static guint32 _bdb_log2(guint32 num)
{
	guint32 i, limit;

	i = 0;
	limit = 1;

	while (limit < num)
	{
		limit <<= 1;
		i++;
	}

	return i;

} /* static guint32 _bdb_log2(guint32 num) */

static void* _bdb_page_get(HTAB *htab, guint32 bucket, int is_bucket)
{
	int	offset, page;
	guint16	*bp;

	page = htab->hdr.hdrpages;

	if (is_bucket == FALSE)
	{
		page  += OPAGENUM(bucket);
		bucket = (1 << SPLITNUM(bucket)) - 1;
	}

	if (bucket)
	{
		page += bucket;
		page += htab->hdr.spares[_bdb_log2(bucket + 1) - 1];
	}

	offset = page << htab->hdr.bshift;

	if (lseek(htab->fd, offset, SEEK_SET) != offset)
	{
		return NULL;
	}

	if (read(htab->fd, htab->pagebuf, htab->hdr.bsize) != htab->hdr.bsize)
	{
		return NULL;
	}

	bp = (guint16 *) htab->pagebuf;

	if ( BYTEORDER_DIFFERS(htab->hdr.lorder) )
	{
		int i, max;

		bp[0] = BYTESWAP_UINT16(bp[0]);
		max = bp[0] + 2;

	    	/*
			bound the size of max by
	     		the maximum number of entries
	     		in the array
	     	*/

		if (max > (htab->hdr.bsize / sizeof(guint16)))
		{
			return NULL;
		}

		/* do the byte order swap */
		for (i = 1; i <= max; i++)
		{
			bp[i] = BYTESWAP_UINT16(bp[i]);
		}
	}

	/*
		check the validity of the page here
		(after doing byte order swaping if necessary)
	*/

	if (bp[0] != 0)
	{
		guint16 i, num_keys, offset;

		num_keys = bp[0];

		/*
			bp[0] is supposed to be the number of
			entries currently in the page.  If
			bp[0] is too large (larger than the whole
			page) then the page is corrupted
		*/

		if (num_keys > (htab->hdr.bsize / sizeof(guint16)))
		{
			return NULL;
		}

		/* bound free space */
		if (FREESPACE(bp) > htab->hdr.bsize)
		{
			return NULL;
		}

		/*
			check each key and data offset to make
 			sure they are all within bounds they
 			should all be less than the previous
 			offset as well.
 		*/

		offset = htab->hdr.bsize;

		for (i = 1; i <= num_keys; i += 2)
  		{
			/* ignore overflow pages etc. */

			if (bp[i + 1] < REAL_KEY)
			{
				/*
					there are no other valid keys
		 			after seeing a non REAL_KEY
		 		*/

				break;
			}

			if (bp[i] > offset || bp[i + 1] > bp[i])			
			{
				return NULL;
			}
	
			offset = bp[i + 1];
		}

	} /* if (bp[0] != 0) */

	return bp;

} /* static void* _bdb_page_get(HTAB *htab, guint32 bucket, int is_bucket) */

static int _bdb_access(HTAB *htab, DBT *key, DBT *data)
{
	guint16	*bp, ndx;
	guint32	bucket, size;

	for (;;)
	{
		bp = (guint16 *) htab->cpage;

		if (bp == NULL)
		{
			htab->cndx	= 1;
			bucket		= htab->cbucket;

			for (;;)
			{
				bucket++;

				if (bucket > htab->hdr.max_bucket)
				{
					return -1;
				}

				bp = _bdb_page_get(htab, bucket, TRUE);

				if (bp == NULL)
				{
					return -1;
				}

				if (bp[0])
				{
					break;
				}

			} /* for (;;) */

			htab->cbucket = bucket;

		} /* if (bp == NULL) */

		while (bp[htab->cndx + 1] == OVFLPAGE)
		{
			bp =  _bdb_page_get(htab, bp[htab->cndx], FALSE);

			if (bp == NULL)
			{
				return -1;
			}

			htab->cndx = 1;
		}

		if (bp[0] == 0)
		{
			htab->cpage = NULL;
			continue;
		}

		htab->cpage = (char *) bp;
		ndx = htab->cndx;

		if (bp[ndx + 1] < REAL_KEY)
		{
			return -1;
		}

		htab->cndx = ndx + 2;

		if (htab->cndx > bp[0])
		{
			htab->cpage = NULL;
		}

		/* key size */
		size  = (ndx > 1) ? bp[ndx - 1] : htab->hdr.bsize;
		size -= bp[ndx];

		if (key->data == NULL)
		{
			key->data  = htab->pagebuf + bp[ndx];
			key->size  = size;
			break;
		}

		/* Real key/data pair */
		if (size == key->size &&
			memcmp(key->data, htab->pagebuf + bp[ndx], size) == 0)
		{
			break;
		}

	} /* for (;;) */

	if (data)
	{
		data->data = htab->pagebuf + bp[ndx + 1];
		data->size = bp[ndx] - bp[ndx + 1];
	}

	return 0;

} /* static int _bdb_access(HTAB *htab, DBT *key, DBT *data) */

static void _bdb_close(HTAB *htab)
{
	if (htab)
	{
		if (htab->fd != -1)
		{
			close(htab->fd);
		}

		g_free(htab->pagebuf);
		g_free(htab);
	}

} /* static void _bdb_close(HTAB *htab) */

static HTAB *_bdb_open(char **dbname, int mode, int errmesg)
{
	int		fd, hdrsize, error;
	const char	*filename;
	HTAB		*htab;

	error = 0;

	filename = *dbname;

	if (filename == NULL)
	{
		static const char *dbtab[] =
		{
			"index.db",
			"fat.db",
			"cache.db",
			"CCache log",
			NULL
		};

		const char **dbptr = dbtab;

		fd = -1;

		while ((filename = *(dbptr++)) != NULL) 
		{
			fd = file_open(filename, mode);

 			if (fd != -1 || errno != ENOENT)
			{
				break;
			}
		}

		*dbname = (gchar *) filename;
	}

	else
	{
		fd = file_open(filename, mode);
	}

 	if (fd == -1)
	{
		error = errno;
		goto error0;
	}

	/* initialize to '\0' */
	htab = g_malloc0(sizeof(HTAB));

	htab->fd = fd;
	hdrsize  = read(fd, (char *) &(htab->hdr), sizeof(HASHHDR));

	if (hdrsize == -1)
	{
		error = errno;
		goto error1;
	}

	if (hdrsize != sizeof(HASHHDR))
	{
		goto error1;
	}

#ifndef UINT32_IS_BIG_ENDIAN
	_bdb_swap_header(htab);
#endif

	/* Verify file type, versions and hash function */

	if (htab->hdr.magic != HASHMAGIC)
	{
		goto error1;
	}

	if (htab->hdr.version != HASHVERSION &&
			htab->hdr.version != OLDHASHVERSION)
	{
		goto error1;
	}

	if (_bdb_hash4(CHARKEY, sizeof(CHARKEY)) != htab->hdr.h_charkey)
	{
		goto error1;
	}

	htab->pagebuf = g_malloc(htab->hdr.bsize);
	htab->cbucket = -1;

	return htab;

error1:
	_bdb_close(htab);

error0:
	errno = error ? error : EINVAL;

	if (errmesg)
	{
		gui_strerror(nscache_err_open_index_file);
	}

	return NULL;

} /* static HTAB *_bdb_open(char **dbname, int mode, int errmesg) */

int berkeley_check(const char *dbname, const DBT *key)
{
	int	error;
	char	*str;
	HTAB	*htab;

	error = -1;

	str = (char *) dbname;
	htab = _bdb_open(&str, O_RDONLY, FALSE);

	if (htab)
	{
		error = 0;

		if (key)
		{
			error = _bdb_access(htab, (DBT *) key, NULL);
		}

		_bdb_close(htab);
	}

	return error;

} /* int berkeley_check(const char *dbname, const DBT *key) */

GSList *berkeley_read(char **dbname, CacheEntryReadFunc func, void *userdata)
{
	char		*filename;
	GSList		*retv;
	HTAB		*htab;
	DBT		key, data;
	DBnscache	nsc;

	nsc.userdata	= userdata;
	nsc.error	= 0;

	filename = *dbname;
	htab = _bdb_open(&filename, O_RDONLY, TRUE);

	retv = NULL;

	if (htab == NULL)
	{
		g_free(*dbname);
		*dbname = NULL;
	}

	else
	{
		if (*dbname == NULL)
		{
			*dbname = g_strdup(filename);
		}

		while (key.data = NULL, _bdb_access(htab, &key, &data) == 0)
		{
			gpointer rec = (*func)(&key, &data, &nsc);

			if (rec)
			{
				retv = g_slist_prepend(retv, rec);
			}
		}

		_bdb_close(htab);

		if (nsc.error)
		{
			gui_errfmt("%s (%d)",
				gettext("Broken record in cache index"),
				nsc.error);
		}

		else
		{
			gui_errfmt(NULL);
		}
	}

	return retv;

} /* GSList *berkeley_read(const char *dbname, SeqFunc func, gpointer userdata) */

#ifndef CACHE_READONLY

static int _bdb_page_put(HTAB *htab)
{
	int	size;
	guint16	*bp;

	bp	= (guint16 *) htab->pagebuf;
	size	= htab->hdr.bsize;

	if ( BYTEORDER_DIFFERS(htab->hdr.lorder) )
	{
		int i, max;

		max = bp[0] + 2;

		/*
			bound the size of max by
			the maximum number of entries
			in the array
		*/

		if (max > (size / sizeof(guint16)))
		{
			return -1;
		}

		for (i = 0; i <= max; i++)
		{
			bp[i] = BYTESWAP_UINT16(bp[i]);
		}
	}

	if (lseek(htab->fd, -size, SEEK_CUR) == -1)
	{
		return -1;
	}


	if (write(htab->fd, htab->pagebuf, size) != size)
	{
		return -1;
	}

	return 0;

} /* static int _bdb_page_put(HTAB *htab) */

static int _bdb_delpair(HTAB *htab)
{
	int	n, keys, ndx, offset;
	guint16	*bp, newoff, pairlen;

	bp = (guint16 *) htab->pagebuf;
	n  = bp[0];

	ndx = htab->cndx - 2;
 
	if (bp[ndx + 1] < REAL_KEY)
	{
		return -1;
	}

	newoff = (ndx != 1) ? bp[ndx - 1] : htab->hdr.bsize;
	pairlen = newoff - bp[ndx + 1];

	if (ndx != (n - 1))
	{
		/* Hard Case -- need to shuffle keys */

		int	i;
		guint32	dst_offset, length, offset;
		char	*dst, *src;

		offset		= (guint32) OFFSET(bp);
		dst_offset	= offset + (guint32) pairlen;
		dst		= htab->pagebuf + dst_offset;
		src		= htab->pagebuf + offset;
		length		= bp[ndx + 1] - offset;

		/*
			+-----------+XXX+---------+XXX+---------+---------> +infinity
			|           |             |             |
			0           src_offset    dst_offset    hdr.bsize

			Dst_offset is > src_offset, so if src_offset were bad,
			dst_offset would be too, therefore we check only dst_offset.

			If dst_offset is >= hdr.bsize, either OFFSET(bp),
			or pairlen, or both is corrupted.

			Once we know dst_offset is < hdr.bsize, we can subtract
			it from hdr.bsize to get an upper bound on length.
		 */

		if (dst_offset > (guint32) htab->hdr.bsize)
		{
			return -1;
		}

		if (length > (guint32) (htab->hdr.bsize - dst_offset))
		{
			return -1;
		}

		memmove(dst, src, length);

		/* Now adjust the pointers */
		for (i = ndx + 2; i <= n; i += 2)
		{
			if (bp[i + 1] == OVFLPAGE)
			{
				bp[i - 2] = bp[i];
				bp[i - 1] = bp[i + 1];
			}

			else
			{
				bp[i - 2] = bp[i] + pairlen;
				bp[i - 1] = bp[i + 1] + pairlen;
			}
		}

	} /* if (ndx != (n - 1)) */

	/* Finally adjust the page data */
	bp[n] = OFFSET(bp) + pairlen;
	bp[n - 1] = bp[n + 1] + pairlen + 2 * sizeof(guint16);
	bp[0] = n - 2;

	if (_bdb_page_put(htab))
	{
		return -1;
	}

	keys = UINT32_FROM_BIG_ENDIAN(htab->hdr.nkeys);

	if (keys > 0)
	{
		keys--;
		htab->hdr.nkeys = UINT32_TO_BIG_ENDIAN(keys);
	}

	offset = (char *) &(htab->hdr.nkeys) - (char *) &(htab->hdr);

	if (lseek(htab->fd, offset, SEEK_SET) != offset ||
		write(htab->fd, &(htab->hdr.nkeys), sizeof(gint32)) != sizeof(gint32))
	{
		return -1;
	}

	return 0;

} /* static int _bdb_delpair(HTAB *htab) */

int berkeley_delete(const char *dbname, const char *filename, const DBT *key)
{
	int	error;
	char	*str;
	HTAB	*htab;

	str = (char *) dbname;
	htab = _bdb_open(&str, O_RDWR, TRUE);

	if (htab == NULL)
	{
		return -1;
	}

	error = _bdb_access(htab, (DBT *) key, NULL);

	if (error == 0)
	{
		error = NSC_ERR_FILE;

		if (file_remove(filename) == 0)
		{
			error = _bdb_delpair(htab);

			if (error)
			{
				error = NSC_ERR_FILE;
			}
		}
	}

	_bdb_close(htab);

	return error;

} /* int berkeley_delete(const char *dbname, const char *filename, const DBT *key) */

#endif /* CACHE_READONLY */

#endif /* CONFIG_BERKELEY_DB */
