/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		created and modified 2005 ... 2008 by Harald Foerster	   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>

#include "file.h"
#include "moz.h"

#ifdef HAVE_ZLIB
#include "zlib.h"
#ifdef HAVE_BZLIB
#include "bzlib.h"
#endif
#else
#include <signal.h>
#include <sys/wait.h>
#endif

#ifdef HAVE_MSIE
#include "msie.h"
#endif

/*
#include "nscache.h"
#include "stringbuf.h"
#include <sys/types.h>
#include <sys/stat.h>
*/


#ifndef ENODATA
#define ENODATA		ENOENT
#endif

#ifndef O_BINARY
#define O_BINARY	0
#endif

#define FILE_OPEN_READ_FLAGS	(O_BINARY | O_RDONLY)
#define FILE_OPEN_WRITE_FLAGS	(O_BINARY | O_WRONLY | O_CREAT | O_EXCL)

/* globals needed in 'gui.c' function 'SaveFolderAs()' */

const char index_html_str[]	= "/index.html";

unsigned int decode;
unsigned int single_folder = 0;
unsigned int url_to_local_selector = 0;

#ifdef HAVE_LINK
unsigned int hardlink = 1;
#endif


#ifdef HAVE_CYGWIN

void file_fname_replace(char* fname)
{
	/*
		MS Windows doesn't like the following
		characters (without '/' and '\')
	*/

	static const char replace_char[] = "?:*|<>\"";

	const char* ptr = replace_char;

	do
	{
		char* s = fname - 1;

		while((s = strchr(++s, *ptr)) != NULL)
		{
			*s = '!';
		}
	}
	while(*(++ptr) != '\0');

} /* void file_fname_replace(char*) */

#endif /* HAVE_CYGWIN */

static int file_make_pathname(const char* filename)
{
	int	slen;
	char*	str;

	slen = nscache_db.path.len;

	str = stringbuf_append(&nscache_db.path, filename, 0);

	if(str == NULL)
	{
		return -1;
	}

	return slen;

} /* static int file_make_pathname(const char*) */

#ifdef HAVE_LINK

static int file_link(const nscache_record_t* rec, char* dst)
{
	int slen, retv;;

	if(rec->content_offset)
	{
		return -1;
	}

	slen = file_make_pathname(rec->filename);

	if(slen == -1)
	{
		return slen;
	}

#ifdef HAVE_CYGWIN

	file_fname_replace(dst);
#endif
	retv = link(nscache_db.path.str, dst);

	stringbuf_truncate(&nscache_db.path, slen);

	return retv;

} /* static int file_link(const nscache_record_t*, char*) */

#endif /* HAVE_LINK */

int file_open(const char* filename, int flags)
{
	int fd, slen;

	slen = file_make_pathname(filename);

	if(slen == -1)
	{
		return slen;
	}

	fd = open(nscache_db.path.str, flags);

	stringbuf_truncate(&nscache_db.path, slen);

	return fd;

} /* int file_open(const char*, int) */

#ifndef CACHE_READONLY

int file_remove(const char* filename)
{
	int slen, retv;

	slen = file_make_pathname(filename);

	if(slen == -1)
	{
		return slen;
	}

	retv = unlink(nscache_db.path.str);

	stringbuf_truncate(&nscache_db.path, slen);

	return retv;

} /* int file_remove(const char*) */

#endif /* CACHE_READONLY */

static int file_access_rdwr(const char* path)
{
	int error = errno;

	if(access(path, R_OK | W_OK))
	{
		int retv = FILE_ACCESS_READ;

		if(access(path, R_OK))
		{
			error = errno;
			retv  = access(path, F_OK);
		}

		if(retv >= 0)
		{
			errno = error;
		}

		return retv;
	}

	return FILE_ACCESS_READ | FILE_ACCESS_WRITE;

} /* static int file_access_rdwr(const char*) */

int file_access_index(void)
{
	int slen, retv;

	slen = file_make_pathname(nscache_db.name);

	if(slen == -1)
	{
		return slen;
	}

	retv = file_access_rdwr(nscache_db.path.str);

	if(retv == 0)
	{
		int error = errno;

		if(access(nscache_db.path.str, W_OK) == 0)
		{
			/* write only ? */
			retv = FILE_ACCESS_WRITE;
		}

		errno = error;
	}

	stringbuf_truncate(&nscache_db.path, slen);

	return retv;

} /* int file_access_index(void) */

int file_access(const nscache_record_t* rec)
{
	int slen, retv;

	if(rec->content_offset)
	{

#ifdef HAVE_MSIE
		if(rec->content_offset == MSIE_CACHE)
		{
			return msie_file_access(rec);
		}
#endif

		retv = moz_file_access(rec);

#ifndef CACHE_READONLY
		if(retv < 0)
#endif
			return retv;
	}

	/* cache file */

	slen = file_make_pathname(rec->filename);

	if(slen == -1)
	{
		return slen;
	}

	retv = file_access_rdwr(nscache_db.path.str);

#ifndef CACHE_READONLY
	if(retv > 0 && (retv & FILE_ACCESS_WRITE))
	{
		/* cache directory */

		char *sep;

		if(rec->content_offset)
		{
			goto index_access;
		}

		sep = strrchr(nscache_db.path.str, '/');

		if(sep)
		{
			int acc;

			*sep = '\0';
			 acc = file_access_rdwr(nscache_db.path.str);
			*sep = '/';

			if(acc > 0 && (acc & FILE_ACCESS_WRITE))
			{
index_access:
				stringbuf_truncate(&nscache_db.path, slen);

				/* index file */

				slen = file_make_pathname(nscache_db.name);

				if(slen == -1)
				{
					return retv;
				}

				acc = file_access_rdwr(nscache_db.path.str);

				if(acc == (FILE_ACCESS_READ | FILE_ACCESS_WRITE))
				{
					retv |= FILE_ACCESS_REMOVE;
				}
			}
		}
	}
#endif /* CACHE_READONLY */

	stringbuf_truncate(&nscache_db.path, slen);

	return retv;

} /* int file_access(const nscache_record_t*) */

int file_chmod(const nscache_record_t* rec, mode_t mode)
{
	int slen, retv;

	if(rec->content_offset)
	{
		return -1;
	}

	slen = file_make_pathname(rec->filename);

	if(slen == -1)
	{
		return slen;
	}

	retv = chmod(nscache_db.path.str, mode);

	stringbuf_truncate(&nscache_db.path, slen);

	return retv;

} /* int file_chmod(const nscache_record_t*, mode_t) */

int file_status(const nscache_record_t* rec, struct stat* statbuf)
{
	int slen, retv;

	if(rec->content_offset)
	{
		return -1;
	}

	slen = file_make_pathname(rec->filename);

	if(slen == -1)
	{
		return slen;
	}

	retv = stat(nscache_db.path.str, statbuf);

	stringbuf_truncate(&nscache_db.path, slen);

	return retv;

} /* int file_status(const nscache_record_t*, struct stat*) */

#ifdef HAVE_ZLIB

static void* file_decode(const char *encoding, const char *buffer, size_t *length)
{

#ifdef HAVE_BZLIB
	char**		next_out;
	unsigned int*	avail_out;
	unsigned int*	total_out;
#else

#define next_out	(&(u.zs.next_out))
#define avail_out	(&(u.zs.avail_out))
#define total_out	(&(u.zs.total_out))

#endif

	int		status, winbits;
	char*		pbuf;

	union
	{
		z_stream	zs;
#ifdef HAVE_BZLIB
		bz_stream	bzs;
#endif
	} u;

	pbuf = NULL;

	if(strcmp(encoding, "gzip") == 0 ||
		strcmp(encoding, "deflate") == 0 ||
			strcmp(encoding, "compress") == 0)
	{
		winbits = 15 + 32;

#ifdef HAVE_BZLIB
		next_out  = (char**) &(u.zs.next_out);
		avail_out = (unsigned int*) &(u.zs.avail_out);
		total_out = (unsigned int*) &(u.zs.total_out);
	}

	else if(strcmp(encoding, "bzip2") == 0)
	{
		winbits   = 0;
		next_out  = (char**) &(u.bzs.next_out);
		avail_out = (unsigned int*) &(u.bzs.avail_out);
		total_out = (unsigned int*) &(u.bzs.total_out_lo32);
#endif
	}

	else
	{
		return pbuf;
	}

	for(;;)
	{
		memset(&u, '\0', sizeof(u));

#ifdef HAVE_BZLIB
		if(winbits == 0)
		{
			u.bzs.next_in	= (char *) buffer;
			u.bzs.avail_in	= *length;
			status		= BZ2_bzDecompressInit(&u.bzs, 0, 0);
		}

		else

#endif /* HAVE_BZLIB */

		{
			u.zs.next_in	= (char *) buffer;
			u.zs.avail_in	= *length;
			status		= inflateInit2(&u.zs, winbits);
		}

		if(status == 0)
		{
			for(;;)
			{
				char* ptr;

				*avail_out = *length << 2;

				ptr  = pbuf;
				pbuf = realloc(pbuf, *avail_out + *total_out);

				if(pbuf == NULL)
				{
					if(ptr)
					{
						free(ptr);
					}

					break;
				}

				*next_out = &pbuf[*total_out];
#ifdef HAVE_BZLIB
				if(winbits == 0)
				{
					status = BZ2_bzDecompress(&u.bzs);

					if(status == BZ_STREAM_END)
					{
						goto _stream_end;
					}
				}
				else
#endif
				if((status = inflate(&u.zs, Z_SYNC_FLUSH)) == Z_STREAM_END)
				{
#ifdef HAVE_BZLIB
_stream_end:
#endif
					ptr  = pbuf;
					pbuf = realloc(pbuf, *total_out + 1);

					if(pbuf == NULL)
					{
						free(ptr);
						break;
					}

					pbuf[*total_out] = '\0';
					*length = *total_out;
					break;
				}

				if(status != 0)
				{
					free(pbuf);
					pbuf = NULL;
					break;
				}

			} /* for(;;) */
#ifdef HAVE_BZLIB
			if(winbits == 0)
			{
				BZ2_bzDecompressEnd(&u.bzs);
				break;
			}
#endif
			inflateEnd(&u.zs);

			if(pbuf != NULL)
			{
				break;
			}

		} /* if(status == 0) */

		if(winbits < 0)
		{
			break;
		}

		winbits = -15;

	} /* for(;;) */

	return pbuf;
}

#else /* ! HAVE_ZLIB */

static void* file_decode(const char* encoding, const char* buffer, size_t* length)
{
	int		fdpin[2], fdpout[2];
	char*		pbuf;
	const char*	args;
	pid_t		child_exec;
	sigset_t	sig_block, sig_unblock;

	pbuf = NULL;

	if(strcmp(encoding, "gzip") == 0 ||
			strcmp(encoding, "bzip2") == 0)
	{
		args = "-cdq";
	}

	else if(strcmp(encoding, "compress") == 0)
	{
		args = "-cd";
	}

	else
	{
		return pbuf;
	}

	if(pipe(fdpin))
	{
		return pbuf;
	}

	if(pipe(fdpout))
	{
		close(fdpin[0]);
		close(fdpin[1]);

		return pbuf;
	}

	/* block SIGCHLD (in fact all signals) */
	sigfillset(&sig_block);
	sigprocmask(SIG_BLOCK, &sig_block, &sig_unblock);

	switch(child_exec = fork())
	{
		case  0:
		{
			close(fdpin[1]);
			close(fdpout[0]);

			if(dup2(fdpin[0], STDIN_FILENO) >= 0 &&
					dup2(fdpout[1], STDOUT_FILENO) >= 0)
			{
				/* "gzip -cdq < buffer" */
				execlp(encoding, encoding, args, NULL);
			}

			_exit(-1);

		} /* case  0 (child_exec) */

		case -1:
		{
			close(fdpin[0]);
			close(fdpin[1]);
			close(fdpout[0]);
			close(fdpout[1]);
			break;
		}

		default:
		{
			pid_t child_write;

			close(fdpin[0]);
			close(fdpout[1]);

			switch(child_write = fork())
			{
				case  0:
				{
					int rlen  = 0;
					int count = *length;

					do
					{
						int retv = write(fdpin[1], &buffer[rlen], count);

						if(retv < 0)
						{
							_exit(retv);
						}

						rlen  += retv;
						count -= retv;
					}
					while(count > 0);

					_exit(0);

				} /* case  0 (child_write) */

				case -1:
				{
					close(fdpin[1]);
					close(fdpout[0]);
					break;
				}

				default:
				{
					int count, plen, rlen, retv, status;

					close(fdpin[1]);

					count = plen = rlen = 0;

					for(;;)
					{
						if(count == 0)
						{
							char* ptr;

							count = *length << 2;
							plen += count;
							ptr   = pbuf;
							pbuf  = realloc(pbuf, plen);

							if(pbuf == NULL)
							{
								free(ptr);
								break;
							}
						}

						retv = read(fdpout[0], &pbuf[rlen], count);

						if(retv == 0)
						{
							break;
						}

						if(retv < 0)
						{
							free(pbuf);
							pbuf = NULL;
							break;
						}

						rlen  += retv;
						count -= retv;

					} /* for(;;) */

					close(fdpout[0]);
					waitpid(child_write, &status, 0);

					if(pbuf != NULL)
					{
						if(WIFEXITED(status) == FALSE ||
								WEXITSTATUS(status) != 0)
						{
							free(pbuf);
							pbuf = NULL;
						}

						else
						{
							char* ptr = pbuf;

							pbuf = realloc(pbuf, rlen + 1);

							if(pbuf == NULL)
							{
								free(ptr);
							}

							else
							{
								pbuf[rlen] = '\0';
								*length = rlen;
							}
						}

					} /* if(pbuf != NULL) */

					break;

				} /* default (parent_write) */

			} /* switch(child_write = fork()) */

			waitpid(child_exec, NULL, 0);

			break;

		} /* default (parent_exec) */

	} /* switch(child_exec = fork()) */

	/* unblock signals */
	sigprocmask(SIG_SETMASK, &sig_unblock, NULL);

	return pbuf;

} /* static void* file_decode(const char*, const char*, size_t*) */

#endif /* #ifdef HAVE_ZLIB .. #else */

void* file_read(const nscache_record_t* rec, size_t* length)
{
	int	fd, offset;
	char*	buffer;

	if(rec->content_offset)
	{
#ifdef HAVE_MSIE
		if(rec->content_offset == MSIE_CACHE)
		{
			return msie_file_read(rec, length);
		}
#endif
		if(moz_file_access(rec) < 0)
		{
			return NULL;
		}
	}

	fd = file_open(rec->filename, FILE_OPEN_READ_FLAGS);

	if(fd < 0)
	{
		return NULL;
	}

	if(rec->content_offset)
	{
		*length = rec->content_length;

		offset = lseek(fd, rec->content_offset, SEEK_SET);

		if(offset != rec->content_offset)
		{
			if(offset != -1)
			{
				errno = ENODATA;
			}

			close(fd);

			return NULL;
		}
	}

	else
	{
		*length = offset = lseek(fd, 0, SEEK_END);

		if(offset == -1 || (offset = lseek(fd, 0, SEEK_SET)) != 0)
		{
			if(offset != -1)
			{
				errno = ENODATA;
			}

			close(fd);

			return NULL;
		}
	}

	buffer = malloc(*length + 1);

	if(buffer)
	{
		int len;

		buffer[*length] = '\0';

		len = read(fd, buffer, *length);

		if(len != *length)
		{
			if(len != -1)
			{
				errno = ENODATA;
			}

			free(buffer);
			buffer = NULL;
		}
	}

	close(fd);

	if(buffer && rec->content_encoding && decode)
	{
		char *ptr;

		ptr = file_decode(rec->content_encoding, buffer, length);

		if(ptr)
		{
			free(buffer);
			buffer = ptr;
		}
	}

	return buffer;

} /* void* file_read(const nscache_record_t*, size_t*) */

static int file_write(char* dst, mode_t file_perm, const char* buffer, size_t length)
{
	int fd, len;

#ifdef HAVE_CYGWIN

	file_fname_replace(dst);
#endif

	fd = open(dst, FILE_OPEN_WRITE_FLAGS, file_perm);

	if(fd < 0)
	{
		return ERR_FILE_OPEN_DST;
        }

	len = write(fd, buffer, length);

	close(fd);

	if(len != length)
	{
		unlink(dst);

		if(len != -1)
		{
			errno = ENOSPC;
		}

		return ERR_FILE_WRITE;
	}

	return 0;

} /* static int file_write(char*, mode_t, const char*, size_t) */

int file_copy(const nscache_record_t* rec, char* dst, mode_t file_perm)
{
	int	error;
	char*	buffer;
	size_t	length;

	buffer = file_read(rec, &length);

	if(buffer == NULL)
	{
		return ERR_FILE_OPEN_SRC;
	}

	error = file_write(dst, file_perm, buffer, length);

	free(buffer);

	return error;

} /* int file_copy(const nscache_record_t*, char*, mode_t) */

int file_save(const nscache_record_t* rec, char* dst)
{

#ifdef HAVE_LINK

	/*
		Try to link this file only if it's 'stand alone',
		and it should'nt be decoded, and the user have
		decided so.
		If it's a Mozilla data cache file, we have to read
		the bytes from the first (which is the offset) to
		the last  (which is the offset+length) byte.
		Files from MS Internet Explorer will always be readed.
	*/

	if(hardlink && rec->content_offset == 0 &&
			(rec->content_encoding == NULL || decode == FALSE))
	{
		if(file_link(rec, dst) == 0)
		{
			return 0;
		}

		if(errno != EPERM && errno != EXDEV)
		{
			return ERR_FILE_LINK;
		}
	}

#endif /*  HAVE_LINK */

	return file_copy(rec, dst, FILE_PERM_USER);

} /* int file_save(nscache_record_t*, char*) */

static int directory_create(const char* path)
{
	int    count, error;
	char*  last_sep;

	/*
		We do create the folders recursively, because doing
		so is very easy. Each time this function is called,
		we first shrinken the path by replacing the last se-
		parator with the string terminator. The first time
		we hide the file name, after then all folders that
		doesn't exist yet. This function is called by itself
		until the first (most top) folder can be created, then
		each instance of this function will add its part (sub
		folder) until getting back to the first instance, where
		the whole path will be completed by creating the last
		(most bottom) sub folder.
	*/

	last_sep = strrchr(path, '/');

	if(last_sep == NULL)
	{
		return ERR_FILE_DIR_CREATE;
	}

	while(last_sep > path && last_sep[-1] == '/')
	{
		last_sep--;
	}

	*last_sep = '\0';
	count	  = 0;

	while((error = mkdir(path, DIR_PERMISSION)) != 0)
	{
		error = errno;

		/*
			The only error we can accept is that the dest-
			ination doesn't exist (that's why we're here).
		*/

		if(error == ENOENT)
		{
			error = 0;

			count = directory_create(path);

			if(count == ERR_FILE_DIR_CREATE)
			{
				error = count;
			}
		}

		if(error)
		{
			break;
		}
	}

	/* don't forget to restore the name */

	*last_sep = '/';

	if(error == 0)
	{
		/* summery of created folders */
		return ++count;
	}

	return ERR_FILE_DIR_CREATE;

} /* static int directory_create(const char*) */

static char* is_index_html(const char* url)
{
	/*
		A name with no dot is treated as folder, with
		one or more dot(s) or capital letter(s) as file.
		This can cause errors, e.g.
			readme		file treated as folder
			README		folder treated as file
			folder.img	folder treated as file
	*/

	unsigned int	chr;
	const char*	str;

	/* get last slash */

	str = strrchr(url, '/');

	if(str == NULL)
	{
		str = url - 1;
	}

	if(str[1] == '\0')
	{
		/* return string end of 'url' for appending 'index.html' */

		return (char *) str + 1;
	}

	while( (chr = *(++str)) != '\0' )
	{
		/* return NULL for file name */

		if(chr == '.' || chr == '#' || isupper(chr))
		{
			return NULL;
		}
	}

	/* return last character of 'url' for appending 'index.html' */

	return (char *) str - 1;

} /* static char* is_index_html(const char*) */

static size_t file_convert(const char* urlstr, size_t service_len, 
				const nscache_record_t** rec_array,
							StringBuf* filebuf)
{
	size_t	converted, file_offs, buf_start;

	buf_start = file_offs = converted = 0;

	do
	{
		unsigned int ch;

		/* find start of instruction */

		ch = filebuf->str[file_offs];

		if(ch == '\0')
		{
			continue;
		}

		if(ch == '<')
		{
			if(filebuf->str[file_offs + 1] == '/')
			{
				file_offs++;
				buf_start = 0;

				continue;
			}

			if(strncmp(&filebuf->str[file_offs + 1], "!--", 3) == 0)
			{
				char* rend = strstr(&filebuf->str[file_offs + 4], "-->");

				if(rend == NULL)
				{
					return converted;
				}

				file_offs = rend - filebuf->str;
				buf_start = 0;

				continue;
			}

			buf_start = file_offs;
		}

		else if(ch == '>' && buf_start > 0)
		{
			static const struct
			{
				char*	str;
				size_t	len;

			} *link_ptr, link_array[] =
				{
					{"href=",	5},
					{"src=",	4},
					{"longdesc=",	9},
					{"background=",11},
					{NULL,		0}
				};

			static const char updir_str[] = "../";
 
			char*		url_start;
			char*		url_keep;
			char*		s;
			int		diff, updir_count;
			unsigned int	is_folder;
			size_t		url_start_offs;
			size_t		url_keep_offs;

			struct
			{
				size_t	offs;
				int	chr;

			} url_end;


			if(file_offs - buf_start < 6)
			{
				/* min. "<src= >" */
				continue;
			}

			if(file_offs - buf_start > 11 + service_len)
			{
				if(g_strncasecmp(&filebuf->str[buf_start],
								"<base href=", 11) == 0)
				{
					/* start of remark */

					if(stringbuf_insert(filebuf, "<!-- ", 5,
									buf_start) == NULL)
					{
						/* out of memory */
						return 0;
					}

					/* end of remark ("> -->") */

					if(stringbuf_insert(filebuf, "> --", 4,
									file_offs + 5) == NULL)
					{
						/* out of memory */
						return 0;
					}

					converted++;
					file_offs += 9;
					buf_start  = 0;

					continue;

				} /* if(g_strncasecmp(&filebuf->str[buf_start] ...) == 0) */

			} /* if(file_offs - buf_start > 11 + service_len) */


			/* now start conversion ... */

			link_ptr = link_array;

			while((url_start = (char*) link_ptr->str) != NULL)
			{
				url_start =
				tl_mem_find_str_igncase(&filebuf->str[buf_start],
					file_offs - buf_start, link_ptr->str, link_ptr->len - 1);

				if(url_start)
				{
					break;
				}

				link_ptr++;

			} /* while((url_start = (char*) link_ptr->str) != NULL) */


			if(url_start == NULL ||
				( url_start[-1] != '<' && ! isspace(url_start[-1]) ))
			{
				goto __loop_continue__;
			}

			filebuf->str[file_offs] = '\0';
			url_start += link_ptr->len;

			if(*url_start == '\"')
			{
				ch = *(url_start++);
				s  = strchr(url_start, ch);

				/* not really neccessary - but safety first */

				if(s == NULL)
				{
					goto __loop_continue__;
				}
			}

			else
			{
				s = url_start;

				do
				{
					ch = *(++s);
				}
				while(ch != '\0' && ! isspace(ch));
			}

			url_start_offs	= url_start - filebuf->str;
			url_end.offs	= s - filebuf->str;
			url_end.chr 	= ch;

			/* terminate url */
			*s = '\0';

			if(strncmp(url_start, urlstr, service_len) == 0)
			{
				/* service found - skip it */

				url_keep_offs = url_start_offs + service_len;

				if(filebuf->str[url_keep_offs] == '/')
				{
					url_keep_offs++;
				}

				else if(filebuf->str[url_keep_offs] != '\0')
				{
					/* differs - e.g. "foo" vs. "foo_1" */
					goto __urlend_restore__;
				}
			}

			else if(*url_start == '/')
			{
				/* relative addressing mode */
				url_keep_offs = url_start_offs + 1;
			}

			else if(*url_start == '#')
			{
				/* nothing to do */
				goto __urlend_restore__;
			}

			else if(strstr(url_start, "://"))
			{
				/* other service */
				goto __urlend_restore__;
			}

			else
			{
				/* relative to current folder */
				url_keep_offs = url_start_offs;

			} /* if(strncmp(url_start, urlstr, service_len) == 0) */


			/*
				Compare the URL with the links for matching
				the complete URL, or whether it is a folder
				and needs appending 'index.html'.
			*/

			url_keep  = &filebuf->str[url_keep_offs];
			is_folder = '\0';

			do
			{
				const nscache_record_t*  cache;
				const nscache_record_t** downloaded;

				size_t	clen, ulen;

				s	= &filebuf->str[url_end.offs];
				ulen	= s - url_keep;

				while((ch = *(--s)) != '/')
				{
					if(ch == '#')
					{
						ulen = s - url_keep;
						break;
					}
				}

				downloaded = rec_array;

				while((cache = *(downloaded++)) != NULL)
				{
					/* no match for "www.foo.bar" "www.foo.bar/index.html" */

					clen = strlen(&cache->urlstr[service_len + 1]);
					is_folder = cache->urlstr[service_len + clen];

					if(url_to_local_selector == URL_CONVERT_ONLY_FILE_REFERENCE &&
												ulen == clen)
					{
						/*
							match "name/" "name/"
							match "name.html#somewhat" "name.html"
						*/

						if(strncmp(url_keep,
							&cache->urlstr[service_len + 1], ulen) == 0)
						{
							/* file was downloaded */

							break;
						}
					}

					else if(clen - ulen == 1 && is_folder == '/')
					{
						/* match "name" "name/" */

						if(strncmp(url_keep,
							&cache->urlstr[service_len + 1], ulen) == 0)
						{
							/* file was downloaded */

							break;
						}
					}

				} /* while((cache = *(downloaded++)) != NULL) */

				if(cache == NULL)
				{
					if(url_to_local_selector == URL_CONVERT_ONLY_FILE_REFERENCE)
					{
						/* Ignored if relative addressing */

						if(url_keep_offs - url_start_offs > 1)
						{
							/* file was not downloaded */
							goto __urlend_restore__;
						}
					}

					is_folder = '\0';
				}
			}
			while(FALSE);

			updir_count = 0;

			if(single_folder)
			{
				/* find last slash */

				if((s = strrchr(url_keep, '/')) != NULL)
				{
					url_keep_offs = &s[1] - filebuf->str;
				}

				/* if appending "index.html" => remove all */

				if(is_index_html(url_keep))
				{
					url_keep_offs = url_end.offs;
				}
			}

			else if(url_keep_offs > url_start_offs)
			{
				const char* url_dest;

				/* skip "http://www.foo.bar/" (including slash) */

				url_dest = &urlstr[service_len + 1];

				/* remaining folders plus filename */

				while((s = strchr(url_dest, '/')) != NULL)
				{
					/* compare including slash */

					size_t len = ++s - url_dest;

					if(strncmp(url_dest, url_keep, len))
					{
						break;
					}


					url_keep_offs	+= len;
					url_keep	+= len;
					url_dest	+= len;
				}

				while((s = strchr(url_dest, '/')) != NULL)
				{
					/* climb one folder up */
					updir_count++;

					/* skip slash */
					url_dest = &s[1];
				}

			} /* if(single_folder) .. else if(url_keep > url_start) */

			/* move down => neg. value */

			diff  = url_start_offs - url_keep_offs;
			diff += updir_count * (sizeof(updir_str) - 1);

			if((diff | updir_count) != 0)
			{
				char* ptr;

				if(stringbuf_move(filebuf, url_start_offs, diff) == NULL)
				{
					/* out of memory */
					return 0;
				}

				ptr = &filebuf->str[url_start_offs];

				while(--updir_count >= 0)
				{
					/* prepend "../" */
					ptr  = memcpy(ptr, updir_str, sizeof(updir_str) - 1);
					ptr += sizeof(updir_str) - 1;
				}


				/*
					If no filename - append "index.html", adjust
					'url_keep_offs', because the string has been
					moved 'diff' bytes up or down.
				*/

				url_keep_offs += diff;

			} /* if((diff | updir_count) != 0) */

			if(is_folder == '/')
			{
				/* get the end of URL */
				url_keep = &filebuf->str[url_end.offs + diff];

				if(url_keep[-1] != '/')
				{
					/* 'index_html' with slash */
					url_keep--;
				}
			}

			else
			{
				url_keep = is_index_html(&filebuf->str[url_keep_offs]);
			}
	
			if(url_keep)
			{
				size_t		index_sz;
				const char*	index_str;

				index_sz  = sizeof(index_html_str) - 2;
				index_str = &index_html_str[1];

				if(*url_keep != '\0')
				{
					/* 'index_html' with slash */
					index_sz++;
					index_str--;

					/* end of URL*/
					url_keep++;
				}

				diff += index_sz;

				if(stringbuf_insert(filebuf, index_str, index_sz,
							url_keep - filebuf->str) == NULL)
				{
					/* out of memory */
					return 0;
				}

			} /* if(url_keep) */

			if((diff | updir_count) != 0)
			{
#ifdef HAVE_CYGWIN
				file_fname_replace(&filebuf->str[url_start_offs]);
#endif
				url_end.offs	+= diff;
				file_offs	+= diff;
				converted++;

			} /* if((diff | updir_count) != 0) */

__urlend_restore__:
			filebuf->str[url_end.offs] = url_end.chr;

__loop_continue__:
			filebuf->str[file_offs] = '>';

			buf_start = 0;

		} /* else if(ch == '>' && buf_start > 0) */
	}
	while(++file_offs < filebuf->len);

	return converted;

} /* static size_t file_convert(const char*, size_t,
					const nscache_record_t**, StringBuf*) */
int folder_save(const char* dst,
		const nscache_record_t** rec_array, int depth, SaveDirResult* sdr)
{
	int		error;
	size_t		dir_length, service_length;
	const char*	service_beg;
	const char*	service_end;
	StringBuf	path;

	const nscache_record_t** recs;
	const nscache_record_t*  cache;

	memset(sdr, '\0', sizeof(SaveDirResult));

	error = ERR_FILE_DIR_CREATE;

	if(rec_array == NULL ||
		*rec_array == NULL ||
			stringbuf_init(&path, dst, 0) == NULL)
	{
		/* out of memory */
		errno = ENOMEM;
		sdr->errors = 1;

		return error;
	}

	if(access(path.str, F_OK) == 0)
	{
		/* folder exists */
		stringbuf_delete(&path);
		errno = EEXIST;
		sdr->errors = 1;

		return error;
	}

	dir_length = path.len;

	cache = *rec_array;

	/* length of e.g. "http://www.foo.bar" */

	if((service_beg = tl_get_url_start(cache->urlstr)) == NULL ||
		(service_end = strchr(&(strchr(service_beg, ':'))[3], '/')) == NULL)
	{
		/* invalid argument */
		stringbuf_delete(&path);
		errno = EINVAL;
		sdr->errors = 1;

		return error;
	}

	/* length without trailing slash */

	service_length = service_end - service_beg;

	recs = rec_array;

	while(error = 0, (cache = *(recs++)) != NULL)
	{
		size_t		converted;
		unsigned int	chr;
		char*		dst;
		char*		urlstr;
		char*		urlfname;
		StringBuf	filebuf;

		stringbuf_truncate(&path, dir_length);

		urlstr = tl_get_url_start(cache->urlstr);

		if(urlstr == NULL)
		{
			sdr->errors++;
			continue;
		}

		dst = &urlstr[service_length];

		/*
			If we've got something like this
			"http://www.foo.bar/load.html?http://www.foo.bar/index.html"
			then the URL will changed to
			"http://www.foo.bar/load.html?http:--www.foo.bar-index.html"
		*/

		urlfname = tl_get_url_filename(dst);
		chr = '\0';

		if(urlfname)
		{
			chr = *urlfname;
			*urlfname = '\0';
		}

		if(single_folder)
		{
			/*
				If writing the files into a single
				directory (without any sub directories)
				we need only the file name.
			*/

			dst = strrchr(dst, '/');
		}

		else
		{
			int loop;

			if(*dst != '/')
			{
				sdr->errors++;
				continue;
			}

			loop = depth;

			while(--loop >= 0)
			{
				if(dst == NULL)
				{
					sdr->errors++;
					continue;
				}

				/* skip all folders that won't be saved */

				while(*(++dst) == '/') {;}

				dst = strchr(dst, '/');
			}

		} /* if(single_folder) .. else */

		if(stringbuf_append(&path, dst, 0) == NULL)
		{
			sdr->errors++;
			continue;
		}

		if(urlfname)
		{
			size_t	start;
			char*	s;

			*urlfname = chr;

			start = path.len;

			if(stringbuf_append(&path, urlfname, 0) == NULL)
			{
				sdr->errors++;
				continue;
			}

			s = &path.str[start];

			/* maybe URL in URL */

			while((s = strchr(++s, '/')) != NULL)
			{
				*s = '-';
			}
		}

		if(path.str[path.len - 1] == '/')
		{
			stringbuf_append(&path, &index_html_str[1],
							sizeof(index_html_str) - 2);
		}

		else
		{
			/*
				Truncate filename to 'NAME_MAX',
				if neccessary (sometimes too long)
			*/

			char* basename = strrchr(path.str, '/');

			if(basename && strlen(++basename) > NAME_MAX)
			{
				stringbuf_truncate(&path, &basename[NAME_MAX] - path.str);
			}
		}

		converted = 0;

		if(url_to_local_selector != URL_CONVERT_NONE &&
			cache->content_encoding == NULL && cache->content_type &&
					strcmp(cache->content_type, "text/html") == 0)
		{
			char*	buffer;
			size_t	length;

			buffer = file_read(cache, &length);

			if(buffer)
			{
				stringbuf_from_string(&filebuf, buffer, length);

				converted = file_convert(urlstr, service_length,
								rec_array, &filebuf);
				if(converted == 0)
				{
					stringbuf_delete(&filebuf);
				}

			} /* if(buffer) */

		} /* if(url_to_local_selector != URL_CONVERT_NONE) */

		while( (error = ( (converted > 0) ?
			file_write(path.str, FILE_PERM_USER, filebuf.str, filebuf.len) :
					file_save(cache, path.str) )) != 0 )
		{

#ifdef HAVE_LINK
			if(error == ERR_FILE_LINK)
			{
				if(file_access(cache) < 0)
				{
					error = ERR_FILE_OPEN_SRC;
					break;
				}
			}
			else
#endif
			if(error != ERR_FILE_OPEN_DST)
			{
				break;
			}

			/*
				This type of error is for us essential:
					ENOENT: folder doesn't exists.

				If we've got another error,
				we'll continue with the next file.
			*/

			if(errno != ENOENT)
			{
				break;
			}

			error = directory_create(path.str);

			if(error == ERR_FILE_DIR_CREATE)
			{
				break;
			}

			sdr->folders += error;

		} /* while( (error = ( (converted > 0) ? ... */

		if(converted > 0)
		{
			stringbuf_delete(&filebuf);
		}

		if(error)
		{
			/*
				Sometimes cache files are stored more than once in
				a database e.g. "http://www.foo.bar/img/image.gif"
				and "http://www.foo.bar/img//image.gif".
				Because were writing in a newly created folder,
				we can ignore the error message, if we attempted
				to overwrite an already existing file.
				If putting all files in one folder, there may be
				the same file names in different folders (e.g.
				'index.html'). In this case we don't ignore this error.
			*/

			if(errno != EEXIST || error != ERR_FILE_OPEN_DST || single_folder)
			{
				sdr->errors++;

				if(error == ERR_FILE_DIR_CREATE)
				{
					break;
				}

				if(sdr->fname == NULL)
				{
					char* err_buff;

					sdr->errnr = errno;

					if(error == ERR_FILE_OPEN_SRC)
					{
						int slen;
#ifdef HAVE_MSIE
						if(cache->content_offset == MSIE_CACHE)
						{
							stringbuf_copy(&nscache_db.path, MSIE_GET_PATHNAME(cache), 0);
							stringbuf_append(&nscache_db.path, cache->filename, 0);
							slen = 0;
						}
						else
#endif
						slen = file_make_pathname(cache->filename);

						if(slen != -1)
						{
							err_buff = malloc(nscache_db.path.sz);

							if(err_buff)
							{
								sdr->fname = strcpy(err_buff,
										nscache_db.path.str);
							}

							stringbuf_truncate(&nscache_db.path, slen);
						}
					}

					else
					{
						err_buff = malloc(path.sz);

						if(err_buff)
						{
							sdr->fname = strcpy(err_buff, path.str);
						}

					} /* if(error == ERR_FILE_OPEN_SRC) .. else */

				} /* if(sdr->fname == NULL) */

			} /* if(errno != EEXIST || error != ERR_FILE_OPEN_DST || single_folder) */
		}

		else
		{
			sdr->files++;
			sdr->converted += converted;

		} /* if(error) .. else */

	} /* while(error = 0, (cache = *(recs++)) != NULL) */

	stringbuf_delete(&path);

	return error;

} /* int folder_save(const char* dst,
		const nscache_record_t**, int, SaveDirResult*) */

/* EOF */
