/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		created 2008 by Harald Foerster				   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#if defined(HAVE_XLOCALE)

#define X_LOCALE 1
#include <X11/Xlocale.h>

#elif defined(HAVE_LOCALE)

#ifdef X_LOCALE
#undef X_LOCALE
#endif

#include <locale.h>

#endif /* #ifdef HAVE_XLOCALE .. #elif */


#include <gdk/gdkx.h>
#include <string.h>

#include "charconv.h"
#include "fonts.h"
#include "nscache.h"


struct FontnameAndLocale
{
	const gchar *fontname;
	const gchar *locale;
};

static const gchar iso_2022_cn[]	= "ISO-2022-CN";
static const gchar iso_2022_jp[]	= "ISO-2022-JP";
static const gchar iso_2022_kr[]	= "ISO-2022-KR";
static const gchar shift_jis[]		= "Shift_JIS";

static const gchar big5[]		= "BIG5";
static const gchar euc_jp[]		= "EUC-JP";
static const gchar euc_kr[]		= "EUC-KR";


#ifdef HAVE_ICONV

gchar *fonts_convert_charset(const gchar *charset)
{
	static const struct FontConvertCharset
	{
		const gchar *from;
		const gchar *to;

	} fcc[] =
	{
		{ iso_2022_cn,	big5 },
		{ iso_2022_jp,	euc_jp },
		{ iso_2022_kr,	euc_kr },
		{ shift_jis,	euc_jp },
		{NULL,		NULL}
	};

	const struct FontConvertCharset *p = fcc;

	do
	{
		if (g_strcasecmp(p->from, charset) == 0)
		{
			break;
		}
	}
	while ((++p)->from != NULL);

	return (gchar *) p->to;

} /* gchar *fonts_convert_charset(const gchar*) */

#endif /* HAVE_ICONV */

static struct FontnameAndLocale *fonts_get_fontset(const gchar *charset)
{
	static const gchar ksc5601_1987_0[]	= "ksc5601.1987-0";
	static const gchar ko_KR[]		= "ko_KR";

	static const gchar tis620_2533_1[]	= "tis620.2533-1";
	static const gchar th_TH_TIS_620[]	= "th_TH.TIS-620";

	static const struct CharsetFontInfo
	{
		const gchar			*mime;
		const struct FontnameAndLocale	fnl;

	} cfi[] =
	{
		{ big5,			{"big5-0", "zh_TW.big5"} },
		{"BIG5-HKSCS",		{"hkscs-1", "zh_HK.hkscs"} },
		{"GB18030",		{"gb18030.2000-0", "zh_CN"} },
		{"GB2312",		{"gb2312.1980-0", "zh_CN.gb2312"} },
		{"GBK",			{"gbk-0", "zh_CN.gbk"} },
		{ euc_jp,		{ "jisx0208.1983-0,jisx0201.1976-0", "ja_JP.eucJP" } },
		{ euc_kr,		{ ksc5601_1987_0, ko_KR } },
		{"KS_C_5601-1987",	{ ksc5601_1987_0, ko_KR } },
		{"x-windows-949",	{ ksc5601_1987_0, ko_KR } },
		{"EUC-TW",		{ tis620_2533_1,  th_TH_TIS_620 } },
		{"TIS-620",		{ tis620_2533_1,  th_TH_TIS_620 } },
		{"VISCII",		{"viscii1-1", "vi_VN.viscii"} },
		{NULL,			{NULL, NULL} }
	};

	const struct CharsetFontInfo *p = cfi;

	do
	{
		if (g_strcasecmp(p->mime, charset) == 0)
		{
			return (struct FontnameAndLocale *) &(p->fnl);
		}
	}
	while ((++p)->mime != NULL);

	return NULL;

} /* struct FontnameAndLocale *fonts_get_fontset(const gchar*) */

static gint fonts_get_name(GdkWindow *window, gchar *buf, gint blen,
				gint pxsize,const gchar *registry, const gchar *encoding)
{
	enum
	{
		kFamilyPosition = 2,
		kPxSizePosition = 7
	};

	gint	i, found, fnum, helv, len, pxlen;
	gchar	**xfontnames, pxbuf[8], pxcpy[8];

	pxlen = g_snprintf(pxbuf, sizeof(pxbuf), "%d", pxsize);

	if (pxlen <= 0)
	{
		return pxlen;
	}

	if (pxlen == 1)
	{
		/* -?- */
		pxcpy[1] = '\0';
	}

	else
	{
		/* -1?- */
		strcpy(pxcpy, pxbuf);
	}

	pxcpy[pxlen - 1] = '?';

	len = g_snprintf(buf, blen,
			"-*-*-medium-r-normal-*-%s-*-*-*-*-*-%s-%s",
				pxcpy, registry, (encoding == NULL) ? "" : encoding);
	if (len <= 0)
	{
		return len;
	}

	if (encoding == NULL)
	{
		gchar *s;

		/* remove last '-' */
		buf[len - 1] = '\0';

		/* more fonts? */
		s = strchr(buf, ',');

		if (s)
		{
			*s = '\0';
		}
	}

	xfontnames = XListFonts(GDK_WINDOW_XDISPLAY(window), buf, 32767, &fnum);

	if (xfontnames == NULL)
	{
		return -1;
	}

	pxcpy[0] = '\0';
	i = found = helv = 0;

	do
	{
		const gchar *spos;

		spos = tl_strchr_nth(xfontnames[i], '-', kPxSizePosition) + 1;

		if (spos)
		{
			gint smatch = strncmp(spos, pxbuf, pxlen);

			if (smatch <= 0)
			{
				gint		ffm, pmatch;
				const gchar	*fpos;

				ffm	= FALSE;
				pmatch	= strncmp(spos, pxcpy, pxlen);
				fpos	= tl_strchr_nth(xfontnames[i], '-', kFamilyPosition);

				if (fpos)
				{
					static const gchar helvetica_str[] = "-helvetica-";

					if (strncmp(fpos, helvetica_str,
							sizeof(helvetica_str) - 1) == 0)
					{
						if (smatch == 0)
						{
							found = i;
							break;
						}

						if (pmatch > 0 || (pmatch == 0 && helv == FALSE))
						{
							pmatch	= 1;
							ffm	= TRUE;
						}
					}
				}

				if (pmatch > 0)
				{
					found = i;
					helv = ffm;
					strncpy(pxcpy, spos, pxlen);
				}

			} /* if (smatch <= 0) */

		} /* if (spos) */
	}
	while (++i < fnum);

	len = g_snprintf(buf, blen, "%s", xfontnames[found]);
	XFreeFontNames(xfontnames);

	return len;

} /* static gint fonts_get_name(GdkWindow*, gchar*, gint,
					gint, const gchar*, const gchar*) */

GtkStyle *fonts_get_style(GtkCList *clist, const gchar *charset)
{
	/*
		"-*-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
	*/

	static GHashTable *fonts_hash = NULL;

	gint		len, pxsize;
	gchar		*encoding, *registry;
	GtkStyle	*style;
	GdkFont		*font;
	GdkWindow	*window;
	gchar		buf[1024], reg[256];

	const struct FontnameAndLocale *fnl;

	if (charset == NULL)
	{
		charset = charconv_iso8859_1_str;
	}

	if (fonts_hash == NULL)
	{
		fonts_hash = g_hash_table_new(g_str_hash, g_str_equal);
	}

	else if ((style = g_hash_table_lookup(fonts_hash, charset)) != NULL)
	{
		goto _search_font_done_;
	}

#ifdef HAVE_ICONV

	registry = fonts_convert_charset(charset);

	if (registry)
	{
		if ((style = g_hash_table_lookup(fonts_hash, registry)) != NULL)
		{
			goto _search_font_done_;
		}

		charset = registry;
	}

	if (g_strcasecmp(charset, charconv_locale_str) == 0)
	{
		style = gtk_widget_get_style(GTK_WIDGET(clist));
		goto _style_insert_;
	}

#endif /* HAVE_ICONV */

	fnl = fonts_get_fontset(charset);

	if (fnl)
	{
#if defined(HAVE_LOCALE) || defined(HAVE_XLOCALE)
		registry = (gchar *) fnl->fontname;
		encoding = NULL;
#else
		goto _font_not_found_;
#endif
	}

	else
	{
		registry = strchr(charset, '-');

		if (registry)
		{
			/* without '-' */
			len = registry - charset;

			registry = strncpy(reg, charset, len);
			encoding = strchr(&charset[len + 1], '-');

			if (encoding)
			{
				/* "ISO-8859-" => "ISO8859" */
				strncpy(&registry[len], &charset[len + 1],
						encoding - &charset[len + 1]);
				registry[encoding - &charset[1]] = '\0';
				encoding++;
			}

			else if (g_strncasecmp(charset, "windows", len) == 0)
			{
				/* "windows-1250" => "microsoft-cp1250" */
				registry = strcpy(reg, "microsoft-cp");
				strcpy(&reg[sizeof("microsoft-cp") - 1], &charset[len + 1]);
				encoding = NULL;
			}

			else
			{
				/* "KOI8-" => "KOI8*" */
				strcpy(&registry[len], "*");
				encoding = (gchar *) &charset[len + 1];

			} /* if (encoding) .. else if .. else */
		}

		else
		{
			registry = strcpy(reg, charset);
			strcat(registry, "*");
			encoding = "*";

		} /* if (registry) .. else */

	} /* if (registry == NULL) */

	style = gtk_widget_get_style(GTK_WIDGET(clist));
	pxsize = (style->font)->ascent;

	if ((style->font)->descent > 2)
	{
		pxsize += (style->font)->descent / 2;
	}

	else if ((style->font)->descent == 2)
	{
		pxsize += pxsize % 2;
	}

	window = GTK_WIDGET(clist)->window;
	len = fonts_get_name(window, buf, sizeof(buf) - 3,
					pxsize, registry, encoding);
	if (len > 0)
	{

#if defined(HAVE_LOCALE) || defined(HAVE_XLOCALE)

		if (fnl)
		{
#ifdef X_LOCALE
			static gchar *user_locale = NULL;
#else
			static gchar *user_locale = "";
#endif
			while ((registry = strchr(registry, ',')) != NULL)
			{
				gint sz;

				buf[len] = ',';
				len++;
				registry++;

				sz = fonts_get_name(window, &buf[len],
								sizeof(buf) - 2 - len,
									pxsize, registry, NULL);
				if (sz > 0)
				{
					len += sz;
				}
			}

			buf[len] = ',';
			len++;

			len = fonts_get_name(window, &buf[len],
							sizeof(buf) - 1 - len,
								pxsize, "iso8859-1", NULL);
#ifdef X_LOCALE
			if (user_locale == NULL)
			{
				user_locale = g_strdup(setlocale(LC_CTYPE, NULL));
			}
#endif

			if (len <= 0 || setlocale(LC_CTYPE, fnl->locale) == NULL)
			{
				goto _font_not_found_;
			}

			font = gdk_fontset_load(buf);
			setlocale(LC_CTYPE, user_locale);
		}
		else

#endif /* HAVE_LOCALE || HAVE_XLOCALE */
		{
			font = gdk_font_load(buf);
		}

		if (font == NULL)
		{
			goto _font_not_found_;
		}

		style = gtk_style_copy(style);
		style->font = font;
	}

	else
	{
_font_not_found_:
		style = (GtkStyle *) -1;

	} /* if (len > 0) .. else */

#ifdef HAVE_ICONV
_style_insert_:
#endif
	g_hash_table_insert(fonts_hash, g_strdup(charset), style);

_search_font_done_:
	return (style == (GtkStyle *) -1) ? NULL : style;

} /* GtkStyle *fonts_get_style(GtkCList *clist, const char *charset) */

/* EOF */
