/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		modified 2005 ... 2008 by Harald Foerster		   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "charconv.h"
#include "gui.h"
#include "gaccel.h"

/*
#include <gtk/gtk.h>
#include "gui-gtk.h"
#include "nls.h"
#include "nscache.h"
*/

enum
{
	kColumnMenu,
	kColumnKey,
#ifndef CONFIG_GTKCLIST_GTKCTREE
	kColumnRowData,
#endif
	kNumberOfColumns
};


#ifdef CONFIG_GTKCLIST_GTKCTREE
#define AccelListNew(list) \
	GuiListInit((list), kNumberOfColumns)
#else
#define AccelListNew(list) \
{ \
	(list)->slist = \
		gtk_list_store_new (kNumberOfColumns, G_TYPE_STRING, \
					G_TYPE_STRING, G_TYPE_POINTER); \
	GuiListInit((list), kColumnRowData); \
}
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define AccelListAppend(list, str, data) \
{ \
	gint row; \
	row = gtk_clist_append((list)->clist, (str)); \
	gtk_clist_set_row_data((list)->clist, row, (data)); \
}
#else
#define AccelListAppend(list, str, data) \
{ \
	GtkTreeIter iter; \
	gtk_list_store_append((list)->slist, &iter); \
	gtk_list_store_set((list)->slist, &iter, \
		kColumnMenu, (str)[0], kColumnKey, (str)[1], \
				kColumnRowData, (data), -1); \
}
#endif


/* gui.c */
extern void AdjustMenu(GtkWidget *w, gpointer data);

static GtkAccelGroup *gaccel_group = NULL;

static gaccel_t gaccel_tab[kGAccelKeyTotal] =
{
	{ "file/open",		NULL, NULL, 0, 0, 0 },
	{ "file/prop",		NULL, NULL, 0, 0, 0 },
	{ "file/saverc",	NULL, NULL, 0, 0, 0 },
	{ "file/quit",		NULL, NULL, 0, 0, 0 },
	{ "config/col_url",	NULL, NULL, 0, 0, 0 },
	{ "config/col_file",	NULL, NULL, 0, 0, 0 },
	{ "config/col_type",	NULL, NULL, 0, 0, 0 },
	{ "config/col_size",	NULL, NULL, 0, 0, 0 },
	{ "config/col_enc",	NULL, NULL, 0, 0, 0 },
	{ "config/col_charset",	NULL, NULL, 0, 0, 0 },
	{ "config/col_mdtm",	NULL, NULL, 0, 0, 0 },
	{ "config/col_atm",	NULL, NULL, 0, 0, 0 },
	{ "config/col_exp",	NULL, NULL, 0, 0, 0 },
	{ "view/entry_infowin",	NULL, NULL, 0, 0, 0 },
	{ "view/find",		NULL, NULL, 0, 0, 0 },
	{ "edit/save",		NULL, NULL, 0, 0, 0 },
#ifndef CACHE_READONLY
	{ "edit/delete",	NULL, NULL, 0, 0, 0 },
#endif
	{ "edit/clipboard",	NULL, NULL, 0, 0, 0 },
	{ "edit/showinfo",	NULL, NULL, 0, 0, 0 },
	{ "edit/readonly",	NULL, NULL, 0, 0, 0 },
	{ "edit/readwrite",	NULL, NULL, 0, 0, 0 },
	{ "opt/viewers",	NULL, NULL, 0, 0, 0 },
	{ "opt/accel",		NULL, NULL, 0, 0, 0 },
	{ "opt/case",		NULL, NULL, 0, 0, 0 },
#ifndef CACHE_READONLY
	{ "opt/rdonly",		NULL, NULL, 0, 0, 0 },
#endif
	{ "opt/decode",		NULL, NULL, 0, 0, 0 },
#ifdef HAVE_ICONV
	{ "opt/wm_utf8",	NULL, NULL, 0, 0, 0 },
#endif
	{ "opt/save",		NULL, NULL, 0, 0, 0 },
	{ "help/about",		NULL, NULL, 0, 0, 0 }

}; /* static gaccel_t gaccel_tab[kGAccelKeyTotal] */



typedef struct
{
	GuiListStore	list;
	GtkWidget	*window;
	gaccel_t	*rowdata;
	gaccel_t	accel_tab[kGAccelKeyTotal];

} GaccelAssignDialog;

static GaccelAssignDialog *gaccel_dialog = NULL;


static gaccel_t *gaccel_find_by_name(gaccel_t *gaccelp, const gchar *name)
{
	gint loop = kGAccelKeyTotal - 1;

	do
	{
		gaccelp--;

		if (!strcmp(gaccelp->name, name))
		{
			return gaccelp;
		}
	}
	while (--loop > kGAccelKeyNone);

	return NULL;
}

static gaccel_t *gaccel_find_by_keys(gaccel_t *gaccelp, guint keyval,
						GdkModifierType modifier)
{
	gint loop = kGAccelKeyTotal - 1;

	do
	{
		gaccelp--;

		if (keyval == gaccelp->keyval &&
				modifier == gaccelp->modifier)
		{
			return gaccelp;
		}
	}
	while (--loop > kGAccelKeyNone);

	return NULL;
}

static gboolean gaccel_key_pressed(GtkWidget *window,
					GdkEventKey *event, gaccel_t *gaccelp)
{
	if (event->length && (event->state & GDK_CONTROL_MASK))
	{
		gaccelp = gaccel_find_by_keys(gaccelp, event->keyval, event->state);

		if (gaccelp)
		{
			AdjustMenu(gaccelp->widget, NULL);
		}
	}

	return FALSE;
}

static void gaccel_remove_accelerator(const gaccel_t *gaccelp,
						GtkAccelGroup *accel_group)
{
	gtk_accel_group_unlock(accel_group);

	gtk_widget_remove_accelerator(gaccelp->widget, accel_group,
						gaccelp->key, gaccelp->modifier);
	gtk_accel_group_lock(accel_group);
}

static void gaccel_add_accelerator(const gaccel_t *gaccelp,
						GtkAccelGroup *accel_group)
{
	gtk_accel_group_unlock(accel_group);

	gtk_widget_add_accelerator(gaccelp->widget, "activate", accel_group,
				gaccelp->key, gaccelp->modifier,  GTK_ACCEL_VISIBLE);

	gtk_accel_group_lock(accel_group);
}

void gaccel_bind_widget(gint accel_id, const gchar *label, GtkWidget *widget, GtkWidget *window)
{
	GtkAccelGroup	*accel_group;
	gaccel_t	*gaccelp;

	if (accel_id == kGAccelKeyNone || widget == NULL)
	{
		return;
	}

	accel_group = gaccel_group;

	if (accel_group == NULL)
	{

#if GTK_MAJOR_VERSION == 1

		accel_group = gtk_accel_group_get_default();
		gtk_accel_group_lock(accel_group);
#endif

		accel_group  = gtk_accel_group_new();
		gaccel_group = accel_group;
		gtk_accel_group_lock(accel_group);

		gtk_window_add_accel_group(GTK_WINDOW(window), accel_group);

		GuiSignalConnect(window, "key-press-event",
			gaccel_key_pressed, &gaccel_tab[kGAccelKeyTotal]);
	}

	gaccelp = &gaccel_tab[accel_id];

	gaccelp->label	= label;
	gaccelp->widget	= widget;

	if (gaccelp->key)
	{
		gaccel_add_accelerator(gaccelp, accel_group);
	}
}

static void gaccel_dialog_destroy(GtkWidget *w, GaccelAssignDialog *acceldlg)
{
	g_free(acceldlg);
	gaccel_dialog = NULL;
}

static void gaccel_dialog_okay(GtkWidget *w, GaccelAssignDialog *acceldlg)
{
	gint	 loop;
	gaccel_t *old, *mod;

	old	= gaccel_tab;
	mod 	= acceldlg->accel_tab;
	loop	= 0;
	
	do
	{
		if (old->keyval != mod->keyval || old->modifier != mod->modifier)
		{
			if (old->keyval)
			{
				gaccel_remove_accelerator(old, gaccel_group);
			}
		}

		old++;
		mod++;
	}
	while (++loop < kGAccelKeyTotal);

	old	= gaccel_tab;
	mod 	= acceldlg->accel_tab;
	loop	= 0;
	
	do
	{
		if (old->keyval != mod->keyval || old->modifier != mod->modifier)
		{
			if (mod->keyval)
			{
				gaccel_add_accelerator(mod, gaccel_group);
			}

			old->keyval   = mod->keyval;
			old->key      = mod->key;
			old->modifier = mod->modifier;
		}

		old++;
		mod++;
	}
	while (++loop < kGAccelKeyTotal);

	gtk_widget_destroy(acceldlg->window);
}

static gboolean gaccel_dialog_key(GtkWidget *window,
			GdkEventKey *event, GaccelAssignDialog *acceldlg)
{
	gaccel_t *gaccelp;

	if (event->keyval == GDK_BackSpace)
	{
		gaccelp = acceldlg->rowdata;

		if (gaccelp->keyval)
		{
			GuiListSetText(&(acceldlg->list), kColumnKey, NULL);

			gaccelp->keyval   = 0;
			gaccelp->key      = 0;
			gaccelp->modifier = 0;
		}

		return TRUE;
	}

	else if (event->length && (event->state & GDK_CONTROL_MASK))
	{
		if (gaccel_find_by_keys(&acceldlg->accel_tab[kGAccelKeyTotal],
							event->keyval, event->state))
		{
			gdk_beep();
		}

		else
		{
			gint key = event->keyval;

			if (event->state & GDK_SHIFT_MASK)
			{
				key = gdk_keyval_to_lower(key);
			}

			if (gtk_accelerator_valid(key, event->state))
			{
				gchar *name = gtk_accelerator_name(key, event->state);

				if (name)
				{
					gchar *str = CHARCONV_UTF8_FROM_LOCALE(name);
#if GTK_MAJOR_VERSION > 1
					if (str == NULL)
					{
						str = tl_replace_non_ascii_chars_in_str(name);
					}
#endif
					GuiListSetText(&(acceldlg->list), kColumnKey, str);
					CHARCONV_UTF8_FREE(str, name);
					g_free(name);

					gaccelp = acceldlg->rowdata;

					gaccelp->keyval   = event->keyval;
					gaccelp->key      = key;
					gaccelp->modifier = event->state;
				}
			}
		}

		return TRUE;
	}

	return FALSE;
}


#ifdef CONFIG_GTKCLIST_GTKCTREE

static void gaccel_dialog_row_selected(GtkCList* clist, gint row, gint col,
			GdkEventButton *e, GaccelAssignDialog *acceldlg)
{
	acceldlg->list.row = row;
	acceldlg->rowdata  = &(acceldlg->accel_tab[row]);
}

#else /* ! CONFIG_GTKCLIST_GTKCTREE */

static void gaccel_dialog_row_selected(GtkTreeSelection *selection, GaccelAssignDialog *acceldlg)
{
	GtkTreeIter	iter;
	GtkTreeModel	*model;

	if (gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		acceldlg->list.iter = iter;
		gtk_tree_model_get(model, &iter, kColumnRowData, &(acceldlg->rowdata), -1);
	}
}

#endif /* #ifdef CONFIG_GTKCLIST_GTKCTREE .. #else */

static void gaccel_build_column(GuiListStore *list, gint colnr, gint height, gchar *title)
{
	gchar *str;

	GuiListColumnTextNew(list, colnr, height, GTK_JUSTIFY_LEFT);
	str = CHARCONV_UTF8_GETTEXT(title);
	GuiListSetColumnTitle(list, colnr, str);
	CHARCONV_UTF8_DESTROY(str);
	GuiListSetColumnAutoResize(list, colnr, TRUE);
}

void gaccel_dialog_open(void)
{
	gint		height, loop;
	GtkWidget	*button, *frame, *hbox, *label, *swin, *vbox, *window;

	GaccelAssignDialog *acceldlg = gaccel_dialog;

	if (acceldlg)
	{
		GuiWindowPresent(acceldlg->window);
		return;
	}

	acceldlg = g_malloc(sizeof(GaccelAssignDialog));
	gaccel_dialog = acceldlg;

	vbox = gtk_vbox_new(FALSE, 2);
	window = guitl_window_new_gettext(gettext_nop("MozCache: Keyboard"), vbox, 4);
	acceldlg->window = window;

	GuiSignalConnect(window, "destroy", gaccel_dialog_destroy, acceldlg);
	GuiSignalConnect(window, "key-press-event", gaccel_dialog_key, acceldlg);

	swin = gtk_scrolled_window_new(NULL, NULL);
	gtk_container_add(GTK_CONTAINER(vbox), swin);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (swin),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	GuiWidgetSetSizeRequest(swin, 320, 180);

	AccelListNew(&(acceldlg->list));
	GuiListContainerAdd(&(acceldlg->list), swin);

	height = guitl_list_calculate_row_height(&(acceldlg->list));
	GuiListSetRowHeight(&(acceldlg->list), height);

	gaccel_build_column(&(acceldlg->list), kColumnMenu, height, gettext_nop("Menu"));
	gaccel_build_column(&(acceldlg->list), kColumnKey,  height, gettext_nop("Key"));

	memcpy(acceldlg->accel_tab, gaccel_tab, sizeof(gaccel_tab));

	loop = 0;

	do
	{
		gchar *str, *p[2];

		p[0] = CHARCONV_UTF8_GETTEXT(acceldlg->accel_tab[loop].label);
		str = gtk_accelerator_name(acceldlg->accel_tab[loop].key,
						acceldlg->accel_tab[loop].modifier);
		p[1] = CHARCONV_UTF8_FROM_LOCALE(str);

#if GTK_MAJOR_VERSION > 1
		if (p[1] == NULL)
		{
			p[1] = tl_replace_non_ascii_chars_in_str(str);
		}
#endif
		AccelListAppend(&(acceldlg->list), p, &(acceldlg->accel_tab[loop]));

		CHARCONV_UTF8_DESTROY(p[0]);
		CHARCONV_UTF8_FREE(p[1], str);
		g_free(str);
	}
	while (++loop < kGAccelKeyTotal);

	GuiListColumnTitlesShow(&(acceldlg->list));
	GuiListColumnTitlesPassive(&(acceldlg->list));
	GuiListSetSelectionMode(&(acceldlg->list), GTK_SELECTION_BROWSE);

	GuiListSignalConnectSelectedRow(&(acceldlg->list),
				gaccel_dialog_row_selected, acceldlg);

	label = guitl_label_new_gettext(
			gettext_nop("Select a menu entry and press your prefered keys,"
							" e.g. Quit => 'Cntrl' + 'Q'"));
	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);

	frame = guitl_frame_new_add_child_gettext(NULL, label);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 1);

	hbox = guitl_hbutton_box_new(vbox, 8, GTK_BUTTONBOX_END);

	button = guitl_button_box_add_icon_button(hbox, kIconButtonCancel);
	GuiSignalConnectSwapped(button, "clicked",
			gtk_widget_destroy, acceldlg->window);

	button = guitl_button_box_add_icon_button(hbox, kIconButtonOkay);
	GuiSignalConnect(button, "clicked", gaccel_dialog_okay, acceldlg);
	gtk_widget_grab_default(button);

	GuiListSelectFirstRow(&(acceldlg->list));
	gtk_widget_show_all(acceldlg->window);
}

gint gaccel_parse_str(gchar *str)
{
	gchar *name = tl_get_1qstr(str);

	if (name)
	{
		gchar *accelerator = tl_get_1qstr(NULL);

		if (accelerator)
		{
			GdkModifierType	modifier;
			guint		accel_key;

			modifier  = 0;
			accel_key = 0;

			gtk_accelerator_parse(accelerator, &accel_key, &modifier);

			if (accel_key != 0 && (modifier & GDK_CONTROL_MASK))
			{
				gaccel_t *gaccelp;

				gaccelp = gaccel_find_by_name(&gaccel_tab[kGAccelKeyTotal], name);

				if (gaccelp)
				{
					gaccelp->modifier = modifier;
					gaccelp->key = accel_key;

					gaccelp->keyval = (modifier & GDK_SHIFT_MASK) ?
						gdk_keyval_to_upper(accel_key) : accel_key;

					return TRUE;
				}
			}
		}
	}

	return FALSE;
}

void gaccel_save_data(FILE *fd, const gchar *id)
{
	gaccel_t *gaccelp = gaccel_tab;

	gint loop = kGAccelKeyTotal - 1;

	do
	{
		if (gaccelp->key)
		{
			gchar *p = gtk_accelerator_name(gaccelp->key, gaccelp->modifier);

			if (p)
			{
				fprintf(fd, "%s \"%s\" \"%s\"\n", id, gaccelp->name, p);
				g_free(p);
			}
		}

		gaccelp++;
	}
	while (--loop > kGAccelKeyNone);
}

/* EOF */
