/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		modified 2005 ... 2008 by Harald Foerster		   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include <glib.h>
#include <string.h>
#include <stdlib.h>

#include "gprop.h"

enum
{
	GPROP_FIRST,
	GPROP_STR  = GPROP_FIRST,
	GPROP_INT,
	GPROP_BOOL,
	GPROP_LAST = GPROP_BOOL
};

typedef struct
{
	int	item;
	void	*value;

} gprop_t;

static GSList *gprop_list[3] = {NULL};

static const char* gprop_type_tab[] =
{
	"str.",
	"int.",
	"bool."
};

static const char* gprop_item_str[] =
{
	"find/pattern",
	"find/mime_type",
	"save/path",

	NULL
};

static const char* gprop_item_int[] =
{
	"tl_width",
	"tl_height",

	"sort_type",

	"tcol_url_width",
	"tcol_file_width",
	"tcol_type_width",
	"tcol_size_width",
	"tcol_enc_width",
	"tcol_charset_width",
	"tcol_mdtm_width",
	"tcol_atm_width",
	"tcol_exp_width",

	"lcol_url_width",
	"lcol_file_width",
	"lcol_type_width",
	"lcol_size_width",
	"lcol_enc_width",
	"lcol_charset_width",
	"lcol_mdtm_width",
	"lcol_atm_width",
	"lcol_exp_width",

	"url_colnr",
	"file_colnr",
	"type_colnr",
	"size_colnr",
	"enc_colnr",
	"charset_colnr",
	"mdtm_colnr",
	"atm_colnr",
	"exp_colnr",

	"find/width",
	"find/height",
	"find/rcol_url_width",
	"find/rcol_found_width",
	"find/rcol_file_width",
	"find/rcol_line_width",

	"info/width",
	"info/height",

	"save/url_to_local",

	NULL
};

static const char* gprop_item_bool[] =
{
	"see_col_url",
	"see_col_file",
	"see_col_type",
	"see_col_size",
	"see_col_enc",
	"see_col_charset",
	"see_col_mdtm",
	"see_col_atm",
	"see_col_exp",

	"opt/fname_to_lower",
#ifndef CACHE_READONLY
	"opt/readonly",
#endif
	"opt/decode",
#ifdef HAVE_ICONV
	"opt/wm_utf8",
#endif
	"opt/save_config",

	"find/case_sensitive",
	"find/full_match",
#ifdef HAVE_REGEX
	"find/with_reg_exp",
#endif

#ifdef HAVE_LINK
	"save/hardlink",
#endif
	"save/single_folder",

	NULL
};

static const char* const* gprop_item_tab[] =
{
	gprop_item_str,
	gprop_item_int,
	gprop_item_bool
};

static const char gprop_false_str[] = "false";
static const char gprop_true_str[]  = "true";

static int	gprop_find(const gprop_t *gp, gconstpointer data);
static int	gprop_sort(const gprop_t *gp, const gprop_t *data);
static gprop_t*	gprop_get_entry(int item, int type);
static void 	gprop_set_value(int item, void *val, int type);
static int	gprop_get_value(int item, void *valp, int type);

static int gprop_find(const gprop_t *gp, gconstpointer data)
{
	int item = (int) data;

	return gp->item - item;
}

static int gprop_sort(const gprop_t *gp, const gprop_t *data)
{
	return gp->item - data->item;
}

static gprop_t *gprop_get_entry(int item, int type)
{
	GSList	*node;

	node = g_slist_find_custom(gprop_list[type],
			(gpointer) item, (GCompareFunc) gprop_find);

	return node == NULL ? (gprop_t *) node : (gprop_t *) node->data;
}

static void gprop_set_value(int item, void *val, int type)
{
	gprop_t *gp = gprop_get_entry(item, type);

	if (gp)
	{
		if (type == GPROP_STR)
		{
			g_free(gp->value);
		}
	}

	else
	{
		GSList **lst;

		gp = g_malloc(sizeof(gprop_t));

		gp->item = item;

		 lst = &gprop_list[type];
		*lst = g_slist_insert_sorted(*lst, gp, (GCompareFunc) gprop_sort);
	}

	gp->value = type == GPROP_STR ? g_strdup(val) : val;
}

void gprop_set_str(int item, const char *sval)
{
	if ((unsigned int) item <= kPropStrItemLast)
	{
		gprop_set_value(item, (void *) sval, GPROP_STR);
	}
}

void gprop_set_int(int item, int ival)
{
	if ((unsigned int) item <= kPropIntItemLast)
	{
		gprop_set_value(item, (void *) ival, GPROP_INT);
	}
}

void gprop_set_bool(int item, int bval)
{
	if ((unsigned int) item <= kPropBoolItemLast)
	{
		gprop_set_value(item, (void *) bval, GPROP_BOOL);
	}
}

static int gprop_get_value(int item, void *valp, int type)
{
	gprop_t *gp = gprop_get_entry(item, type);

	if (gp == NULL)
	{
		return FALSE;
	}

	if (sizeof(char*) == sizeof(int))
	{
		int *p = (int *) valp;

		*p = (int) gp->value;
	}

	else
	{
		if (type == GPROP_STR)
		{
			char **s = (char **) valp;

			*s = (char *) gp->value;
		}

		else
		{
			int *p = (int *) valp;

			*p = (int) gp->value;
		}
	}

	return TRUE;
}

int gprop_get_str(int item, char **svalp)
{
	return gprop_get_value(item, (void *) svalp, GPROP_STR);
}

int gprop_get_int(int item, int *ivalp)
{
	return gprop_get_value(item, (void *) ivalp, GPROP_INT);
}

int gprop_get_bool(int item, int *bvalp)
{
	return gprop_get_value(item, (void *) bvalp, GPROP_BOOL);
}

int gprop_parse_str(char *str)
{
	int type = GPROP_LAST;

	do
	{
		const char	*ptr;
		int		len;

		ptr	= gprop_type_tab[type];
		len	= strlen(ptr);

		if (strncmp(str, ptr, len) == 0)
		{
			int			item;
			const char* const*	itab;
			void			*value;

			str += len;

			len = strcspn(str, "=");

			if (len == 0)
			{
				break;
			}

			itab = gprop_item_tab[type];
			item = 0;

			while (*itab)
			{
				if (strncmp(str, *itab, len) == 0)
				{
					break;
				}

				itab++;
				item++;
			}

			if (*itab == NULL)
			{
				break;
			}

			str += len + 1;

			if (*str == '\0')
			{
				break;
			}

			if (type == GPROP_STR)
			{
				value = (void *) str;
			}

			else if (type == GPROP_INT)
			{
				value = (void *) atoi(str);
			}

			else
			{
				if (strcmp(str, gprop_false_str) == 0)
				{
					value = (void *) FALSE;
				}

				else if (strcmp(str, gprop_true_str) == 0)
				{
					value = (void *) TRUE;
				}

				else
				{
					break;
				}
			}

			gprop_set_value(item, value, type);

			return TRUE;

		} /* if (strncmp(str, p, len) == 0) */
	}
	while (--type >= GPROP_FIRST);

	return FALSE;
}

static void gprop_dump(gprop_t *gp, gpointer data)
{
	struct
	{
		FILE		*fd;
		const char	*id;
		int		type;

	} *s;

	const char* const* p;

	s = data;

	if (s->type == GPROP_STR && gp->value == NULL)
	{
		return;
	}

	p = gprop_item_tab[s->type];

	fprintf(s->fd, "%s %s%s=", s->id, gprop_type_tab[s->type], p[gp->item]);

	switch (s->type)
	{
		case GPROP_STR:
			if (gp->value)
			{
				fprintf(s->fd, "%s\n", (char *) gp->value);
			}
			break;

		case GPROP_INT:
			fprintf(s->fd, "%d\n", (int) gp->value);
			break;

		default: /* GPROP_BOOL */
			fprintf(s->fd, "%s\n", (gp->value ?
					gprop_true_str : gprop_false_str));
			break;
	}
}

void gprop_save_data(FILE *fd, const char *id)
{
	struct
	{
		FILE		*fd;
		const char	*id;
		int		type;
	} s;

	s.fd   = fd;
	s.id   = id;
	s.type = GPROP_FIRST;

	do
	{
		g_slist_foreach(gprop_list[s.type], (GFunc) gprop_dump, &s);
	}
	while (++s.type <= GPROP_LAST);
}

/* EOF */
