/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		modified 2005 ... 2008 by Harald Foerster		   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

#include "gui.h"
#include "apassign.h"
#include "file.h"
#include "gaccel.h"
#include "gprop.h"
#include "utf8-tools.h"
#include "viewer.h"

#if GTK_MAJOR_VERSION == 1 || defined(HAVE_GLOB)
#include <gdk/gdkkeysyms.h>
#endif

#ifdef HAVE_MSIE
#include "msie.h"
#endif

/*
#include <stdio.h>
#include <time.h>
#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "gui-gtk.h"
#include "nls.h"
#include "charconv.h"
#include "nscache.h"
#include "stringbuf.h"
*/

enum
{
	kColumnURL,
	kColumnFile,
	kColumnType,
	kColumnSize,
	kColumnEncoding,
	kColumnCharset,
	kColumnModify,
	kColumnAccess,
	kColumnExpire,
	kColumnsTotal,
#ifdef CONFIG_GTKCLIST_GTKCTREE
	kNumberOfColumns = kColumnsTotal,
#else
	kColumnRowData = kColumnsTotal,
	kColumnPixMime,
	kColumnPixClose = kColumnPixMime,
	kColumnPixOpen,
	kNumberOfColumns,
#endif
	kColumnInfoLastIndex = kColumnsTotal - 1
};



#ifdef CONFIG_SWAP_COLUMNS

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define MainSetColumnEntryArray(array, colnr, str) \
	(array)[column_tab_array[colnr].position] = (str)
#else
#define MainSetColumnEntryArray(array, colnr, str) \
	(array)[column_tab_array[colnr].initcolnr] = (str)
#endif

#else /* ! CONFIG_SWAP_COLUMNS */

#define MainSetColumnEntryArray(array, colnr, str) \
	(array)[colnr] = (str)

#endif /* #ifdef CONFIG_SWAP_COLUMNS .. #else */

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define MainSetColumnTitle(nbook_page, index) \
{ \
	GtkWidget *label = \
	guitl_label_new_gettext(column_info[index].label); \
	if ((nbook_page) == kTreeList) \
	{ \
		GuiTreeSetColumnWidget( \
			&(notebook_list[kTreeList].store.tree), \
					(index), label); \
	} \
	else \
	{ \
		GuiListSetColumnWidget( \
			&(notebook_list[kSortedList].store.list), \
					(index), label); \
	} \
}
#else /* ! CONFIG_GTKCLIST_GTKCTREE */
#define MainSetColumnTitle(nbook_page, index) \
{ \
	gchar *str = CHARCONV_UTF8_GETTEXT(column_info[index].label); \
	if ((nbook_page) == kTreeList) \
	{ \
		GuiTreeSetColumnTitle( \
			&(notebook_list[kTreeList].store.tree), \
						(index), str); \
	} \
	else \
	{ \
		GuiListSetColumnTitle( \
			&(notebook_list[kSortedList].store.list), \
						(index), str); \
	} \
	CHARCONV_UTF8_DESTROY(str); \
}
#endif /* #ifdef CONFIG_GTKCLIST_GTKCTREE .. #else */

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define MainListNew(list) \
	GuiListInit((list), kNumberOfColumns)
#else
#define MainListNew(list) \
{ \
	(list)->slist = \
		gtk_list_store_new (kNumberOfColumns - 1, \
			G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, \
			G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, \
			G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, \
			G_TYPE_POINTER, GDK_TYPE_PIXBUF); \
	GuiListInit((list), kColumnRowData); \
}
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define MainListAppend(list, str, icon, data) \
{ \
	gint row; \
	row = gtk_clist_append((list)->clist, (str)); \
	gtk_clist_set_pixtext((list)->clist, row, 0, (str)[0], \
			3, (icon)->pixmap, (icon)->mask); \
	gtk_clist_set_row_data((list)->clist, row, (data)); \
}
#else
#define MainListAppend(list, str, icon, data) \
{ \
	GtkTreeIter iter; \
	gtk_list_store_append((list)->slist, &iter); \
	gtk_list_store_set((list)->slist, &iter, \
		kColumnURL, (str)[0], kColumnFile, (str)[1], \
		kColumnType, (str)[2], kColumnSize, (str)[3], \
		kColumnEncoding, (str)[4], kColumnCharset, (str)[5], \
		kColumnModify, (str)[6], kColumnAccess, (str)[7], \
		kColumnExpire, (str)[8], \
		kColumnPixMime, (icon)->pixbuf, kColumnRowData, (data), -1); \
}
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define MainTreeNew(tree) \
	GuiTreeInit((tree), kNumberOfColumns, kColumnURL)
#else
#define MainTreeNew(tree) \
{ \
	(tree)->stree = \
		gtk_tree_store_new (kNumberOfColumns, \
			G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, \
			G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, \
			G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, \
			G_TYPE_POINTER, GDK_TYPE_PIXBUF, GDK_TYPE_PIXBUF); \
	GuiTreeInit((tree), kColumnRowData, kColumnPixMime); \
}
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define MainTreeAppendNode(tree, parent, str, icon_close, icon_open, child) \
{ \
	char *centry[9]; \
	centry[0] = (str); \
	centry[1] = centry[2] = centry[3] = centry[4] = \
	centry[5] = centry[6] = centry[7] = centry[8] = NULL; \
	(child) = gtk_ctree_insert_node((tree)->ctree, (parent), NULL, \
				centry, 8, \
				(icon_close).pixmap, (icon_close).mask, \
				(icon_open).pixmap, (icon_open).mask, \
				FALSE, ((parent) == NULL)); \
}
#else
#define MainTreeAppendNode(tree, parent, str, icon_close, icon_open, child) \
{ \
	gtk_tree_store_append((tree)->stree, (child), (parent)); \
	gtk_tree_store_set ((tree)->stree, (child), kColumnURL, (str), \
				kColumnPixClose, (icon_close).pixbuf, \
				kColumnPixOpen, (icon_open).pixbuf, -1); \
}
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define MainTreeInsertNode(tree, parent, str, icon, data) \
{ \
	GtkCTreeNode *node; \
	node = gtk_ctree_insert_node((tree)->ctree, (parent), NULL, \
			(str), 8, (icon)->pixmap, (icon)->mask, \
			(icon)->pixmap, (icon)->mask, FALSE, FALSE); \
	gtk_ctree_node_set_row_data((tree)->ctree, node, (data)); \
}
#else
#define MainTreeInsertNode(tree, parent, str, icon, data) \
{ \
	GtkTreeIter iter; \
	gtk_tree_store_append((tree)->stree, &iter, (parent)); \
	gtk_tree_store_set((tree)->stree, &iter, \
		kColumnURL, (str)[0], kColumnFile, (str)[1], \
		kColumnType, (str)[2], kColumnSize, (str)[3], \
		kColumnEncoding, (str)[4], kColumnCharset, (str)[5], \
		kColumnModify, (str)[6], kColumnAccess, (str)[7], \
		kColumnExpire, (str)[8], \
		kColumnPixMime, (icon)->pixbuf, kColumnRowData, (data), -1); \
}
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define MainTreeExpandTopLevelNodes(tree) {;}
#else
#define MainTreeExpandTopLevelNodes(tree) \
{ \
	GtkTreeIter iter; \
	GtkTreeModel *model = GTK_TREE_MODEL((tree)->stree); \
	gint row = -1; \
	while (gtk_tree_model_iter_nth_child(model, &iter, NULL, ++row)) \
	{ \
		GtkTreePath *path = gtk_tree_model_get_path(model, &iter); \
		gtk_tree_view_expand_row((tree)->view,  path,  FALSE); \
		gtk_tree_path_free(path); \
	} \
}
#endif

static const char invalid_filename_str[] = gettext_nop("Invalid filename!");

#if GTK_MAJOR_VERSION == 1
#define gui_fname_error(fname)	gui_strerror(fname)
#endif

#if GTK_MAJOR_VERSION == 2

/*
	----------------------------------------------------
	 Problems using GTK+-2.2.10 with WindowMaker 0.91.0
	   and XFree86 Version 3.3.6 Protocol Version 11
	----------------------------------------------------

	Sometimes no 'focus-in-event' was received. So I use
	for GTK+-2.0 the 'property-notify-', 'map-', 'unmap-'
	and 'destroy' events, too.
	This seems to work well, but requires two additional
	lists of the window order (mapped and unmapped).
	Maybe this is buggy using other configurations ...
*/

#ifdef FOCUS_IN_BUGFIX
#define _NET_WM_STATE		"_NET_WM_STATE"
#endif

#endif /* GTK_MAJOR_VERSION == 2 */


#define kMessageBufferSize	4096

enum
{
	SEL_TYPE_FIRST		= 1,
	SEL_TYPE_STRING		= SEL_TYPE_FIRST,
	SEL_TYPE_TEXT,
	SEL_TYPE_COMPOUND_TEXT,
	SEL_TYPE_LAST		= SEL_TYPE_COMPOUND_TEXT
};


static const GtkTargetEntry targetlist[] =
{
	{ "STRING", 0, SEL_TYPE_STRING },
	{ "TEXT", 0, SEL_TYPE_TEXT },
	{ "COMPOUND_TEXT", 0, SEL_TYPE_COMPOUND_TEXT }
};


typedef struct
{
	ListSelection	*toplist;	/* list in topwin */
	GtkWidget	*topwin;	/* window with focus */
	GtkWidget	*mainwin;	/* this (gui.c) window */
#ifdef FOCUS_IN_BUGFIX
	GdkAtom		_net_wm_state;	/* seems to be WM  top window property */
	GSList		*ismapped;	/* order of mapped windows (top first) */
	GSList		*unmapped;	/* reverse order of unmapped windows */
#endif

} Windata;

static Windata	windata;

/* both kTreeList and kSortedList */
ListSelection	notebook_list[kNotebookPages];

/* either kTreeList or kSortedList */
ListSelection	*mainlist;

GSList *list_selection_slist = NULL;
GSList *combo_mime_slist = NULL;

static GtkEntry  *entry_db;
static GtkWidget *nscache_type;

static char *selection_urlstr = NULL;
static gboolean gui_is_active = FALSE;

typedef struct
{
#ifdef CONFIG_GTKCLIST_GTKCTREE
	GtkCTreeNode	*node;
#else
	GtkTreeIter	iter;
#endif
	gint		slen;
	gchar		str[1];

} NodeHash;

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define NodeHashGetNodePointer(nhash) \
		(((NodeHash *) nhash)->node)
#else
#define NodeHashGetNodePointer(nhash) \
		&(((NodeHash *) nhash)->iter)
#endif

typedef struct
{
	GNode		*folder_node;
	GtkProgressBar	*progress_bar;
	GSList		*slist;
	guint		processed;
	guint		total;

} OpenData;

typedef struct
{
	GtkWidget	*dialog;

	enum
	{
		kInfoDialogRec,
		kInfoDialogNode

	} type;

	union
	{
		gpointer	rec;
		gpointer	node;
	} u;

} InfoDialogEntry;

GSList *info_dialog_slist = NULL;

struct msg_tmr
{
	guint		tmr_id;
	guint		sig_id;
	gint		width;
	gboolean	do_beep;
	gchar		*str;
	GtkLabel	*label;
	GtkStyle	*style;
	GtkWidget	*frame;
};

static struct msg_tmr status_msg = { 0 };
static struct msg_tmr error_msg  = { 0 };

/* block key and button events while opening a cache database */
guint msg_button_sig_id, msg_key_sig_id;

enum
{
	kNodeInfoFormatSingleLine,
	kNodeInfoFormatMultiLine
};

struct node_info
{
	unsigned int	servicenum;
	unsigned int	domainnum;
	unsigned int	foldernum;
	unsigned int	filenum;
	unsigned long	contentsize;
};

typedef struct
{
	GtkWidget	*left;
	GtkWidget	*right;

} ColumnArrows;

typedef struct
{
	gint		index;
	gint		position;
	gboolean	visible;
	GtkWidget	*menu;
#ifdef CONFIG_GTKCLIST_GTKCTREE
#ifdef CONFIG_COLUMN_BUTTONS
	ColumnArrows	arrow[kNotebookPages];
#endif
#else
#ifdef CONFIG_COLUMN_POPUP
	gint		initcolnr;
#endif
#endif

} ColumnTab;


#ifdef CONFIG_SWAP_COLUMNS

typedef struct
{
#ifdef CONFIG_COLUMN_POPUP
	GtkWidget *menu;
#else
	gboolean  is_active;
#endif
	GtkWidget *collapse;
	GtkWidget *expand;

} CollapseExpandWidgets;

#ifdef CONFIG_COLUMN_BUTTONS
static CollapseExpandWidgets button_collexp;
#endif

#ifdef CONFIG_COLUMN_POPUP

static CollapseExpandWidgets popup_collexp;

static struct popup_colswap
{
	GtkWidget *menu;
	GtkWidget *left;
	GtkWidget *right;
	ColumnTab *selected;

} popup_colswap;

#endif /* CONFIG_COLUMN_POPUP */

#endif /* CONFIG_SWAP_COLUMNS */


typedef struct
{
	gchar		*label;
	gint		 show_column_prop;
	gint		 tree_width_prop;
	gint		 list_width_prop;
	gint		 colnrprop;
	gint		 accel_id;
	gint  		 dwidth;
	GtkJustification justify;

} ColumnInfo;

static ColumnTab	column_tab_array[kColumnsTotal];
static ColumnTab	*column_tab[kColumnsTotal];

static const ColumnInfo column_info[kColumnsTotal] =
{
	{gettext_nop("Cache URL entry"),
		kPropBoolShowColUrl, kPropIntTreeColWidthUrl,
		kPropIntListColWidthUrl, kPropIntColNumUrl,
		kGAccelColumnURL, 200, GTK_JUSTIFY_LEFT},
	{gettext_nop("Local file"),
		kPropBoolShowColFile, kPropIntTreeColWidthFile,
		kPropIntListColWidthFile, kPropIntColNumFile,
		kGAccelColumnFile, 150, GTK_JUSTIFY_LEFT}, 
	{gettext_nop("Type"),
		kPropBoolShowColType, kPropIntTreeColWidthType,
		kPropIntListColWidthType, kPropIntColNumType,
		kGAccelColumnType, 90, GTK_JUSTIFY_LEFT},
	{gettext_nop("Size"),
		kPropBoolShowColSize, kPropIntTreeColWidthSize,
		kPropIntListColWidthSize, kPropIntColNumSize,
		kGAccelColumnSize, 70, GTK_JUSTIFY_RIGHT},
	{gettext_nop("Encoding"),
		kPropBoolShowColEnc, kPropIntTreeColWidthEnc,
		kPropIntListColWidthEnc, kPropIntColNumEnc,
		kGAccelColumnEncoding, 90, GTK_JUSTIFY_LEFT},
	{gettext_nop("Charset"),
		kPropBoolShowColCharset, kPropIntTreeColWidthCharset,
		kPropIntListColWidthCharset, kPropIntColNumCharset,
		kGAccelColumnCharset, 90, GTK_JUSTIFY_LEFT},
	{gettext_nop("Mod. time"),
		kPropBoolShowColMdtm, kPropIntTreeColWidthMdtm,
		kPropIntListColWidthMdtm, kPropIntColNumMdtm,
		kGAccelColumnMod, 90, GTK_JUSTIFY_LEFT},
	{gettext_nop("Access time"),
		kPropBoolShowColAtm, kPropIntTreeColWidthAtm,
		kPropIntListColWidthAtm, kPropIntColNumAtm,
		kGAccelColumnAcc, 90, GTK_JUSTIFY_LEFT},
	{gettext_nop("Expires"),
		kPropBoolShowColExp, kPropIntTreeColWidthExp,
		kPropIntListColWidthExp, kPropIntColNumExp,
		kGAccelColumnExp, 90, GTK_JUSTIFY_LEFT}

}; /* static const ColumnInfo column_info[kColumnsTotal] */

/*
	-----------------------
	 CHECK BUTTONS IN MENU
	-----------------------
*/

enum
{
	kOptionConvertToLower,
#ifndef CACHE_READONLY
	kOptionReadOnly,
#endif
	kOptionDecode,
#ifdef HAVE_ICONV
	kOption_WM_UTF8,
#endif
	kOptionSaveConfigOnExit,
	kOptionEntriesTotal
};

static struct gui_option
{
	gint		active;
	gint		prop;
	GtkWidget	*widget;

} gui_option[] = {

	{ FALSE, kPropBoolOptFnameToLower, NULL },
#ifndef CACHE_READONLY
	{ FALSE, kPropBoolOptReadOnly, NULL },
#endif
	{  TRUE, kPropBoolOptDecode, NULL },
#ifdef HAVE_ICONV
	{  TRUE, kPropBoolOpt_WM_UTF8, NULL },
#endif
	{  TRUE, kPropBoolOptSaveConfig, NULL }
};

/*
	------------------------------------------------
	 enable/disable CheckButton in SaveFolderDialog
	------------------------------------------------
*/

#ifdef HAVE_LINK
static gint hardlink_enable;
#endif


/*
	------------
	 PROTOTYPES
	------------
*/

/* File Menu */
static void Browse(GtkWidget *w, GtkWidget **mbar);
static void PropertiesDialog(GtkWidget *w, GtkWidget **mbar);
static void SaveRC(GtkWidget *w, gpointer data);
static gint Quit(GtkWidget *w);

/* View Menu */
static void ViewInfo(GtkWidget *w, ListSelection **selected);
static void GenViewerMenu(GtkWidget *submenu, gpointer data);

/* Edit Menu */
static void SaveAs(GtkWidget *w, GtkWidget **mbar);

#ifndef CACHE_READONLY
static void DeleteDialog(GtkWidget *w, GtkWidget **mbar);
#endif

static void SelectionSetURL(GtkWidget *w, ListSelection **selected);
static void NodeInfo(GtkWidget *w, ListSelection **selected);
static void RdOnly(GtkWidget *w, ListSelection **selected);
static void Writable(GtkWidget *w, ListSelection **selected);

/* Options Menu */
static void ToggleBool(GtkWidget *w, struct gui_option *opt);

/* Help Menu */
static void About(GtkWidget *w, gpointer data);

/* Misc Prototypes */
static gint gui_read_cache(gchar *dbname);
static gint gui_selected_node_to_url(gchar *buf, gint max, GuiTreeStore *tree);
static gint msg_clr(struct msg_tmr *msg);
static void gui_msg(const gchar *str);
static void gui_window_resize(GtkWidget *w, GtkAllocation *sallocation, gpointer data);

#ifndef CACHE_READONLY
static void info_dialog_remove_entry(gconstpointer rec_or_node);
#endif

#ifdef CONFIG_SWAP_COLUMNS

static void
gui_test_expand_recursive(GuiTreeStore *tree, nscache_record_t *rec, gboolean *can_expand);
static void
gui_collapse_expand_set_sensitive(ListSelection *list, CollapseExpandWidgets *cew);

static void gui_tree_collapse_recursive(GtkWidget *w, GuiTreeStore *tree);

#endif /* CONFIG_SWAP_COLUMNS */

#if defined(CONFIG_GTKCLIST_GTKCTREE) || defined(CONFIG_COLUMN_POPUP)
static void gui_tree_expand_recursive(GtkWidget *w, GuiTreeStore *tree);
#endif

/* called from gaccel.c and gui.c */
void AdjustMenu(GtkWidget *w, ListSelection **selected);


/*
	--------------
	 MENU WIDGETS
	--------------
*/

enum
{
	kMenuBarFile,
	kMenuBarView,
	kMenuBarEdit,
	kMenuBarOption,
	kMenuBarHelp,
	kMenuBarTotal
};

enum
{
	kMenuViewFile,
	kMenuViewURL,
	kMenuFind,

	kMenuSave,
#ifndef CACHE_READONLY
	kMenuDelete,
#endif
	kMenuClipboard,
	kMenuReadOnly,
	kMenuReadWrite,
	kMenuItemsTotal
};

static GtkWidget *gui_menu_bar[kMenuBarTotal];
static GtkWidget *gui_menu_widget[kMenuItemsTotal];
static GtkWidget *gui_popup_widget[kMenuItemsTotal];

GtkWidget *gui_popup_menu;

static const char mn_save_file_str[]	= gettext_nop("Save to file ...");
static const char mn_delete_str[]	= gettext_nop("Delete file ...");
static const char mn_clipboard_str[]	= gettext_nop("Copy url to clipboard");
static const char mn_entry_info_str[]	= gettext_nop("Entry informations ...");
static const char mn_read_only_str[]	= gettext_nop("read-only permissions");
static const char mn_read_write_str[]	= gettext_nop("read/write permissions");
static const char mn_view_file_str[]	= gettext_nop("View file with");
static const char mn_view_url_str[]	= gettext_nop("View URL with");
static const char mn_find_str[]		= gettext_nop("Find ...");


/*
	------------
	 MENU TYPES
	------------
*/

enum
{
	kMTypeEndOfList,
	kMTypeItem,
	kMTypeCheckItem,
	kMTypeSeparator,
	kMTypeSubmenu,
	kMTypeSubmenuCheckItem

};

/*
	------------
	 MENU ITEMS
	------------
*/

typedef struct
{
	gint		type;		/* see above */
	const gchar	*label;		/* name to display */
	gint		accel_id;	/* accel key */
	GtkWidget	**storage;	/* gui_menu_widget */
	GuiSignalFunc	callback;	/* function pointer */
	gpointer	data;		/* data for it */

} GuiMenuItem;

static const GuiMenuItem menuFile[] =
{
	{kMTypeItem, gettext_nop("Open cache index file ..."),
			kGAccelFileOpen, NULL,
			(GuiSignalFunc) Browse, &gui_menu_bar[kMenuBarFile]},
	{kMTypeItem, gettext_nop("Properties ..."),
			kGAccelFileProp, NULL,
			(GuiSignalFunc) PropertiesDialog, &gui_menu_bar[kMenuBarFile]},
	{kMTypeSeparator, NULL, kGAccelKeyNone, NULL, NULL, NULL},
	{kMTypeItem, gettext_nop("Save rc file"),
			kGAccelFileSaveRC, NULL,
			(GuiSignalFunc) SaveRC, NULL},
	{kMTypeSeparator, NULL, kGAccelKeyNone, NULL, NULL, NULL},
	{kMTypeItem, gettext_nop("Quit"),
			kGAccelFileQuit, NULL,
			(GuiSignalFunc) Quit, NULL},
	{kMTypeEndOfList, NULL, kGAccelKeyNone, NULL, NULL, NULL}

}; /* static const GuiMenuItem menuFile[] */

static const GuiMenuItem menuView[] =
{
	{kMTypeItem, mn_entry_info_str,
			kGAccelViewEntryInfo, NULL,
			(GuiSignalFunc) ViewInfo, &windata.toplist},
	{kMTypeSeparator, NULL, kGAccelKeyNone, NULL, NULL, NULL},
	{kMTypeSubmenuCheckItem, gettext_nop("Columns"),
			kGAccelKeyNone, NULL, NULL, NULL},
	{kMTypeSeparator, NULL, kGAccelKeyNone, NULL, NULL, NULL},
	{kMTypeSubmenu, mn_view_file_str,
			kGAccelKeyNone, &gui_menu_widget[kMenuViewFile],
			(GuiSignalFunc) GenViewerMenu, (gpointer) VIEWER_FILE},
	{kMTypeSubmenu, mn_view_url_str,
			kGAccelKeyNone, &gui_menu_widget[kMenuViewURL],
			(GuiSignalFunc) GenViewerMenu, (gpointer) VIEWER_URL},
	{kMTypeSeparator, NULL, kGAccelKeyNone, NULL, NULL, NULL},
	{kMTypeItem, mn_find_str,
			kGAccelViewFind, &gui_menu_widget[kMenuFind],
			(GuiSignalFunc) search_dialog_open, NULL},
	{kMTypeEndOfList, NULL, kGAccelKeyNone, NULL, NULL, NULL}

}; /* static const GuiMenuItem menuView[] = */

static const GuiMenuItem menuEdit[] =
{
	{kMTypeItem, mn_save_file_str,
			kGAccelEditSave, &gui_menu_widget[kMenuSave],
			(GuiSignalFunc) SaveAs, &gui_menu_bar[kMenuBarEdit]},
#ifndef CACHE_READONLY
	{kMTypeItem, mn_delete_str,
			kGAccelEditDelete, &gui_menu_widget[kMenuDelete],
			(GuiSignalFunc) DeleteDialog, &gui_menu_bar[kMenuBarEdit]},
#endif
	{kMTypeSeparator, NULL, kGAccelKeyNone, NULL, NULL, NULL},
	{kMTypeItem, mn_clipboard_str,
			kGAccelEditClipboard, &gui_menu_widget[kMenuClipboard],
			(GuiSignalFunc) SelectionSetURL, &windata.toplist},
	{kMTypeItem, gettext_nop("Show entry informations"),
			kGAccelEditShowInfo, NULL,
			(GuiSignalFunc) NodeInfo, &windata.toplist},
	{kMTypeSeparator, NULL, kGAccelKeyNone, NULL, NULL, NULL},
	{kMTypeItem, mn_read_only_str,
			kGAccelEditReadOnly, &gui_menu_widget[kMenuReadOnly],
			(GuiSignalFunc) RdOnly, &windata.toplist},
	{kMTypeItem, mn_read_write_str,
			kGAccelEditReadWrite, &gui_menu_widget[kMenuReadWrite],
			(GuiSignalFunc) Writable, &windata.toplist},
	{kMTypeEndOfList, NULL, kGAccelKeyNone, NULL, NULL, NULL}

}; /* static const GuiMenuItem menuEdit[] */

static const GuiMenuItem menuOptions[] =
{
	{kMTypeItem, gettext_nop("Viewers setup ..."),
			kGAccelOptViewers, NULL, (GuiSignalFunc) apassign_dialog_open, NULL},
	{kMTypeItem, gettext_nop("Keyboard ..."),
			kGAccelOptAccel, NULL, (GuiSignalFunc) gaccel_dialog_open, NULL},
	{kMTypeSeparator, NULL, kGAccelKeyNone, NULL, NULL, NULL},
	{kMTypeCheckItem, gettext_nop("Convert filenames to lowercase"),
			kGAccelOptCase, &gui_option[kOptionConvertToLower].widget,
			(GuiSignalFunc) ToggleBool, &gui_option[kOptionConvertToLower]},
#ifndef CACHE_READONLY
	{kMTypeCheckItem, gettext_nop("Readonly"),
			kGAccelOptRdOnly, &gui_option[kOptionReadOnly].widget,
			(GuiSignalFunc) ToggleBool, &gui_option[kOptionReadOnly]},
#endif
	{kMTypeCheckItem, gettext_nop("Decode files"),
			kGAccelOptDecode, &gui_option[kOptionDecode].widget,
			(GuiSignalFunc) ToggleBool, &gui_option[kOptionDecode]},
#ifdef HAVE_ICONV
	{kMTypeCheckItem, gettext_nop("Window titles in UTF-8"),
			kGAccelOpt_WM_UTF8, &gui_option[kOption_WM_UTF8].widget,
			(GuiSignalFunc) ToggleBool, &gui_option[kOption_WM_UTF8]},
#endif
	{kMTypeCheckItem, gettext_nop("Save rc on exit"),
			kGAccelOptSave, &gui_option[kOptionSaveConfigOnExit].widget,
			(GuiSignalFunc) ToggleBool, &gui_option[kOptionSaveConfigOnExit]},
	{kMTypeEndOfList, NULL, kGAccelKeyNone, NULL, NULL, NULL}

}; /* static const GuiMenuItem menuOptions[] */

static const GuiMenuItem menuHelp[] =
{
	{kMTypeItem, gettext_nop("About ..."),
		kGAccelHelpAbout, NULL, (GuiSignalFunc) About, NULL},
	{kMTypeEndOfList, NULL, kGAccelKeyNone, NULL, NULL, NULL}

}; /* static const GuiMenuItem menuHelp[] */

static const GuiMenuItem menuPopup[] =
{
	{kMTypeItem, mn_save_file_str,
			kGAccelKeyNone, &gui_popup_widget[kMenuSave],
			(GuiSignalFunc) SaveAs, &gui_menu_bar[kMenuBarEdit]},
#ifndef CACHE_READONLY
	{kMTypeItem, mn_delete_str,
			kGAccelKeyNone, &gui_popup_widget[kMenuDelete],
			(GuiSignalFunc) DeleteDialog, &gui_menu_bar[kMenuBarEdit]},
#endif
	{kMTypeSeparator, NULL, kGAccelKeyNone, NULL, NULL, NULL},
	{kMTypeItem, mn_clipboard_str,
			kGAccelKeyNone, &gui_popup_widget[kMenuClipboard],
			(GuiSignalFunc) SelectionSetURL, &windata.toplist},
	{kMTypeItem, mn_entry_info_str,
			kGAccelKeyNone, NULL,
			(GuiSignalFunc) ViewInfo, &windata.toplist},
	{kMTypeSeparator, NULL, kGAccelKeyNone, NULL, NULL, NULL},
	{kMTypeItem, mn_read_only_str,
			kGAccelKeyNone, &gui_popup_widget[kMenuReadOnly],
			(GuiSignalFunc) RdOnly, &windata.toplist},
	{kMTypeItem, mn_read_write_str,
			kGAccelKeyNone, &gui_popup_widget[kMenuReadWrite],
			(GuiSignalFunc) Writable, &windata.toplist},
	{kMTypeSeparator, NULL, kGAccelKeyNone, NULL, NULL, NULL},
	{kMTypeSubmenu, mn_view_file_str,
			kGAccelKeyNone, &gui_popup_widget[kMenuViewFile],
			(GuiSignalFunc) GenViewerMenu, (gpointer) VIEWER_FILE},
	{kMTypeSubmenu, mn_view_url_str,
			kGAccelKeyNone, &gui_popup_widget[kMenuViewURL],
			(GuiSignalFunc) GenViewerMenu, (gpointer) VIEWER_URL},
	{kMTypeSeparator, NULL, kGAccelKeyNone, NULL, NULL, NULL},
	{kMTypeItem, mn_find_str,
			kGAccelKeyNone, &gui_popup_widget[kMenuFind],
			(GuiSignalFunc) search_dialog_open, NULL},
	{kMTypeEndOfList, NULL, kGAccelKeyNone, NULL, NULL, NULL}

}; /* static const GuiMenuItem menuPopup[] */



#ifdef FOCUS_IN_BUGFIX

static void gui_window_topped(Windata *windat)
{
	GtkWidget *window;

	window = (windat->ismapped == NULL) ?
				windat->unmapped->data :
					windat->ismapped->data;
	if (window != windat->topwin)
	{
		ListSelection *list;

		windat->topwin = window;

		list = GuiObjectGetUserData(window);

		if (list)
		{
			windat->toplist = list;
		}
	}
}

static void gui_window_destroyed(GtkWidget *window, Windata *windat)
{
	if (window != windat->mainwin)
	{
		GSList **list = &windat->ismapped;

		if (g_slist_find(*list, window) == NULL)
		{
			list = &windat->unmapped;
		}

		*list = g_slist_remove(*list, window);

		gui_window_topped(windat);
	}
}

static gint gui_window_focus_property(GtkWidget *window,
					GdkEventProperty *e, Windata *windat)
{
	if (window != windat->topwin &&
		(e->type == GDK_FOCUS_CHANGE ||
			GDK_ATOM_TO_POINTER(e->atom) ==
				GDK_ATOM_TO_POINTER(windat->_net_wm_state)))
	{
		GSList *top = g_slist_find(windat->ismapped, window);

		if (top)
		{
			windat->ismapped = g_slist_remove_link(windat->ismapped, top);
			windat->ismapped = g_slist_concat(top, windat->ismapped);

			gui_window_topped(windat);
		}
	}

	return FALSE;
}

static gint gui_window_map(GtkWidget *window,
				GdkEvent *e, Windata *windat)
{
	GSList *top = g_slist_find(windat->unmapped, window);

	if (top)
	{
		windat->unmapped = g_slist_remove_link(windat->unmapped, top);
		windat->ismapped = g_slist_concat(windat->ismapped, top);

		gui_window_topped(windat);
	}

	else
	{
		windat->ismapped = g_slist_prepend(windat->ismapped, window);
	}

	return FALSE;
}

static gint gui_window_unmap(GtkWidget *window,
				GdkEvent *e, Windata *windat)
{
	GSList *top = g_slist_find(windat->ismapped, window);

	if (top)
	{
		windat->ismapped = g_slist_remove_link(windat->ismapped, top);
		windat->unmapped = g_slist_concat(top, windat->unmapped);

		gui_window_topped(windat);
	}

	return FALSE;
}

#else /* ! FOCUS_IN_BUGFIX */

#if GTK_MAJOR_VERSION > 1
static void gui_window_destroyed(GtkWidget *window, Windata *windat)
{
	if (window == windat->topwin)
	{
		windat->topwin  = NULL;
		windat->toplist = NULL;
	}
}
#endif /* GTK_MAJOR_VERSION > 1 */

static gint gui_window_focus_in(GtkWidget *window,
				GdkEventFocus *e, Windata *windat)
{
	if (window != windat->topwin)
	{
		ListSelection *list;

		windat->topwin = window;

		list = GuiObjectGetUserData(window);

		if (list)
		{
			windat->toplist = list;
		}
	}

	return FALSE;
}

#endif /* #ifdef FOCUS_IN_BUGFIX ... #else */


static void gui_window_setup(GtkWidget *widget, gint prop)
{
	gint		width, height;
	GtkWindow	*window;

	if ( (gprop_get_int(prop, &width) == FALSE ||
			width <= 0 || gdk_screen_width() < width) ||
		(gprop_get_int(prop + 1, &height) == FALSE ||
			height <= 0 || gdk_screen_height() < height) )
	{
		width  = -1;
		height = -1;
	}

	window = GTK_WINDOW(widget);

	GuiWindowSetResizable(window, TRUE);
	gtk_window_set_default_size(window, width, height);

	GuiSignalConnect(window, "size-allocate", gui_window_resize, (gpointer) prop);

} /* static void gui_window_setup(GtkWidget*, gint) */

GtkWidget *gui_window_new(const gchar *title, GtkWidget *child, gint border, gint prop)
{
	GtkWidget *window;

	window = guitl_window_new_gettext(title, child, border);
	gui_window_setup(window, prop);

#ifdef FOCUS_IN_BUGFIX

	gtk_widget_set_events(window, GDK_FOCUS_CHANGE_MASK |
					GDK_PROPERTY_CHANGE_MASK);

	GuiSignalConnect(window, "focus-in-event",
				gui_window_focus_property, &windata);

	GuiSignalConnect(window, "property-notify-event",
				gui_window_focus_property, &windata);

	GuiSignalConnect(window, "map-event", gui_window_map, &windata);
	GuiSignalConnect(window, "unmap-event", gui_window_unmap, &windata);
	GuiSignalConnect(window, "destroy", gui_window_destroyed, &windata);

#else /* ! FOCUS_IN_BUGFIX */

#if GTK_MAJOR_VERSION > 1
	GuiSignalConnect(window, "destroy", gui_window_destroyed, &windata);
#endif
	GuiSignalConnect(window, "focus-in-event", gui_window_focus_in, &windata);
#endif

	return window;

} /* GtkWidget *gui_window_new(const gchar*, gint) */

static void _noop(void)
{
	return;
}

static gint gui_button_key_grab_add(void)
{
	GtkWidget *widget;

	if (gtk_grab_get_current() != NULL)
	{
		return FALSE;
	}

	widget = error_msg.frame;

	gtk_grab_add(widget);
	GuiSignalHandlerUnblock(widget, msg_button_sig_id);
	GuiSignalHandlerUnblock(widget, msg_key_sig_id);

	return TRUE;
}

static void gui_button_key_grab_remove(void)
{
	GtkWidget *widget = error_msg.frame;

	GuiSignalHandlerBlock(widget, msg_button_sig_id);
	GuiSignalHandlerBlock(widget, msg_key_sig_id);
	gtk_grab_remove(widget);
}

static gint gui_button_key_grab_func(GtkWidget *widget, GdkEventAny *e, const gchar *name)
{
	GuiSignalEmitStopByName(widget, name);

	return TRUE;
}

static guint gui_button_key_grab_init(GtkWidget *widget, const gchar *name)
{
	guint sig_id = GuiSignalConnect(widget, name,
				gui_button_key_grab_func, (gpointer) name);
	GuiSignalHandlerBlock(widget, sig_id);

	return sig_id;
}

#if defined(CONFIG_GTKCLIST_GTKCTREE) && ! defined(CACHE_READONLY)
static void gui_clist_freeze_or_thaw(ListSelection *list, gpointer func)
{
	void (*freeze_or_thaw)(GtkCList*) = func;

	(*freeze_or_thaw)((list->type == kTreeList) ?
				GTK_CLIST(list->store.tree.ctree) :
						list->store.list.clist);
}
#endif /* CONFIG_GTKCLIST_GTKCTREE && ! CACHE_READONLY */

static void compute_node_info_recursive(GuiTreeStore *tree,
				nscache_record_t *rec, gpointer data)
{
	struct node_info *nfo = data;

	if (rec)
	{
		nfo->filenum++;
		nfo->contentsize += rec->content_length;
	}

	else 
	{
		gint level = GuiTreeGetLevel(tree);

		if (level == kTreeLevelService)
		{
			nfo->servicenum++;
		}

		else if (level == kTreeLevelDomain)
		{
			nfo->domainnum++;
		}

		else
		{
			nfo->foldernum++;
		}
	}

} /* static void compute_node_info_recursive(GuiTreeStore*,
						nscache_record_t*, gpointer) */

static void compute_node_info(ListSelection *list, struct node_info *nfo)
{
	memset(nfo, '\0', sizeof(*nfo));

	if (list && list->nscrec)
	{
		nfo->filenum = 1;
		nfo->contentsize = list->nscrec->content_length;
	}

	else
	{
		GuiTreeStore	*tree;
		gboolean	selected;

		if (list == NULL)
		{
			tree = &(notebook_list[kTreeList].store.tree);
			selected = FALSE;
		}

		else
		{
			if (list->name == NULL || list->type != kTreeList)
			{
				return;
			}

			tree = &(list->store.tree);
			selected = TRUE;
		}

		guitl_tree_recursive(tree, selected,
			(GuiTreeStoreRecursiveFunc) compute_node_info_recursive, nfo);
	}

} /* static void compute_node_info(ListSelection*, struct node_info*) */

static int get_node_info(char *buf, int sz, int fmt, ListSelection *list)
{
	static const char *format_str[] = 
	{
		"%s: %u  %s: %u  %s: %u  %s: %u  %s: %lu %s",
		"%s: %u\n%s: %u\n%s: %u\n%s: %u\n%s: %lu %s"
	};

	int			len;
	char			*byte_str;
	unsigned long		size;
	struct node_info	nfo;

	compute_node_info(list, &nfo);

	size = nfo.contentsize;

	if (size >= 1024)
	{
		size /= 1024;
		byte_str = gettext("kB");
	}

	else
	{
		byte_str = gettext("Bytes");
	}

	len = g_snprintf(buf, sz, format_str[fmt],
		gettext("Services"), nfo.servicenum,
			gettext("Domains"), nfo.domainnum,
				gettext("Folders"), nfo.foldernum,
					gettext("Files"), nfo.filenum,
						gettext("Size"), size, byte_str);
	buf[sz - 1] = '\0';

	return len;

} /* static int get_node_info(char*, int, int, ListSelection*) */

static void print_node_info(ListSelection *list)
{
	char pom[kMessageBufferSize];

	get_node_info(pom, sizeof(pom), kNodeInfoFormatSingleLine, list);
	gui_msg(pom);
}

static gint gui_error_msg_expose(GtkWidget *label, GdkEventExpose *e, gpointer p)
{
	gdk_draw_rectangle(label->window,
		(label->style)->bg_gc[GTK_STATE_NORMAL],
		TRUE,
		(e->area).x,
		(e->area).y,
		(e->area).width,
		(e->area).height);

	return FALSE;

} /* static gint gui_error_msg_expose(GtkWidget*, GdkEvent*, gpointer) */

static GtkWidget *gui_msg_label_replace_with(GtkWidget *replace, GtkWidget *with)
{
	GtkContainer *frame;

	frame = GTK_CONTAINER(replace->parent);
	gtk_widget_hide(replace);
	gtk_container_remove(frame, replace);
	gtk_container_add(frame, with);
	gtk_widget_show(with);

	return with;
}

static void gui_msg_set_label_text(GtkWidget *label,
				GtkAllocation *allocation, struct msg_tmr *msg)
{
	gchar *str;

	if (GTK_WIDGET_VISIBLE(msg->label) == FALSE)
	{
		msg->width = 0;
		return;
	}

	str = msg->str;

	if (str && (label == NULL || allocation->width != msg->width))
	{
		gint txt_width;

#if GTK_MAJOR_VERSION == 1

		GdkFont *font;

		font = (msg->style)->font;
		txt_width = gdk_string_width(font, str);
#else
		PangoLayout* layout;

		layout = gtk_widget_create_pango_layout(GTK_WIDGET(msg->label), str);
		pango_layout_get_pixel_size(layout, &txt_width, NULL);
#endif
		msg->width = allocation->width;

		if (txt_width > msg->width)
		{
			static const gchar elip_str[] = " (...)";

			gint  chr_width, elip_width, clen, slen;

#if GTK_MAJOR_VERSION == 1
			elip_width = gdk_text_width(font, elip_str, sizeof(elip_str) - 1);
#else
			gchar *s;

			pango_layout_set_text(layout, elip_str, sizeof(elip_str) - 1);
			pango_layout_get_pixel_size(layout, &elip_width, NULL);
#endif
			txt_width += elip_width;
			slen = strlen(str);

#if GTK_MAJOR_VERSION == 1

			clen = slen;

			while (--slen > 0)
			{
				chr_width = gdk_text_width(font, &str[slen], clen - slen);

				if (chr_width == 0)
				{
					continue;
				}

				clen = slen;

#else /* GTK_MAJOR_VERSION > 1 */

			while ((s = g_utf8_find_prev_char(str, &str[slen])) != NULL)
			{
				clen  = slen;
				slen  = s - str;

				pango_layout_set_text(layout, s, clen - slen);
				pango_layout_get_pixel_size(layout, &chr_width, NULL);

#endif /* #if GTK_MAJOR_VERSION == 1 .. #else */

				txt_width -= chr_width;

				if (msg->width > txt_width)
				{
					break;
				}
			}

			str = g_malloc(slen + sizeof(elip_str));
			strncpy(str, msg->str, slen);
			strcpy(&str[slen], elip_str);

		} /* if (txt_width > msg->width) */

#if GTK_MAJOR_VERSION > 1
		g_object_unref(layout);
#endif
		gtk_label_set_text(msg->label, str);

		if (str != msg->str)
		{
			g_free(str);
		}

		if (msg->tmr_id)
		{
			GuiTimeoutRemove(msg->tmr_id);
		}

		msg->tmr_id = GuiTimeoutAdd(10000, (GtkFunction) msg_clr, msg);

		if (msg->do_beep)
		{
			msg->do_beep = FALSE;
			gdk_beep();
		}

	} /* if (str && (label == NULL || allocation->width != msg->width)) */

} /* static void gui_msg_set_label_text(GtkWidget*, GtkAllocation*, struct msg_tmr*) */

static gint msg_clr(struct msg_tmr *msg)
{
	if (msg->str)
	{
		if (msg->sig_id)
		{
			GuiSignalHandlerBlock(msg->label, msg->sig_id);
			/* draw the background */
			gtk_label_set_text(msg->label, " ");
		}

		g_free(msg->str);
	}

	msg->str = NULL;
	msg->tmr_id = 0;

	gtk_label_set_text(msg->label, NULL);
	msg->do_beep = FALSE;

	return FALSE;

} /* static gint msg_clr(struct msg_tmr*) */

static void msg_set(struct msg_tmr *msg, const gchar *str)
{
	if (msg->tmr_id)
	{
		GuiTimeoutRemove(msg->tmr_id);
	}

	if (str == NULL)
	{
		msg_clr(msg);
	}

	else
	{
		if (msg->sig_id && msg->str == NULL)
		{
			GuiSignalHandlerUnblock(msg->label, msg->sig_id);
			/* draw the background */
			gtk_label_set_text(msg->label, " ");
		}

		msg->tmr_id = 0;
		g_free(msg->str);
		msg->str = CHARCONV_UTF8_FROM_LOCALE_DUP(str);
		gui_msg_set_label_text(NULL, &(GTK_WIDGET(msg->label)->allocation), msg);
	}

} /* static void msg_set(struct msg_tmr*, const gchar*) */

static void gui_msg(const gchar *str)
{
	msg_set(&status_msg, str);
}

void gui_err(const gchar *str)
{
	if (gui_is_active == FALSE)
	{
		if (str)
		{
			fprintf(stderr, "%s\n", str);
		}
	}

	else
	{
		if (str && error_msg.do_beep == FALSE)
		{
			error_msg.do_beep = TRUE;
		}

		msg_set(&error_msg, str);
	}

} /* void gui_err(const gchar*) */

void gui_errfmt(const gchar *format, ...)
{
	gchar *errmsg = (gchar *) format;

	if (format)
	{
		va_list	args;
		gchar	pom[kMessageBufferSize];

#if GTK_MAJOR_VERSION == 1

		va_start(args, format);
		g_vsnprintf(pom, sizeof(pom), format, args);
		va_end(args);
		pom[sizeof(pom) - 1] = '\0';
#else
		gint chr, len;

		len = 0;
		va_start(args, format);

		while ((chr = *(format++)) != '\0')
		{
			gint  sz, num;
			gchar *str, *utf;

			if (chr != '%')
			{
				pom[len] = chr;
				sz = 1;
			}

			else switch (*(format++))
			{
				case 'd':
					num = va_arg(args, gint);
					sz  = g_snprintf(&pom[len], sizeof(pom) - len, "%d", num);
					break;

				case 's':
					str = va_arg(args, gchar*);

					if (str == NULL)
					{
						continue;
					}

					utf = (gui_is_active == FALSE) ?
							str : CHARCONV_UTF8_FROM_LOCALE(str);
					sz  = g_snprintf(&pom[len], sizeof(pom) - len, "%s", utf);
					CHARCONV_UTF8_FREE(utf, str);
					break;

				default:
					sz = -1;
					break;
			}

			if (sz < 0 || (len += sz) >= sizeof(pom))
			{
				break;
			}
		}

		va_end(args);
		pom[len] = '\0';

#endif /* #if GTK_MAJOR_VERSION == 1 .. #else */

		errmsg = pom;

	} /* if (format) */

	gui_err(errmsg);

} /* void gui_errfmt(const gchar*, ...) */

void gui_strerror(const gchar *errmsg)
{
	if (errmsg == NULL)
	{
		gui_err(errmsg);
		return;
	}

	gui_errfmt("%s : %s", gettext(errmsg), strerror(errno));
}

#if GTK_MAJOR_VERSION > 1

static void gui_fname_error(const gchar *fname)
{
	gchar *fn;

	if (fname == NULL)
	{
		msg_set(&error_msg, fname);
		return;
	}

	fn = CHARCONV_UTF8_FROM_FILENAME(fname, NULL);
	gui_errfmt("%s : %s", fn, strerror(errno));
	CHARCONV_UTF8_DESTROY(fn);

} /* static void gui_fname_error(const gchar*) */

#endif /* #if GTK_MAJOR_VERSION > 1 */

static void gui_quit(GtkWidget *w)
{
	gui_is_active = FALSE;
	gtk_main_quit();
}

static gint Quit(GtkWidget *w)
{
	gui_is_active = FALSE;
	guitl_window_toplevel_destroy_all();

	return TRUE;
}

static void About(GtkWidget *w, gpointer data)
{
	static GtkWidget *about_win = NULL;

	gchar		*str, pom[kMessageBufferSize];
	GtkWidget	*button, *image, *frame, *hbox, *label, *vbox;

	GtkWidget *window = about_win;

	if (window)
	{
		GuiWindowPresent(window);
		return;
	}

	vbox = gtk_vbox_new(FALSE, 1);
	window = guitl_window_new_gettext(gettext_nop("MozCache: About"), vbox, 4);
	about_win = window;

	GuiWindowSetResizable(GTK_WINDOW(window), FALSE);
	GuiSignalConnect(window, "destroy", gtk_widget_destroyed, &about_win);

	hbox = gtk_hbox_new(FALSE, 1);
	frame = guitl_frame_new_add_child_gettext(NULL, hbox);
	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, FALSE, 10);

	image = guitl_get_icon_dialog(kIconDialogNsCacheLogo);
	gtk_box_pack_start(GTK_BOX(hbox), image, TRUE, FALSE, 5);

	g_snprintf(pom, sizeof(pom), "%s\n\n"
					"%s\n\n"
					"%s %s\n\n"
					"%s\n"
					"%s %s\n"
					"%s\n\n"
					"%s\n"
					"%s %s\n"
					"%s",
		gettext("Cache viewer for Mozilla, Netscape, Firefox & SeaMonkey browsers"),
		gettext("Distributed under GNU-GPL 2 or later"),
		gettext("Version"), nscache_version_str,
		gettext("Previous versions (NScache)"),
		gettext("Author: Stefan Ondrejicka"), "<ondrej@users.sourceforge.net>",
			"URL: http://nscache.sourceforge.net/",
		gettext("Modifications in later versions (MozCache)"),
		gettext("Author: Harald Foerster"), "<harald_foerster@users.sourceforge.net>",
			"URL: http://mozcache.sourceforge.net/");

	pom[sizeof(pom) - 1] = '\0';
	str = CHARCONV_UTF8_FROM_LOCALE(pom);

#if GTK_MAJOR_VERSION > 1
	if (str == NULL)
	{
		str = tl_replace_non_ascii_chars_in_str(pom);
	}
#endif
	label = gtk_label_new(str);
	CHARCONV_UTF8_FREE(str, pom);

	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment(GTK_MISC(label), 0.1, 0.5);
	gtk_misc_set_padding(GTK_MISC(label), 15, 15);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, FALSE, 5);

	hbox = guitl_hbutton_box_new(vbox, 8, GTK_BUTTONBOX_END);
	button = guitl_button_box_add_icon_button(hbox, kIconButtonClose);
	gtk_widget_grab_default(button);
	GuiSignalConnectSwapped(button, "clicked", gtk_widget_destroy, window);

	gtk_widget_show_all(window);

} /* static void About(GtkWidget*, gpointer) */

static void BrowseOK(GtkFileSelection *fs)
{
	gchar		*dbname, *str;
	guint		db_type;
	const gchar	*fname;
	GtkWidget	*browse_type;

	fname	= gtk_file_selection_get_filename(fs);
	str	= CHARCONV_UTF8_FROM_FILENAME(fname, NULL);

	if (str == NULL)
	{
		gui_err(gettext(invalid_filename_str));
		gtk_widget_destroy(GTK_WIDGET(fs));

		return;
	}

	gtk_entry_set_text(entry_db, str);
	CHARCONV_UTF8_DESTROY(str);

	browse_type = GuiObjectGetUserData(fs);
	db_type = GuiOptionMenuGetIndex(browse_type);

	if (nscache_db.type != db_type)
	{
		nscache_db.type = db_type;
		gtk_option_menu_set_history(GTK_OPTION_MENU(nscache_type), db_type);
	}

	dbname = g_strdup(fname);
	gtk_widget_destroy(GTK_WIDGET(fs));

	gui_read_cache(dbname);
}

static void Browse(GtkWidget *w, GtkWidget **mbar)
{
	gint		i;
	const gchar	*fname;
	GtkWidget	*box, *browse_type, *label, *menu, *widget;
	GtkFileSelection *fs;

	widget = guitl_file_selection_new_gettext(gettext_nop("Select cache index ..."));
	guitl_dialog_set_modal(widget, mbar);

	fs = GTK_FILE_SELECTION(widget);
	GuiSignalConnectSwapped(fs->ok_button, "clicked", BrowseOK, fs);

	fname = gtk_entry_get_text(entry_db);

	if (fname && *fname)
	{
		gchar *str = CHARCONV_UTF8_TO_FILENAME(fname);

		if (str != NULL)
		{
			gtk_file_selection_set_filename(fs, str);
			CHARCONV_UTF8_DESTROY(str);
		}
	}

	box = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(fs->main_vbox), box, FALSE, FALSE, 0);

	label = guitl_label_new_gettext(gettext_nop("Cache index file type: "));
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 2);

	browse_type = gtk_option_menu_new();
	GuiObjectSetUserData(fs, browse_type);

#ifndef MULTI_DB_TYPES
	gtk_widget_set_sensitive(browse_type, FALSE);
#endif
	menu	= gtk_menu_new();
	i	= NSCACHE_RT_FIRST;

	do
	{
		GtkWidget *mi;

		mi = guitl_get_icon_menu(i);
		GuiMenuAppend(menu, mi);

#ifdef HAVE_MSIE
		if (i == NSCACHE_RT_MSIE)
		{
			gtk_widget_set_sensitive(mi, FALSE);
		}
#endif
		GuiOptionMenuSetIndex(mi, i);
	}
	while (++i <= NSCACHE_RT_LAST);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(browse_type), menu);

#ifdef HAVE_MSIE
	gtk_option_menu_set_history(GTK_OPTION_MENU(browse_type),
					(nscache_db.type == NSCACHE_RT_MSIE ?
							NSCACHE_RT_FIRST : nscache_db.type));
#else
	gtk_option_menu_set_history(GTK_OPTION_MENU(browse_type), nscache_db.type);
#endif
	gtk_box_pack_start(GTK_BOX(box), browse_type, FALSE, FALSE, 2);

	gtk_widget_show_all(widget);

} /* static void Browse(GtkWidget*, GtkWidget**) */

static void Save(GtkWidget *ok_btn, nscache_record_t *rec)
{
	static gchar* err_str[] =
	{
		NULL,
		gettext_nop("Error opening source file"),
		gettext_nop("Error opening destination file"),
		gettext_nop("Error writing to destinantion file"),
		gettext_nop("Error linking files")
	};

	GtkWidget	*fs;
	const gchar	*dst;
	gint		error;

	fs	= gtk_widget_get_toplevel(ok_btn);
	dst	= gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));
	error	= file_save(rec, (char *) dst);

	if (error == 0)
	{
		gui_msg(gettext("File copied"));
	}

	/* abs(error) */
	gui_strerror(err_str[-error]);

	gtk_widget_destroy(fs);
}

#ifdef HAVE_LINK
static void gui_hardlink_callback(GtkWidget *toggle_button)
{
	hardlink = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle_button));
	gprop_set_bool(kPropBoolSaveHardlink, hardlink);
}
#endif

static void gui_single_folder_callback(GtkWidget *toggle_button)
{
	single_folder = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle_button));
	gprop_set_bool(kPropBoolSaveSingleFolder, single_folder);
}

static void gui_url_to_local_callback(GtkWidget *option_menu)
{
	url_to_local_selector = GuiOptionMenuGetIndex(option_menu);
	gprop_set_int(kPropIntSaveUrlToLocal, url_to_local_selector);
}

static void gui_filename_callback(GtkFileSelection *fs)
{
	gchar *path, *sep;

	path = g_strdup(gtk_file_selection_get_filename(fs));

	while ((sep = strrchr(path, '/')) != NULL)
	{
		struct stat dirstat;

		sep[1] = '\0';

		if ( stat(path, &dirstat) == 0 &&
					S_ISDIR(dirstat.st_mode) )
		{
			gprop_set_str(kPropStrSavePath, path);
			break;
		}

		*sep = '\0';
	}

	g_free(path);
}

static gchar *gui_save_get_path(const gchar *fname)
{
	gchar *path, *sep;

	if (gprop_get_str(kPropStrSavePath, &path) == FALSE)
	{
		path = "";
	}

	sep = strrchr(path, '/');

	sep = (sep == NULL || sep[1] == '\0') ? "" : "/";

	return g_strconcat(path, sep, (fname == NULL) ? "noname" : fname, NULL);
}

static gchar *SaveFileDialog(GtkFileSelection *fs, GtkWidget *button, ListSelection *list)
{
	gchar *fname, *path, *str;

#ifdef HAVE_LINK
	/* check button - hardlink */
	gtk_box_pack_start(GTK_BOX(fs->main_vbox), button, FALSE, FALSE, 0);
#endif
	GuiSignalConnect(fs->ok_button, "clicked", Save, list->nscrec);

	str = tl_get_url_filename( tl_get_url_start(list->nscrec->urlstr) );

	if (str)
	{
		/* maybe URL in URL */

		fname = g_strdup(str);
		str   = fname;

		while ((str = strchr(++str, '/')) != NULL)
		{
			/* replace '/' with '-' */
			*str = '-';
		}

		str = (gchar *) fname;
	}

	else
	{
		fname = (gchar *) &index_html_str[1];
	}

	path = gui_save_get_path(fname);

	g_free(str);

	return path;

} /* static gchar *SaveFileDialog(GtkFileSelection*, GtkWidget*, ListSelection*) */

static void collect_nscache_records(GuiTreeStore *tree, nscache_record_t* rec,
							nscache_record_t*** rec_array)
{
	if (rec)
	{
		**rec_array = rec;
		(*rec_array)++;
	}
}

static void SaveFolder(GtkWidget *ok_btn, ListSelection *list)
{
	static const char fail_mesg[] = gettext_nop("Cannot create folder");

	size_t			rec_items;
	nscache_record_t**	rec_array;
	nscache_record_t**	rec_malloc;

	union
	{
		struct node_info  nfo;
		char		  mesg[4096];
		nscache_record_t* recs[1024];
	} u;

	/* file selector */
	GtkWidget *fs = gtk_widget_get_toplevel(ok_btn);

	rec_array  = u.recs;
	rec_malloc = NULL;

	/* number of files plus terminating 'NULL' */
	compute_node_info(list, &u.nfo);
	rec_items = u.nfo.filenum + 1;

	if ( (rec_items <= sizeof(u.recs) / sizeof(nscache_record_t*)) ||
		(rec_malloc = rec_array =
			malloc(rec_items * sizeof(nscache_record_t*))) != NULL )
	{
		static const char saved_fmt_str[] = "%s: %d   %s: %d   %s: %d   %s: %d";

		int		error;
		const char	*str;
		SaveDirResult	sdr;

		nscache_record_t** array_start = rec_array;

		/* fill array with pointers ... */
		guitl_tree_recursive(&(list->store.tree), TRUE,
			(GuiTreeStoreRecursiveFunc) collect_nscache_records, &rec_array);

		/* ... and terminate it with 'NULL' */
		*rec_array = NULL;

		/* write folder to this destination */
		str = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));

		error = folder_save(str,
			(const nscache_record_t**) array_start,
			GuiTreeGetLevel(&(list->store.tree)) - kTreeLevelDomain, &sdr);

		if (url_to_local_selector == URL_CONVERT_NONE)
		{
			g_snprintf(u.mesg, sizeof(u.mesg), &saved_fmt_str[9],
				gettext("folders created"), sdr.folders,
				gettext("files copied"), sdr.files,
				gettext("errors"), sdr.errors);
		}

		else
		{
			g_snprintf(u.mesg, sizeof(u.mesg), saved_fmt_str,
				gettext("folders created"), sdr.folders,
				gettext("files copied"), sdr.files,
				gettext("URLs converted"), sdr.converted,
				gettext("errors"), sdr.errors);
		}

		u.mesg[sizeof(u.mesg) - 1] = '\0';
		gui_msg(u.mesg);

		if (error == 0)
		{
			str = sdr.fname;

			if (str)
			{
				errno = sdr.errnr;
			}

			gui_fname_error(str);
		}

		else
		{
			gui_strerror(fail_mesg);
		}

		if (sdr.fname)
		{
			free(sdr.fname);
		}

	} /* if( (rec_items <= ... */

	else
	{
		gui_strerror(fail_mesg);
	}


	if (rec_malloc)
	{
		free(rec_malloc);
	}


	gtk_widget_destroy(fs);

} /* static void SaveFolder(GtkWidget*, ListSelection*) */

static gchar *SaveFolderDialog(GtkFileSelection *fs, GtkWidget *button, ListSelection *list)
{
	GtkWidget	*label, *menu, *option, *frame, *widget;
	GtkTable	*table;
	gint		index;

	widget = gtk_table_new(2, 2, FALSE);
	frame  = guitl_frame_new_add_child_gettext(NULL, widget);
	gtk_box_pack_start(GTK_BOX(fs->main_vbox), frame, FALSE, FALSE, 0);

	table = GTK_TABLE(widget);

	/* single_folder + hardlink */
	
#ifdef HAVE_LINK
	/* check button - hardlink */
	gtk_table_attach(table, button, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 4, 4);
#endif

	/* check button - single_folder */

	button = guitl_check_button_new_gettext(gettext_nop("Single Folder"));

	gtk_table_attach(table, button, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 4, 4);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), single_folder);
	GuiSignalConnectSwapped(fs->ok_button, "clicked", gui_single_folder_callback, button);

	/* convert URLs */

	label = guitl_label_new_gettext(gettext_nop("Convert URLs to local:"));
	gtk_table_attach(table, label, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 4, 4);

	option = gtk_option_menu_new();
	gtk_table_attach(table, option, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 4, 4);

	GuiSignalConnectSwapped(fs->ok_button, "clicked", gui_url_to_local_callback, option);
	GuiSignalConnect(fs->ok_button, "clicked", SaveFolder, list);

	menu = gtk_menu_new();
	gtk_widget_realize(menu);

	index = URL_CONVERT_NONE;

	do
	{
		static const gchar *url_to_local_str[] =
		{
			gettext_nop("none"),
			gettext_nop("all URLs with domain reference"),
			gettext_nop("only URLs with cache file reference")
		};

		GtkWidget *mi;

		mi = guitl_menu_item_new_gettext(url_to_local_str[index]);
		GuiMenuAppend(menu, mi);
		GuiOptionMenuSetIndex(mi, index);

		gtk_widget_show(mi);
	}
	while (++index <= URL_CONVERT_ONLY_FILE_REFERENCE);

	/* option */

	gtk_option_menu_set_menu(GTK_OPTION_MENU(option), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(option),
		(url_to_local_selector > URL_CONVERT_ONLY_FILE_REFERENCE) ?
					URL_CONVERT_NONE : url_to_local_selector);

	return gui_save_get_path(list->name);

} /* static gchar *SaveFolderDialog(GtkFileSelection*, GtkWidget*, ListSelection*) */

static void SaveAs(GtkWidget *w, GtkWidget **mbar)
{
	static const gchar save_file_str[]   = gettext_nop("Save cache file as ...");
	static const gchar save_folder_str[] = gettext_nop("Save cache folder as ...");

	gboolean	active;
	gchar		*str;
	const gchar	*title;
	GtkWidget	*button, *widget;
	GtkFileSelection *fs;

	ListSelection *list = windata.toplist;

	if (list == NULL || (list->nscrec == NULL && list->type != kTreeList))
	{
		return;
	}

	title  = (list->nscrec == NULL) ? save_folder_str : save_file_str;
	widget = guitl_file_selection_new_gettext(title);
	guitl_dialog_set_modal(widget, mbar);

	fs = GTK_FILE_SELECTION(widget);
	GuiSignalConnectSwapped(fs->ok_button, "clicked", gui_filename_callback, fs);

#ifdef HAVE_LINK

	/* check button - hardlink */

	button = guitl_check_button_new_gettext(gettext_nop("Hard Link"));

	if (hardlink_enable == FALSE)
	{
		gtk_widget_set_sensitive(button, FALSE);
		active = FALSE;
	}

	else
	{
		GuiSignalConnectSwapped(fs->ok_button, "clicked", gui_hardlink_callback, button);
		active = hardlink;
	}

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), active);
#endif
	if (list->nscrec == NULL)
	{
		str = SaveFolderDialog(fs, button, list);
	}

	else
	{
		str = SaveFileDialog(fs, button, list);
	}

#ifdef HAVE_CYGWIN
	/* fileselector will crash if name is too long */
	gtk_entry_set_max_length(GTK_ENTRY(fs->selection_entry), NAME_MAX / 2);
#endif
	gtk_file_selection_set_filename(fs, str);

	g_free(str);

	gtk_widget_show_all(widget);

} /* static void SaveAs(GtkWidget*, GtkWidget**) */

#ifndef CACHE_READONLY

static GSList *gui_tree_append_parents(GSList *node_list, GuiTreeStore *tree)
{
	GSList *last;

#ifdef CONFIG_GTKCLIST_GTKCTREE

	GtkCTreeNode	*parent;
	GtkCTreeRow	*row;

	last = g_slist_last(node_list);
	row  = GTK_CTREE_ROW(last->data);

	while ((parent = row->parent) != NULL)
	{
		info_dialog_remove_entry(parent);
		node_list = g_slist_append(node_list, parent);
		row	  = GTK_CTREE_ROW(parent);
	}
#else
 	GtkTreeIter	*child, *parent;
	GtkTreeModel	*model;

	/* the selected node */
	last   = g_slist_last(node_list);
	child  = (GtkTreeIter *) last->data;
	model  = GTK_TREE_MODEL(tree->stree);

	/* all parent nodes till reaching toplevel */
	while (parent = g_malloc(sizeof(GtkTreeIter)),
			gtk_tree_model_iter_parent(model, parent, child))
	{
		info_dialog_remove_entry(parent);
		node_list = g_slist_append(node_list, parent);
		child     = parent;
	}

	g_free(parent);
#endif
	return node_list;

} /* static GSList *gui_tree_append_parents(GSList*, GuiTreeStore*) */

#ifndef CONFIG_GTKCLIST_GTKCTREE

static void gui_store_remove_node(ListSelection *list, GtkTreeIter *iter)
{
	GtkTreeModel		*model;
	GtkTreeSelection	*selection;
	GtkTreeIter		*parent;
	GtkTreeIter		parent_iter;

	if (list->type == kTreeList)
	{
		model = GTK_TREE_MODEL(list->store.tree.stree);
		selection = list->store.tree.selection;
	}

	else
	{
		model = GTK_TREE_MODEL(list->store.list.slist);
		selection = list->store.list.selection;
	}

	 parent = &parent_iter;
	*parent = *iter;

	if (gtk_tree_model_iter_has_child(model, parent) == FALSE)
	{
		GtkTreeIter next_iter;

		GtkTreeIter *next = NULL;

		if (gtk_tree_selection_iter_is_selected(selection, parent))
		{
			for (;;)
			{
				gint num;

				next_iter = *parent;

				if (gtk_tree_model_iter_next(model, &next_iter))
				{
					next = &next_iter;
					break;
				}

				next_iter = *parent;

				if (gtk_tree_model_iter_parent(model, parent,
								&next_iter) == FALSE)
				{
					parent = NULL;
				}

				num = gtk_tree_model_iter_n_children(model, parent);

				if (num > 1)
				{
					if (gtk_tree_model_iter_nth_child(model,
							&next_iter, parent, num - 2))
					{
						next = &next_iter;
					}

					break;
				}

				if (parent == NULL)
				{
					break;
				}

			} /* for (;;) */

		} /* if (gtk_tree_selection_iter_is_selected(selection, parent)) */

		if (list->type == kTreeList)
		{
			gtk_tree_store_remove(list->store.tree.stree, iter);
		}

		else
		{
			gtk_list_store_remove(list->store.list.slist, iter);
		}

		if (next)
		{
			gtk_tree_selection_select_iter(selection, next);
		}

	} /* if (gtk_tree_model_iter_has_child(model, parent) == FALSE) */

} /* static void gui_store_remove_node(ListSelection*, GtkTreeIter*) */

#endif /* #ifndef CONFIG_GTKCLIST_GTKCTREE */

static void gui_list_remove_node_func(ListSelection *list, nscache_record_t *rec)
{
	if (rec && list->type == kSortedList)
	{

#ifdef CONFIG_GTKCLIST_GTKCTREE

		gint     row;
		GtkCList *clist;

		clist = (list->store).list.clist;
		row   = gtk_clist_find_row_from_data(clist, rec);

		if (row >= 0)
		{
			GList *selection = clist->selection;

			if (selection && selection->data == (gpointer) row)
			{
				gint next = row + 1;

				if (next >= clist->rows)
				{
					next = row - 1;
				}

				gtk_clist_select_row(clist, next, 0);
			}

			gtk_clist_remove(clist, row);
		}
#else
		GtkTreeIter *iter = guitl_list_find_by_row_data(&(list->store.list), rec);

		if (iter)
		{
			gui_store_remove_node(list, iter);
			g_free(iter);
		}

#endif /* #ifdef CONFIG_GTKCLIST_GTKCTREE */

	} /* if (rec && list->type == kSortedList) */

} /* static void gui_list_remove_node_func(ListSelection*, gpointer) */

static void gui_tree_remove_node_func(gpointer node, ListSelection *list)
{

#ifdef CONFIG_GTKCLIST_GTKCTREE

	if (GTK_CTREE_ROW(node)->children == NULL)
	{
		GtkCTree	*ctree;
		GList		*selection;

		ctree     = (list->store).tree.ctree;
		selection = GTK_CLIST(ctree)->selection;

		if (selection && selection->data == node)
		{
			GtkCTreeNode *next, *parent;

			next = parent = node;

			for (;;)
			{
				GtkCTreeNode *child = GTK_CTREE_ROW(parent)->children;

				if (child)
				{
					if (next != child)
					{
						next = child;
						break;
					}

					next = GTK_CTREE_ROW(child)->sibling;

					if (next)
					{
						break;
					}
				}

				next = GTK_CTREE_ROW(parent)->sibling;

				if (next)
				{
					break;
				}

				next   = parent;
				parent = GTK_CTREE_ROW(parent)->parent;

				if (parent == NULL)
				{
					next = gtk_ctree_node_nth(ctree, 0);
					break;
				}
			}

			if (next)
			{
				gtk_ctree_select(ctree, next);
			}
		}

		gtk_ctree_remove_node(ctree, node);

	} /* if (GTK_CTREE_ROW(node)->children == NULL) */
#else
	gui_store_remove_node(list, node);
	g_free(node);

#endif /* #ifdef CONFIG_GTKCLIST_GTKCTREE .. #else */

} /* static void gui_tree_remove_node_func(gpointer, gpointer) */

static void gui_remove_nscache_record_recursive(GuiTreeStore *tree,
				nscache_record_t* rec, GSList **node_list)
{
	if (rec)
	{
		if (nscache_record_delete(rec))
		{
			return;
		}

		info_dialog_remove_entry(rec);

		g_slist_foreach(list_selection_slist,
			(GFunc) gui_list_remove_node_func, rec);
	}

#ifdef CONFIG_GTKCLIST_GTKCTREE
	*node_list = g_slist_prepend(*node_list, tree->node);
#else
	{
		GtkTreeIter *iter = g_memdup(&(tree->iter), sizeof(GtkTreeIter));
		*node_list = g_slist_prepend(*node_list, iter);
	}
#endif

} /* static void gui_remove_nscache_record_recursive(GuiTreeStore*,
						nscache_record_t*, GSList**) */

static void Delete(GtkWidget *ok_btn, ListSelection *list)
{
	GuiTreeStore	*tree;
	GSList		*node_list;

	GtkWidget *dialog = gtk_widget_get_toplevel(ok_btn);
	gtk_widget_destroy(dialog);

#ifdef CONFIG_GTKCLIST_GTKCTREE
	g_slist_foreach(list_selection_slist,
		(GFunc) gui_clist_freeze_or_thaw, (gpointer) gtk_clist_freeze);
#endif
	tree      = &(notebook_list[kTreeList].store.tree);
	node_list = NULL;

	if (list->nscrec)
	{
		if (nscache_record_delete(list->nscrec) == 0)
		{
			gpointer node;

			info_dialog_remove_entry(list->nscrec);
			node = guitl_tree_find_by_row_data(tree, list->nscrec);

			if (node)
			{
				node_list = g_slist_append(node_list, node);
			}

			g_slist_foreach(list_selection_slist,
				(GFunc) gui_list_remove_node_func, list->nscrec);
		}
	}

	else if (list->type == kTreeList)
	{
		guitl_tree_recursive(tree, TRUE, (GuiTreeStoreRecursiveFunc)
				gui_remove_nscache_record_recursive, &node_list);
	}

	if (node_list)
	{
		node_list = gui_tree_append_parents(node_list, tree);
		g_slist_foreach(node_list, (GFunc) gui_tree_remove_node_func,
							&notebook_list[kTreeList]);
		g_slist_free(node_list);

		g_slist_foreach(search_dialog_slist,
			(GFunc) search_update_window_title, NULL);

		print_node_info(NULL);
	}

#ifdef CONFIG_GTKCLIST_GTKCTREE
	g_slist_foreach(list_selection_slist,
		(GFunc) gui_clist_freeze_or_thaw, (gpointer) gtk_clist_thaw);
#endif

} /* static void Delete(GtkWidget*, ListSelection*) */

static GtkWidget *delete_dialog_get_label(ListSelection *list)
{
	static const gchar rm_file[] =
		gettext_nop("You are in progress to delete the following file:");

	static const gchar rm_folder[] =
		gettext_nop("You are in progress to delete the following folder:");

	GtkWidget	*label;
	const gchar	*what;
	gchar		*str, *name;
	gchar		pom[kMessageBufferSize], buf[64];

	name = list->name;

	if (list->nscrec == NULL)
	{
		what = rm_folder;
	}

	else
	{
		what = rm_file;

		if (list->type == kSortedList)
		{
			name = strrchr(name, '/');
		}
	}

	if (name == NULL || (*(name) == '/' && *(++name) == '\0'))
	{
		name = (gchar *) &index_html_str[1];
	}

	if (strlen(name) > 32)
	{
		name = strncpy(buf, name, 27);
		strcpy(&buf[27], "(...)");
	}

	g_snprintf(pom, sizeof(pom), "%s\n<<< %s >>>\n%s",
			gettext(what), name, gettext("Do you like to continue?"));

	str = CHARCONV_UTF8_FROM_LOCALE(pom);
	label = gtk_label_new(str == NULL ? name : str);
	CHARCONV_UTF8_FREE(str, pom);

	return label;
}

static void DeleteDialog(GtkWidget *w, GtkWidget **mbar)
{
	GtkWidget *button, *dialog, *hbox, *image, *label;

	ListSelection *list = windata.toplist;

	if(list == NULL || list->name == NULL)
	{
		return;
	}

	hbox = gtk_hbox_new(FALSE, 0);
	dialog = guitl_dialog_new_gettext(gettext_nop("MozCache: Delete"), hbox, 10);

	guitl_dialog_set_modal(dialog, mbar);
	GuiWindowSetResizable(GTK_WINDOW(dialog), FALSE);

	image = guitl_get_icon_dialog(kIconDialogWarning);
	gtk_container_add(GTK_CONTAINER(hbox), image);

	label = delete_dialog_get_label(list);
	gtk_container_add(GTK_CONTAINER(hbox), label);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_CENTER);
	GuiWidgetSetSizeRequest(label, -1, 75);

	hbox = guitl_hbutton_box_new(GTK_DIALOG(dialog)->action_area,
							8, GTK_BUTTONBOX_END);

	button = guitl_button_box_add_icon_button(hbox, kIconButtonCancel);
	GuiSignalConnectSwapped(button, "clicked", gtk_widget_destroy, dialog);
	gtk_widget_grab_default(button);

	button = guitl_button_box_add_icon_button(hbox, kIconButtonDelete);
	GuiSignalConnect(button, "clicked", Delete, windata.toplist);

	gtk_widget_show_all(dialog);

} /* static void DeleteDialog(GtkWidget*, GtkWidget**) */

#endif /* ! CACHE_READONLY */

static GtkWidget *gui_properties_fill_frame(gchar *title, gchar *str)
{
	gchar		*utf;
	GtkWidget	*label;

	utf = CHARCONV_UTF8_FROM_LOCALE(str);
	label = gtk_label_new(utf);
	CHARCONV_UTF8_FREE(utf, str);

	gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
	gtk_misc_set_padding(GTK_MISC(label), 8, 8);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_CENTER);

	return guitl_frame_new_add_child_gettext(title, label);
}

static void PropertiesDialog(GtkWidget *w, GtkWidget **mbar)
{
	gint		icon_idx, len;
	GtkWidget	*bbox, *button, *dlg, *frame, *hbox, *image, *vbox;
	const gchar	*noav_str;
	gchar		*str;
	CacheDB		*cache_db;
	gchar		pom[kMessageBufferSize], timebuf[30];

	union
	{
		nscache_record_t	rec;
		struct stat		pstat;
	} u;

	hbox = gtk_hbox_new(FALSE, 5);
	dlg = guitl_dialog_new_gettext(gettext_nop("MozCache: Properties"), hbox, 10);

	guitl_dialog_set_modal(dlg, mbar);
	gtk_window_set_default_size(GTK_WINDOW(dlg), -1, -1);
	GuiWindowSetResizable(GTK_WINDOW(dlg), FALSE);

	bbox = guitl_hbutton_box_new(GTK_DIALOG(dlg)->action_area, 8, GTK_BUTTONBOX_END);
	button = guitl_button_box_add_icon_button(bbox, kIconButtonClose);
	GuiSignalConnectSwapped(button, "clicked", gtk_widget_destroy, dlg);
	gtk_widget_grab_default(button);

	noav_str = gettext(na_str);
	icon_idx = kIconDialogError;
	cache_db = &nscache_db;

	if (cache_db->name == NULL)
	{
		len = g_snprintf(pom, sizeof(pom), "%s",
			gettext("Cache Index File not specified!"));
	}

	else
	{
		len = g_snprintf(pom, sizeof(pom), "%s\n%s: %s",
				cache_db->name, gettext("Version"),
					(cache_db->version == NULL ?
						noav_str : cache_db->version));

		if (len > 0 && len < sizeof(pom))
		{
#ifdef HAVE_MSIE
		if (nscache_db.type == NSCACHE_RT_MSIE)
		{
			icon_idx = kIconDialogInfo;
		}
		else
		{
#endif
			if (u.rec.filename = cache_db->name,
				u.rec.content_offset = 0,
					file_status(&(u.rec), &(u.pstat)) != 0)
			{
				len += g_snprintf(&pom[len], sizeof(pom) - len, "\n%s\n%s",
					gettext("Can't get status of Cache Index File"),
								strerror(errno));
			}

			else
			{
				gint		index_acc;
				const gchar	*byte_ptr, *srd, *swr;

				unsigned long size = u.pstat.st_size;

				if (size >= 1024)
				{
					size /= 1024;
					byte_ptr = gettext("kB");
				}

				else
				{
					byte_ptr = gettext("Bytes");
				}

				index_acc = file_access_index();

				if (index_acc < 0)
				{
					srd = swr = noav_str;
				}

				else
				{
					srd = swr = gettext(yes_str);

					if ( ! (index_acc & FILE_ACCESS_READ))
					{
						srd = gettext(no_str);
					}

					if ( ! (index_acc & FILE_ACCESS_WRITE))
					{
						swr = gettext(no_str);
					}
				}

				str = time_to_string(timebuf, sizeof(timebuf),
								&(u.pstat.st_mtime));
				if (str == NULL)
				{
					str = (gchar *) noav_str;
				}

				len += g_snprintf(&pom[len], sizeof(pom) - len,
							"\n%s: %lu %s\n"
							"%s: %s\n"
							"%s: %s\n"
							"%s: %s",
					gettext("Size"), size, byte_ptr,
					gettext("Readable"), srd,
					gettext("Writable"), swr,
					gettext("Modification time"), str);

				icon_idx = kIconDialogInfo;
			}
#ifdef HAVE_MSIE
		} /* if (nscache_db.type == NSCACHE_RT_MSIE) .. else */
#endif
		} /* if (len > 0 && len < sizeof(pom)) */
	}

	image = guitl_get_icon_dialog(icon_idx);
	gtk_container_add(GTK_CONTAINER(hbox), image);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(hbox), vbox);

	str = pom;
	str[sizeof(pom) - 1] = '\0';
	frame = gui_properties_fill_frame(gettext_nop("Cache Index File"), str);
	gtk_container_add(GTK_CONTAINER(vbox), frame);

	len = get_node_info(pom, sizeof(pom), kNodeInfoFormatMultiLine, NULL);

	if (len > 0 && len < sizeof(pom) && cache_db->name)
	{
		str = time_to_string(timebuf, sizeof(timebuf),
						&(cache_db->access_time));
		if (str == NULL)
		{
			str = (gchar *) noav_str;
		}

		g_snprintf(&pom[len], sizeof(pom) - len,
				"\n%s: %s", gettext("Access time"), str);
	}

	str = pom;
	str[sizeof(pom) - 1] = '\0';
	frame = gui_properties_fill_frame(gettext_nop("Contents"), str);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 10);

	gtk_widget_show_all(dlg);

} /* static void PropertiesDialog(GtkWidget*, GtkWidget**) */

static void NodeInfo(GtkWidget *w, ListSelection **selection)
{
	print_node_info(*selection);
}

static void chmod_recursive_func(GuiTreeStore *tree, nscache_record_t *rec, gpointer data)
{
	if (rec && rec->content_offset == 0)
	{
		struct stat estat;

		struct
		{
			gint		rdwr;
			const gchar	*errstr;

		} *s = data;

		if (file_status(rec, &estat) == 0)
		{
			if (s->rdwr)
			{
				estat.st_mode |= S_IWUSR;
			}

			else
			{
				estat.st_mode &= ~S_IWUSR;
			}

			estat.st_mode |= S_IRUSR;

			if (file_chmod(rec, estat.st_mode))
			{
				s->errstr = "Can't change mode for any of cache files";
			}
		}

		else
		{
			s->errstr = "Some of cache file doesn't exist";
		}
	}
}

static void chmod_recursive(gint rdwr, ListSelection **selected)
{
	ListSelection *list = *selected;

	if (list)
	{
		struct
		{
			gint		rdwr;
			const gchar	*errstr;
		} s;

		s.rdwr   = rdwr;
		s.errstr = NULL;

		if (list->nscrec == NULL && list->type == kTreeList)
		{
			/* folder */
			guitl_tree_recursive(&(list->store.tree), TRUE,
				(GuiTreeStoreRecursiveFunc) chmod_recursive_func, &s);
		}

		else
		{
			/* single file */
			chmod_recursive_func(NULL, list->nscrec, &s);
		}

		gui_err(gettext(s.errstr));
	}
}

static void RdOnly(GtkWidget *w, ListSelection **selected)
{
	chmod_recursive(FALSE, selected);
}

static void Writable(GtkWidget *w, ListSelection **selected)
{
	chmod_recursive(TRUE, selected);
}

static gint watch_viewer_status(gpointer data)
{
	struct
	{
		guint	tid;
		gint	limit;
		gchar*	appassign;

	} *viewer_timeout = data;

	if (viewer_status)
	{
		gui_errfmt("%s : %s", viewer_timeout->appassign, strerror(viewer_status));
		viewer_timeout->limit = 0;
	}

	if (--viewer_timeout->limit <= 0)
	{
		g_free(viewer_timeout->appassign);
		viewer_timeout->appassign = NULL;
		viewer_timeout->tid = 0;

		return FALSE;
	}

	return TRUE;
}

static void _start_viewer(const gchar *appassign, gint viewer_type)
{
	static struct
	{
		guint	tid;
		gint	limit;
		gchar*	appassign;

	} viewer_timeout = {0, 0, NULL};

	ListSelection		*list = windata.toplist;
	const nscache_record_t	*rec  = list->nscrec;
	const gchar		*url  = NULL;

	if (viewer_type == VIEWER_URL)
	{
		if (rec == NULL)
		{
			gchar buf[2048];

			if (list->type != kTreeList)
			{
				return;
			}

			gui_selected_node_to_url(buf, sizeof(buf),
							&(list->store).tree);
			buf[sizeof(buf) - 1] = '\0';
			url = buf;
		}

		else
		{
			url = rec->urlstr;
		}
	}

	if (viewer_start(appassign, url, rec))
	{
		static gchar *viewer_error_msg[] =
		{
			gettext_nop("URL viewer can't be launched"),
			gettext_nop("File viewer can't be launched")
		};

		gui_strerror(viewer_error_msg[viewer_type]);
	}

	else
	{
		static gchar *viewer_launched_msg[] =
		{
			gettext_nop("URL viewer launched"),
			gettext_nop("File viewer launched")
		};

		/*
			The viewer was successfully started. Now
			we have a look at the status every 200 ms
			25 times - maybe the viewer will finish
			with an error, and we can inform the user
			with a message - if so.
		*/

		viewer_timeout.limit = 25;

		if(viewer_timeout.tid)
		{
			GuiTimeoutRemove(viewer_timeout.tid);
			g_free(viewer_timeout.appassign);
		}

		viewer_timeout.appassign = g_strdup(appassign);

		viewer_timeout.tid = GuiTimeoutAdd(200,
				watch_viewer_status, &viewer_timeout);

		gui_msg(gettext(viewer_launched_msg[viewer_type]));
		gui_err(NULL);
	}
}

static void ViewFile(GtkWidget *w, const gchar *args_file)
{
	_start_viewer(args_file, VIEWER_FILE);
}

static void ViewURL(GtkWidget *w, const gchar *args_url)
{
	_start_viewer(args_url, VIEWER_URL);
}

static gint ViewerDefault(gint viewer_type)
{
	ListSelection *list = windata.toplist;

	if (list && list->nscrec)
	{
		GSList *ptr, *slist;

		slist = ptr =
		apassign_get_matching_viewers((list->nscrec)->content_type);

		while (ptr)
		{
			const gchar *app;
			apassign_t  *e;

			e = ptr->data;

			if (viewer_type == VIEWER_FILE)
			{
				int cache_acc = file_access(list->nscrec);

				if (cache_acc <= 0 || ! (cache_acc & FILE_ACCESS_READ))
				{
					break;
				}

				app = e->args_file;
			}

			else
			{
				app = e->args_url;
			}

			if (app)
			{
				_start_viewer(app, viewer_type);
				break;
			}

			ptr = ptr->next;
		}

		g_slist_free(slist);
	}

	return TRUE;
}

static void GenViewerMenu(GtkWidget *submenu, gpointer data)
{
	gint		viewer_type;
	const gchar	*content_type;
	GSList		*ptr, *mlabels, *slist;
	ListSelection	*list;

	while (GTK_MENU_SHELL(submenu)->children)
	{
		gtk_widget_destroy(GTK_WIDGET(GTK_MENU_SHELL(submenu)->children->data));
	}

	list = windata.toplist;

	if (list == NULL)
	{
		return;
	}

	viewer_type = (gint) data;

	if (list->nscrec == NULL)
	{
		if (viewer_type == VIEWER_FILE)
		{
			return;
		}

		content_type = "text/html";
	}

	else
	{
		content_type = (list->nscrec)->content_type;
	}

	mlabels	= NULL;
	slist	= apassign_get_matching_viewers(content_type);
	ptr	= slist;

	while (ptr)
	{
		apassign_t *e = ptr->data;

		gchar *app = (viewer_type == VIEWER_FILE) ?
					e->args_file : e->args_url;
		if (app)
		{
			gchar *s, *str;

			str = strrchr(app, ' ');

			if ((s = str) != NULL)
			{
				while (*(++s) == ' ') {;}

				if (strcmp(s, uof_param_str[viewer_type]))
				{
					str = NULL;
				}
			}

			if (str > app)
			{
				str = g_strndup(app, str - app);
			}

			else
			{
				str = g_strdup(app);
			}

			if (g_slist_find_custom(mlabels, str,
					(GCompareFunc) strcmp) == NULL)
			{
				gchar *utf;

				mlabels = g_slist_append(mlabels, str);

				utf = CHARCONV_UTF8_FROM_LOCALE(str);

				if (utf)
				{
					GtkWidget *mi;

					mi = gtk_menu_item_new_with_label(utf);
					CHARCONV_UTF8_FREE(utf, str);

					GuiMenuAppend(submenu, mi);
					gtk_widget_show(mi);

					GuiSignalConnect(mi, "activate",
						(viewer_type == VIEWER_FILE ?
							ViewFile : ViewURL), app);
				}
			}
		}

		ptr = ptr->next;
	}

	g_slist_foreach(mlabels, (GFunc) g_free, NULL);
	g_slist_free(mlabels);
	g_slist_free(slist);
}

static void SelectionSetURL(GtkWidget *w, ListSelection **selected)
{
	gchar		*url;
	ListSelection	*list;

	list = *selected;

	if (list && list->name)
	{
		if (list->nscrec == NULL)
		{
			gchar buf[2048];

			if (list->type != kTreeList)
			{
				return;
			}

			gui_selected_node_to_url(buf, sizeof(buf),
							&(list->store).tree);
			buf[sizeof(buf) - 1] = '\0';
			url = buf;
		}

		else
		{
			url = (list->nscrec)->urlstr;
		}

		if (url)
		{
			g_free(selection_urlstr);

			if (gtk_selection_owner_set(windata.mainwin,
					GDK_SELECTION_PRIMARY, GDK_CURRENT_TIME))
			{
				selection_urlstr = g_strdup(url);
			}

			else
			{
				selection_urlstr = NULL;

				if (gdk_selection_owner_get(GDK_SELECTION_PRIMARY) ==
								windata.mainwin->window)
				{
					gtk_selection_owner_set(NULL,
						GDK_SELECTION_PRIMARY, GDK_CURRENT_TIME);
				}
			}
		}
	}
}

static gint SelectionClear(GtkWidget *w, GdkEventSelection *event, gchar **urlstr)
{
	gchar *str = *urlstr;

	*urlstr = NULL;
	g_free(str);

	return FALSE;
}

static void SelectionSend(GtkWidget *w, GtkSelectionData *selection_data,
						guint info, guint time, gchar **urlstr)
{
	GdkAtom type;

	if (info >= SEL_TYPE_FIRST && info <= SEL_TYPE_LAST)
	{
		type = gdk_atom_intern(targetlist[info - 1].target, FALSE);
	}

	else
	{
		type = GDK_NONE;
	}

	gtk_selection_data_set(selection_data, type, 8 * sizeof(gchar),
						*urlstr, strlen(*urlstr));
}

static gint info_dialog_find_node_func(const InfoDialogEntry *entry,
						gconstpointer rec_or_node)
{
#ifdef CONFIG_GTKCLIST_GTKCTREE
	return entry->u.node - rec_or_node;
#else
	if (entry->type == kInfoDialogRec ||
			entry->u.node == NULL || rec_or_node == NULL)
	{
		/* either 'kInfoDialogRec' or at least one NULL */
		return entry->u.node - rec_or_node;
	}

	/* none NULL */
	return memcmp(entry->u.node, rec_or_node, sizeof(GtkTreeIter));
#endif
}

static gint info_dialog_find_dialog_func(const InfoDialogEntry *entry, GtkWidget *widget)
{
	return entry->dialog - widget;
}

static gboolean info_dialog_free_entry_func(InfoDialogEntry *entry, gboolean no_entry)
{
	if (entry->type == kInfoDialogNode)
	{
		/* keep 'no entry' dialog (if wanted) */
		if (entry->u.node == NULL && no_entry == FALSE)
		{
			return FALSE;
		}

#ifndef CONFIG_GTKCLIST_GTKCTREE
		g_free(entry->u.node);
#endif
	}

	g_free(entry);

	return TRUE;
}

#ifndef CACHE_READONLY
static void info_dialog_remove_entry(gconstpointer rec_or_node)
{
	GSList *found, *slist;

	slist = info_dialog_slist;
	found = g_slist_find_custom(slist, (gpointer) rec_or_node,
				(GCompareFunc) info_dialog_find_node_func);

	if (found)
	{
		InfoDialogEntry *entry = found->data;

		if (info_dialog_free_entry_func(entry, FALSE))
		{
			info_dialog_slist = g_slist_remove(slist, entry);
		}
	}
}
#endif /* ! CACHE_READONLY */

static void info_dialog_remove_all(void)
{
	GSList *slist, *no_entry;

	/* keep 'no entry' dialog */
	slist    = info_dialog_slist;
	no_entry = g_slist_find_custom(slist, NULL,
				(GCompareFunc) info_dialog_find_node_func);
	slist = g_slist_remove_link(slist, no_entry);
	g_slist_foreach(slist, (GFunc) info_dialog_free_entry_func, FALSE);
	g_slist_free(slist);
	info_dialog_slist = no_entry;
}

static void info_dialog_destroyed(GtkWidget *dialog, GSList **slist)
{
	GSList *found;

	found = g_slist_find_custom(*slist, (gpointer) dialog,
				(GCompareFunc) info_dialog_find_dialog_func);
	if (found)
	{
		InfoDialogEntry *entry = found->data;

		info_dialog_free_entry_func(entry, TRUE);
		*slist = g_slist_remove(*slist, entry);
	}
}

static GtkWidget *info_dialog_label_new(const gchar *pom)
{
	gchar		*str;
	GtkWidget	*label;

	str = CHARCONV_UTF8_FROM_LOCALE(pom);
	label = gtk_label_new(str);
	CHARCONV_UTF8_FREE(str, pom);

	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
	gtk_misc_set_padding(GTK_MISC(label), 12, 12);

	return label;
}

static void gui_content_type_add_func(GuiTreeStore *tree,
				nscache_record_t *rec, gpointer data)
{
	nscache_content_types_add(rec, data);
}

static GtkWidget *gui_content_type_label_new(gchar *buf, gint max,
					GSList *ctype, const gchar *title)
{
	gint len = g_snprintf(buf, max, "%s\n\n", gettext(title));

	while (ctype)
	{
		gint sz;
		nscache_typecount_t *tcount = ctype->data;

		sz = g_snprintf(&buf[len], max - len,
				"%s: %u\n", tcount->str, tcount->num);

		if (sz <= 0 || (len += sz) >= max)
		{
			break;
		}

		ctype = ctype->next;
	}

	buf[max - 1] = '\0';

	return info_dialog_label_new(buf);
}

static gint gui_selected_node_to_url(gchar *buf, gint max, GuiTreeStore *tree)
{
	gint		len, sz;
	gchar		*fname;
	const gchar	*fmt_str;
	GSList		*url, *ptr;

	url = NULL;

#ifdef CONFIG_GTKCLIST_GTKCTREE
	{
		GtkCTreeNode	*parent;
		GtkCTreeRow	*row;

		parent = tree->node;

		do
		{
			gtk_ctree_node_get_pixtext(tree->ctree,
						parent, kColumnURL,
							&fname, NULL, NULL, NULL);

			url = g_slist_prepend(url, fname);
			row = GTK_CTREE_ROW(parent);
		}
		while ((parent = row->parent) != NULL);
	}
#else
	{
 		GtkTreeIter	i1, i2, *child, *parent;
		GtkTreeModel	*model;

		/* the selected node */
		parent = memcpy(&i1, &(tree->iter), sizeof(GtkTreeIter));
		child  = &i2;

		model = GTK_TREE_MODEL(tree->stree);

		/* all parent nodes till reaching toplevel */

		do
		{
 			GtkTreeIter *temp;

			gtk_tree_model_get(model, parent, kColumnURL, &fname, -1);

			/* fname must be free'd */
			url 	= g_slist_prepend(url, fname);
			temp 	= child;
			child	= parent;
			parent	= temp;
		}
		while (gtk_tree_model_iter_parent(model, parent, child));
	}
#endif

	len	= 0;
	fmt_str	= "%s://";
	ptr	= url;

	while (ptr)
	{
		sz = g_snprintf(&buf[len], max - len, fmt_str, ptr->data);

		if (sz <= 0 || (len += sz) >= max)
		{
			break;
		}

		fmt_str	= "%s/";
		ptr	= ptr->next;
	}

#ifndef CONFIG_GTKCLIST_GTKCTREE
	g_slist_foreach(url, (GFunc) g_free, NULL);
#endif
	g_slist_free(url);

	return len;
}

static GtkWidget *info_dialog_label_node(ListSelection *list, gconstpointer node)
{
	gint		len;
	GtkBox		*hbox;
	GtkWidget	*label, *vbox, *widget;
	gchar		pom[kMessageBufferSize];

	if (node == NULL)
	{
		g_snprintf(pom, sizeof(pom), "%s\n\n", gettext("no entry"));
	}

	else
	{
		gui_selected_node_to_url(pom, sizeof(pom), &(list->store).tree);
	}

	vbox = gtk_vbox_new(FALSE, 0);
	label = info_dialog_label_new(pom);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	widget = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), widget, FALSE, FALSE, 0);
	hbox = GTK_BOX(widget);

	len = g_snprintf(pom, sizeof(pom), "%s\n\n", gettext("Contents"));
	get_node_info(&pom[len], sizeof(pom) - len, kNodeInfoFormatMultiLine, list);
	pom[sizeof(pom) - 1] = '\0';
	label = info_dialog_label_new(pom);
	gtk_box_pack_start(hbox, label, FALSE, FALSE, 0);

	if (node != NULL)
	{
		nscache_content_types_t *content_types = NULL;

		guitl_tree_recursive(&(list->store).tree, TRUE,
			(GuiTreeStoreRecursiveFunc) gui_content_type_add_func,
								&content_types);
		if (content_types == NULL)
		{
			return vbox;
		}

		label = gui_content_type_label_new(pom, sizeof(pom),
				content_types->content_type, gettext_nop("MIME type"));
		gtk_box_pack_start(hbox, label, FALSE, FALSE, 0);

		label = gui_content_type_label_new(pom, sizeof(pom),
				content_types->content_encoding, gettext_nop("Encoding"));
		gtk_box_pack_start(hbox, label, FALSE, FALSE, 0);

		label = gui_content_type_label_new(pom, sizeof(pom),
				content_types->charset, gettext_nop("Charset"));
		gtk_box_pack_start(hbox, label, FALSE, FALSE, 0);

		nscache_content_types_free(content_types);

	} /* if (list != NULL) */

	return vbox;
}

static GtkWidget *info_dialog_label_rec(const nscache_record_t *rec)
{
	const gchar	*noav_str, *path;
	gchar		mdtm[30], atm[30], exptm[30];
	gchar		pom[kMessageBufferSize];

#ifdef HAVE_MSIE
	if (rec->content_offset == MSIE_CACHE)
	{
		path = MSIE_GET_PATHNAME(rec);
	}
	else
#endif
		path = nscache_db.path.str;

	noav_str = gettext(na_str);

	g_snprintf(pom, sizeof(pom), "%s: %s\n"
						"%s: %s\n"
						"%s: %s\n"
						"%s: %s\n"
						"%s: %d %s\n"
						"%s: %s\n"
						"%s: %s\n"
						"%s: %s\n"
						"%s: %s\n"
						"%s: %s\n",
		gettext("Source URL"), rec->urlstr,
		gettext("Local cache file"), rec->filename,
		gettext("Path"), path,
		gettext("MIME type"), (rec->content_type ?
				rec->content_type : noav_str),
		gettext("Size"), rec->content_length, gettext("Bytes"),
		gettext("Encoding"), (rec->content_encoding ?
				rec->content_encoding : noav_str),
		gettext("Charset"), (rec->charset ?
					rec->charset : noav_str),
		gettext("Modification time"),
			(time_to_string(mdtm, sizeof(mdtm), &(rec->last_modified)) ?
									mdtm : noav_str),
		gettext("Access time"),
			(time_to_string(atm, sizeof(atm), &(rec->last_accessed)) ?
									atm : noav_str),
		gettext("Expiration time"),
			(time_to_string(exptm, sizeof(exptm), &(rec->expires)) ?
									exptm : noav_str));
	pom[sizeof(pom) - 1] = '\0';

	return info_dialog_label_new(pom);
}

static void ViewInfo(GtkWidget *w, ListSelection **selected)
{
	static const gchar info_title[] = gettext_nop("MozCache: entry info window");

	gpointer	node;
	GSList		*found;
	GtkWidget	*button, *dialog, *hbox, *swin, *widget;
	ListSelection	*list;
	InfoDialogEntry	*entry;

	nscache_record_t *rec;

	if (selected == NULL)
	{
		return;
	}

	rec  = NULL;
	list = *selected;

	if (list == NULL || list->name == NULL || list->name[0] == '\0')
	{
		node = NULL;
	}

	else if ((node = rec = list->nscrec) == NULL)
	{
#ifdef CONFIG_GTKCLIST_GTKCTREE
		node = (list->store.tree).node;
#else
		node = &(list->store.tree).iter;
#endif
	}

	found = g_slist_find_custom(info_dialog_slist, node,
				(GCompareFunc) info_dialog_find_node_func);
	if (found)
	{
		GuiWindowPresent(((InfoDialogEntry *) found->data)->dialog);
		return;
	}

	entry = g_malloc(sizeof(InfoDialogEntry));

	if (rec)
	{
		entry->type  = kInfoDialogRec;
		entry->u.rec = rec;

		widget = info_dialog_label_rec(rec);
	}

	else
	{
		entry->type   = kInfoDialogNode;
#ifdef CONFIG_GTKCLIST_GTKCTREE
		entry->u.node = node;
#else
		entry->u.node = g_memdup(node, sizeof(GtkTreeIter));
#endif
		widget = info_dialog_label_node(list, node);
	}

	swin = gtk_scrolled_window_new(NULL, NULL);
	dialog = guitl_dialog_new_gettext(info_title, swin, 0);

	entry->dialog = dialog;
	info_dialog_slist = g_slist_prepend(info_dialog_slist, entry);

	gui_window_setup(dialog, kPropIntInfoWinWidth);
	GuiSignalConnect(dialog, "destroy", info_dialog_destroyed, (gpointer) &info_dialog_slist);

	gtk_container_set_border_width(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), 3);
	GuiWidgetSetSizeRequest(swin, 300, 150);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swin),
					GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swin), widget);

	hbox = guitl_hbutton_box_new(GTK_DIALOG(dialog)->action_area, 0, GTK_BUTTONBOX_END);
	button = guitl_button_box_add_icon_button(hbox, kIconButtonClose);
	GuiSignalConnectSwapped(button, "clicked", gtk_widget_destroy, dialog);
	gtk_widget_grab_default(button);

	gtk_widget_show_all(dialog);

} /* static void ViewInfo(GtkWidget*, ListSelection**) */

#ifdef CONFIG_GTKCLIST_GTKCTREE
static gint gui_tree_compare_func(GtkCTree *ctree,
				GtkCTreeRow *r1, GtkCTreeRow *r2)
{
	if (r1->children && r2->children == NULL)
	{
		/* r1 is folder */
		return -1;
	}

	if (r2->children && r1->children == NULL)
	{
		/* r2 is folder */
		return 1;
	}

	/* r1 and r2 are both of the same type */
	return strcmp((r1->row.cell)->u.text, (r2->row.cell)->u.text);
}
#else
static gint gui_tree_compare_func(GtkTreeModel *model, GtkTreeIter *iter1,
					GtkTreeIter *iter2, gpointer userdata)
{
	gint		retv;
	gboolean	expander1, expander2;
	gchar		*name1, *name2;

	expander1 = gtk_tree_model_iter_has_child(model, iter1);
	expander2 = gtk_tree_model_iter_has_child(model, iter2);

	if (expander1 && expander2 == FALSE)
	{
		/* expander1 is folder */
		return -1;
	}

	if (expander2 && expander1 == FALSE)
	{
		/* expander2 is folder */
		return 1;
	}

	/* expander1 and expander2 are both of the same type */
	gtk_tree_model_get(model, iter1, kColumnURL, &name1, -1);
	gtk_tree_model_get(model, iter2, kColumnURL, &name2, -1);

	retv = strcmp(name1, name2);

	g_free(name1);
	g_free(name2);

	return retv;
}
#endif /* #ifdef CONFIG_GTKCLIST_GTKCTREE .. #else */

#ifdef CONFIG_GTKCLIST_GTKCTREE
static gint gui_list_compare_func(GtkCList *clist, GtkCListRow *r1, GtkCListRow *r2)
#else
static gint gui_list_compare_func(GtkTreeModel *model, GtkTreeIter *iter1,
					GtkTreeIter *iter2, gpointer userdata)
#endif
{
	nscache_record_t *rec1, *rec2;

#ifdef CONFIG_GTKCLIST_GTKCTREE
	rec1     = r1->data;
	rec2     = r2->data;
#else
	gtk_tree_model_get(model, iter1, kColumnRowData, &rec1, -1);
	gtk_tree_model_get(model, iter2, kColumnRowData, &rec2, -1);
#endif

	return nscache_compare_func(rec1, rec2);
}

static void gui_list_sort(GtkWidget *menu, gint *stype_ptr)
{
	if (stype_ptr)
	{
		gint sort_type = *stype_ptr;

		if (sort_type == nscache_sort_type)
		{
			sort_type ^= STYPE_REVERS_MASK;
			*stype_ptr = sort_type;
		}

		nscache_sort_type = sort_type;
		gprop_set_int(kPropIntSortType, sort_type);
	}

	GuiListColumnSort(&(notebook_list[kSortedList].store.list),
						0, gui_list_compare_func, NULL);
}

static gint traverse_free_node_func(GNode *node, gpointer data)
{
	g_free(node->data);

	return FALSE;
}

static gpointer get_parent_node(char **dname, GNode *parent)
{
	GNode	*node;
	char	*beg, *end, *fname;

	beg = tl_get_url_start(*dname);

	if (beg == NULL)
	{
		return NULL;
	}

	fname	= tl_get_url_filename(beg);
	end	= strchr(beg, ':');
	node	= parent;

	do
	{
		gint		slen;
		guint		ch;
		NodeHash	*child;

		 ch   = *end;
		*end  = '\0';
		 slen = end - beg;

		if ((node = g_node_first_child(node)) != NULL)
		{
			do
			{
				NodeHash *i = node->data;

				if (i && i->slen == slen && strcmp(i->str, beg) == 0)
				{
					parent = node;
					goto _next_node;
				}
			}
			while ((node = g_node_next_sibling(node)) != NULL);
		}

		child = g_malloc(sizeof(NodeHash) + slen);

		child->slen = slen;
		strcpy(child->str, beg);

		MainTreeAppendNode(&(notebook_list[kTreeList].store.tree),
						(parent->data == NULL ?
							parent->data :
							NodeHashGetNodePointer(parent->data)),
						beg, folder_icon[kIconFolder],
						folder_icon[kIconFolderOpen],
						NodeHashGetNodePointer(child));

		parent = g_node_append_data(parent, child);

_next_node:
		*end	= ch;
		beg 	= end;

		while (*(++beg) == '/') {;}

	}
	while (beg != fname && (end = strchr(beg, '/')) != NULL);

	if (*beg == '\0') 
	{
		beg--;
	}

	*dname = beg;

	return NodeHashGetNodePointer(parent->data);

} /* gpointer get_parent_node(char**, GNode*) */

static gint nscrec_into_list_and_tree(OpenData *data)
{
	if (data->slist == NULL)
	{
		gtk_main_quit();
		return FALSE;
	}

	do
	{
		gpointer	parent;
		gchar		*centry[10];
		gchar		size[15], mdtm[30], atm[30], exptm[30];

		nscache_record_t *rec;
		const PixbufIcon *icon;

		rec = (data->slist)->data;
		data->slist = g_slist_remove(data->slist, rec);

		if (gui_option[kOptionConvertToLower].active &&
						rec->content_offset == 0)
		{
			g_strdown(rec->filename);
		}

		centry[kColumnURL] = rec->urlstr;
		parent = get_parent_node(&centry[kColumnURL], data->folder_node);

		MainSetColumnEntryArray(centry, kColumnFile, rec->filename);
		MainSetColumnEntryArray(centry, kColumnType, rec->content_type);

		g_snprintf(size, sizeof(size), "%d", rec->content_length);
		MainSetColumnEntryArray(centry, kColumnSize, size);

		MainSetColumnEntryArray(centry, kColumnEncoding, rec->content_encoding);
		MainSetColumnEntryArray(centry, kColumnCharset, rec->charset);

		MainSetColumnEntryArray(centry, kColumnModify,
			time_to_string(mdtm, sizeof(mdtm), &(rec->last_modified)));

		MainSetColumnEntryArray(centry, kColumnAccess,
			time_to_string(atm, sizeof(atm), &(rec->last_accessed)));

		MainSetColumnEntryArray(centry, kColumnExpire,
			time_to_string(exptm, sizeof(exptm), &(rec->expires)));

		icon = guitl_get_icon_mime(rec->content_type);

		MainTreeInsertNode(&(notebook_list[kTreeList].store.tree),
							parent, centry, icon, rec);
		centry[kColumnURL] = rec->urlstr;

		MainListAppend(&(notebook_list[kSortedList].store.list),
							centry, icon, rec);
		data->processed++;
	}
	while (data->slist != NULL && data->processed % 5 != 0);

	GuiProgressBarSetValue(data->progress_bar,
		((gdouble) data->processed) / ((gdouble) data->total));

	return TRUE;

} /* static gint nscrec_into_list_and_tree(OpenData*) */

static void gui_list_disable_entries(ListSelection *list, gpointer data)
{
#ifndef CONFIG_GTKCLIST_GTKCTREE
	g_free(list->name);
#endif
	list->name   = NULL;
	list->nscrec = NULL;
}

static void gui_combo_set_popdown_mime(GtkCombo *combo, GList *mime_types)
{
	gchar *str = g_strdup(gtk_entry_get_text(GTK_ENTRY(combo->entry)));

	gtk_combo_set_popdown_strings(combo, mime_types);
	gtk_entry_set_text(GTK_ENTRY(combo->entry), str);

	g_free(str);
}

#ifndef CONFIG_GTKCLIST_GTKCTREE
static GSList *gui_columns_set_visible(GtkTreeView *view)
{
	gint colnr;
	GSList *slist;


	colnr = kColumnsTotal - 1;
	slist = NULL;

	do
	{
		ColumnTab *col_tab = &column_tab_array[colnr];

		if (col_tab->visible)
		{
			GtkTreeViewColumn* column;

			column = gtk_tree_view_get_column(view, col_tab->position);
			slist  = g_slist_prepend(slist, column);
		}
	}
	while (--colnr >= kColumnURL);

	return slist;
}
#endif

static gint gui_read_cache(gchar *dbname)
{
	gchar		*name;
	gint		width, height;
	GtkWidget	*dummy, *err_label, *progress_bar;
	GuiListStore	*list;
	GuiTreeStore	*tree;
	OpenData	data;

	if (gui_button_key_grab_add() == FALSE)
	{
		return FALSE;
	}

	gui_err(NULL);

	/* deferred message(s) */

	dummy = gui_msg_label_replace_with(GTK_WIDGET(status_msg.label),
							gtk_label_new(NULL));
	err_label = GTK_WIDGET(error_msg.label);
	width  = (err_label->allocation).width;
	height = (err_label->allocation).height;
	progress_bar = gui_msg_label_replace_with(err_label,
						gtk_progress_bar_new());

	GuiWidgetSetSizeRequest(progress_bar, width, height);
	data.progress_bar = GTK_PROGRESS_BAR(progress_bar);

	/* results are invalid now */

	info_dialog_remove_all();

	g_slist_foreach(search_dialog_slist, (GFunc) search_disable_entries, NULL);
	g_slist_free(search_dialog_slist);
	search_dialog_slist = NULL;

	list = &(notebook_list[kSortedList].store.list);
	tree = &(notebook_list[kTreeList].store.tree);

	/* free allocated memory */
	guitl_list_foreach_row_data(list, (GFunc) nscache_record_free, NULL);

	g_slist_foreach(list_selection_slist, (GFunc) gui_list_disable_entries, NULL);

	GuiTreeClear(tree);
	GuiListClear(list);

#ifdef CONFIG_COLUMN_BUTTONS
	button_collexp.is_active = FALSE;
	gtk_widget_set_sensitive(button_collexp.collapse, FALSE);
	gtk_widget_set_sensitive(button_collexp.expand, FALSE);
	GuiTreeSetSelectionMode(tree, GTK_SELECTION_EXTENDED);
#endif

	gtk_main_iteration_do(FALSE);

	data.slist = nscache_read_cache(dbname, TRUE);
	g_free(dbname);

	if (data.slist)
	{

#ifdef CONFIG_GTKCLIST_GTKCTREE

		/*
			Sometimes rows or parts of them will be drawn
			on the screen. CTree and CList are both frozen,
			but this still happens (expose-event?).
			The function pointers will now be replaced with
			'_noop', and after filling the lists they
			will be restored.
		*/

#if GTK_MAJOR_VERSION == 1
#define GUI_CLIST_GET_CLASS(obj)	GTK_CLIST_CLASS(GTK_OBJECT(obj)->klass)
#else
#define GUI_CLIST_GET_CLASS(obj)	GTK_CLIST_GET_CLASS(obj)
#endif
		GtkCListClass	*clist_class, *ctree_class;
		gpointer	clist_draw_row, ctree_draw_row;

		ctree_class = GUI_CLIST_GET_CLASS(tree->ctree);
		clist_class = GUI_CLIST_GET_CLASS(list->clist);

		ctree_draw_row = ctree_class->draw_row;
		clist_draw_row = clist_class->draw_row;

		ctree_class->draw_row = (gpointer) _noop;
		clist_class->draw_row = (gpointer) _noop;
#else
		GtkTreeView	*view;
		GSList		*columns;

		view  = (mainlist->type == kTreeList) ? tree->view : list->view;
		columns = gui_columns_set_visible(view);

		g_slist_foreach(columns,
			(GFunc) gtk_tree_view_column_set_visible, (gpointer) FALSE);

#endif /* CONFIG_GTKCLIST_GTKCTREE */

		data.folder_node = g_node_new(NULL);
		data.processed = 0;
		data.total = g_slist_length(data.slist);

		GuiTreeFreeze(tree);
		GuiListFreeze(list);

		GuiIdleAddPriority(GUI_PRIORITY_REDRAW,
					nscrec_into_list_and_tree, &data);
		gtk_main();

		if (gui_is_active == FALSE)
		{
			/* user has decided to quit */
			gtk_main_quit();
			return FALSE;
		}

		/* free strings */
		g_node_traverse(data.folder_node, G_PRE_ORDER, G_TRAVERSE_ALL, -1,
				(GNodeTraverseFunc) traverse_free_node_func, NULL);
		g_node_destroy(data.folder_node);

		GuiTreeColumnSort(tree, kColumnURL, gui_tree_compare_func, NULL);
		gui_list_sort(NULL, NULL);

#ifdef CONFIG_GTKCLIST_GTKCTREE

		/* restore func pointers */
		ctree_class->draw_row = ctree_draw_row;
		clist_class->draw_row = clist_draw_row;
#else
		g_slist_foreach(columns,
			(GFunc) gtk_tree_view_column_set_visible, (gpointer) TRUE);
		g_slist_free(columns);
#endif
		GuiTreeThaw(tree);
		GuiListThaw(list);

		MainTreeExpandTopLevelNodes(tree);

#ifdef CONFIG_COLUMN_BUTTONS
		GuiTreeSetSelectionMode(tree, GTK_SELECTION_BROWSE);
#endif
		GuiTreeSelectFirstRow(tree);
		GuiListSelectFirstRow(list);

	} /* if (data.slist) */

	name = nscache_db.name;

	if (name)
	{
		gchar *path = (nscache_db.path).str;

		if (path == NULL)
		{
			path = name;
			name = NULL;
		}

		path = g_strconcat(path, name, NULL);
		dbname = CHARCONV_UTF8_FROM_FILENAME(path, NULL);

		if (dbname)
		{
			gtk_entry_set_text(entry_db, dbname);
			CHARCONV_UTF8_DESTROY(dbname);
		}

		g_free(path);
	}

	g_slist_foreach(combo_mime_slist,
		(GFunc) gui_combo_set_popdown_mime, (gpointer) mime_types_available);

	/* print message(s) */
	print_node_info(NULL);
	gui_msg_label_replace_with(dummy, GTK_WIDGET(status_msg.label));
	gui_msg_label_replace_with(progress_bar, GTK_WIDGET(error_msg.label));

	gui_button_key_grab_remove();

#if GTK_MAJOR_VERSION == 1
	gtk_editable_select_region(GTK_EDITABLE(entry_db), 0, -1);
#endif
	gtk_widget_grab_focus(GTK_WIDGET(entry_db));

	return FALSE;

} /* static gint gui_read_cache(gchar*) */

static void gui_open_cache(void)
{
	gchar *dbname;

	if (gtk_main_level() != 1)
	{
		return;
	}

#ifdef HAVE_MSIE
	dbname = NULL;

	if (nscache_db.type != NSCACHE_RT_MSIE)
#endif
	{
		const gchar *str = gtk_entry_get_text(entry_db);

#if GTK_MAJOR_VERSION == 1
		dbname = g_strdup(str);
#else
		dbname = CHARCONV_UTF8_TO_FILENAME(str);
#endif
		if (dbname == NULL)
		{
			gui_err(gettext(invalid_filename_str));
			return;
		}
	}

	gui_read_cache(dbname);

} /* static void gui_open_cache(void) */

#ifdef MULTI_DB_TYPES
static void TypeOpen(GtkWidget *w, gpointer index)
{
	nscache_db.type = (gint) index;
	gui_open_cache();
}
#endif

static void ToggleBool(GtkWidget *w, struct gui_option *opt)
{
	opt->active = GTK_CHECK_MENU_ITEM(w)->active;

	if (opt->prop == kPropBoolOptDecode)
	{
		decode = opt->active;
	}

	gprop_set_bool(opt->prop, opt->active);
}

static void SaveRC(GtkWidget *w, gpointer data)
{
	gui_strerror(save_rc());
}

static void notebook_page_switched(GtkNotebook *nbook, GtkNotebookPage *page,
							gint nr, Windata *windat)
{
	ListSelection *list = &notebook_list[nr];

	GuiObjectSetUserData(windat->mainwin, list);

	windat->toplist = list;
	mainlist	= list;
}

#ifdef CONFIG_GTKCLIST_GTKCTREE
void gui_list_row_selected(GtkCList* clist, gint row, gint col,
					GdkEventButton *e, ListSelection *list)
{
	if (gtk_clist_get_selectable(clist, row))
	{
		list->store.list.row = row;

		gtk_clist_get_pixtext(clist, row, kColumnURL,
					&(list->name), NULL, NULL, NULL);

		list->nscrec = gtk_clist_get_row_data(clist, row);
	}
}

void gui_list_row_unselected(GtkCList* clist, gint row, gint col,
					GdkEventButton *e, ListSelection *list)
{
	list->name   = NULL;
	list->nscrec = NULL;
}

void gui_tree_row_selected(GtkCTree* ctree, GList *node, gint col, ListSelection *list)
{
	list->store.tree.node = GTK_CTREE_NODE(node);

	gtk_ctree_node_get_pixtext(ctree, list->store.tree.node, kColumnURL,
						&(list->name), NULL, NULL, NULL);

	list->nscrec = gtk_ctree_node_get_row_data(ctree, list->store.tree.node);

#ifdef CONFIG_COLUMN_BUTTONS
	gui_collapse_expand_set_sensitive(list, &button_collexp);
#endif
}

void gui_tree_row_unselected(GtkCTree* ctree, GList *node, gint col, ListSelection *list)
{
	list->name   = NULL;
	list->nscrec = NULL;

#ifdef CONFIG_COLUMN_BUTTONS
	gtk_widget_set_sensitive(button_collexp.collapse, FALSE);
	gtk_widget_set_sensitive(button_collexp.expand, FALSE);
#endif
}

#else /* ! CONFIG_GTKCLIST_GTKCTREE */

void gui_tree_row_selected(GtkTreeSelection *selection, ListSelection *list)
{
	GtkTreeIter	iter;
	GtkTreeModel	*model;

	if (gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		gint data_col;

		list->store.tree.iter = iter;

		g_free(list->name);

		data_col = list->type == kTreeList ?
				list->store.tree.data_column :
					list->store.list.data_column;

		gtk_tree_model_get(model, &iter, kColumnURL, &(list->name),
						data_col, &(list->nscrec), -1);
	}

	else
	{
		g_free(list->name);
		list->name   = NULL;
		list->nscrec = NULL;
	}
}

#endif /* #ifdef CONFIG_GTKCLIST_GTKCTREE .. #else */

#ifdef CONFIG_GTKCLIST_GTKCTREE

static void gui_tree_collapsed(GtkCTree *ctree, GtkCTreeNode *collapse_node, GuiTreeStore *tree)
{
	/*
		A folder was collapsed. If the selected
		entry is inside it, select the folder
		itself - avoids hidden entries.
	*/

	if (gtk_ctree_is_ancestor(ctree, collapse_node, tree->node))
	{
		gtk_ctree_select(ctree, collapse_node);
	}

#ifdef CONFIG_COLUMN_BUTTONS
	else if (button_collexp.is_active == FALSE)
	{
		gboolean can_collapse, can_expand;

		can_collapse = FALSE;
		can_expand   = FALSE;

		if (collapse_node == tree->node)
		{
			can_expand = TRUE;
		}

		else if (gtk_ctree_is_ancestor(ctree, tree->node, collapse_node))
		{
			can_collapse = TRUE;
			can_expand   = TRUE;
		}

		if (can_expand)
		{
			gtk_widget_set_sensitive(button_collexp.collapse, can_collapse);
			gtk_widget_set_sensitive(button_collexp.expand, can_expand);
		}
	}
#endif
}
	
#ifdef CONFIG_COLUMN_BUTTONS
static void gui_tree_expanded(GtkCTree *ctree, GtkCTreeNode *expand_node, GuiTreeStore *tree)
{
	if (button_collexp.is_active == FALSE &&
			(expand_node == tree->node ||
				gtk_ctree_is_ancestor(ctree, tree->node, expand_node)))
	{
		gboolean can_expand = FALSE;

		guitl_tree_recursive(tree, TRUE,
			(GuiTreeStoreRecursiveFunc) gui_test_expand_recursive, &can_expand);

		gtk_widget_set_sensitive(button_collexp.expand, can_expand);
		gtk_widget_set_sensitive(button_collexp.collapse, TRUE);
	}
}
#endif

#else /* ! CONFIG_GTKCLIST_GTKCTREE */

static void gui_tree_collapsed(GtkTreeView *view, GtkTreeIter *collapse_iter,
					GtkTreePath *collapse_path, GuiTreeStore *tree)
{
	if (gtk_tree_store_is_ancestor(tree->stree, collapse_iter, &(tree->iter)))
	{
		gtk_tree_selection_select_iter (tree->selection, collapse_iter);
	}
}
#endif /* #ifdef CONFIG_GTKCLIST_GTKCTREE .. #else */

#ifdef CONFIG_GTKCLIST_GTKCTREE

static void gui_list_resize_column(GtkWidget *w, gint col, gint width,
							gpointer data)
{
	int index, prop, type;

	index = column_tab[col]->index;
	type  = (int) data;
	prop  = type == kTreeList ?
			column_info[index].tree_width_prop :
				column_info[index].list_width_prop;

	gprop_set_int(prop, width);
}

#else /* ! CONFIG_GTKCLIST_GTKCTREE */

void gui_list_resize_column(GObject *column, GParamSpec *param, gpointer data)
{
	/*
		either main window or search dialog
		-----------------------------------
		kPropIntTreeColWidthUrl ... kPropIntTreeColWidthExp
		kPropIntListColWidthUrl ... kPropIntListColWidthExp
		kPropIntFindColWidthUrl ... kPropIntFindColWidthFile
	*/

	int prop, width;

	/* param->name -> "width" */
	g_object_get(column, param->name, &width, NULL);
	prop = (int) data;
	gprop_set_int(prop, width);
}

#endif /* #ifdef CONFIG_GTKCLIST_GTKCTREE .. #else */

static void gui_window_resize(GtkWidget *w, GtkAllocation *sallocation, gpointer data)
{
	/*
		either main window, search or info dialog
		-----------------------------------------
		kPropIntMainWinWidth + kPropIntMainWinHeight
		kPropIntFindWinWidth + kPropIntFindWinHeight
		kPropIntInfoWinWidth + kPropIntInfoWinHeight
	*/

	int prop = (int) data;

	gprop_set_int(prop, sallocation->width);
	gprop_set_int(prop + 1, sallocation->height);
}

static void adjust_recursive(GuiTreeStore *tree, nscache_record_t *rec, gboolean *sensitive)
{
	if (rec)
	{
		int cache_acc = file_access(rec);

		if (cache_acc > 0)
		{
			if (rec->content_offset == 0)
			{
				if (cache_acc & FILE_ACCESS_WRITE)
				{
					sensitive[kMenuReadOnly] = TRUE;
				}

				else
				{
					sensitive[kMenuReadWrite] = TRUE;
				}
#ifdef HAVE_LINK
				if (rec->content_encoding == NULL ||
					gui_option[kOptionDecode].active == FALSE)
				{
					hardlink_enable = TRUE;
				}
#endif
			}

			if (cache_acc & FILE_ACCESS_READ)
			{
				sensitive[kMenuSave] = TRUE;
			}
#ifndef CACHE_READONLY
			if (cache_acc & FILE_ACCESS_REMOVE)
			{
				sensitive[kMenuDelete] = TRUE;
			}
#endif
		}
	}

} /* static void adjust_recursive(GuiTreeStore*, nscache_record_t*, gboolean*) */

static void adjust_viewer_func(const apassign_t *e, gboolean *sensitive)
{
	if (e)
	{
		if (e->args_file != NULL && e->args_file[0] != '\0')
		{
			sensitive[kMenuViewFile] = sensitive[kMenuSave];
		}

		if (e->args_url != NULL && e->args_url[0] != '\0')
		{
			sensitive[kMenuViewURL] = TRUE;
		}
	}
}

static void adjust_viewer(const char *content_type, gboolean *sensitive)
{
	GSList *app;

	app = apassign_get_matching_viewers(content_type);
	g_slist_foreach(app, (GFunc) adjust_viewer_func, sensitive);
	g_slist_free(app);
	sensitive[kMenuClipboard] = TRUE;
}

void AdjustMenu(GtkWidget *mni, ListSelection **selected)
{
	ListSelection		*list;
	nscache_record_t	*rec;
	GtkWidget		**menu, **popup;
	gint			item;
	gboolean		sensitive[kMenuItemsTotal];

	if (selected == NULL)
	{
		menu = &gui_menu_widget[kMenuItemsTotal];
		item = kMenuItemsTotal - 1;

		do
		{
			if (*(--menu) == mni)
			{
				break;
			}
		}
		while (--item >= 0);

		if (item < 0)
		{
			return;
		}

		list = windata.toplist;
	}

	else
	{
		list = *selected;
	}

#ifdef HAVE_LINK
	hardlink_enable = FALSE;
#endif
	memset(sensitive, '\0', sizeof(sensitive));

	rec = (list == NULL) ? NULL : list->nscrec;

	if (rec == NULL)
	{
		if (list && list->name && list->type == kTreeList)
		{
			if (GuiTreeGetLevel(&(list->store.tree)) > kTreeLevelService)
			{
				/* value of 'sensitive[kMenuSave]' is FALSE */
				adjust_viewer("text/html", sensitive);

#ifdef HAVE_MSIE
				/* no db path */
				if (nscache_db.type == NSCACHE_RT_MSIE ||
							access(nscache_db.path.str, F_OK) == 0)
#else
				/* still mounted ? */
				if (access(nscache_db.path.str, F_OK) == 0)
#endif
				{
					guitl_tree_recursive(&(list->store.tree), TRUE,
						(GuiTreeStoreRecursiveFunc) adjust_recursive,
											sensitive);
				}

			} /* if (GuiTreeGetLevel(list->store.tree) > kTreeLevelService) */

		} /* if (list && list->name && list->type == kTreeList) */

	} /* if (rec == NULL) */

	else
	{
		adjust_recursive(NULL, rec, sensitive);
		/* adjusting of 'sensitive[kMenuSave]' done */
		adjust_viewer(rec->content_type, sensitive);

	} /* if (rec == NULL) .. else */

#ifndef CACHE_READONLY

	if (nscache_db.readonly || gui_option[kOptionReadOnly].active)
	{
		sensitive[kMenuDelete] = FALSE;
	}
#endif
	sensitive[kMenuFind] = TRUE;

	menu	= &gui_menu_widget[kMenuItemsTotal];
	popup	= &gui_popup_widget[kMenuItemsTotal];

	item = kMenuItemsTotal - 1;

	do
	{
		gtk_widget_set_sensitive(*(--menu),  sensitive[item]);
		gtk_widget_set_sensitive(*(--popup), sensitive[item]);

	}
	while (--item >= 0);

} /* static void AdjustMenu(GtkWidget*, ListSelection**) */

gboolean gui_mouse_button_pressed(GtkWidget *widget, GdkEvent *event,
							GtkWidget *popup_menu)
{
	guint		button;
	guint32		time;
	GdkEventButton	*bevent;

	bevent = (GdkEventButton *) event;

	if (bevent == NULL)
	{
		time	= GDK_CURRENT_TIME;
		button	= 0;
		goto display_popup;
	}

	button = bevent->button;

	if (button == 1 && bevent->state == GDK_SHIFT_MASK)
	{
		button = 2;
	}

	if (bevent->type == GDK_BUTTON_PRESS)
	{
		if (button == 3 || (button == 2 && (windata.toplist)->type == kTreeList))
		{
			if ((windata.toplist)->name)
			{

#ifdef CONFIG_GTKCLIST_GTKCTREE
				gint     row, column;
				GtkCList *clist;

				clist = GTK_CLIST(widget);

				if (gtk_clist_get_selection_info(clist, (gint) bevent->x,
							(gint) bevent->y, &row, &column))
				{
					if (button == 2)
					{
						GtkCTreeNode *node =
						gtk_ctree_node_nth(GTK_CTREE(widget), row);

						if (GTK_CTREE_ROW(node)->children == FALSE)
						{
							return FALSE;
						}
					}
#if GTK_MAJOR_VERSION == 1
					GuiSignalEmitStopByName(widget, "button-press-event");
#endif
					gtk_clist_select_row(clist, row, 0);

					if (button == 2)
					{
						GuiTreeStore *tree =
							&notebook_list[kTreeList].store.tree;

						if (GTK_CTREE_ROW(tree->node)->expanded)
						{
#ifdef CONFIG_COLUMN_BUTTONS
							gui_tree_collapse_recursive(NULL, tree);
#else
							GuiTreeCollapseRecursive(tree);
#endif
						}

						else
						{
							gui_tree_expand_recursive(NULL, tree);
						}

						return TRUE;

					} /* if (button == 2) */
				}

#else /* ! CONFIG_GTKCLIST_GTKCTREE */

				GtkTreeView *view;
				GtkTreePath *path;

				view = GTK_TREE_VIEW(widget);

				if (gtk_tree_view_get_path_at_pos(view,
						(gint) bevent->x, (gint) bevent->y, &path,
						NULL, NULL, NULL))
				{
					GtkTreeSelection *selection;

					if (button == 2)
					{
						GuiTreeStore	*tree;
						GtkTreeModel	*model;
						GtkTreeIter	iter;

						tree  = &notebook_list[kTreeList].store.tree;
						model = GTK_TREE_MODEL(tree->stree);

						if (gtk_tree_model_get_iter(model,
								&iter, path) == FALSE ||
							gtk_tree_model_iter_has_child(model,
										&iter) == FALSE)
						{
							gtk_tree_path_free(path);
							return FALSE;
						}
					}

					selection = gtk_tree_view_get_selection(view);
					gtk_tree_selection_select_path(selection, path);

					if (button == 2)
					{
						if (gtk_tree_view_row_expanded(view, path))
						{
							gtk_tree_view_collapse_row(view, path);
						}

						else
						{
							gtk_tree_view_expand_row(view, path, TRUE);
						}
					}

					gtk_tree_path_free(path);
				}

#endif /* #ifdef CONFIG_GTKCLIST_GTKCTREE .. #else */

			} /* if ((windata.toplist)->name) */

			if (button == 2)
			{
				return TRUE;

			}

			time = bevent->time;
display_popup:
			gtk_menu_popup(GTK_MENU(popup_menu), NULL, NULL, NULL, NULL, button, time);

			return TRUE;

		} /* if (button == 3 || (button == 2 && (windata.toplist)->type == kTreeList)) */

	} /* if (bevent->type == GDK_BUTTON_PRESS) */

	else if (bevent->type == GDK_2BUTTON_PRESS)
	{
		if (button == 1)
		{
			return ViewerDefault(VIEWER_FILE);
		}

		if (button == 2)
		{
			return ViewerDefault(VIEWER_URL);
		}
	}

	return FALSE;

} /* gboolean gui_mouse_button_pressed(GtkWidget*, GdkEvent*, GtkWidget*) */

#if GTK_MAJOR_VERSION == 1

gboolean gui_sf10_show_popup_menu(GtkWidget *widget, GdkEventKey *event,
							GtkWidget *popup_menu)
{
	if (event->state == GDK_SHIFT_MASK && event->keyval == GDK_F10)
	{
		return gui_mouse_button_pressed(widget, NULL, popup_menu);
	}

	return FALSE;
}

#else

gboolean gui_sf10_show_popup_menu(GtkWidget *widget, GtkWidget *popup_menu)
{
	return gui_mouse_button_pressed(widget, NULL, popup_menu);
}

#endif /* #if GTK_MAJOR_VERSION == 1 .. #else */


#ifdef CONFIG_GTKCLIST_GTKCTREE

static void gui_tree_expand_recursive_func(GuiTreeStore *tree,
				nscache_record_t *rec, gpointer data)
{
	/*
		Sometimes 'gtk_ctree_expand_recursive' redraws
		the bottom line, if scrolled out, not propper.
		Seems to work only when all sub nodes are collapsed.
	*/

	if (rec == NULL)
	{
		gtk_ctree_expand(tree->ctree, tree->node);
	}
}

#endif /* CONFIG_GTKCLIST_GTKCTREE */


#if defined(CONFIG_GTKCLIST_GTKCTREE) || defined(CONFIG_COLUMN_POPUP)

static void gui_tree_expand_recursive(GtkWidget *w, GuiTreeStore *tree)
{
#ifdef CONFIG_COLUMN_BUTTONS
	button_collexp.is_active = TRUE;
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
	gtk_clist_freeze(GTK_CLIST(tree->ctree));
	guitl_tree_recursive(tree, TRUE,
		(GuiTreeStoreRecursiveFunc) gui_tree_expand_recursive_func, NULL);
	gtk_clist_thaw(GTK_CLIST(tree->ctree));
#else
	GuiTreeExpandRecursive(tree);
#endif

#ifdef CONFIG_COLUMN_BUTTONS
	button_collexp.is_active = FALSE;
	gtk_widget_set_sensitive(button_collexp.collapse, TRUE);
	gtk_widget_set_sensitive(button_collexp.expand, FALSE);
#endif
}

#endif /* defined(CONFIG_GTKCLIST_GTKCTREE) || defined(CONFIG_COLUMN_POPUP) */


#ifdef CONFIG_SWAP_COLUMNS

static void gui_tree_collapse_recursive(GtkWidget *w, GuiTreeStore *tree)
{
#ifdef CONFIG_COLUMN_BUTTONS
	button_collexp.is_active = TRUE;
#endif
	GuiTreeCollapseRecursive(tree);

#ifdef CONFIG_COLUMN_BUTTONS
	button_collexp.is_active = FALSE;
	gtk_widget_set_sensitive(button_collexp.collapse, FALSE);
	gtk_widget_set_sensitive(button_collexp.expand, TRUE);
#endif
}

static void gui_test_expand_recursive(GuiTreeStore *tree, nscache_record_t *rec,
									gboolean *can_expand)
{
	if (rec == NULL && *can_expand == FALSE)
	{
		gboolean is_expanded;

		GuiTreeGetExpanded(tree, is_expanded);

		if (is_expanded == FALSE)
		{
			*can_expand = TRUE;
		}
	}
}

static void gui_collapse_expand_set_sensitive(ListSelection *list, CollapseExpandWidgets *cew)
{
		gboolean can_collapse, can_expand;

		can_collapse = can_expand = FALSE;

		if (list->name && list->nscrec == NULL)
		{
			GuiTreeGetExpanded(&(list->store.tree), can_collapse);

			if (can_collapse == FALSE)
			{
				can_expand = TRUE;
			}

			else
			{
				guitl_tree_recursive(&(list->store.tree), TRUE,
					(GuiTreeStoreRecursiveFunc)
						gui_test_expand_recursive, &can_expand);
			}
		}

		gtk_widget_set_sensitive(cew->collapse, can_collapse);
		gtk_widget_set_sensitive(cew->expand, can_expand);
}

static gboolean column_is_left_most(int position)
{
	while (--position > kColumnURL)
	{
		if (column_tab[position]->visible)
		{
			return FALSE;
		}
	}

	return TRUE;
}

static gboolean column_is_right_most(int position)
{
	while (++position <= kColumnInfoLastIndex)
	{
		if (column_tab[position]->visible)
		{
			return FALSE;
		}
	}

	return TRUE;
}

#ifdef CONFIG_COLUMN_BUTTONS

static void column_arrows_set_visibility(int nbook_page, int position)
{
	ColumnTab *col_tab = column_tab[position];
	ColumnArrows *arrow = &col_tab->arrow[nbook_page];

	if (column_is_left_most(position))
	{
		gtk_widget_hide(arrow->left);
	}
	else
	{
		gtk_widget_show(arrow->left);
	}

	if (column_is_right_most(position))
	{
		gtk_widget_hide(arrow->right);
	}
	else
	{
		gtk_widget_show(arrow->right);
	}
}

#endif /* CONFIG_COLUMN_BUTTONS */

static int column_get_right_visible(int position)
{
	while (++position <= kColumnInfoLastIndex)
	{
		if (column_tab[position]->visible)
		{
			return position;
		}
	}

	return -1;
}

static int column_get_left_visible(int position)
{
	while (--position >= kColumnURL)
	{
		if (column_tab[position]->visible)
		{
			break;
		}
	}

	return position;
}

#endif /* CONFIG_SWAP_COLUMNS */

static void column_set_visibility_list(gint nbook_page, gint index, gint active)
{
	int position;

	position = column_tab_array[index].position;

	if (nbook_page == kTreeList)
	{
		GuiTreeSetColumnVisiblility(&(notebook_list[kTreeList].store.tree),
								position, active);
	}

	else
	{
		GuiListSetColumnVisiblility(&(notebook_list[kSortedList].store.list),
								position, active);
	}

#ifdef CONFIG_COLUMN_BUTTONS

	if (index != kColumnURL)
	{
		int left_pos, right_pos;

		column_arrows_set_visibility(nbook_page, position);

		right_pos = column_get_right_visible(position);

		if (right_pos > 0)
		{
			column_arrows_set_visibility(nbook_page, right_pos);
		}

		left_pos = column_get_left_visible(position);

		if (left_pos > 0)
		{
			column_arrows_set_visibility(nbook_page, left_pos);
		}
	}

#endif /* CONFIG_COLUMN_BUTTONS */

} /* static void column_set_visibility_list(gint, gint, gint) */

static void column_menu_set_sensitive(gint active)
{
	static GtkWidget *last_visible_column = NULL;

	if (active == FALSE)
	{
		gint index, last;

		last	= -1;
		index	= kColumnInfoLastIndex;

		do
		{
			if (column_tab_array[index].visible)
			{
				if (last > kColumnURL)
				{
					return;
				}

				last = index;
			}
		}
		while (--index >= kColumnURL);

		last_visible_column = column_tab_array[last].menu;
	}

	if (last_visible_column)
	{
		gtk_widget_set_sensitive(last_visible_column, active);

		if (active)
		{
			last_visible_column = NULL;
		}
	}

} /* static void column_menu_set_sensitive(gint active) */

static void column_set_visibility(GtkWidget *menu_check_item, gpointer data)
{
	gint index  = (gint) data;
	gint active = GTK_CHECK_MENU_ITEM(menu_check_item)->active;

	gprop_set_bool(column_info[index].show_column_prop, active);

	column_tab_array[index].visible = active;
	column_menu_set_sensitive(active);

	column_set_visibility_list(kTreeList, index, active);
	column_set_visibility_list(kSortedList, index, active);

} /* static void column_set_visibility(GtkWidget*, gpointer) */

#ifdef CONFIG_SWAP_COLUMNS

#ifdef CONFIG_GTKCLIST_GTKCTREE

static void columns_do_swap(gint nbook_page, ColumnTab* col1, ColumnTab* col2)
{
	int      w1, w2;
	GtkCList *clist;

	if (nbook_page == kTreeList)
	{
		if (!gprop_get_int(column_info[col1->index].tree_width_prop, &w1))
		{
			w1 = column_info[col1->index].dwidth;
		}

		if (!gprop_get_int(column_info[col2->index].tree_width_prop, &w2))
		{
			w2 = column_info[col2->index].dwidth;
		}

		clist = GTK_CLIST(notebook_list[nbook_page].store.tree.ctree);
		gtk_clist_freeze(clist);
		guitl_ctree_swap_cols(notebook_list[nbook_page].store.tree.ctree,
							col1->position, col2->position);
	}

	else
	{
		if (!gprop_get_int(column_info[col1->index].list_width_prop, &w1))
		{
			w1 = column_info[col1->index].dwidth;
		}

		if (!gprop_get_int(column_info[col2->index].list_width_prop, &w2))
		{
			w2 = column_info[col2->index].dwidth;
		}

		clist = notebook_list[nbook_page].store.list.clist;
		gtk_clist_freeze(clist);
		guitl_clist_swap_cols(clist, col1->position, col2->position);
	}

	gtk_clist_set_column_justification(clist, col1->position,
					column_info[col1->index].justify);
	gtk_clist_set_column_justification(clist, col2->position,
					column_info[col2->index].justify);

	gtk_clist_set_column_width(clist, col1->position, w1);
	gtk_clist_set_column_width(clist, col2->position, w2);

	gtk_clist_thaw(clist);

#ifdef CONFIG_COLUMN_BUTTONS
	column_arrows_set_visibility(nbook_page, col1->position);
	column_arrows_set_visibility(nbook_page, col2->position);
#endif

} /* static void columns_do_swap(gint, ColumnTab*, ColumnTab*) */

#endif /* CONFIG_GTKCLIST_GTKCTREE */

static void columns_swap(ColumnTab* col1, gint direction)
{
	ColumnTab*	col2;
	gint		pos1, pos2;

	pos2 = (direction > 0) ?
			column_get_right_visible(col1->position) :
				column_get_left_visible(col1->position);
	if (pos2 < 0)
	{
		return;
	}

	/* swap pointer and position */

	col2 = column_tab[pos2];
	column_tab[pos2] = col1;

	pos1 = col1->position;
	column_tab[pos1] = col2;

	col1->position = col2->position;
	col2->position = pos1;

	/* swap columns */

#ifdef CONFIG_GTKCLIST_GTKCTREE
	columns_do_swap(kTreeList, col1, col2);
	columns_do_swap(kSortedList, col1, col2);
#else
	guitl_view_swap_cols(notebook_list[kTreeList].store.tree.view, pos1, pos2);
	guitl_view_swap_cols(notebook_list[kSortedList].store.list.view, pos1, pos2);
#endif
	gprop_set_int(column_info[col1->index].colnrprop, col1->position);
	gprop_set_int(column_info[col2->index].colnrprop, col2->position);

} /* static void columns_swap(ColumnTab*, gint) */

static void column_move_right(GtkWidget *widget, gpointer data)
{
	ColumnTab *column_tab;

#ifdef CONFIG_COLUMN_BUTTONS
	gtk_button_leave(GTK_BUTTON(widget));
	column_tab = data;
#else
	column_tab = *(ColumnTab **) data;
#endif
	columns_swap(column_tab, +1);
}

static void column_move_left(GtkWidget *widget, gpointer data)
{
	ColumnTab *column_tab;

#ifdef CONFIG_COLUMN_BUTTONS
	gtk_button_leave(GTK_BUTTON(widget));
	column_tab = data;
#else
	column_tab = *(ColumnTab **) data;
#endif
	columns_swap(column_tab, -1);
}

#ifdef CONFIG_COLUMN_BUTTONS

static GtkWidget *column_make_arrow(GtkArrowType direction, ColumnTab *col_tab)
{
	GtkWidget *arrow, *button;

	button = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);

	GuiSignalConnect(button, "clicked",
		(direction == GTK_ARROW_LEFT ?
			column_move_left : column_move_right), col_tab);

	arrow = gtk_arrow_new(direction, GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(button), arrow);

	return button;

} /* static GtkWidget *column_make_arrow(GtkArrowType, ColumnTab*) */

static void column_make_widget(gint nbook_page, gint index)
{
	GtkWidget	*arrow, *label, *widget;
	GtkTable	*table;
	ColumnTab	*col_tab;

	col_tab = column_tab[index];

	widget = gtk_table_new(3, 1, FALSE);
	table = GTK_TABLE(widget);

	arrow = column_make_arrow(GTK_ARROW_LEFT, col_tab);
	col_tab->arrow[nbook_page].left = arrow;
	gtk_table_attach(table, arrow, 0, 1, 0, 1,
				GTK_FILL, GTK_FILL, 0, 0);

	label = guitl_label_new_gettext(column_info[col_tab->index].label);

	gtk_table_attach(table, label, 1, 2, 0, 1,
				GTK_EXPAND, GTK_EXPAND, 0, 0);

	arrow = column_make_arrow(GTK_ARROW_RIGHT, col_tab);
	col_tab->arrow[nbook_page].right = arrow;
	gtk_table_attach(table, arrow, 2, 3, 0, 1,
				GTK_FILL, GTK_FILL, 0, 0);

	if (nbook_page == kTreeList)
	{
		GuiTreeSetColumnWidget(&(notebook_list[kTreeList].store.tree),
									index, widget);
	}

	else
	{
		GuiListSetColumnWidget(&(notebook_list[kSortedList].store.list),
									index, widget);
	}

	gtk_widget_show_all(widget);

	column_arrows_set_visibility(nbook_page, index);

} /* static void column_make_widget(gint, gint) */

#else /* ! CONFIG_COLUMN_BUTTONS */

#ifdef CONFIG_COLUMN_POPUP

static gboolean gui_popup_collapse_expand(GtkWidget *w, GdkEvent *event, GtkWidget **popup_menu)
{
	GdkEventButton *bevent = (GdkEventButton *) event;

	/* w == NULL => called by 'gui_mouse_button_pressed' */

	if (w == NULL || (bevent->type == GDK_BUTTON_PRESS && bevent->button == 3))
	{
		gui_collapse_expand_set_sensitive(&notebook_list[kTreeList], &popup_collexp);

		gtk_menu_popup(GTK_MENU(*popup_menu), NULL, NULL, NULL, NULL,
							bevent->button, bevent->time);
	}

	return TRUE;
}

static gboolean gui_popup_colswap(GtkWidget *widget, GdkEvent *event, ColumnTab *col_tab)
{
	GdkEventButton *bevent = (GdkEventButton *) event;

	if (bevent->type == GDK_BUTTON_PRESS && bevent->button == 3)
	{
		gboolean sensitive;

#ifdef CONFIG_GTKCLIST_GTKCTREE
		col_tab = *(ColumnTab **) col_tab;
#endif
		popup_colswap.selected = col_tab;

		sensitive = !column_is_left_most(col_tab->position);
		gtk_widget_set_sensitive(popup_colswap.left, sensitive);

		sensitive = !column_is_right_most(col_tab->position);
		gtk_widget_set_sensitive(popup_colswap.right, sensitive);

		gtk_menu_popup(GTK_MENU(popup_colswap.menu), NULL, NULL, NULL, NULL,
							bevent->button, bevent->time);
	}

	return TRUE;
}

static void column_make_widget(gint nbook_page, gint index)
{
	gint		position;
	GtkWidget	*label, *button;

	position = column_tab[index]->index;
	label = guitl_label_new_gettext(column_info[position].label);

	if (nbook_page == kTreeList)
	{
		GuiTreeSetColumnWidget(&(notebook_list[kTreeList].store.tree),
								index, label);
		GuiTreeColumnTitleActive(&(notebook_list[kTreeList].store.tree),
								index, TRUE);
	}

	else
	{
		GuiListSetColumnWidget(&(notebook_list[kSortedList].store.list),
								index, label);
		GuiListColumnTitleActive(&(notebook_list[kSortedList].store.list),
								index, TRUE);
	}

	button = gtk_widget_get_ancestor(label, GTK_TYPE_BUTTON);

	if (button)
	{
		if (index == kColumnURL)
		{
			GuiSignalConnect(button, "button-press-event",
				gui_popup_collapse_expand, &popup_collexp.menu);
		}

		else
		{
			gpointer data;

#ifdef CONFIG_GTKCLIST_GTKCTREE
			/* widget and button may differ */
			data = &column_tab[index];
#else
			/* widget and button always equal */
			data = column_tab[index];
#endif
			GuiSignalConnect(button, "button-press-event",
							gui_popup_colswap, data);
		}
	}

	gtk_widget_show(label);
}

#endif /* CONFIG_COLUMN_POPUP */

#endif /* #ifdef CONFIG_COLUMN_BUTTONS .. #else */

#else /* ! CONFIG_SWAP_COLUMNS */

#define column_make_widget(nbook_page, index) \
		MainSetColumnTitle(nbook_page, index)

#endif /* #if CONFIG_SWAP_COLUMNS .. #else */


static GtkWidget *gui_build_notebook_page(GtkWidget *notebook, GtkWidget *parent)
{
	static const gchar *page_label[] =
	{
		gettext_nop("Tree view"),
		gettext_nop("Sorted view")
	};

	gint		type;
	GtkWidget	*label, *swin;

	swin = gtk_scrolled_window_new(NULL, NULL);
	GuiWidgetSetSizeRequest(swin, -1, 150);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swin),
					GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
	if (parent == NULL)
	{
		parent = swin;
		type = kTreeList;
	}

	else
	{
		gtk_box_pack_start(GTK_BOX(parent), swin, TRUE, TRUE, 1);
		type = kSortedList;
	}

	label = guitl_label_new_gettext(page_label[type]);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), parent, label);

	list_selection_slist =
		g_slist_append(list_selection_slist, &notebook_list[type]);

	notebook_list[type].type = type;

	return swin;
}

static void gui_build_sorted_list(GtkWidget *notebook)
{
	static const gchar *stype_label[] =
	{
		gettext_nop("URL"),		/* STYPE_URL  */
		gettext_nop("Local file"),	/* STYPE_FILE  */
		gettext_nop("Size"), 		/* STYPE_SIZE */
		gettext_nop("Access time"),	/* STYPE_ATM  */
		gettext_nop("Mime type")	/* STYPE_MIME */
	};

	static gint stype_array[STYPE_MIME + 1] = { 0 };

	GuiListStore	*list;
	GtkBox		*hbox;
	GtkWidget	*label, *menu, *swin, *stype_menu, *vbox, *widget;
	gint		height, i, sort_type;

	vbox = gtk_vbox_new(FALSE, 0);

	widget = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), widget, FALSE, FALSE, 1);

	hbox = GTK_BOX(widget);
	label = guitl_label_new_gettext(gettext_nop("Sort by: "));
	gtk_box_pack_start(hbox, label, FALSE, FALSE, 1);
	
	stype_menu = gtk_option_menu_new();
	gtk_box_pack_start(hbox, stype_menu, FALSE, FALSE, 1);

	menu = gtk_menu_new();

	i = STYPE_URL;

	do
	{
		GtkWidget *mi;

		mi = guitl_menu_item_new_gettext(stype_label[i]);
		GuiMenuAppend(menu, mi);
		gtk_widget_show(mi);

		stype_array[i] = i;
		GuiSignalConnect(mi, "activate", gui_list_sort, &stype_array[i]);
	}
	while (++i <= STYPE_MIME);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(stype_menu), menu);

	if (gprop_get_int(kPropIntSortType, &sort_type) == FALSE)
	{
		i = sort_type = STYPE_URL;
	}

	else
	{
		i = sort_type & ~STYPE_REVERS_MASK;

		if (i < STYPE_URL || i > STYPE_MIME)
		{
			i = sort_type = STYPE_URL;
		}
	}

	stype_array[i] = sort_type;
	nscache_sort_type = sort_type;
	gtk_option_menu_set_history(GTK_OPTION_MENU(stype_menu), i);

	swin = gui_build_notebook_page(notebook, vbox);
	list = &(notebook_list[kSortedList].store.list);

	MainListNew(list);
	GuiListContainerAdd(list, swin);

	height = guitl_list_calculate_row_height(list);
	GuiListSetRowHeight(list, height);

	i = kColumnURL;

	do
	{
		int w;

		if (i == kColumnURL)
		{

#ifndef CONFIG_GTKCLIST_GTKCTREE
			GuiListColumnPixtextNew(list, kColumnURL, height, kColumnPixMime);
#endif
			MainSetColumnTitle(kSortedList, i);
		}

		else
		{

#ifndef CONFIG_GTKCLIST_GTKCTREE
			GuiListColumnTextNew(list, i, height,
				column_info[column_tab[i]->index].justify);
#endif
			column_make_widget(kSortedList, i);
		}

#ifdef CONFIG_GTKCLIST_GTKCTREE
		gtk_clist_set_column_justification(list->clist, i,
					column_info[column_tab[i]->index].justify);
#else
		GuiListSignalConnectResizeColumn(list, i, gui_list_resize_column,
			(gpointer) column_info[column_tab[i]->index].list_width_prop);

		GuiListSetColumnAlignment(list, i, 0.5);
#endif
		GuiListSetColumnAutoResize(list, i, FALSE);
		GuiListSetColumnResizable(list, i, TRUE);

		GuiListSetColumnWidth(list, i,
			gprop_get_int(column_info[column_tab[i]->index].list_width_prop, &w) ?
								w : column_info[i].dwidth);
	}
	while (++i <= kColumnInfoLastIndex);

	GuiListSetSelectionMode(list, GTK_SELECTION_BROWSE);
	GuiListColumnTitlesShow(list);

#ifdef CONFIG_COLUMN_POPUP
	GuiListColumnTitleActive(list, kColumnURL, FALSE);
#else
	GuiListColumnTitlesPassive(list);
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
	GuiListSignalConnectResizeColumn(list,
				gui_list_resize_column, (gpointer) kSortedList);
#endif
	GuiListSignalConnectSelectedRow(list,
				gui_list_row_selected, &notebook_list[kSortedList]);
	GuiListSignalConnectUnselectedRow(list,
				gui_list_row_unselected, &notebook_list[kSortedList]);

} /* static void gui_build_sorted_list(GtkWidget*) */

#ifdef CONFIG_SWAP_COLUMNS

#ifdef CONFIG_COLUMN_BUTTONS

static GtkWidget *tree_list_make_expand_collapse_button(gint pixid)
{
	GtkWidget *button;

	button = guitl_get_icon_button(pixid);
	gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);

	GuiSignalConnect(button, "clicked",
		(gpointer) (pixid == kIconButtonExpand ?
			gui_tree_expand_recursive : gui_tree_collapse_recursive),
				&(notebook_list[kTreeList].store.tree));
	return button;

} /* static GtkWidget *tree_list_make_expand_collapse_button(gint) */

static void tree_list_make_first_column(GuiTreeStore *tree)
{
	GtkWidget *button, *label, *widget;
	GtkTable  *table;

	widget = gtk_table_new(3, 1, FALSE);
	table  = GTK_TABLE(widget);

	button = tree_list_make_expand_collapse_button(kIconButtonExpand);

#ifdef CONFIG_COLUMN_BUTTONS
	button_collexp.is_active = FALSE;
	button_collexp.expand    = button;
#endif
	gtk_table_attach(table, button, 0, 1, 0, 1,
				GTK_FILL, GTK_FILL, 0, 0);
 
	button = tree_list_make_expand_collapse_button(kIconButtonCollapse);

#ifdef CONFIG_COLUMN_BUTTONS
	button_collexp.collapse = button;
#endif
	gtk_table_attach(table, button, 1, 2, 0, 1,
				GTK_FILL, GTK_FILL, 0, 0);

	label = guitl_label_new_gettext(column_info[kColumnURL].label);
	gtk_table_attach(table, label, 2, 3, 0, 1,
				GTK_FILL, GTK_FILL, 1, 1);

	GuiTreeSetColumnWidget(tree, kColumnURL, widget);
	gtk_widget_show_all(widget);

} /* static void tree_list_make_first_column(GuiTreeStore*) */

#else /* ! CONFIG_COLUMN_BUTTONS */

#define tree_list_make_first_column(tree) \
		column_make_widget(kTreeList, kColumnURL)

#endif /* CONFIG_COLUMN_BUTTONS */

#else /* ! CONFIG_SWAP_COLUMNS */

#define tree_list_make_first_column(tree) \
		MainSetColumnTitle(kTreeList, kColumnURL)

#endif /* #ifdef CONFIG_SWAP_COLUMNS .. #else */

static void gui_build_tree_list(GtkWidget *notebook)
{
	GuiTreeStore	*tree;
	GtkWidget	*swin;
	gint		height, i;

	swin = gui_build_notebook_page(notebook, NULL);
	tree = &notebook_list[kTreeList].store.tree;

	MainTreeNew(tree);
	GuiTreeContainerAdd(tree, swin);

	height = guitl_tree_calculate_row_height(tree);
	GuiTreeSetRowHeight(tree, height);

	i = kColumnURL;

	do
	{
		int w;

		if (i == kColumnURL)
		{
#ifndef CONFIG_GTKCLIST_GTKCTREE
			GuiTreeColumnPixtextNew(tree, kColumnURL, height,
				kColumnPixMime, kColumnPixClose, kColumnPixOpen);
#endif
			tree_list_make_first_column(tree);
		}

		else
		{
#ifndef CONFIG_GTKCLIST_GTKCTREE
			GuiTreeColumnTextNew(tree, i, height,
				column_info[column_tab[i]->index].justify);
#endif
			column_make_widget(kTreeList, i);
		}

#ifdef CONFIG_GTKCLIST_GTKCTREE
		gtk_clist_set_column_justification(GTK_CLIST(tree->ctree), i,
					column_info[column_tab[i]->index].justify);
#else
		GuiTreeSignalConnectResizeColumn(tree, i, gui_list_resize_column,
			(gpointer) column_info[column_tab[i]->index].tree_width_prop);

		GuiTreeSetColumnAlignment(tree, i, 0.5);
#endif
		GuiTreeSetColumnAutoResize(tree, i, FALSE);
		GuiTreeSetColumnResizable(tree, i, TRUE);

		GuiTreeSetColumnWidth(tree, i,
			gprop_get_int(column_info[column_tab[i]->index].tree_width_prop, &w) ?
								w : column_info[i].dwidth);
	}
	while (++i <= kColumnInfoLastIndex);

	GuiTreeColumnTitlesShow(tree);

#ifndef CONFIG_COLUMN_POPUP
	GuiTreeColumnTitlesPassive(tree);
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
	gtk_ctree_set_expander_style(tree->ctree, GTK_CTREE_EXPANDER_SQUARE);
	gtk_ctree_set_line_style(tree->ctree, GTK_CTREE_LINES_DOTTED);

	GuiTreeSignalConnectResizeColumn(tree,
				gui_list_resize_column, (gpointer) kTreeList);
#endif
	GuiTreeSignalConnectSelectedRow(tree,
				gui_tree_row_selected, &notebook_list[kTreeList]);
	GuiTreeSignalConnectUnselectedRow(tree,
				gui_tree_row_unselected, &notebook_list[kTreeList]);

	GuiTreeSignalConnectAfterCollapse(tree, gui_tree_collapsed, tree);

#ifdef CONFIG_COLUMN_BUTTONS
	GuiTreeSignalConnectAfterExpand(tree, gui_tree_expanded, tree);
#else
	GuiTreeSetSelectionMode(tree, GTK_SELECTION_BROWSE);
#endif

} /* static void gui_build_tree_list(GtkWidget*) */

#ifdef HAVE_GLOB

struct CompletitionPopup
{
	GtkWidget *popup;
	GtkLabel  *label;
};

static void gui_completition_hide(struct CompletitionPopup *data)
{
	gdk_keyboard_ungrab(GDK_CURRENT_TIME);
	gtk_widget_hide_all(data->popup);
	gtk_label_set_text(data->label, NULL);
}

static gint gui_completition_button_press(GtkWidget *popup, GdkEventButton *e,
						struct CompletitionPopup *data)
{
	gint		x, y;
	GdkWindow	*window;

	gui_completition_hide(data);
	window = gdk_window_at_pointer(&x, &y);

	if (window)
	{
		GdkWindow *wnd;

		e = (GdkEventButton *) gdk_event_copy((GdkEvent *) e);

		e->x = x;
		e->y = y;

		wnd = e->window;
		e->window = window;
		gdk_event_put((GdkEvent *) e);
		e->window = wnd;

		gdk_event_free((GdkEvent *) e);
	}

	return TRUE;
}

static gint gui_completition_expose(GtkWidget *label, GdkEvent *e, GtkWidget *popup)
{
	gdk_draw_rectangle(popup->window,
		(popup->style)->fg_gc[GTK_STATE_NORMAL],
		FALSE,
		(popup->allocation).x,
		(popup->allocation).y,
		(popup->allocation).width - 1,
		(popup->allocation).height - 1);

	return TRUE;
}

static gint gui_completition_show_matches(GtkWidget *entry, gchar **fnamev,
							struct CompletitionPopup *data)
{
	static const gchar spaces[] = "   ";

	gint		x, y;
	gchar		*str;
	StringBuf	strbuf;

	stringbuf_new(&strbuf, 0);

	while ((str = *(fnamev++)) != NULL)
	{
		gint slen;

		str = CHARCONV_UTF8_FROM_FILENAME(str, &slen);

		if (str)
		{
			stringbuf_append(&strbuf, str, slen);
			CHARCONV_UTF8_DESTROY(str);
			stringbuf_append(&strbuf, spaces, sizeof(spaces) - 1);
		}
	}

	if (strbuf.len == 0)
	{
		stringbuf_delete(&strbuf);
		return -1;
	}

	stringbuf_truncate(&strbuf, strbuf.len - (sizeof(spaces) - 1));

#if GTK_MAJOR_VERSION > 1
	gtk_window_resize(GTK_WINDOW(data->popup), 1, 1);
#endif
	gtk_label_set_text(data->label, strbuf.str);
	stringbuf_delete(&strbuf);

	gtk_widget_realize(data->popup);
	gdk_window_get_origin(entry->window, &x, &y);
	GuiWindowMove(GTK_WINDOW(data->popup), x, y + entry->allocation.height);

	gtk_widget_show_all(data->popup);

	if (gdk_pointer_grab((data->popup)->window, FALSE,
		GDK_BUTTON_PRESS_MASK, NULL, NULL, GDK_CURRENT_TIME) ||
			gdk_keyboard_grab(entry->window, FALSE, GDK_CURRENT_TIME))
	{
		gui_completition_hide(data);
		gdk_beep();
	}

	return 0;

} /* static gint gui_completition_show_matches(GtkWidget*, gchar**,
							CompletitionPopup*) */

static gboolean gui_filename_completition(GtkWidget *ew, GdkEventKey *event,
							struct CompletitionPopup *data)
{
	gint		chrpos, visible;
	gchar		*dbname, *fname;
	const gchar	*errstr;
	GtkEditable	*editable;

	visible = GTK_WIDGET_VISIBLE(data->popup);

	if (event->keyval != GDK_Tab || (event->state & GDK_CONTROL_MASK))
	{
		if (visible)
		{
			gui_completition_hide(data);
		}

		if (event->keyval == GDK_Return)
		{
			gui_open_cache();
			return TRUE;
		}

		return FALSE;
	}

	GuiSignalEmitStopByName(ew, "key-press-event");

	if (visible)
	{
		if (event->keyval == GDK_Tab)
		{
			return TRUE;
		}

		gui_completition_hide(data);
	}

	editable = GTK_EDITABLE(ew);

	if (GuiEditableGetSelectionStart(editable, &chrpos))
	{
		gtk_editable_delete_selection(editable);
	}

	else
	{
		chrpos = gtk_editable_get_position(editable);
	}

	fname  = gtk_editable_get_chars(editable, 0, chrpos);
	dbname = CHARCONV_UTF8_TO_FILENAME(fname);
	errstr = gettext(invalid_filename_str);

	if (dbname)
	{
		CompletitonData	*ptr;

		ptr = tl_fname_completion(dbname);
		CHARCONV_UTF8_DESTROY(dbname);

		if (ptr == NULL)
		{
			errstr = (errno != 0) ?
					strerror(errno) :
						gettext("No match found");
		}

		else
		{
			gint	slen;
			gchar	*str;

			str = CHARCONV_UTF8_FROM_FILENAME(ptr->append, &slen);

			if (str)
			{
				gint		chrlen;
				const gchar	*tmperr;

				chrlen = UTF8_TOOLS_STRLEN(str);
				tmperr = NULL;

				if (chrlen <= 0)
				{
					if (ptr->fnamec == 1)
						tmperr = gettext("Match");
				}

				else
				{
					gint ipos;

					if (ptr->fnamec == 1)
					{
						if (*UTF8_TOOLS_OFFSET_TO_POINTER(str,
										chrlen - 1) == '/')
						{
							gchar *s = gtk_editable_get_chars(editable,
									chrpos, chrpos + 1);
							if (*s == '/')
							{
								slen--;
							}

							g_free(s);
						}
					}

					ipos = chrpos;
					gtk_editable_insert_text(editable, str, slen, &ipos);
					gtk_editable_set_position(editable, chrlen + chrpos);

				} /* if (chrlen <= 0) .. else */

				CHARCONV_UTF8_DESTROY(str);

				if (ptr->fnamec > 1 &&
					gui_completition_show_matches(ew, ptr->fnamev, data))
				{
					tmperr = errstr;
					
				}

				errstr = tmperr;

			} /* if (str) */

			tl_completion_free(ptr);

		} /* if (ptr == NULL) .. else */

	} /* if (dbname) */

	g_free(fname); /* free here (if using GTK+-1.2) */

	gui_err(errstr);

	return TRUE;

} /* static gboolean gui_filename_completition(GtkWidget*, GdkEventKey*, CompletitionPopup*) */

static void gui_completition_popup_init(GtkWidget *entry)
{
	static struct CompletitionPopup completition_popup;

	struct CompletitionPopup *data = &completition_popup;

	GtkWidget	*label, *popup;
	GtkStyle 	*style;

	GuiSignalConnect(entry, "key-press-event",
					gui_filename_completition, data);
	label = gtk_label_new(NULL);
	data->label = GTK_LABEL(label);
	gtk_label_set_line_wrap(data->label, TRUE);
	gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);

	popup = gtk_window_new(GTK_WINDOW_POPUP);
	data->popup = popup;
	gtk_container_add(GTK_CONTAINER(popup), label);
	gtk_container_set_border_width(GTK_CONTAINER(popup), 4);
	gtk_widget_set_events(popup, GDK_BUTTON_PRESS_MASK);

	GuiSignalConnect(popup, "button-press-event",
					gui_completition_button_press, data);
	GuiSignalConnectAfter(label, "expose-event",
					gui_completition_expose, popup);
	GuiSignalConnectSwapped(windata.mainwin, "destroy",
					gtk_widget_destroy, popup);

	style = gtk_style_copy(guitl_rc_get_style(label));

	gdk_color_parse("Black", &(style->fg[GTK_STATE_NORMAL]));
	gdk_color_parse("LightYellow", &(style->bg[GTK_STATE_NORMAL]));

	gtk_widget_set_style(label, style);
	gtk_widget_set_style(popup, style);

} /* static void gui_completition_popup_init(GtkWidget*) */

#endif /* HAVE_GLOB */

static GtkWidget *gui_build_entry_select_browse(void)
{
	GtkBox		*box;
	GtkWidget	*button, *ew, *frame, *label, *menu, *widget;
	gint		i;

	widget = gtk_hbox_new(FALSE, 4);
	gtk_container_set_border_width(GTK_CONTAINER(widget), 4);

	frame = guitl_frame_new_add_child_gettext(NULL, widget);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 2);

	box = GTK_BOX(widget);
	label = guitl_label_new_gettext(gettext_nop("Cache index file: "));
	gtk_box_pack_start(box, label, FALSE, FALSE, 5);

	ew	 = gtk_entry_new();
	entry_db = GTK_ENTRY(ew);
	gtk_box_pack_start(box, ew, TRUE, TRUE, 1);

#ifdef HAVE_GLOB
	gui_completition_popup_init(ew);
#else
	GuiSignalConnect(ew, "activate", gui_open_cache, NULL);
#endif
	nscache_type = gtk_option_menu_new();
	gtk_box_pack_start(box, nscache_type, FALSE, FALSE, 2);

#ifndef MULTI_DB_TYPES
	gtk_widget_set_sensitive(nscache_type, FALSE);
#endif
	menu	= gtk_menu_new();
	i	= NSCACHE_RT_FIRST;

	do
	{
		GtkWidget *mi;

		mi = guitl_get_icon_menu(i);
		GuiMenuAppend(menu, mi);

#ifdef MULTI_DB_TYPES
		GuiSignalConnect(mi, "activate", TypeOpen, (gpointer) i);
#endif
	}
	while (++i <= NSCACHE_RT_LAST);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(nscache_type), menu);

	button = guitl_get_icon_button(kIconButtonOpen);
	gtk_box_pack_start(box, button, FALSE, FALSE, 1);
	GuiSignalConnect(button, "clicked", Browse, &gui_menu_bar[kMenuBarFile]);

	return frame;

} /* static GtkWidget *gui_build_entry_select_browse(void) */

static void gui_build_submenu_checkitems(GtkWidget *submenu)
{
	int index = kColumnURL;

	do
	{
		GtkWidget *mi;

		mi = guitl_check_menu_item_new_gettext(column_info[index].label);
		GuiMenuAppend(submenu, mi);
		column_tab_array[index].menu = mi;
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mi),
						column_tab_array[index].visible);
		gaccel_bind_widget(column_info[index].accel_id,
						column_info[index].label, mi, NULL);

		column_set_visibility(mi, (gpointer) index);

		GuiSignalConnect(mi, "activate", column_set_visibility, (gpointer) index);
	}
	while (++index <= kColumnInfoLastIndex);

	column_menu_set_sensitive(FALSE);

} /* static void gui_build_submenu_checkitems(GtkWidget*) */

static void gui_build_menu_item(const GuiMenuItem *item, GtkWidget *menu, GtkWidget *window)
{
	for (;;)
	{
		GtkWidget *mi, *submenu;

		switch (item->type)
		{
			case kMTypeItem:
			case kMTypeSubmenu:
			case kMTypeSubmenuCheckItem:
				mi = guitl_menu_item_new_gettext(item->label);

				if (item->type == kMTypeItem)
				{
					break;
				}

				submenu = gtk_menu_new();
				gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), submenu);
				gtk_widget_realize(submenu);

				if (item->callback)
				{
					GuiSignalConnect(submenu, "show",
						item->callback, item->data);
				}

				if (item->type == kMTypeSubmenuCheckItem)
				{
					gui_build_submenu_checkitems(submenu);
				}

				break;

			case kMTypeCheckItem:
				mi = guitl_check_menu_item_new_gettext(item->label);
				gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mi),
						((struct gui_option *) item->data)->active);
				break;

			case kMTypeSeparator:
				mi = gtk_menu_item_new();
				break;

			default:
				return;
		}

		GuiMenuAppend(menu, mi);
		gtk_widget_show(mi);

		if (item->accel_id != kGAccelKeyNone)
		{
			gaccel_bind_widget(item->accel_id, item->label, mi, window);
		}

		if (item->callback && item->type != kMTypeSubmenu)
		{
			GuiSignalConnect(mi, "activate", item->callback, item->data);
		}

		if (item->storage)
		{
			*item->storage = mi;
		}

		item++;

	} /* for (;;) */

} /* static void gui_build_menu_item(const GuiMenuItem*, GtkWidget*, GtkWidget*) */

static GtkWidget *gui_build_menu(GtkWidget *window)
{
	static const struct build_menu_array
	{
		const gchar		*label;
		const GuiMenuItem	*entries;
		gboolean		adjust_menu;

	} build_menu_array[] = {

		{gettext_nop("File"), menuFile, FALSE},
		{gettext_nop("View"), menuView, TRUE},
		{gettext_nop("Edit"), menuEdit, TRUE},
		{gettext_nop("Options"), menuOptions, FALSE},
		{gettext_nop("Help"), menuHelp, FALSE},
		{NULL}

	};

	GtkWidget *mbar, *mbb, **mbbptr, *menu;

	const struct build_menu_array *build_tab;

	struct gui_option *opt = &gui_option[kOptionEntriesTotal - 1];

	do
	{
		gprop_get_bool(opt->prop, &(opt->active));
	}
	while (--opt >= gui_option);

	decode = gui_option[kOptionDecode].active;

	mbar = gtk_menu_bar_new();
	GuiMenuBarDisableShadow(mbar);

	build_tab = build_menu_array;
	mbbptr = gui_menu_bar;

	do
	{
		mbb = guitl_menu_item_new_gettext(build_tab->label);
		GuiSignalConnect(mbb, "destroy", gtk_widget_destroyed, mbbptr);
		*(mbbptr++) = mbb;
		GuiMenuBarAppend(mbar, mbb);

		if (build_tab->adjust_menu)
		{
			GuiSignalConnect(mbb, "activate",
					AdjustMenu, &(windata.toplist));
		}

		menu = gtk_menu_new();
		gtk_widget_realize(menu);

		gtk_menu_item_set_submenu(GTK_MENU_ITEM(mbb), menu);

		gui_build_menu_item(build_tab->entries, menu, window);
	}
	while ((++build_tab)->label);

	GuiMenuItemSetRightJustified(GTK_MENU_ITEM(mbb), TRUE);

	menu		= gtk_menu_new();
	gui_popup_menu	= menu;

	GuiSignalConnect(menu, "show", AdjustMenu, &(windata.toplist));

	gtk_widget_realize(menu);

	gui_build_menu_item(menuPopup, menu, NULL);

#ifdef CONFIG_COLUMN_POPUP

	menu = gtk_menu_new();
	popup_colswap.menu = menu;

	popup_colswap.left = guitl_menu_item_new_gettext(gettext_nop("left"));
	GuiMenuAppend(menu, popup_colswap.left);
	GuiSignalConnect(popup_colswap.left, "activate",
				column_move_left, &(popup_colswap.selected));

	popup_colswap.right = guitl_menu_item_new_gettext(gettext_nop("right"));
	GuiMenuAppend(menu, popup_colswap.right);
	GuiSignalConnect(popup_colswap.right, "activate",
				column_move_right, &(popup_colswap.selected));

	gtk_widget_show_all(menu);

	menu = gtk_menu_new();
	popup_collexp.menu = menu;

	mbb = guitl_menu_item_new_gettext(gettext_nop("collapse"));
	popup_collexp.collapse = mbb;
	GuiMenuAppend(menu, mbb);
	GuiSignalConnect(mbb, "activate", gui_tree_collapse_recursive,
				&(notebook_list[kTreeList].store.tree));

	mbb = guitl_menu_item_new_gettext(gettext_nop("expand"));
	popup_collexp.expand = mbb;
	GuiMenuAppend(menu, mbb);
	GuiSignalConnect(mbb, "activate", gui_tree_expand_recursive,
				&(notebook_list[kTreeList].store.tree));

	gtk_widget_show_all(menu);

#endif /* CONFIG_COLUMN_POPUP */

	return mbar;

} /* static GtkWidget *gui_build_menu(GtkWidget*) */

static void gui_build_msg_tmr(GtkWidget *box, struct msg_tmr *msg)
{
	GtkStyle  *style;
	GtkWidget *label;

	label = gtk_label_new(NULL);
	gtk_widget_ref(label);

	msg->label = GTK_LABEL(label);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	GuiSignalConnect(label, "size-allocate", gui_msg_set_label_text, (gpointer) msg);

	msg->frame = guitl_frame_new_add_child_gettext(NULL, label);
	gtk_container_add(GTK_CONTAINER(box), msg->frame);

	style = guitl_rc_get_style(label);

	if (msg == &error_msg)
	{
		style = gtk_style_copy(style);
		gdk_color_parse("Red", &(style->fg[GTK_STATE_NORMAL]));
		gdk_color_parse("LightGray", &(style->bg[GTK_STATE_NORMAL]));
		gtk_widget_set_style(label, style);

		msg_button_sig_id = gui_button_key_grab_init(msg->frame, "button-press-event");
		msg_key_sig_id = gui_button_key_grab_init(msg->frame, "key-press-event");

		msg->sig_id = GuiSignalConnect(label, "expose-event",
						gui_error_msg_expose, NULL);
		GuiSignalHandlerBlock(label, msg->sig_id);
	}

	msg->style = style;
}

void gui_init(char *dbname)
{
	int		i, visible;
	GtkBox		*vbox;
	GtkWidget	*entry, *menu_bar, *notebook, *widget, *window;

	g_log_set_handler("Gdk", G_LOG_LEVEL_MASK, (GLogFunc) _noop, NULL);

#ifdef CONFIG_SWAP_COLUMNS

	visible	= FALSE;
	i	= kColumnInfoLastIndex;

	do
	{
		ColumnTab *col_tab = &column_tab_array[i];

		if ( ! gprop_get_int(column_info[i].colnrprop, &col_tab->position))
		{
			col_tab->position = i;
		}

		visible += col_tab->position;
	}
	while (--i > kColumnURL);

	column_tab_array[kColumnURL].position = i;

	if (visible != 1+2+3+4+5+6+7+8)
	{
		ColumnTab *col_tab = &column_tab_array[kColumnInfoLastIndex];

		do
		{
			col_tab->position = 0;
		}
		while (--col_tab > &column_tab_array[kColumnURL]);
	}

#endif /* CONFIG_SWAP_COLUMNS */

	visible	= FALSE;
	i	= kColumnURL;

	do
	{
		ColumnTab *col_tab = &column_tab_array[i];

		col_tab->index = i;

#ifdef CONFIG_SWAP_COLUMNS
		if (col_tab->position == 0)
#endif
		{
			col_tab->position = i;
		}

#ifndef CONFIG_GTKCLIST_GTKCTREE
#ifdef CONFIG_COLUMN_POPUP
		col_tab->initcolnr = col_tab->position;
#endif
#endif
		column_tab[col_tab->position] = col_tab;

		if ( ! gprop_get_bool(column_info[i].show_column_prop, &col_tab->visible))
		{
			col_tab->visible = TRUE;
		}

		visible |= col_tab->visible;
	}
	while (++i <= kColumnInfoLastIndex);

	if (visible == FALSE)
	{
		column_tab_array[kColumnURL].visible = TRUE;
	}

#ifdef FOCUS_IN_BUGFIX
	windata._net_wm_state = gdk_atom_intern( _NET_WM_STATE, TRUE );
	windata.ismapped = NULL;
	windata.unmapped = NULL;
#endif

	windata.topwin = NULL;

	widget = gtk_vbox_new(FALSE, 0);
	window = gui_window_new("MozCache", widget, 2, kPropIntMainWinWidth);
	windata.mainwin = window;

	GuiSignalConnect(window, "delete-event", Quit, NULL);
	GuiSignalConnect(window, "destroy", gui_quit, NULL);

	gtk_selection_add_targets(window, GDK_SELECTION_PRIMARY,
			targetlist, sizeof(targetlist) / sizeof(GtkTargetEntry));

	GuiSignalConnect(window, "selection-clear-event",
					SelectionClear, &selection_urlstr);
	GuiSignalConnect(window, "selection-get",
					SelectionSend, &selection_urlstr);
	gtk_widget_realize(window);

	memset(notebook_list, '\0', sizeof(notebook_list));
	windata.toplist = &notebook_list[kTreeList];
	mainlist	= &notebook_list[kTreeList];
	GuiObjectSetUserData(window, &notebook_list[kTreeList]);

	guitl_create_icons(window);

	vbox = GTK_BOX(widget);
	widget = gtk_vbox_new(FALSE, 3);
	gtk_box_pack_end(vbox, widget, FALSE, FALSE, 0);

	gui_build_msg_tmr(widget, &status_msg);
	gui_build_msg_tmr(widget, &error_msg);

	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_box_pack_end(vbox, notebook, TRUE, TRUE, 2);

	gui_build_tree_list(notebook);
	gui_build_sorted_list(notebook);

	entry = gui_build_entry_select_browse();
	gtk_box_pack_end(vbox, entry, FALSE, FALSE, 0);

	menu_bar = gui_build_menu(window);
	gtk_box_pack_start(vbox, menu_bar, FALSE, FALSE, 0);

	GuiSignalConnect(notebook, "switch-page", notebook_page_switched, &windata);

	GuiTreeSignalConnect(&(notebook_list[kTreeList].store.tree),
		"button-press-event", gui_mouse_button_pressed, gui_popup_menu);

	GuiListSignalConnect(&(notebook_list[kSortedList].store.list),
		"button-press-event", gui_mouse_button_pressed, gui_popup_menu);

#if GTK_MAJOR_VERSION == 1

	GuiSignalConnect(window, "key-press-event",
			gui_sf10_show_popup_menu, gui_popup_menu);
#else
	GuiSignalConnect(window, "popup-menu",
			gui_sf10_show_popup_menu, gui_popup_menu);
#endif

	gui_is_active = TRUE;

	if (nscache_db.type > NSCACHE_RT_LAST)
	{
		nscache_read_cache(dbname, FALSE);
	}

	gtk_option_menu_set_history(GTK_OPTION_MENU(nscache_type), nscache_db.type);

	if (dbname)
	{
#if GTK_MAJOR_VERSION == 1
		gtk_entry_set_text(entry_db, dbname);
#else
		gchar *str = CHARCONV_UTF8_FROM_FILENAME(dbname, NULL);

		if (str)
		{
			gchar *fname;

			gtk_entry_set_text(entry_db, str);
			fname = CHARCONV_UTF8_TO_FILENAME(dbname);
			CHARCONV_UTF8_FREE(str, dbname);

			if (fname)
			{
				g_free(dbname);
				dbname = fname;
			}
		}
#endif
	}

	gtk_widget_show_all(window);

#ifndef CACHE_READONLY

	if (nscache_db.readonly)
	{
		gtk_widget_hide(gui_menu_widget[kMenuDelete]);
		gtk_widget_hide(gui_popup_widget[kMenuDelete]);
		gtk_widget_hide(gui_option[kOptionReadOnly].widget);
	}
#endif

	GuiIdleAddPriority(G_PRIORITY_LOW, gui_read_cache, dbname);

} /* void gui_init(char*) */

/* EOF */
