/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		modified 2005 ... 2008 by Harald Foerster		   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <sys/stat.h>

#ifdef HAVE_GETOPT_LONG
#include <getopt.h>
#else
#include "getopt.h"
#endif

#include "apassign.h"
#include "charconv.h"
#include "file.h"
#include "gaccel.h"
#include "gprop.h"
#include "nscache.h"
#include "viewer.h"

/*
#include "nls.h"
*/


#define kPrintVersion	(1 << 0)
#define kPrintHelp	(1 << 1)

enum
{
	NSC_FIRST,
	NSC_TYPE = NSC_FIRST,
	NSC_INDEXDB,
	NSC_VIEWER,
	NSC_MENUACCEL,
	NSC_PROP,
	NSC_LAST = NSC_PROP
};

static const char *cfgrc_name[] =
{
	"CacheType:",
	"IndexDB:",
	"Viewer:",
	"MenuAccel:",
	"Property:"
};

static const char* nscache_cache_type_str[] =
{
#ifdef CONFIG_BERKELEY_DB
	"ns",
	"ns6",
#endif
	"moz",

#ifdef HAVE_MSIE
	"ie",
#endif
	"auto",
	NULL
};

static const char illegal_cache_type_str[] = gettext_nop("Illegal cache type");

static const char* nscache_sort_type_str[] =
{
	"url",
	"file",
	"size",
	"access",
	"mime",
	NULL
};

static int get_str_index(const char **tab, const char *pattern)
{
	int		idx;
	const char	*str;

	idx = 0;

	while ((str = *(tab++)) != NULL)
	{
		if (strcmp(str, pattern) == 0)
		{
			return idx;
		}

		idx++;
	}

	return -1; /* no match */

} /* static int get_str_index(const char**, const char*) */

static void fprn_stderr(const char *format, const char *mesg, const char *detail)
{
	if (format == NULL)
	{
		format = "%s : %s\n";
	}

	fprintf(stderr, format, gettext(mesg), detail);

} /* static void fprn_stderr(const char*, const char*, const char*) */

static FILE *open_rc(const char *modus)
{
	FILE		*fd;
	char		*rc;
	const char	*home;

	home = g_get_home_dir();

	if (home == NULL)
	{
		return (FILE *) home;
	}

	rc = g_strconcat(home, "/.mozcacherc", NULL);
	fd = fopen(rc, modus);

	g_free(rc);

	return fd;

} /* static FILE *open_rc(const char*) */

char *save_rc(void)
{
	FILE *fd = open_rc("w");

	if (fd == NULL)
	{
		return gettext_nop("Unable to open ~/.mozcacherc file");
	}

	fprintf(fd, "# %s\n# %s\n\n",
			gettext("This .mozcacherc file was generated by MozCache"),
					gettext("You may edit it if you're careful!"));
	if (nscache_db.name)
	{
		const char *path;

		if (nscache_db.type < NSCACHE_RT_UNKNOWN)
		{
			fprintf(fd, "%s %s\n", cfgrc_name[NSC_TYPE],
					nscache_cache_type_str[nscache_db.type]);
		}

		path = nscache_db.path.str;

		fprintf(fd, "%s %s%s\n\n", cfgrc_name[NSC_INDEXDB],
				(path == NULL ? "" : path), nscache_db.name);
	}

	apassign_save_data(fd, cfgrc_name[NSC_VIEWER]);
	gaccel_save_data(fd, cfgrc_name[NSC_MENUACCEL]);
	gprop_save_data(fd, cfgrc_name[NSC_PROP]);

	fclose(fd);

	return NULL;

} /* char *save_rc(void) */

static void load_rc(int errmsg, char **dbname)
{
	FILE *fd;
	char *line, pom[2048];

	fd = open_rc("r");

	if (fd == NULL)
	{
		return;
	}

	while ((line = fgets(pom, sizeof(pom), fd)) != NULL)
	{
		static const char parse_msg[] = gettext_nop("Unable to parse");
		static const char parse_fmt[] = "%s : \"%s\"\n";
		static const char white_spaces[] = " \t";

		int i;

		line += strspn(line, white_spaces);

		if (*line == '#' || (i = strcspn(line, "\n")) == 0)
		{
			continue;
		}

		line[i]	= '\0';
		i	= NSC_LAST;

		do
		{
			int		len;
			const char	*str;

			str = cfgrc_name[i];
			len = strlen(str);

			if (g_strncasecmp(line, str, len) == 0)
			{
				line += len;
				line += strspn(line, white_spaces);
				break;
			}
		}
		while (--i >= NSC_FIRST);

		switch (i)
		{
			case NSC_TYPE:
			{
				int type = get_str_index(nscache_cache_type_str, line);

				if (type < 0)
				{
					if (errmsg)
					{
						fprn_stderr(NULL, illegal_cache_type_str, line);
					}

					break;
				}

				nscache_db.type = type;
				break;
			}

			case NSC_INDEXDB:
			{
				if (*line != '\0')
				{
					*dbname = g_strdup(line);
				}

				break;
			}

			case NSC_VIEWER:
			{
				apassign_parse_str(line);
				break;
			}

			case NSC_MENUACCEL:
			{
				if (gaccel_parse_str(line) == FALSE && errmsg)
				{
					fprn_stderr(parse_fmt, parse_msg, line);
				}

				break;
			}

			case NSC_PROP:
			{
				if (gprop_parse_str(line) == FALSE && errmsg)
				{
					fprn_stderr(parse_fmt, parse_msg, line);
				}

				break;
			}

			default:
			{
				if (errmsg)
				{
					fprn_stderr(parse_fmt, parse_msg, line);
				}

				break;
			}

		} /* switch (i) */

	} /* while ((line = fgets(pom, sizeof(pom), fd)) != NULL) */

	fclose(fd);

} /* static void load_rc(int, char**) */

static void print_version_help(int what, const char *prgname)
{

#ifdef CONFIG_BERKELEY_DB
#define BERKELEY_DB_STR	"ns|ns6|"
#else
#define BERKELEY_DB_STR
#endif

#ifdef HAVE_MSIE
#define MSIE_STR	"ie|"
#else
#define MSIE_STR
#endif

	if (what & kPrintVersion)
	{
		printf("%s-%s\n", nscache_package_str, nscache_version_str);
	}

	if ( ! (what & kPrintHelp))
	{
		exit(0);
	}

	printf("%s %s %s %s\n", gettext("Usage:"), prgname,
				gettext("[OPTION]"), gettext("[FILENAME]"));

	printf(
		"  -t,  --db-type=[" BERKELEY_DB_STR "moz|" MSIE_STR "auto]\n"
		"                           %s\n"
#ifdef CONFIG_BERKELEY_DB
		"                           ns   = %s\n"
		"                           ns6  = %s\n"
		"                                 %s\n"
#endif
		"                           moz  = %s\n"
		"                                 %s\n"
#ifdef HAVE_MSIE
		"                           ie   = %s\n"
#endif
		"                           auto = %s\n"
		"  -f,  --db-file=%s %s\n"
		"  -d,  --temp-dir=%s %s\n"
		"  -s,  --summary           %s\n"
		"  -c,  --content-types     %s\n"
		"  -l,  --list              %s\n"
		"  -m,  --multi-line        %s\n"
		"  -S,  --sort=[url|file|size|access|mime]\n"
		"                           %s\n"
		"  -R,  --sort-reverse      %s\n"
#ifndef CACHE_READONLY
		"  -r,  --readonly          %s\n"
#endif
		"  -v,  --version           %s\n"
		"  -h,  --help              %s\n",
		gettext("set cache type"),
#ifdef CONFIG_BERKELEY_DB
			gettext(NSC_GET_BROWSER_NAME(NSCACHE_RT_NETSCAPE_4)),
			gettext(NSC_GET_BROWSER_NAME(NSCACHE_RT_NETSCAPE_6)),
			gettext("(also Mozilla < 0.9)"),
#endif
			gettext(NSC_GET_BROWSER_NAME(NSCACHE_RT_MOZILLA)),
			gettext("(also Firefox, SeaMonkey and Netscape)"),
#ifdef HAVE_MSIE
			gettext(NSC_GET_BROWSER_NAME(NSCACHE_RT_MSIE)),
#endif
			gettext("unknown (default)"),
		gettext("FILENAME "),
			gettext("set cache file to FILENAME"),
		gettext("DIR     "),
			gettext("store temporary files in DIR (default: /tmp)"),
		gettext("print number of entries and total size"),
		gettext("print info about cache content types"),
		gettext("list cache directory contents to stdout"),
		gettext("print multi line info about each cache entry"),
		gettext("print cache entries sorted by type (default: url)"),
		gettext("print cache entries in reverse order"),
#ifndef CACHE_READONLY
		gettext("activate readonly mode"),
#endif
		gettext("display version information and exit"),
		gettext("display this help and exit")

	); /* printf() */

	exit(0);

} /* static void print_version_help(intt, const char*) */

static int cmd_setup(int argc, char **argv, char **dbname)
{
	const struct option options[] =
	{
		{"db-type", 1, NULL, 't'},
		{"db-file", 1, NULL, 'f'},
		{"temp-dir", 1, NULL, 'd'},
		{"summary", 0, NULL, 's'},
		{"content-types", 0, NULL, 'c'},
		{"list", 0, NULL, 'l'},
		{"multi-line", 0, NULL, 'm'},
		{"sort", 1, NULL, 'S'},
		{"sort-reverse", 0, NULL, 'R'},
#ifndef CACHE_READONLY
		{"readonly", 0, NULL, 'r'},
#endif
		{"version", 0, NULL, 'v'},
		{"help", 0, NULL, 'h'},
		/* !!! GTK&GDK OPTIONS !!! */
		{"gtk-module", 1, NULL, 'g'},
		{"g-fatal-warnings", 0, NULL, 'g'},
		{"gtk-debug", 2, NULL, 'g'},
		{"gtk-no-debug", 2, NULL, 'g'},
		{"gdk-debug", 2, NULL, 'g'},
		{"gdk-no-debug", 2, NULL, 'g'},
		{"display", 1, NULL, 'g'},
		{"sync", 0, NULL, 'g'},
		{"no-xshm", 0, NULL, 'g'},
		{"name", 1, NULL, 'g'},
		{"class", 1, NULL, 'g'},
		{"gxid_host", 1, NULL, 'g'},
		{"gxid_port", 1, NULL, 'g'},
		{"xim-preedit", 1, NULL, 'g'},
		{"xim-status", 1, NULL, 'g'},
		{NULL, 0, NULL, 0},
	};

	/* viewer.c */
	extern char *viewer_temp_dir;

	int	c, index, print_list, print_info, sort_reverse;
	char	*db_name;
	char	*db_type;
	char	*sort_type;
	char	*temp_dir;

	print_list = kNoStdOut;
	print_info = 0;
	sort_reverse = 0;

	db_name	   = NULL;
	db_type    = NULL;
	sort_type  = NULL;
	temp_dir   = NULL;

	*dbname		= NULL;
	nscache_db.type	= NSCACHE_RT_UNKNOWN;

#ifndef CACHE_READONLY
#define OPTION_RDONLY	"r"
#else
#define OPTION_RDONLY
#endif
	while ((c = getopt_long(argc, argv, OPTION_RDONLY "sclmvhRd:f:t:S:",
							options, &index)) != - 1)
	{
		switch (c)
		{
			case 't':
				db_type = optarg;
				break;
			case 'f':
				db_name = optarg;
				break;
			case 'd':
				temp_dir = optarg;
				break;
			case 's':
				print_list = kOnlyStdOutSummary;
				break;
			case 'c':
				print_list = kOnlyStdOutConentTypes;
				break;
			case 'l':
				print_list = kOnlyStdOutList;
				break;
			case 'm':
				print_list = kOnlyStdOutMultiLine;
				break;
			case 'S':
				sort_type = optarg;
				break;
			case 'R':
				sort_reverse = STYPE_REVERS_MASK;
				break;
			case 'v':
				print_info |= kPrintVersion;
				break;
			case 'h':
				print_info |= kPrintHelp;
				break;
#ifndef CACHE_READONLY
			case 'r':
				nscache_db.readonly = TRUE;
				break;
#endif
			case 'g':
				break;

			default:
				return EINVAL;

		} /* switch (c) */

	} /* while ((c = getopt_long(argc, argv, ... )) != - 1) */

	if (optind < argc)
	{
		if (db_name != NULL || argc > optind + 1)
		{
			fprn_stderr(NULL, gettext_nop("Illegal argument"), argv[optind]);
			return EINVAL;
		}

		db_name = argv[optind];
	}

	if (print_info)
	{
		print_version_help(print_info, argv[0]);
	}

	load_rc(print_list == kNoStdOut, dbname);

	if (db_name)
	{
		g_free(*dbname);

		*dbname		= g_strdup(db_name);
		nscache_db.type	= NSCACHE_RT_UNKNOWN;
	}

	if (db_type)
	{
		int type = get_str_index(nscache_cache_type_str, db_type);

		if (type < 0)
		{
			fprn_stderr(NULL, illegal_cache_type_str, db_type);
			return EINVAL;
		}

		nscache_db.type = type;
	}

	if (print_list)
	{
		int error, stype;

		if (*dbname == NULL
#ifdef HAVE_MSIE
			&& (nscache_db.type != NSCACHE_RT_MSIE)
#endif
			)
		{
			fprn_stderr("%s%s", gettext_nop("Cache file not specified!"), "\n");
			return EINVAL;
		}

		stype = STYPE_URL;

		if (sort_type && (print_list == kOnlyStdOutList ||
					print_list == kOnlyStdOutMultiLine))
		{
			stype = get_str_index(nscache_sort_type_str, sort_type);

			if (stype < 0)
			{
				fprn_stderr(NULL, gettext_nop("Illegal sort type"), sort_type);
				return EINVAL;
			}
		}

		error = nscache_dump(*dbname, print_list, stype | sort_reverse);
		exit(error);

	} /* if (print_list) */


	if (temp_dir)
	{
		struct stat	tmpstat;
		const char	*home = NULL;

		if (temp_dir[0] == '~' &&
			(temp_dir[1] == '\0' || temp_dir[1] == '/'))
		{
			temp_dir++;
			home = g_get_home_dir();
		}

		if (home == NULL)
		{
			home = "";
		}

		temp_dir = g_strconcat(home, temp_dir, NULL);

		if ( stat(temp_dir, &tmpstat) ||
			! (S_ISDIR(tmpstat.st_mode)) ||
				access(temp_dir, X_OK | W_OK))
		{
			g_free(temp_dir);
			temp_dir = NULL;
		}

	} /* if (temp_dir) */

	if (temp_dir == NULL)
	{
		temp_dir = strdup(g_get_tmp_dir());
	}

	if (temp_dir)
	{
		int len = strlen(temp_dir) - 1;

		while (len >= 0 && temp_dir[len] == '/')
		{
			temp_dir[len] = '\0';
			len--;
		}

		viewer_temp_dir = temp_dir;

	} /* if (temp_dir) */

	if (*dbname == NULL)
	{
		const char *home = g_get_home_dir();

		if (home)
		{

#ifdef CONFIG_BERKELEY_DB
			*dbname = g_strconcat(home, "/.netscape/cache", NULL);

			if (access(*dbname, F_OK) == 0)
			{
				return 0;
			}

			g_free(*dbname);
#endif
			*dbname = g_strconcat(home, "/.mozilla", NULL);
		}

	} /* if (*dbname == NULL) */

	return 0;

} /* static int cmd_setup(int, char**, char**) */

static void main_sigterm(int signum)
{
	int save_flag;

	/* store settings */
	if ( ! gprop_get_bool(kPropBoolOptSaveConfig, &save_flag) || save_flag)
	{
		const char *errstr = save_rc();

		if (errstr && signum != SIGTERM)
		{
			gui_strerror(errstr);
		}
	}

	/* Remove all temporary files */
	viewer_atexit();

	if (signum == SIGTERM)
	{
		_exit(0);
	}
}

static void main_atexit(void)
{
	/* user lost interrest */
	main_sigterm(SIGQUIT);
}

static void main_signal_init(void)
{
	struct sigaction sigact;

	parent_pid = getpid();
	parent_uid = getuid();

	/* block other signals, while executing 'sa_handler' */
	sigfillset(&sigact.sa_mask);
	sigact.sa_flags = 0;

	/* ignore broken pipes */
	sigact.sa_handler = SIG_IGN;
	sigaction(SIGPIPE, &sigact, NULL);

	/* MozCache was terminated */
	sigact.sa_handler = main_sigterm;
	sigaction(SIGTERM, &sigact, NULL);

	/* child was terminated */
	sigact.sa_handler = viewer_sigchld;
	sigact.sa_flags = SA_NOCLDSTOP;
	sigaction(SIGCHLD, &sigact, NULL);

} /* static void main_signal_init(void) */

int main(int argc, char **argv)
{
	int	error;
	char	*dbname;

	NLS_INIT_LOCALE();

	error = cmd_setup(argc, argv, &dbname);

	if (error)
	{
		return error;
	}

#ifdef HAVE_LINK
	gprop_get_bool(kPropBoolSaveHardlink, &hardlink);
#endif
	gprop_get_bool(kPropBoolSaveSingleFolder, &single_folder);
	gprop_get_int(kPropIntSaveUrlToLocal, &url_to_local_selector);

	apassign_set_default_viewers();

	gtk_init(&argc, &argv);
	charconv_init();
	gui_init(dbname);

	main_signal_init();
	atexit(main_atexit);

	gtk_main();

	return 0;

} /* int main(int, char**) */

/* EOF */
