/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		modified 2005 ... 2008 by Harald Foerster		   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>

#include "moz.h"
#include "bswap32.h"
#include "file.h"
#include "nls.h"

/*
#include <sys/types.h>
#include <sys/stat.h>
#include "nscache.h"
#include "stringbuf.h"
#include <gtk/gtk.h>
#include <time.h>
*/


#define MOZ_MAX_RECS_PER_BUCKET	256
#define MOZ_BITMAP_SIZE		4096

#ifdef MOZ_ENABLE_MAX_VERSION
#define MOZ_VERSION(v)		(((v) >= MOZ_MIN_VERSION) && ((v) <= MOZ_MAX_VERSION))
#else
#define MOZ_VERSION(v)		((v) >= MOZ_MIN_VERSION)
#endif

#define MOZ_LOC_INITED(v)	 ((v) & 0x80000000)
#define MOZ_LOC_FILE_INDEX(v)	(((v) & 0x30000000) >> 28)
#define MOZ_LOC_FILE_SIZE(v) 	 ((v) & 0x00FFFF00)
#define MOZ_LOC_BLOCK_COUNT(v)	(((v) & 0x03000000) >> 24)
#define MOZ_LOC_BLOCK_NUMBER(v)  ((v) & 0x00FFFFFF)
#define MOZ_LOC_GENERATION(v)	 ((v) & 0xFF)

#define MOZ_GET_BLOCK_OFFSET(block_size, block_number)	\
		(((block_size) * (block_number)) + MOZ_BITMAP_SIZE)

/* includes terminating null byte */
#define MOZ_FILENAME_LENGTH	(12)

#define kRecordArraySize	MOZ_MAX_RECS_PER_BUCKET + 1

#define kBuckets		(1 << 5)
#define kBucketStart		(sizeof(MozCacheMapHeader))
#define kBucketStart_1v6	(sizeof(MozCacheMapHeader_1v6))

#define GetBucketIndex(hashnr)	((hashnr) & (kBuckets - 1))

#if 0
#define GetRecordsPerBucket()	 	 (MOZ_MAX_RECS_PER_BUCKET)
#define GetRecordsPerBucket_1v6(records) ((records) / kBuckets)
#endif /* 0 */

typedef struct
{
	guint32	version;
	gint32	size;
	gint32	nentries;
	guint32	dirty;

} MozMapFileHeader;

typedef struct
{
	MozMapFileHeader	filehead;
	guint32			erank[kBuckets];
	guint8			reserved[MOZ_BITMAP_SIZE -
					sizeof(MozMapFileHeader) -
					(sizeof(guint32) * kBuckets)];
} MozCacheMapHeader;

typedef struct
{
	MozMapFileHeader	filehead;
	gint32			record_count;
	guint32			erank[kBuckets];
	guint32			bucket_usage[kBuckets];

} MozCacheMapHeader_1v6;

typedef struct
{
	guint32	hashnr;
	guint32	erank;
	guint32	data_loc;
	guint32	meta_loc;

} MozCacheMapRecord;

typedef struct
{
	guint32	version;	/* usefull for stand-alone metadata files */
	guint32	meta_loc;	/* for verification */
	gint32	fetch_count;
	guint32	last_fetched;
	guint32	last_modified;
	guint32	expiration;
	guint32	data_size;
	guint32	key_size;	/* includes terminating null byte */
	guint32	meta_size;	/* includes terminating null byte */
	guint8	key_start[1];	/* start of key data */

} MozCacheEntryRecord;

#define CMR_BYTESWAP_UNKNOWN	((ByteSwap32Func) -1)

enum
{
	MOZ_FIRST_FILE,
	MOZ_MAP_FILE	= MOZ_FIRST_FILE,
	MOZ_B1_FILE,
	MOZ_B2_FILE,
	MOZ_B3_FILE,
	MOZ_LAST_FILE	= MOZ_B3_FILE
};

typedef struct
{
	int		fd;
	const size_t	block_size;
	const char	*name;

} MozCacheFile;

typedef struct
{
	guint32		version_orig;
	guint32		version_swap;
	guint32		dirty;
	guint32		recs_per_bucket;
	ByteSwap32Func	cmr_byteswap;
	int		err_map;
	int		err_block;

} MozMapFile;

typedef struct
{
	MozCacheFile	cache_file[4];
	MozMapFile	map_file;

} MozDatabase;

static MozDatabase moz_database =
{
	/*cache_file[4] */
	{
		{ -1,    0, "_CACHE_MAP_" },
		{ -1,  256, "_CACHE_001_" },
		{ -1, 1024, "_CACHE_002_" },
		{ -1, 4096, "_CACHE_003_" }
	},

	/*map_file */
	{
		0
	}
};

static const gchar moz_err_cache_in_use[] =
			gettext_nop("WARNING: cache in use");

static GSList*	moz_read_db(char **dbname);
static int	moz_cache_check(const char *dbname);

#ifndef CACHE_READONLY
static int	moz_record_delete(nscache_record_t *);
#endif

NSC_BROWSER moz_browser =
{
	moz_read_db,

#ifndef CACHE_READONLY
	moz_record_delete,
#endif
	moz_cache_check,

	gettext_nop("Mozilla")
};



static void moz_close_db(void)
{
	MozCacheFile *moz_file = &moz_database.cache_file[MOZ_LAST_FILE];

	do
	{
		if (moz_file->fd >= 0)
		{
			close(moz_file->fd);
			moz_file->fd = -1;
		}
	}
	while (--moz_file >= &moz_database.cache_file[MOZ_FIRST_FILE]);

} /* static void moz_close_db(void) */

static MozDatabase *moz_open_db(const char *dbname, MozCacheFile *moz_file, int mode, int errmesg)
{
	gint			dirty;
	gchar			*str;
	MozDatabase		*moz_db;
	MozCacheMapHeader_1v6	header;

	moz_db = &moz_database;

	if (moz_file == NULL)
	{
		moz_file = &moz_db->cache_file[MOZ_LAST_FILE];
	}

	do
	{
		moz_file->fd = file_open(moz_file->name, mode);

		if (moz_file->fd < 0)
		{
			if (errmesg)
				gui_strerror(nscache_err_open_index_file);

			moz_close_db();

			return NULL;
		}
	}
	while (--moz_file >= &moz_db->cache_file[MOZ_FIRST_FILE]);

	dirty = 0;

	if (read((moz_db->cache_file[MOZ_MAP_FILE]).fd,
				&header, sizeof(header)) != sizeof(header))
	{
		str = gettext_nop("Error reading cache information");
	}

	else
	{
		guint32	version;

		version = header.filehead.version;

		if ((moz_db->map_file).version_orig == 0)
		{
			(moz_db->map_file).version_orig = version;
			(moz_db->map_file).version_swap = UINT32_FROM_BIG_ENDIAN(version);

			if ( ! MOZ_VERSION((moz_db->map_file).version_swap) )
			{
				if (errmesg)
				{
					gui_errfmt("%s : version %d.%d",
						gettext("Mozilla cache map file not recognized"),
						(unsigned int) ((moz_db->map_file).version_swap >> 16),
						(unsigned int) ((moz_db->map_file).version_swap & 0x0000FFFF));
				}

				moz_close_db();

				return NULL;
			}
		}

		dirty = header.filehead.dirty;
		(moz_db->map_file).dirty = dirty;

		if (version != (moz_db->map_file).version_orig)
		{
			str = gettext_nop("Version number differs from previous");
		}

		else if ((moz_db->map_file).version_swap < 0x00010006)
		{
			if (lseek((moz_db->cache_file[MOZ_MAP_FILE]).fd,
					kBucketStart, SEEK_SET) == kBucketStart)
			{
				(moz_db->map_file).recs_per_bucket =
							MOZ_MAX_RECS_PER_BUCKET;
				return moz_db;
			}

			str = gettext_nop("Error executing \'lseek\'");
		}

		else
		{
			guint32	hdr_recs, max_recs;

			hdr_recs = UINT32_FROM_BIG_ENDIAN(header.record_count);
			max_recs = 16 * kBuckets;

			do
			{
				if (max_recs == hdr_recs)
				{
					(moz_db->map_file).recs_per_bucket =
									max_recs / kBuckets;
					return moz_db;
				}

				max_recs <<= 1;
			}
			while (max_recs <= 256 * kBuckets);

			str = gettext_nop("Unknown number of max records");
		}
	}

	if (errmesg)
	{
		str = gettext(str);

		if (dirty)
		{
			gui_errfmt("%s - %s",
				gettext(moz_err_cache_in_use), str);
		}

		else
		{
			gui_err(str);
		}
	}

	moz_close_db();

	return NULL;

} /* static int moz_open_db(const char*, MozCacheFile*, int, int) */

static int moz_generate_filename(MozCacheMapRecord *cmr, int type, char* fname)
{
	guint32 generation =
		MOZ_LOC_GENERATION(type == 'm' ? cmr->meta_loc : cmr->data_loc);

	return sprintf(fname, "%08X%c%02X", cmr->hashnr, (char) type, generation);
}

static moz_record_t *moz_compose_record(MozDatabase *moz_db, MozCacheMapRecord *cmr,
						MozCacheEntryRecord *entry, size_t size)
{
	moz_record_t	*rec;
	char		*key, *str, *end;
	unsigned int	index;
	guint32		len;

	if (sizeof(MozCacheEntryRecord) > size)
	{
		return NULL;
	}

	end = &((char *) entry)[size];
	key = entry->key_start;

	if (key <= (char *) entry || key > end)
	{
		return NULL;
	}

	key += UINT32_FROM_BIG_ENDIAN(entry->key_size);

	if (key <= (char *) entry || key > end)
	{
		return NULL;
	}

	len = UINT32_FROM_BIG_ENDIAN(entry->meta_size);

	if (&key[len] <= (char *) entry || &key[len] > end)
	{
		return NULL;
	}

	if (entry->version != (moz_db->map_file).version_orig)
	{
		return NULL;
	}

	/* initialize to '\0' */
	rec = g_malloc0(sizeof(moz_record_t));

	rec->recordid = cmr->hashnr;
	rec->record.content_length = UINT32_FROM_BIG_ENDIAN(entry->data_size);

	index = MOZ_LOC_FILE_INDEX(cmr->data_loc);

	if (index)
	{
		MozCacheFile *moz_file = &moz_db->cache_file[index];

		rec->record.content_offset =
			MOZ_GET_BLOCK_OFFSET(moz_file->block_size,
						MOZ_LOC_BLOCK_NUMBER(cmr->data_loc));

		rec->record.filename = (char *) moz_file->name;
	}

	else
	{
		char fname[MOZ_FILENAME_LENGTH];
#if 0
		rec->record.content_offset = 0;
#endif
		moz_generate_filename(cmr, 'd', fname);
		rec->record.filename = g_strdup(fname);
	}

	str = strchr(entry->key_start, ':');

	if (str && str[1] != '\0')
	{
		rec->record.urlstr = g_strdup(&str[1]);

		rec->record.last_accessed = UINT32_FROM_BIG_ENDIAN(entry->last_fetched);
		rec->record.last_modified = UINT32_FROM_BIG_ENDIAN(entry->last_modified);
		rec->record.expires = UINT32_FROM_BIG_ENDIAN(entry->expiration);

		str = key;
		key = tl_mem_find_str(key, len, "response-head", strlen("response-head"));

		if (key)
		{
			time_t tmp;

			key += strlen("response-head") + 1;
			len -= key - str;

			/* Content-Type: */
			str = tl_get_mime_value_str(key, len, "content-type:",
							sizeof("content-type:") - 1);
			rec->record.content_type = content_type_into_glist(str);

			/* "charset" */
			str = tl_get_mime_value_str(key, len, "charset",
								sizeof("charset") - 1);
			rec->record.charset = content_misc_into_gslist(str);

			/* Content-Encoding: */
			str = tl_get_mime_value_str(key, len, "content-encoding:",
							sizeof("content-encoding:") - 1);
			rec->record.content_encoding = content_misc_into_gslist(str);

			/* Date: */
			str = tl_get_mime_value_str(key, len, "date:", sizeof("date:") - 1);
			tmp = time_from_string(str);

			if (tmp)
			{
				rec->record.last_accessed = tmp;
			}

			/* Last-Modified: */
			str = tl_get_mime_value_str(key, len, "last-modified:",
							sizeof("last-modified:") - 1);
			tmp = time_from_string(str);

			if (tmp)
			{
				rec->record.last_modified = tmp;
			}

			/* Expires: */
			str = tl_get_mime_value_str(key, len, "expires:", sizeof("expires:") - 1);
			tmp = time_from_string(str);

			if (tmp)
			{
				rec->record.expires = tmp;
			}

		} /* if (key) */

		return rec;

	} /* if (str && str[1] != '\0') */

	nscache_record_free(&rec->record);

	return NULL;

} /* static moz_record_t *moz_compose_record(MozDatabase*, MozCacheMapRecord*,
							MozCacheEntryRecord*, size_t) */

static MozCacheEntryRecord *moz_load_entry_record(MozDatabase *moz_db,
							MozCacheMapRecord *cmr, size_t *size)
{
	unsigned int	index, offset;
	guint32		blocknr;
	MozCacheFile	*moz_file;

	index = MOZ_LOC_FILE_INDEX(cmr->meta_loc);

	if (index == 0)
	{
		nscache_record_t	nscrec;
		char			fname[MOZ_FILENAME_LENGTH];

		/* read entry record stored in stand alone file */
		nscrec.content_offset = 0;
		moz_generate_filename(cmr, 'm', fname);
		nscrec.filename = fname;

		return file_read(&nscrec, size);
	}

	moz_file	= &moz_db->cache_file[index];
	blocknr		= MOZ_LOC_BLOCK_NUMBER(cmr->meta_loc);
	offset		= MOZ_GET_BLOCK_OFFSET(moz_file->block_size, blocknr);

	if (lseek(moz_file->fd, offset, SEEK_SET) == offset)
	{
		guint32			block_count;
		MozCacheEntryRecord	*entry;

		block_count = MOZ_LOC_BLOCK_COUNT(cmr->meta_loc) + 1;

		*size	= moz_file->block_size * block_count;
		entry	= malloc(*size + 1); /* use malloc() here - like file_read() */

		if (entry)
		{
			int sz = read(moz_file->fd, entry, *size);

			if (sz != -1)
			{
				((char *) entry)[sz] = '\0';
				*size = sz;

				return entry;
			}

			free(entry);
		}
	}

	return NULL;

} /* static MozCacheEntryRecord *moz_load_entry_record(MozDatabase*, MozCacheMapRecord*, size_t*) */

static void moz_map_record_byteswap(ByteSwap32Func swap, MozCacheMapRecord *cmr)
{
	cmr->hashnr	= (*swap)(cmr->hashnr);
	/*cmr->erank	= (*swap)(cmr->erank);*/
	cmr->data_loc	= (*swap)(cmr->data_loc);
	cmr->meta_loc	= (*swap)(cmr->meta_loc);
}

static moz_record_t *moz_get_entry_info(MozDatabase *moz_db, MozCacheMapRecord *cmr)
{
	static const ByteSwap32Func swapfunc_tab[] =
	{
		BSWAP32_BIG_ENDIAN_FUNC,
		BSWAP32_LITTLE_ENDIAN_FUNC,
#ifdef CONFIG_PDP_ENDIAN
		BSWAP32_PDP_ENDIAN_FUNC,
#endif
		CMR_BYTESWAP_UNKNOWN
	};

	const ByteSwap32Func *swapfunc_ptr = swapfunc_tab;

	for (;;)
	{
		ByteSwap32Func		swap;
		MozCacheMapRecord	cmrrec, *cmrptr;

		swap = (moz_db->map_file).cmr_byteswap;

		if (swap == CMR_BYTESWAP_UNKNOWN)
		{
			swap = *(swapfunc_ptr++);

			if (swap == CMR_BYTESWAP_UNKNOWN)
			{
				return NULL;
			}

			cmrrec = *cmr;
			cmrptr = &cmrrec;
		}

		else
		{
			cmrptr = cmr;
		}

		if (swap)
		{
			moz_map_record_byteswap(swap, cmrptr);
		}

		if ( ! MOZ_LOC_INITED(cmrptr->data_loc) ||
				! MOZ_LOC_INITED(cmrptr->meta_loc))
		{
			if ((moz_db->map_file).cmr_byteswap != CMR_BYTESWAP_UNKNOWN)
			{
				return NULL;
			}
		}

		else
		{
			size_t			size;
			MozCacheEntryRecord	*entry;

			entry = moz_load_entry_record(moz_db, cmrptr, &size);

			if (entry)
			{
				moz_record_t *rv = moz_compose_record(moz_db, cmrptr, entry, size);

				free(entry); /* malloc'd */

				if (rv)
				{
					(moz_db->map_file).cmr_byteswap = swap;
					return rv;
				}
			}

			if ((moz_db->map_file).cmr_byteswap != CMR_BYTESWAP_UNKNOWN)
			{
				break;
			}
		}

	} /* for (;;) */

	(moz_db->map_file).err_block++;

	return NULL;

} /* static moz_record_t *moz_get_entry_info(MozDatabase *moz_db, MozCacheMapRecord *cmr) */

static GSList *moz_read_db(char **dbname)
{
	int		buckets, bucket_size, len;
	GSList		*retv;
	MozDatabase	*moz_db;

	union
	{
		char			pom[2048];
		MozCacheMapRecord	record_array[kRecordArraySize];
	} u;

	g_free(*dbname);
	*dbname = NULL;

	moz_db = &moz_database;
	(moz_db->map_file).version_orig	= 0;
	(moz_db->map_file).err_map 	= 0;
	(moz_db->map_file).err_block 	= 0;
	(moz_db->map_file).cmr_byteswap	= CMR_BYTESWAP_UNKNOWN;

	moz_db = moz_open_db(*dbname, NULL, O_RDONLY, TRUE);

	if (moz_db == NULL)
	{
		return (GSList *) moz_db;
	}

	*dbname = g_strdup((moz_db->cache_file[MOZ_MAP_FILE]).name);

	nscache_db.version = g_strdup_printf("%hu.%hu",
				(unsigned short) ((moz_db->map_file).version_swap >> 16),
				(unsigned short) ((moz_db->map_file).version_swap & 0x0000FFFF));

	retv		= NULL;
	buckets		= kBuckets;
	bucket_size	= (moz_db->map_file).recs_per_bucket * sizeof(MozCacheMapRecord);

	do
	{
		int i, size;

		size = read(moz_db->cache_file[MOZ_MAP_FILE].fd,
					&u.record_array, bucket_size);

		if (size != bucket_size)
		{
			(moz_db->map_file).err_map = 1;
			break;
		}

		u.record_array[(moz_db->map_file).recs_per_bucket].hashnr = 0;

		for (i = 0; u.record_array[i].hashnr != 0; i++)
		{
			moz_record_t *nr;

			nr = moz_get_entry_info(moz_db, &u.record_array[i]);

			if (nr)
			{
				retv = g_slist_prepend(retv, nr);
			}
		}
	}
	while (--buckets > 0);

	moz_close_db();

	len = 0;

	if ((moz_db->map_file).dirty)
	{
		len = g_snprintf(u.pom, sizeof(u.pom), "%s",
			gettext(moz_err_cache_in_use));
	}

	if ((moz_db->map_file).err_block && len >= 0)
	{
		len += g_snprintf(&u.pom[len], sizeof(u.pom) - len, "%s%s (%d)",
			(len > 0 ? " - " : ""),
			gettext("Corrupted blockfile"),
			(moz_db->map_file).err_block);
	}

	if ((moz_db->map_file).err_map && len >= 0)
	{
		len += g_snprintf(&u.pom[len], sizeof(u.pom) - len, "%s%s",
			(len > 0 ? " - " : ""),
			gettext("Corrupted cache map file"));
	}

	gui_err((len > 0) ? u.pom : NULL);

	return retv;

} /* static GSList *moz_read_db(const char *dbname) */

static int moz_read_bucket(MozCacheFile *moz_file, int mode, int errmesg,
				guint32 recordid, MozCacheMapRecord *record_array)
{
	int		bucket, bucket_size, offset, rv;
	MozDatabase	*moz_db;

	moz_db = moz_open_db(nscache_db.name, moz_file, mode, errmesg);

	if (moz_db == NULL)
	{
		return NSC_ERR_SILENT;
	}

	bucket_size	= (moz_db->map_file).recs_per_bucket * sizeof(MozCacheMapRecord);
	bucket		= GetBucketIndex(recordid);

	rv	= NSC_ERR_HASHNR;
	offset  = lseek((moz_db->cache_file[MOZ_MAP_FILE]).fd, 0, SEEK_CUR);
	offset += bucket * bucket_size;

	if (bucket == 0 || lseek((moz_db->cache_file[MOZ_MAP_FILE]).fd,
						offset, SEEK_SET) == offset)
	{
		int sz = read((moz_db->cache_file[MOZ_MAP_FILE]).fd,
							record_array, bucket_size);
		if (sz == bucket_size)
		{
			int i;

			if ((moz_db->map_file).cmr_byteswap)
			{
				recordid = (*(moz_db->map_file).cmr_byteswap)(recordid);
			}

			record_array[(moz_db->map_file).recs_per_bucket].hashnr = 0;

			for (i = 0; record_array[i].hashnr != 0; i++)
			{
				if (record_array[i].hashnr == recordid)
				{
					if (mode == O_RDONLY ||
						lseek((moz_db->cache_file[MOZ_MAP_FILE]).fd,
								offset, SEEK_SET) == offset)
					{
						rv = i;
					}

					break;
				}
			}

			if (mode == O_RDONLY || rv < 0)
			{
				moz_close_db();
			}

		} /* if (sz == bucket_size) */

	} /* if (bucket == 0 || lseek((moz_db->cache_file[MOZ_MAP_FILE]).fd,
							offset, SEEK_SET) == offset) */

	return rv;

} /* static int moz_read_bucket(MozCacheFile*, int, int, guint32, MozCacheMapRecord*) */

#ifndef CACHE_READONLY

static gint32 moz_cache_free_blocks(MozDatabase *moz_db, MozCacheMapRecord *cmr, int type)
{
	unsigned int	index, offset;
	guint32		blocknr, block_count, location, mask;
	MozCacheFile	*moz_file;

	union
	{
		guint32	block_alloc;
		char	block_data[MOZ_BITMAP_SIZE];
	} u;


	location = type == 'm' ? cmr->meta_loc : cmr->data_loc;

	if ( ! MOZ_LOC_INITED(location) )
	{
		return NSC_ERR_ENTRY;
	}


	index = MOZ_LOC_FILE_INDEX(location);

	if (index == 0)
	{
		/* remove meta or data file */

		guint32	rv;
		char	fname[MOZ_FILENAME_LENGTH];

		moz_generate_filename(cmr, type, fname);

		if (file_remove(fname) != 0)
		{
			return type == 'm' ? NSC_ERR_ENTRY : NSC_ERR_FILE;
		}

		rv = MOZ_LOC_FILE_SIZE(location);

		/* KB or pages of 256 Bytes*/
		return ((moz_db->map_file).version_swap < 0x0001000B) ?
							(rv << 2) : (rv >> 8);
	}


	/* free meta or data blocks */

	moz_file	= &moz_db->cache_file[index];
	blocknr		= MOZ_LOC_BLOCK_NUMBER(location);
	block_count	= MOZ_LOC_BLOCK_COUNT(location) + 1;
	offset		= (blocknr / 8) & ~(0x00000003);

	if ( (lseek(moz_file->fd, offset, SEEK_SET) != offset) ||
		(read(moz_file->fd, &u.block_alloc, sizeof(guint32)) != sizeof(guint32)) ||
						(lseek(moz_file->fd, offset, SEEK_SET) != offset) )
	{
		return NSC_ERR_ENTRY;
	}

	mask   = (1 << block_count) - 1;
	mask <<= blocknr % 32;

	if ((moz_db->map_file).version_swap < 0x0001000B)
	{
		/* 8-bit => reverse byte order */
		mask = UINT32_TO_LITTLE_ENDIAN(mask);
	}

	else
	{
		/* 32bit => big endian */
		mask = UINT32_TO_BIG_ENDIAN(mask);
	}

	if (((u.block_alloc ^= mask) & mask) == 0 &&
			write(moz_file->fd, &u.block_alloc, sizeof(guint32)) == sizeof(guint32))
	{
		/* clear blocks */

		gint32 size;

		size   = moz_file->block_size * block_count;
		offset = MOZ_GET_BLOCK_OFFSET(moz_file->block_size, blocknr);

		if (lseek(moz_file->fd, offset, SEEK_SET) == offset)
		{
			gint32 sz, len;

			sz = size;

			if (size > MOZ_BITMAP_SIZE)
			{
				sz = MOZ_BITMAP_SIZE;
			}

			memset(u.block_data, '\0', sz);
			len = 0;

			do
			{
				if (write(moz_file->fd, u.block_data, sz) != sz)
				{
					break;
				}

				len += sz;
			}
			while (len < size);
		}

		return ((moz_db->map_file).version_swap < 0x0001000B) ? size : size >> 8;
	}

	return NSC_ERR_ENTRY;

} /* static gint32 moz_cache_free_blocks(MozDatabase*, MozCacheMapRecord*, int) */

static int moz_record_delete(nscache_record_t *nsc_rec)
{
	int		i, bucket, moz_map_fd, size;
	gint32		rv, nentries, total_size;
	guint32		erank, *bucket_erank;
	MozDatabase	*moz_db;
	ByteSwap32Func	swap;

	MozCacheMapRecord *cmr;

	union
	{
		MozCacheMapHeader	header;
		MozCacheMapHeader_1v6	header_1v6;
		MozCacheMapRecord	record_array[kRecordArraySize];
	} u;

	moz_record_t *moz_rec = (moz_record_t *) nsc_rec;

	i = moz_read_bucket(NULL, O_RDWR, TRUE, moz_rec->recordid, u.record_array);

	if (i < 0)
	{
		return i;
	}

	rv = NSC_ERR_ENTRY;

	cmr = &u.record_array[i];

	moz_db = &moz_database;

	if ((moz_db->map_file).cmr_byteswap)
	{
		moz_map_record_byteswap((moz_db->map_file).cmr_byteswap, cmr);
	}

	/* first free data blocks */

	rv = moz_cache_free_blocks(moz_db, cmr, 'd');

	if (rv < 0)
	{
		goto _close_db;
	}

	total_size = rv;

	/* then free meta blocks */

	rv = moz_cache_free_blocks(moz_db, cmr, 'm');

	if (rv < 0)
	{
		goto _close_db;
	}

	total_size += rv;

	/* find last record */

	do
	{
		i++;
	}
	while (u.record_array[i].hashnr);

	/* fill the gap with the last record */

	i--;

	if (cmr != &u.record_array[i])
	{
		*cmr =  u.record_array[i];
		 cmr = &u.record_array[i];
	}

	/* clear last record */

	cmr->hashnr 	= 0;
	cmr->erank	= 0;
	cmr->data_loc	= 0;
	cmr->meta_loc	= 0;

	/* update eviction rank */

	erank = 0;

	while (--cmr >= &u.record_array[0])
	{
		guint32 tmp = cmr->erank;

		if ((moz_db->map_file).cmr_byteswap)
		{
			tmp = (*(moz_db->map_file).cmr_byteswap)(tmp);
		}

		if (erank < tmp)
		{
			erank = tmp;
		}
	}

	rv	= NSC_ERR_ENTRY;
	size	= (i + 1) * sizeof(MozCacheMapRecord);

	moz_map_fd = (moz_db->cache_file[MOZ_MAP_FILE]).fd;

	if (write(moz_map_fd, &u.record_array, size) != size)
	{
		goto _close_db;
	}

	if (lseek(moz_map_fd, 0, SEEK_SET) != 0)
	{
		goto _close_db;
	}

	if (read(moz_map_fd, &u.header_1v6,
		sizeof(u.header_1v6)) != sizeof(u.header_1v6))
	{
		goto _close_db;
	}

	if (lseek(moz_map_fd, 0, SEEK_SET) != 0)
	{
		goto _close_db;
	}

	/* update cache size */

	total_size = UINT32_FROM_BIG_ENDIAN(u.header.filehead.size) - total_size;

	if (total_size < 0)
	{
		total_size = 0;
	}

	u.header.filehead.size = UINT32_TO_BIG_ENDIAN(total_size);

	/* update number of entries */

	nentries = UINT32_FROM_BIG_ENDIAN(u.header.filehead.nentries) - 1;

	if (nentries < 0)
	{
		nentries = 0;
	}

	u.header.filehead.nentries = UINT32_TO_BIG_ENDIAN(nentries);

	bucket = GetBucketIndex(moz_rec->recordid);
	swap   = NULL;

	if ((moz_db->map_file).version_swap < 0x00010006)
	{
		bucket_erank	= &u.header.erank[bucket];
		swap		= (moz_db->map_file).cmr_byteswap;
	}

	else
	{

		/* update number of records in bucket */

		guint32 bcount = u.header_1v6.bucket_usage[bucket];

#ifdef CONFIG_PDP_ENDIAN

		/*
			BE or PDP?
			A value readed as '1' could actually be '256'!
			If so, the new number of buckets should be
			greater than one (255) , else zero.
		*/

#define BUCKET_COUNTER_OVERFLOW(oldval, newval) \
		( (oldval) > MOZ_MAX_RECS_PER_BUCKET || \
		((oldval) == (MOZ_MAX_RECS_PER_BUCKET >> 8) && \
		(newval) > (MOZ_MAX_RECS_PER_BUCKET >> 8)) )

		if ((bcount & (guint32) 0x0000FFFF) == 0)
		{

#ifndef UINT32_IS_LITTLE_ENDIAN
			/* LE => 1234 */
			swap = BSWAP32_LITTLE_ENDIAN_FUNC;
#else
			/* BE => 1234 - PDP => 2143 */
			bcount >>= 16;

			if (bcount)
			{
				swap = BUCKET_COUNTER_OVERFLOW(bcount, i) ?
					BSWAP32_BIG_ENDIAN_FUNC : BSWAP32_LE_PDP_FUNC;
			}
#endif
		}

#ifndef UINT32_IS_LITTLE_ENDIAN
		else if (bcount &= (guint32) 0x0000FFFF,
					BUCKET_COUNTER_OVERFLOW(bcount, i))
		{
			/* BE => 4321 (3412) - PDP => 3412 (4321) */
			swap = BSWAP32_BE_PDP_FUNC;
		}
#endif

		u.header_1v6.bucket_usage[bucket] = (swap == NULL) ? i : (*swap)(i);

#else /* ! CONFIG_PDP_ENDIAN */

		if ((bcount & (guint32) 0x000001FF) == 0)
		{
			swap	= BSWAP32_BE_LE_FUNC;
			i	= (*swap)(i);
		}

		u.header_1v6.bucket_usage[bucket] = i;

#endif /* CONFIG_PDP_ENDIAN */

		bucket_erank = &u.header_1v6.erank[bucket];
	}

	*bucket_erank = (swap == NULL) ? erank : (*swap)(erank);

	write(moz_map_fd, &u.header_1v6, sizeof(u.header_1v6));

	rv = 0;

_close_db:
	moz_close_db();

	return (int) rv;

} /* static int moz_record_delete(nscache_record_t *nsc_rec) */

#endif /* ! CACHE_READONLY */

int moz_file_access(const nscache_record_t *nsc_rec)
{
	int			dirty, rv;
	moz_record_t		*moz_rec;
	MozCacheMapRecord	record_array[kRecordArraySize];

	moz_rec	= (moz_record_t *) nsc_rec;
	dirty	= (moz_database.map_file).dirty;
	rv	= moz_read_bucket(&moz_database.cache_file[MOZ_MAP_FILE],
				O_RDONLY, FALSE, moz_rec->recordid, record_array);
	if (rv < 0)
	{
		if (rv != NSC_ERR_HASHNR)
		{
			return rv;
		}

		errno = ENODATA;
	}

	else
	{
		rv = FILE_ACCESS_READ;
	}

	if (dirty == 0 && (moz_database.map_file).dirty)
	{
		gui_err(gettext(moz_err_cache_in_use));
	}

	return rv;

} /* int moz_file_access(const nscache_record_t *nsc_rec) */

static int moz_cache_check(const char *dbname)
{
	if (NSCACHE_RT_MOZILLA != NSCACHE_RT_LAST)
	{
		if (moz_open_db(dbname, NULL, O_RDONLY, FALSE) == NULL)
		{
			return NSC_ERR_SILENT;
		}

		moz_close_db();
	}

	return 0;

} /* static int moz_cache_check(const char *dbname) */

/* EOF */
