/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		modified 2005,2006,2007 by Harald Foerster		   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#ifdef HAVE_MSIE

#include <errno.h>
#include <string.h>
#include <wininet.h>

#include "msie.h"
#include "nls.h"
#include "file.h"

/*
#include "nscache.h"
#include <glib.h>
#include <sys/types.h>
#include <time.h>
#include <windows.h>
#include <winbase.h>
*/


typedef struct
{
	HMODULE wininet_hModule;
	HANDLE	(WINAPI *FindFirstUrlCacheEntry)
				(LPCSTR, LPINTERNET_CACHE_ENTRY_INFOA, PDWORD);
	BOOL	(WINAPI *FindNextUrlCacheEntry)
				(HANDLE, LPINTERNET_CACHE_ENTRY_INFOA, PDWORD);
	BOOL	(WINAPI *FindCloseUrlCache) (HANDLE);
	HANDLE	(WINAPI *RetrieveUrlCacheEntryStream)
				(LPCSTR, LPINTERNET_CACHE_ENTRY_INFOA, PDWORD, BOOL, DWORD);
	BOOL	(WINAPI *ReadUrlCacheEntryStream)
				(HANDLE, DWORD, LPVOID, PDWORD, DWORD);
	BOOL	(WINAPI *UnlockUrlCacheEntryStream) (HANDLE, DWORD);
	BOOL	(WINAPI *GetUrlCacheEntryInfo)
				(LPCSTR, LPINTERNET_CACHE_ENTRY_INFOA, PDWORD);
#ifndef CACHE_READONLY
	BOOL	(WINAPI *DeleteUrlCacheEntry) (LPCSTR);
#endif

} MSIE_WININET;

static MSIE_WININET msie_wininet = { NULL };
static const char msie_wininet_dll_str[] = "wininet.dll";

static GSList*	msie_read_db(char **dbname);
static int	msie_cache_check(const char *dbname);

#ifndef CACHE_READONLY
static int	msie_record_delete(nscache_record_t *);
#endif

NSC_BROWSER msie_browser =
{
	msie_read_db,

#ifndef CACHE_READONLY
	msie_record_delete,
#endif
	msie_cache_check,

	gettext_nop("MSIE")
};



static void msie_set_errno(DWORD error)
{
	int err;

	switch (error)
	{
		case NO_ERROR:
			err = 0;
			break;

		/* No such file or directory */
		case ERROR_FILE_NOT_FOUND:
		case ERROR_PATH_NOT_FOUND:
		case ERROR_DLL_NOT_FOUND:
			err = ENOENT;
			break;

		/* Permission denied */
		case ERROR_ACCESS_DENIED:
			err = EACCES;
			break;

		/* Invalid argument */
		case ERROR_INVALID_PARAMETER:
			err = EINVAL;
			break;

		/* Accessing a corrupted shared lib */
		case ERROR_INVALID_DLL:
			err = ELIBBAD;
			break;

		/* Function not implemented */
		case ERROR_PROC_NOT_FOUND:
			err = ENOSYS;
			break;

		/* Can't access a needed shared lib */
		case ERROR_DLL_INIT_FAILED:
			err = ELIBACC;
			break;

		default:
			err = EPERM;
			break;
	}

	errno = err;

} /* static void msie_set_errno(DWORD error) */

static void msie_close_db(MSIE_WININET *dll)
{
	if (dll->wininet_hModule)
	{
		FreeLibrary(dll->wininet_hModule);

		dll->wininet_hModule = NULL;
	}

} /* static void msie_close_db(MSIE_WININET *dll) */

static MSIE_WININET *msie_open_db(int errmesg)
{
	MSIE_WININET *dll = &msie_wininet;

	if (dll->wininet_hModule)
	{
		return dll;
	}

	dll->wininet_hModule = LoadLibraryA(msie_wininet_dll_str);

	if (dll->wininet_hModule)
	{
		do
		{
			if ( (dll->FindFirstUrlCacheEntry =
				(void *) GetProcAddress(dll->wininet_hModule, 
						"FindFirstUrlCacheEntryA")) == NULL )
			{
				break;
			}

			if ( (dll->FindNextUrlCacheEntry =
				(void *) GetProcAddress(dll->wininet_hModule, 
						"FindNextUrlCacheEntryA")) == NULL )
			{
				break;
			}

			if ( (dll->FindCloseUrlCache =
				(void *) GetProcAddress(dll->wininet_hModule, 
						"FindCloseUrlCache")) == NULL )
			{
				break;
			}

			if ( (dll->RetrieveUrlCacheEntryStream =
				(void *) GetProcAddress(dll->wininet_hModule, 
						"RetrieveUrlCacheEntryStreamA")) == NULL )
			{
				break;
			}

			if ( (dll->ReadUrlCacheEntryStream =
				(void *) GetProcAddress(dll->wininet_hModule, 
						"ReadUrlCacheEntryStream")) == NULL )
			{
				break;
			}

			if ( (dll->UnlockUrlCacheEntryStream =
				(void *) GetProcAddress(dll->wininet_hModule, 
						"UnlockUrlCacheEntryStream")) == NULL )
			{
				break;
			}

			if ( (dll->GetUrlCacheEntryInfo =
				(void *) GetProcAddress(dll->wininet_hModule, 
						"GetUrlCacheEntryInfoA")) == NULL )
			{
				break;
			}

#ifndef CACHE_READONLY
			if ( (dll->DeleteUrlCacheEntry =
				(void *) GetProcAddress(dll->wininet_hModule, 
						"DeleteUrlCacheEntry")) == NULL )
			{
				break;
			}
#endif /* ! CACHE_READONLY */

			return dll;
		}
		while (FALSE);

	} /* if (dll->wininet_hModule) */

	msie_set_errno(GetLastError());
	msie_close_db(dll);

	if (errmesg)
	{
		gui_strerror(gettext_nop("Error opening 'wininet.dll'"));
	}

	return NULL;

} /* static MSIE_WININET *msie_open_db(int) */

static time_t filetime_to_time_t(const FILETIME *ftm)
{
	if (ftm->dwLowDateTime | ftm->dwHighDateTime)
	{
		FILETIME	ltm;
		SYSTEMTIME	stm;

		if (FileTimeToLocalFileTime(ftm, &ltm))
		{
			ftm = &ltm;
		}

		if (FileTimeToSystemTime(ftm, &stm))
		{
			struct tm	utm;

			utm.tm_isdst = -1;
			utm.tm_sec   = stm.wSecond;
			utm.tm_min   = stm.wMinute;
			utm.tm_hour  = stm.wHour;
			utm.tm_mday  = stm.wDay;

			/* utm: 0 .. 11 -- stm: 1 .. 12 */
			utm.tm_mon   = stm.wMonth - 1;

			if (utm.tm_mon < 0)
			{
				utm.tm_mon = 0;
			}

			/* utm: years after 1900 -- stm: 1601 .. 30827 */
			utm.tm_year = stm.wYear - 1900;

			if (utm.tm_year < 0)
			{
				utm.tm_year = 0;
			}

			return mktime(&utm);
		}
	}

	return 0;

} /* static time_t filetime_to_time_t(const FILETIME *ftm) */

static msie_record_t *msie_dbt_to_record(LPINTERNET_CACHE_ENTRY_INFOA info)
{
	char		*str;
	msie_record_t	*msie_rec;

	if (info->lpszLocalFileName == NULL ||
		info->lpszLocalFileName[0] == '\0' ||
			info->lpszSourceUrlName == NULL ||
				strstr(info->lpszSourceUrlName, "://") == NULL)
	{
		return NULL;
	}

	/* initialize to '\0' */
	msie_rec = g_malloc0(sizeof(msie_record_t));

	msie_rec->record.urlstr = g_strdup(info->lpszSourceUrlName);

	str = strrchr(info->lpszLocalFileName, '\\');

	if (str == NULL)
	{
		msie_rec->record.filename = g_strdup(info->lpszLocalFileName);
		msie_rec->pathname = content_misc_into_gslist(na_str); /* "n/a" */
	}

	else
	{
		str++;
		msie_rec->record.filename = g_strdup(str);

		/* always be nul-terminated */
		info->lpszLocalFileName[str - info->lpszLocalFileName] = '\0';
		msie_rec->pathname = content_misc_into_gslist(info->lpszLocalFileName);
	}

	msie_rec->record.last_modified = filetime_to_time_t(&info->LastModifiedTime);
	msie_rec->record.last_accessed = filetime_to_time_t(&info->LastAccessTime);
	msie_rec->record.expires = filetime_to_time_t(&info->ExpireTime);

	msie_rec->record.content_length = info->dwSizeLow;

	/* avoid accessing files directly - use always 'wininet.dll' */
	msie_rec->record.content_offset = MSIE_CACHE;

	/* Content-Type: */
	str = tl_get_mime_value_str(info->lpHeaderInfo, info->dwHeaderInfoSize,
					"content-type:", sizeof("content-type:") - 1);
	msie_rec->record.content_type = content_type_into_glist(str);

#if 0
	/* no entry for "charset" */
	str = tl_get_mime_value_str(info->lpHeaderInfo, info->dwHeaderInfoSize,
						"charset", sizeof("charset") - 1);
	msie_rec->record.charset = content_misc_into_gslist(str);

	/* no entry for "Content-Encoding:" */
	str = tl_get_mime_value_str(info->lpHeaderInfo, info->dwHeaderInfoSize,
				"content-encoding:", sizeof("content-encoding:") - 1);
	msie_rec->record.content_encoding = content_misc_into_gslist(str);

#endif /* 0 */

	return msie_rec;

} /* static msie_record_t *msie_dbt_to_record(info) */

static GSList *msie_read_db(char **dbname)
{
	int				max_size;
	GSList    			*retv;
	DWORD				error;
	DWORD				info_size;
	HANDLE				handle;
	MSIE_WININET			*dll;
	LPINTERNET_CACHE_ENTRY_INFOA	info;

	*dbname = g_strdup(msie_wininet_dll_str);

	dll = msie_open_db(TRUE);

	if (dll == NULL)
	{
		return (GSList *) dll;
	}

	max_size  = MAX_CACHE_ENTRY_INFO_SIZE;
	info	  = g_malloc(max_size);

	/* "cookie:", "visited:"  or NULL ("*.*) */
	while ( (handle = dll->FindFirstUrlCacheEntry(NULL, info,
				(info_size = max_size, &info_size))) == NULL )
	{
		error = GetLastError();

		if (error == ERROR_INSUFFICIENT_BUFFER)
		{
			max_size = info_size;
			info = g_realloc(info, max_size);

			continue;
		}

		g_free(info);
		msie_set_errno(error);

		return NULL;
	}


	error	= 0;
	retv	= NULL;

	do
	{
		msie_record_t *msie_rec = msie_dbt_to_record(info);

		if (msie_rec)
		{
			retv = g_slist_prepend(retv, msie_rec);
		}

		while (dll->FindNextUrlCacheEntry(handle, info,
				(info_size = max_size, &info_size)) == FALSE)
		{
			error = GetLastError();

			if (error == ERROR_INSUFFICIENT_BUFFER)
			{
				error	 = 0;
				max_size = info_size;
				info	 = g_realloc(info, max_size);

				continue;
			}

			break;
		}
	}
	while (error == 0);


	dll->FindCloseUrlCache(handle);

	g_free(info);

	if (error != ERROR_NO_MORE_ITEMS)
	{
		msie_set_errno(error);
		gui_strerror(gettext_nop("Error reading 'wininet.dll'"));
	}

	else
	{
		gui_strerror(NULL);
	}

	return retv;

} /* static GSList *msie_read_db(const char *dummy) */

#ifndef CACHE_READONLY

static int msie_record_delete(nscache_record_t *nsc_rec)
{
	if (msie_wininet.wininet_hModule &&
		msie_wininet.DeleteUrlCacheEntry(nsc_rec->urlstr) == FALSE)
	{
		msie_set_errno(GetLastError());

		return NSC_ERR_ENTRY;
	}

	return 0;

} /* static int msie_record_delete(nscache_record_t *) */

#endif /* ! CACHE_READONLY */

static int msie_cache_check(const char *dummy)
{
	MSIE_WININET *dll = msie_open_db(FALSE);

	if (dll)
	{
		msie_close_db(dll);

		return 0;
	}

	return -1;
}

int msie_file_access(const nscache_record_t *nsc_rec)
{
	DWORD				info_size;
	LPINTERNET_CACHE_ENTRY_INFOA	info;
	char				*mem;
	char				infobuf[MAX_CACHE_ENTRY_INFO_SIZE];

	mem	  = NULL;
	info_size = sizeof(infobuf);
	info	  = (LPINTERNET_CACHE_ENTRY_INFOA) &infobuf;


	while (msie_wininet.GetUrlCacheEntryInfo(nsc_rec->urlstr,
						info, &info_size) == FALSE)
	{
		DWORD error = GetLastError();

		if (error == ERROR_INSUFFICIENT_BUFFER)
		{
			void *ptr = realloc(mem, info_size);

			if (ptr)
			{
				mem  = ptr;
				info = ptr;

				continue;
			}
		}

		if (mem)
		{
			free(mem);
		}

		return -1;
	}

	if (mem)
	{
		free(mem);
	}

#ifndef CACHE_READONLY
	return FILE_ACCESS_READ | FILE_ACCESS_WRITE | FILE_ACCESS_REMOVE;
#else
	return FILE_ACCESS_READ;
#endif

} /* int msie_file_access(const nscache_record_t *nsc_rec) */

void *msie_file_read(const nscache_record_t *nsc_rec, size_t *length)
{
	HANDLE				handle;
	DWORD				error;
	LPINTERNET_CACHE_ENTRY_INFOA	info;
	char				*mem;
	char				infobuf[MAX_CACHE_ENTRY_INFO_SIZE];

	union
	{
		DWORD	info_size;
		DWORD	file_size;
	} u;

	mem	  	= NULL;
	u.info_size	= sizeof(infobuf);
	info	  	= (LPINTERNET_CACHE_ENTRY_INFOA) &infobuf;

	while ( (handle = msie_wininet.RetrieveUrlCacheEntryStream(nsc_rec->urlstr,
						info, &u.info_size, FALSE, 0)) == NULL )
	{
		error = GetLastError();

		if (error == ERROR_INSUFFICIENT_BUFFER)
		{
			void *ptr = realloc(mem, u.info_size);

			if (ptr)
			{
				mem  = ptr;
				info = ptr;

				continue;
			}
		}

		if (mem)
		{
			free(mem);
		}

		msie_set_errno(error);

		return NULL;
	}


	u.file_size = info->dwSizeLow;

	if (mem)
	{
		free(mem);
	}

	mem = malloc(u.file_size + 1);

	if (mem == NULL)
	{
		return mem;
	}


	while (error = 0, msie_wininet.ReadUrlCacheEntryStream(handle,
					0, mem, &u.file_size, 0) == FALSE)
	{
		error = GetLastError();

		if (error == ERROR_INSUFFICIENT_BUFFER)
		{
			void *ptr = realloc(mem, u.file_size + 1);

			if (ptr)
			{
				mem = ptr;

				continue;
			}
		}

		break;
	}


	msie_wininet.UnlockUrlCacheEntryStream(handle, 0);

	if (error)
	{
		free(mem);

		msie_set_errno(error);

		return NULL;
	}

	*length = u.file_size;
	mem[*length] = '\0';

	return mem;

} /* void *msie_file_read(const nscache_record_t *, size_t *) */

#endif /* HAVE_WININET_H */

/* EOF */
