/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		modified 2005 ... 2008 by Harald Foerster		   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "nscache.h"
#include "nls.h"
#include "file.h"

#ifdef CONFIG_BERKELEY_DB
#include "ns.h"
#endif

#include "moz.h"

#ifdef HAVE_MSIE
#include "msie.h"
#endif

/*
#include "nscache.h"
#include <gtk/gtk.h>
#include <time.h>
*/


NSC_BROWSER *nsc_browser[] =
{
#ifdef CONFIG_BERKELEY_DB
	&ns_browser,
	&ns6_browser,
#endif

	&moz_browser,

#ifdef HAVE_MSIE
	&msie_browser,
#endif
};


int	nscache_sort_type;
GList	*mime_types_available	= NULL;
static GSList *misc_content	= NULL;

CacheDB nscache_db = { {0} };

const gchar *nscache_package_str = PACKAGE;
const gchar *nscache_version_str = VERSION;

const gchar yes_str[]	= gettext_nop("yes");
const gchar no_str[]	= gettext_nop("no");
const gchar na_str[]	= gettext_nop("n/a");

const gchar nscache_err_open_index_file[] =
			gettext_nop("Error opening cache index file");

char *content_type_into_glist(const char *content_type)
{
	char  *str;
	GList *glist;

	if (content_type == NULL)
	{
		return (char *) content_type;
	}

	glist = g_list_find_custom(mime_types_available,
					(gpointer) content_type,
					(GCompareFunc) g_strcasecmp);
	if (glist)
	{
		return glist->data;
	}

	/* lower case */
	str = tl_strndup_to_lower(content_type, -1);

	mime_types_available =
		g_list_insert_sorted(mime_types_available,
					str, (GCompareFunc) strcmp);
	return str;
}

char *content_misc_into_gslist(const char *encoding_or_charset)
{
	char   *str;
	GSList *slist;

	if (encoding_or_charset == NULL)
	{
		return (char *) encoding_or_charset;
	}

	slist = g_slist_find_custom(misc_content,
			(gpointer) encoding_or_charset,
				(GCompareFunc) g_strcasecmp);
	if (slist)
	{
		return slist->data;
	}

	str = g_strdup(encoding_or_charset);

	misc_content = g_slist_prepend(misc_content, str);

	return str;
}

void nscache_record_free(nscache_record_t *rec)
{
	if (rec->content_offset == 0)
	{
		g_free(rec->filename);
	}

#ifdef HAVE_MSIE
	else if (rec->content_offset == MSIE_CACHE)
	{
		g_free(rec->filename);
	}
#endif

	g_free(rec->urlstr);

	g_free(rec);
}

#ifndef CACHE_READONLY

int nscache_record_delete(nscache_record_t *rec)
{
	int	cache_acc;
	char	*msg, *src;

	cache_acc = file_access(rec);

	if (cache_acc >= 0)
	{
		if (cache_acc & FILE_ACCESS_REMOVE)
		{
			int		err;
			unsigned int	db_type;

			db_type = nscache_db.type;

			err = db_type <= NSCACHE_RT_LAST ?
					nsc_browser[db_type]->record_delete(rec) :
									NSC_ERR_ENTRY;
			switch (err)
			{
				case 0:
					nscache_record_free(rec);
					return 0;

				case NSC_ERR_ENTRY:
					msg = gettext_nop("Unable to remove cache index entry");
					src = rec->urlstr;
					break;

				case NSC_ERR_FILE:
					msg = gettext_nop("Error removing cache file");
					src = rec->filename;
					break;

				case NSC_ERR_HASHNR:
					goto _print_err_hashnr;

				default: /* NSC_ERR_SILENT */
					return -1;
			}

		} /* if (cache_acc & FILE_ACCESS_REMOVE) */

		else
		{
			msg = gettext_nop("read only file");
			src = rec->filename;
		}

	} /* if (cache_acc >= 0) */

	else if(rec->content_offset && cache_acc == NSC_ERR_HASHNR)
	{
_print_err_hashnr:
		msg = gettext_nop("Cannot find cache index entry");
		src = rec->urlstr;
	}

	else
	{
		gui_strerror(rec->filename);
		return -1;
	}

	gui_errfmt("%s - %s", gettext(msg), src);

	return -1;

} /* int nscache_record_delete(nscache_record_t *rec) */

#endif /* CACHE_READONLY */

static unsigned int nscache_guess_type(const char *dbname)
{
	unsigned int db_type = NSCACHE_RT_FIRST;

	do
	{
		if (nsc_browser[db_type]->cache_check(dbname) == 0)
		{
			return db_type;
		}
	}
	while (++db_type <= NSCACHE_RT_LAST);

	return NSCACHE_RT_MOZILLA;
}

static void nscache_path_append_slash(StringBuf *path, const char *fname)
{
	size_t last_chr = path->len - 1;

	if (fname == NULL || fname[last_chr] != '/')
	{
		stringbuf_character_append(path, '/');
	}
}

static void nscache_path_remove_dot_dir(StringBuf *path)
{
	char *str;

	str = path->str;

	while ((str = strstr(str, "//")) != NULL)
	{
		str = stringbuf_move(path, str - path->str, -1);
	}

	str = path->str;

	while ((str = strstr(str, "/./")) != NULL)
	{
		str = stringbuf_move(path, str - path->str, -2);
	}

	str = path->str;

	while ((str = strstr(str, "/../")) != NULL)
	{
		char *slash;

		*str = '\0';
		slash = strrchr(path->str, '/');

		if (slash == NULL)
		{
			slash = str;
		}

		*str = '/';
		str += 3;

		str = stringbuf_move(path, slash - path->str, -(str - slash));
	}

} /* static void nscache_path_remove_dot_dir(StringBuf *path) */
	
GSList *nscache_read_cache(const char *dbname, gboolean slist)
{
	unsigned int	db_type;
	GSList		*retv;
	GList		*mime_types;
	CacheDB		*cache_db;

	g_list_foreach(mime_types_available, (GFunc) g_free, NULL);
	g_list_free(mime_types_available);

	g_slist_foreach(misc_content, (GFunc) g_free, NULL);
	g_slist_free(misc_content);

	cache_db = &nscache_db;

	if(cache_db->path.str == NULL)
	{
		if (stringbuf_new(&(cache_db->path), 0) == NULL)
		{
			return NULL;
		}
	}

	stringbuf_truncate(&(cache_db->path), 0);
	g_free(cache_db->name);
	g_free(cache_db->version);

	cache_db->name		= NULL;
	cache_db->version	= NULL;
	cache_db->access_time	= 0;

	db_type = cache_db->type;

#ifdef HAVE_MSIE
	if (db_type != NSCACHE_RT_MSIE)
#endif
	{
		char		*fname;
		struct stat	dbstat;

		fname = NULL;

		if (dbname == NULL || *dbname != '/')
		{
			if (dbname && dbname[0] == '~' &&
				(dbname[1] == '\0' || dbname[1] == '/'))
			{
				/* skip '~' */
				if (*(++dbname) == '/')
				{
					/* skip '/' */
					dbname++;
				}

				fname = (char *) g_get_home_dir();
			}

			else
			{
				fname = (char *) g_get_current_dir();
			}

			if (fname)
			{
				fname = stringbuf_copy(&(cache_db->path), fname, 0);

				if (fname == NULL)
				{
					return (GSList *) fname;
				}

				nscache_path_append_slash(&(cache_db->path), fname);
			}

		} /* if (dbname == NULL || *dbname != '/') */

		fname = stringbuf_append(&(cache_db->path), dbname, 0);

		if (fname == NULL)
		{
			return (GSList *) fname;
		}

		if ( stat(fname, &dbstat) == 0 &&
					(S_ISDIR(dbstat.st_mode)) )
		{
			nscache_path_append_slash(&(cache_db->path), fname);

			fname = NULL;
		}

		if (fname)
		{
			fname = strrchr(fname, '/');

			if (fname == NULL)
			{
				nscache_path_append_slash(&(cache_db->path), fname);
			}

			else if (*(++fname) != '\0')
			{
				size_t plen;

				cache_db->name = g_strdup(fname);
				plen = fname - (cache_db->path).str;

				stringbuf_truncate(&(cache_db->path), plen);
			}
		}

		nscache_path_remove_dot_dir(&(cache_db->path));

		if (db_type > NSCACHE_RT_LAST)
		{
			db_type = nscache_guess_type(cache_db->name);
#ifdef HAVE_MSIE
			if (db_type == NSCACHE_RT_MSIE)
			{
				stringbuf_truncate(&(cache_db->path), 0);
				g_free(cache_db->name);
				cache_db->name = NULL;
			}
#endif
		}

	} /* if (db_type != NSCACHE_RT_MSIE) */

	cache_db->access_time = time(NULL);
	cache_db->type = db_type;

	mime_types_available = NULL;
	misc_content = NULL;

	if (slist == FALSE)
	{
		return NULL;
	}

	retv = nsc_browser[db_type]->read_db(&(cache_db->name));

	mime_types = mime_types_available;
	mime_types = g_list_prepend(mime_types, g_strdup("text/*"));
	mime_types = g_list_prepend(mime_types, g_strdup("image/*"));
	mime_types = g_list_prepend(mime_types, g_strdup("application/*"));
	mime_types = g_list_prepend(mime_types, g_strdup("*"));
	mime_types = g_list_prepend(mime_types, g_strdup(""));

	mime_types_available = mime_types;

	return retv;

} /* GSList *nscache_read_cache(const char *dbname, gboolean) */

static gint nscache_typecount_find_func(const nscache_typecount_t *tcount,
								const gchar *str)
{
	return strcmp(tcount->str, str);
}

static gint nscache_typecount_sort_func(const nscache_typecount_t *tcount,
						const nscache_typecount_t *data)
{
	return strcmp(tcount->str, data->str);
}

static GSList *nscache_content_type_func(GSList *ctype, const gchar *str)
{
	GSList *list;

	if (str == NULL)
	{
		str = na_str;
	}

	list = g_slist_find_custom(ctype, (gchar *) str,
				(GCompareFunc) nscache_typecount_find_func);
	if (list == NULL)
	{
		nscache_typecount_t *entry;

		list = g_slist_last(ctype);

		if (list)
		{
			if (((nscache_typecount_t *) list->data)->str == na_str)
			{
				ctype = g_slist_remove_link(ctype, list);
			}

			else
			{
				list = NULL;
			}
		}

		entry = g_malloc(sizeof(nscache_typecount_t));

		entry->str = str;
		entry->num = 1;

		if (list == NULL && str == na_str)
		{
			return g_slist_append(ctype, entry);
		}

		ctype = g_slist_insert_sorted(ctype, entry,
				(GCompareFunc) nscache_typecount_sort_func);
		if (list)
		{
			ctype = g_slist_concat(ctype, list);
		}

		return ctype;
	}

	((nscache_typecount_t *) list->data)->num++;

	return ctype;

} /* static GSList *nscache_content_type_func(GSList*, const char*) */

void nscache_content_types_add(nscache_record_t *rec, nscache_content_types_t **content_types)
{
	nscache_content_types_t *ctypes;

	if (rec == NULL || content_types == NULL)
	{
		return;
	}

	ctypes = *content_types;

	if (ctypes == NULL)
	{
		ctypes = g_malloc0(sizeof(nscache_content_types_t));
		*content_types = ctypes;
	}

	ctypes->content_type =
		nscache_content_type_func(ctypes->content_type, rec->content_type);

	ctypes->content_encoding =
		nscache_content_type_func(ctypes->content_encoding, rec->content_encoding);

	ctypes->charset =
		nscache_content_type_func(ctypes->charset, rec->charset);

} /* void nscache_content_types_add(nscache_record_t*, nscache_content_types_t**) */

static void nscache_content_type_free(GSList *slist)
{
	g_slist_foreach(slist, (GFunc) g_free, NULL);
	g_slist_free(slist);
}

void nscache_content_types_free(nscache_content_types_t *ctypes)
{
	if (ctypes)
	{
		nscache_content_type_free(ctypes->content_type);
		nscache_content_type_free(ctypes->content_encoding);
		nscache_content_type_free(ctypes->charset);

		g_free(ctypes);
	}

} /* void nscache_content_types_free(nscache_content_types_t*) */

static void print_summary(GSList *slist)
{
	int entries = 0;
	int size = 0;
	const char *str;

	while (slist)
	{
		nscache_record_t *rec = slist->data;

		entries++;
		size += rec->content_length;

		slist = slist->next;
	}

	if (size > 1024)
	{
		size /= 1024;
		str = gettext("kB");
	}
	else
	{
		str = gettext("Bytes");
	}

	printf("\t%s: %d\n\t%s: %d %s\n",
		gettext("Number of entries"),
		entries,
		gettext("Total size"),
		size,
		str);

} /* static void print_summary(GSList *ptr) */

static void print_content_type(nscache_typecount_t *ctype, gpointer data)
{
	printf("%s: %u\n", ctype->str, ctype->num);
}

static void print_short_list(nscache_record_t *rec, gpointer data)
{
	printf("%s\t%s\n", rec->filename, rec->urlstr);
}

static void print_multiline(nscache_record_t *rec, const gchar *noav_str)
{
	char mdtm[30], atm[30], exptm[30];

	printf("%s: %s\n", gettext("File"), rec->filename);

#ifdef HAVE_MSIE
	if (nscache_db.type == NSCACHE_RT_MSIE)
	{
		printf("%s: %s\n", gettext("Path"), MSIE_GET_PATHNAME(rec));
	}
#endif

	printf("\t%s: %s\n"
		"\t%s: %s\n"
		"\t%s: %d\n"
		"\t%s: %s\n"
		"\t%s: %s\n"
		"\t%s: %s\n"
		"\t%s: %s\n"
		"\t%s: %s\n",
		gettext("URL"), rec->urlstr,
		gettext("MIME type"), (rec->content_type ?
					rec->content_type : noav_str),
		gettext("Size"), rec->content_length,
		gettext("Encoding"), (rec->content_encoding ?
					rec->content_encoding : noav_str),
		gettext("Charset"), (rec->charset ?
						rec->charset : noav_str),
		gettext("Modification time"),
			(time_to_string(mdtm, sizeof(mdtm), &(rec->last_modified)) ?
									mdtm : noav_str),
		gettext("Access time"),
			(time_to_string(atm, sizeof(atm), &(rec->last_accessed)) ?
									atm : noav_str),
		gettext("Expiration time"),
			(time_to_string(exptm, sizeof(exptm), &(rec->expires)) ?
									exptm : noav_str));

} /* static void print_multiline(nscache_record_t *rec, const gchar *noav_str) */

static gint domain_len(const gchar *str)
{
	const gchar *end = strstr(str, "://");

	if (end)
	{
		end = strchr(&end[3], '/');
	}

	return (end == NULL) ? strlen(str) : ++end - str;
}

gint nscache_compare_func(const nscache_record_t *rec1, const nscache_record_t *rec2)
{
	gint rv = 0;

	switch (nscache_sort_type & ~STYPE_REVERS_MASK)
	{
		case STYPE_FILE:
			if (rec1->filename && rec2->filename)
			{
				rv = strcmp(rec1->filename, rec2->filename);
			}

			break;

		case STYPE_SIZE:
			rv = rec1->content_length - rec2->content_length;
			break;

		case STYPE_ATM:
			rv = rec1->last_accessed - rec2->last_accessed;
			break;

		case STYPE_MIME:
			if (rec1->content_type != rec2->content_type)
			{
				if (rec1->content_type && rec2->content_type)
				{
					rv = strcmp(rec1->content_type, rec2->content_type);
				}

				else
				{
					rv = (rec1->content_type == NULL) ? 1 : -1;
				}
			}

			break;

		default: /* case STYPE_URL: */
			if (nscache_sort_type & STYPE_REVERS_MASK)
			{
				gint len1, len2;

				len1 = domain_len(rec1->urlstr);
				len2 = domain_len(rec2->urlstr);

				if (len2 > len1)
				{
					len2 = len1;
				}

				rv = strncmp(rec1->urlstr, rec2->urlstr, len2);
			}

			break;

	} /* switch (nscache_sort_type & ~STYPE_REVERS_MASK) */

	if (rv == 0)
	{
		return strcmp(rec1->urlstr, rec2->urlstr);
	}

	return (nscache_sort_type & STYPE_REVERS_MASK) ? -rv : rv;

} /* gint nscache_compare_func(const nscache_record_t*, const nscache_record_t*) */

int nscache_dump(const char *dbname, int what, int sort_type)
{
	const char	*path;
	GSList		*slist;

	nscache_sort_type = sort_type;
	slist  = nscache_read_cache(dbname, TRUE);
	dbname = nscache_db.name;

	if (dbname == NULL)
	{
		return errno == 0 ? EINVAL : errno;
	}

	path = nscache_db.path.str;

	if (path == NULL)
	{
		path = "";
	}

	printf("%s%s\n", path, dbname);

	if (what == kOnlyStdOutSummary)
	{
		print_summary(slist);
	}

	else if (what == kOnlyStdOutConentTypes)
	{
		static gchar fmt_str[] = "\n%s\n";

		nscache_content_types_t *content_types = NULL;

		g_slist_foreach(slist,
			(GFunc) nscache_content_types_add,
						&content_types);
		if (content_types)
		{
			printf(fmt_str, gettext("MIME type"));
			g_slist_foreach(content_types->content_type,
					(GFunc) print_content_type, NULL);

			printf(fmt_str, gettext("Encoding"));
			g_slist_foreach(content_types->content_encoding,
					(GFunc) print_content_type, NULL);

			printf(fmt_str, gettext("Charset"));
			g_slist_foreach(content_types->charset,
					(GFunc) print_content_type, NULL);
		}
	}

	else
	{
		slist = g_slist_sort(slist, (GCompareFunc) nscache_compare_func);

		if (what == kOnlyStdOutMultiLine)
		{
			g_slist_foreach(slist,
				(GFunc) print_multiline,
					(gpointer) gettext(na_str));
		}

		else /* if (what == kOnlyStdOutList) */
		{
			g_slist_foreach(slist, (GFunc) print_short_list, NULL);
		}
	}

	return 0;

} /* int nscache_dump(const char *dbname, int what) */

/* EOF */
