/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		modified 2005,2008 by Harald Foerster			   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#ifdef HAVE_REGEX

#include <glib.h>
#include <string.h>

#include "re.h"

void re_free(re_entry *ree)
{

#if defined(HAVE_POSIX_REGEX)

	regfree(&(ree->preg));

#elif defined(HAVE_PCRE_REGEX)

	if (ree->preg_extra) free(ree->preg_extra);
	if (ree->preg) free(ree->preg);

#elif defined(HAVE_GNU_REGEX)

	regfree(&(ree->preg));

#elif defined(HAVE_V8_REGEX)

	if (ree->preg) free(ree->preg);

#elif defined(HAVE_BSD_REGEX)

	g_free(ree->pattern);
#endif

	g_free(ree);

} /* void re_free(re_entry*) */

re_entry *re_make(const char *pattern, int option, char **error)
{

#if defined(HAVE_PCRE_REGEX)
	static const unsigned char *pcretable = NULL;
#endif

#if defined(HAVE_PCRE_REGEX) || defined(HAVE_POSIX_REGEX)
	int  ec;
#endif

#if  defined(HAVE_PCRE_REGEX) || defined(HAVE_GNU_REGEX) || defined(HAVE_BSD_REGEX)
	const char *errmsg;
#endif

	/* initialize to '\0' */
	re_entry *ree = g_malloc0(sizeof(re_entry));

#if defined(HAVE_POSIX_REGEX)

	if ((ec = regcomp(&(ree->preg), pattern, option)))
	{
		char errmsg[512];

		regerror(ec, &(ree->preg), errmsg, sizeof(errmsg));

#elif defined(HAVE_PCRE_REGEX)

	if (pcretable == NULL)
	{
		pcretable = pcre_maketables();
	}

	if ((ree->preg = pcre_compile(pattern, option, &errmsg, &ec, pcretable)))
	{
		ree->preg_extra = pcre_study(ree->preg, 0, &errmsg);
	}

	else
	{

#elif defined(HAVE_GNU_REGEX)

#ifdef HAVE_GNU_REGEX_SYNTAX_OPTION
	re_syntax_options = HAVE_GNU_REGEX_SYNTAX_OPTION;
#endif

	errmsg = re_compile_pattern(pattern, strlen(pattern), &(ree->preg));

	if (errmsg == NULL)
	{
		ree->preg.fastmap = malloc(256);
		re_compile_fastmap(&(ree->preg));
		re_set_registers(&(ree->preg), &(ree->regs), 0, NULL, NULL);
	}

	else
	{

#elif defined(HAVE_V8_REGEX)

	if (!(ree->preg = regcomp(pattern)))
	{

#elif defined(HAVE_BSD_REGEX)

	ree->pattern = g_strdup(pattern);

#ifdef HAVE_BSD_REGEX_SYNTAX_OPTION
	re_syntax_options = HAVE_BSD_REGEX_SYNTAX_OPTION;
#endif

	if ((errmsg = re_comp(ree->pattern)))
	{

#endif /* #if defined(HAVE_PCRE_REGEX) .. #elif */

		if (error)
		{
#ifdef HAVE_V8_REGEX
			*error = NULL;
#else
			*error = g_strdup(errmsg);
#endif
		}

		re_free(ree);

		return NULL;
	}

	return ree;

} /* re_entry *re_make(const char*, int, char**) */

int re_pmatch(re_entry *ree, const char *str, int slen, int *end_offs)
{
	int beg_offs;

	if (ree->str_lower_func)
	{
		str = (*ree->str_lower_func)(str, slen);
	}

#if defined(HAVE_POSIX_REGEX)

{
	regmatch_t offset;

	/* regexec returns 0 for success */

	beg_offs = regexec(&(ree->preg), str, 1, &offset, 0);

	if (beg_offs != 0)
	{
		beg_offs = -1;
	}

	else
	{
		*end_offs = offset.rm_eo;
		 beg_offs = offset.rm_so;
	}
}

#elif defined(HAVE_PCRE_REGEX)

{
	int offset[6];

	/* If  pcre_exec fails, it returns a negative number */

	beg_offs = pcre_exec(ree->preg, ree->preg_extra, str, slen, 0, 0, offset, 3);

	if (beg_offs >= 0)
	{
		*end_offs = offset[1];
		 beg_offs = offset[0];
	}
}

#elif defined(HAVE_GNU_REGEX)

	/* re_search returns the index where the match began */

	beg_offs = re_search(&(ree->preg), str, slen, 0, slen, &(ree->regs));

	if (beg_offs >= 0)
	{
		*end_offs = ree->regs.end[0];
	}

#elif defined(HAVE_V8_REGEX)

	/* regexec returns either 1 for success or 0 for failure */

	beg_offs = regexec(ree->preg, str) - 1;

	if (beg_offs == 0)
	{
		*end_offs = ree->preg->endp[0] - str;
		 beg_offs = ree->preg->startp[0] - str;
	}

#elif defined(HAVE_BSD_REGEX)

	*end_offs = -1;

	re_comp(ree->pattern);

	/* re_exec returns either 1 for success or 0 for failure */

	beg_offs = re_exec(str) - 1;

#endif /* #if defined(HAVE_POSIX_REGEX) .. #elif */

	if (ree->str_lower_func)
	{
		g_free((char *) str);
	}

	return beg_offs;

} /* int re_pmatch(re_entry*, const char*, int, int*) */

#endif /* HAVE_REGEX */

/* EOF */
