/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		modified 2005 ... 2008 by Harald Foerster		   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include <ctype.h>
#include <string.h>
#include <unistd.h>

#include "charconv.h"
#include "file.h"
#include "gprop.h"
#include "gui.h"
#include "utf8-tools.h"

#if GTK_MAJOR_VERSION == 1
#include "fonts.h"
#endif

#ifdef HAVE_REGEX
#include "re.h"
#endif

#ifdef HAVE_FNMATCH
#include <fnmatch.h>
#else
#include "fnmatch.h"
#endif

/*
#include <gtk/gtk.h>
#include "gui-gtk.h"
#include "nls.h"
#include "nscache.h"
*/


enum
{
	kColumnURL,
	kColumnFound,
	kColumnFile,
	kColumnLine,
#ifndef CONFIG_GTKCLIST_GTKCTREE
	kColumnPixbuf,
	kColumnRowHeight,
	kColumnRowData,
#endif
	kNumberOfColumns
};

/* 'x-windows-949' used by Mozilla */
#define XWindows949ToCP949(charset) \
	( (g_strcasecmp((charset), "x-windows-949") == 0) ? \
						"CP949" : (charset) )

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define SearchListNew(list) \
	GuiListInit((list), kNumberOfColumns)
#else
#define SearchListNew(list) \
{ \
	(list)->slist = \
		gtk_list_store_new (kNumberOfColumns, \
					G_TYPE_STRING, G_TYPE_STRING, \
					G_TYPE_STRING, G_TYPE_STRING, \
					GDK_TYPE_PIXBUF, \
					G_TYPE_INT, G_TYPE_POINTER); \
	GuiListInit((list), kColumnRowData); \
}
#endif

#if GTK_MAJOR_VERSION == 1
#ifdef HAVE_ICONV
#ifdef HAVE_UNICODE
#define ReplaceNonAsciiCharsInString(is_utf8, str) \
{ \
	if (is_utf8) \
		utf8tl_replace_non_ascii_chars_in_str(str); \
	else \
		tl_replace_non_ascii_chars_in_str(str); \
}
#else
#define ReplaceNonAsciiCharsInString(is_utf8, str) \
	tl_replace_non_ascii_chars_in_str(str)
#endif
#define Search_CList_Append(list, str, is_utf8, data, row) \
{ \
	const gchar *from, *to; \
	GtkStyle *style; \
	from = ((nscache_record_t *) (data))->charset; \
	style = fonts_get_style((list)->clist, from); \
	if (style == NULL || (from && (to = fonts_convert_charset(from)))) \
	{ \
		gchar *s; \
		if (style == NULL) \
		{ \
			to = charconv_locale_str; \
			from = XWindows949ToCP949(from); \
		} \
		s = charconv((str)[kColumnFound], to, from, NULL); \
		if (s == NULL) \
		{ \
			style = NULL; \
			ReplaceNonAsciiCharsInString((is_utf8), (str)[kColumnFound]); \
		} \
		else \
		{ \
			(str)[kColumnFound] = s; \
		} \
	} \
	(row) = gtk_clist_append((list)->clist, (str)); \
	if (style) \
	{ \
		gtk_clist_set_cell_style((list)->clist, \
					(row), kColumnFound, style); \
	} \
}
#else /* ! HAVE_ICONV */
#define Search_CList_Append(list, str, is_utf8, data, row) \
{ \
	GtkStyle *style = fonts_get_style((list)->clist, \
				((nscache_record_t *) (data))->charset); \
	if (style == NULL) \
	{ \
		tl_replace_non_ascii_chars_in_str((str)[kColumnFound]); \
	} \
	(row) = gtk_clist_append((list)->clist, (str)); \
	if (style) \
	{ \
		gtk_clist_set_cell_style((list)->clist, \
					(row), kColumnFound, style); \
	} \
}
#endif
#else /* GTK_MAJOR_VERSION > 1 */
#define Search_CList_Append(list, str, is_utf8, data, row) \
	(row) = gtk_clist_append((list)->clist, (str))
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
#define SearchListAppend(list, str, is_utf8, icon, data) \
{ \
	gint freeze_count, row; \
	freeze_count = (list)->clist->freeze_count; \
	if (freeze_count == 0) \
	{ \
		gtk_clist_freeze((list)->clist); \
	} \
	Search_CList_Append((list), (str), (is_utf8), (data), (row)); \
	gtk_clist_set_pixtext((list)->clist, row, kColumnURL, (str)[0], \
					3, (icon)->pixmap, (icon)->mask); \
	gtk_clist_set_row_data((list)->clist, row, (data)); \
	if (freeze_count == 0 &&  \
		gtk_clist_row_is_visible((list)->clist, row) != GTK_VISIBILITY_NONE) \
	{ \
		gtk_clist_thaw((list)->clist); \
	} \
}
#else
#define SearchListAppend(list, str, is_utf8, icon, data) \
{ \
	GtkTreeIter iter; \
	gtk_list_store_append((list)->slist, &iter); \
	gtk_list_store_set((list)->slist, &iter, \
		kColumnURL, (str)[0], kColumnFound, (str)[1], \
		kColumnFile, (str)[2], kColumnLine, (str)[3], \
		kColumnPixbuf, (icon)->pixbuf, \
		kColumnRowHeight, 0, kColumnRowData, (data), -1); \
}
#endif


typedef gchar* (*StrStrFunc)    (const gchar*, gint, const gchar*, gint);
typedef gint   (*IsAlNumFunc)   (const gchar*, const gchar*, gint);
typedef gchar* (*LineBreakFunc) (const gchar*, const gchar*);

typedef struct
{
	gint		is_utf8;
	IsAlNumFunc	alnumfunc;
	LineBreakFunc	startfunc;
	LineBreakFunc	stopfunc;

} FuncPtr;

static const FuncPtr tl_funcs =
{
	FALSE,
	tl_is_alnum_str,
	tl_get_linestart,
	tl_get_linestop
};

#ifdef HAVE_ICONV
static const FuncPtr utf8tl_funcs =
{
	TRUE,
	utf8tl_is_alnum_str,
	utf8tl_get_linestart,
	utf8tl_get_linestop
};
#endif

typedef struct
{
	gint		ref;
	gchar		*str;
	size_t		len;
	gint		split;
	StrStrFunc	strstrfunc;
	const FuncPtr	*funcptr;

#ifdef HAVE_REGEX
	re_entry	*reg_exp;
#endif

} search_pattern_t;

typedef struct
{
	GSList		*slist;
	guint		processed;
	guint		total;

} search_docs_t;

typedef struct
{
	search_pattern_t	pattern;

#ifdef HAVE_ICONV
	search_pattern_t	*pattern_locale;
	GHashTable		*hash_locale;
#endif
	gboolean		full_match;
	guint			idle_id;
	gchar			*mime_type;
	search_docs_t		docs;

} search_info_t;

typedef struct
{
	GtkWidget	*window;
	GtkCombo	*combo;
	GtkEntry	*pattern;
	GtkEntry	*mime_type;
	GtkWidget	*case_sensitive;
	GtkWidget	*full_match;

#ifdef HAVE_REGEX
	GtkWidget	*reg_exp;
#endif

	GtkWidget	*progress_bar;
	GtkWidget	*do_search;
	GtkWidget	*do_stop;
	GtkWidget	*do_close;

	GtkWidget	*search_all;
	GtkWidget	*search_selected;
	GtkWidget	*search_results;

} search_widget_t;

typedef struct
{
	search_widget_t	w;
	ListSelection	search_selection;
	search_info_t	i;

} search_data_t;

GSList *search_dialog_slist = NULL;

static char search_dialog_title[] = gettext_nop("MozCache: Find");


#ifdef HAVE_REGEX

static re_entry *search_make_regex(search_pattern_t *pattern,
						gint case_sensitive, gchar **error)
{
	re_entry *reg_exp;

	gint option = REGEX_SEARCH_OPTION;

#ifdef REGEX_SEARCH_IGNORE_CASE

	if (case_sensitive == FALSE)
	{

#if defined(HAVE_POSIX_REGEX)
		if (tl_is_ascii_str(pattern->str))
#elif defined(HAVE_PCRE_REGEX) && ! defined(REGEX_SEARCH_UTF8)
		if ((pattern->funcptr)->is_utf8 == FALSE ||
				tl_is_ascii_str(pattern->str))
#endif
		option |= REGEX_SEARCH_IGNORE_CASE;

	} /* if (case_sensitive == FALSE) */

#endif /* REGEX_SEARCH_IGNORE_CASE */

#ifdef REGEX_SEARCH_UTF8

	if ((pattern->funcptr)->is_utf8)
	{
		option |= REGEX_SEARCH_UTF8;
	}

	while ((reg_exp = re_make(pattern->str, option, error)) == NULL)
	{
		if ((option & REGEX_SEARCH_UTF8) == FALSE)
		{
			return reg_exp;
		}

		option &= ~REGEX_SEARCH_UTF8;

#ifdef REGEX_SEARCH_IGNORE_CASE
		if ((option & REGEX_SEARCH_IGNORE_CASE) &&
				tl_is_ascii_str(pattern->str) == FALSE)
		{
			option &= ~REGEX_SEARCH_IGNORE_CASE;
		}
#endif
	}

#else /* ! REGEX_SEARCH_UTF8 */

	if ((reg_exp = re_make(pattern->str, option, error)) == NULL)
	{
		return reg_exp;
	}

#endif /* #ifdef REGEX_SEARCH_UTF8 .. #else */

	if (case_sensitive == FALSE)

#ifdef REGEX_SEARCH_IGNORE_CASE
	if ((option & REGEX_SEARCH_IGNORE_CASE) == FALSE)
#endif
	{

#ifdef HAVE_ICONV
		if ((pattern->funcptr)->is_utf8)
		{
			reg_exp->str_lower_func =
				(StrDupToLowerFunc) utf8tl_strndup_to_lower;
		}
		else
#endif
		{
			reg_exp->str_lower_func =
				(StrDupToLowerFunc) tl_strndup_to_lower;
		}

		pattern->split = TRUE;
	}

#ifndef HAVE_BSD_REGEX
	if (pattern->split == FALSE && strpbrk(pattern->str, "^$"))
#endif
	{
		/*
			Start is after, end is before a newline, also
			non UNIX nl and '\0' (in fact all cntrl chars)
		*/

		pattern->split = TRUE;
	}

	return reg_exp;

} /* static re_entry *search_make_regex(search_pattern_t*, gint, gchar**) */

#endif /* HAVE_REGEX */


#ifdef HAVE_ICONV

static void search_hash_locale_find(gpointer key, gpointer value, gpointer data)
{
	struct
	{
		search_pattern_t 	*val;
		gchar			*str;

	} *ptr = data;

	search_pattern_t *pattern = value;

	if (ptr->val == NULL &&
		pattern != (search_pattern_t *) -1 &&
			(pattern->funcptr)->is_utf8 == FALSE &&
				strcmp(ptr->str, pattern->str) == 0)
	{
		ptr->val = pattern;
	}

} /* static void search_hash_locale_find(gpointer, gpointer, gpointer) */

static search_pattern_t *search_hash_locale_new(search_info_t *info, const gchar *charset)
{

#if GTK_MAJOR_VERSION == 1
#define FROM_STR	charconv_locale_str
#else
#define FROM_STR	charconv_utf8_str
#endif

	gchar			*str;
	gint			len;
	search_pattern_t	*pattern;

	len = info->pattern.len;
	str = charconv(info->pattern.str, XWindows949ToCP949(charset), FROM_STR, &len);

	if (str || tl_is_ascii_str(info->pattern.str))
	{
		if (str != info->pattern.str)
		{
			struct
			{
				search_pattern_t 	*val;
				gchar			*str;

			} hash_retv;

#if GTK_MAJOR_VERSION == 1
			gint is_utf8;

			if (g_strcasecmp(charset, charconv_utf8_str) == 0)
			{
				is_utf8 = TRUE;
				pattern = NULL;
			}

			else
			{
				is_utf8 = FALSE;
#endif
				hash_retv.val = NULL;
				hash_retv.str = (str == NULL) ? info->pattern.str : str;

				g_hash_table_foreach(info->hash_locale,
				(GHFunc) search_hash_locale_find, &hash_retv);

				pattern = hash_retv.val;

#if GTK_MAJOR_VERSION == 1
			}
#endif
			if (pattern == NULL)
			{
				gint case_sensitive;

				if (str == NULL)
				{
					str = g_strdup(info->pattern.str);
				}

				pattern = g_memdup(&(info->pattern), sizeof(search_pattern_t));

				/* pattern reference count starts with zero */
				pattern->ref = 0;
				pattern->str = str;
				pattern->len = len;

				case_sensitive = (info->pattern.strstrfunc ==
							(StrStrFunc) tl_mem_find_str);
#if GTK_MAJOR_VERSION == 1
				if (is_utf8)
				{
					pattern->funcptr = &utf8tl_funcs;

					if (case_sensitive == FALSE &&
						tl_is_ascii_str(pattern->str) == FALSE)
					{
						pattern->split = TRUE;
						pattern->strstrfunc =
							(StrStrFunc) utf8tl_strstr_igncase;
					}
				}
#else
				pattern->split = FALSE;
				pattern->funcptr = &tl_funcs;

				if (case_sensitive == FALSE)
				{
					pattern->strstrfunc =
							(StrStrFunc) tl_mem_find_str_igncase;
				}

#endif /* #if GTK_MAJOR_VERSION == 1 .. #else */

#ifdef HAVE_REGEX
				if (pattern->reg_exp)
				{
					pattern->reg_exp =
						search_make_regex(pattern, case_sensitive, NULL);

					if (pattern->reg_exp == NULL)
					{
						g_free(pattern->str);
						g_free(pattern);
						pattern = (search_pattern_t *) -1;
					}
				}
#endif /* HAVE_REGEX */

			}

			else
			{
				g_free(str);
				pattern->ref++;

			} /* if (pattern == NULL) .. else */
		}

		else
		{
			pattern = &(info->pattern);
			pattern->ref++;

		} /* if (str != info->pattern.str) .. else */
	}

	else
	{
		pattern = (search_pattern_t *) -1;

	} /* if (str || tl_is_ascii_str(info->pattern.str)) ..else */

	g_hash_table_insert(info->hash_locale, (gchar *) charset, pattern);

	return pattern;

} /* static search_pattern_t *search_hash_locale_new(search_info_t*, const gchar*) */

static void search_hash_locale_free(gpointer key, gpointer value, gpointer data)
{
	search_pattern_t *pattern = value;

	if (pattern != (search_pattern_t *) -1 &&
					--pattern->ref < 0)
	{
		g_free(pattern->str);

#ifdef HAVE_REGEX
		if (pattern->reg_exp)
		{
			re_free(pattern->reg_exp);
		}
#endif
		g_free(pattern);
	}

} /* static void search_hash_locale_free(gpointer, gpointer, gpointer) */

#endif /* HAVE_ICONV */


static void search_append_cache_doc(nscache_record_t *nscrec, search_info_t *info)
{
	if (nscrec)
	{
		if ( info->mime_type == NULL ||
			(nscrec->content_type != NULL &&
					fnmatch(info->mime_type,
						nscrec->content_type, 0) == 0) )
					
		{
			info->docs.slist = g_slist_append(info->docs.slist, nscrec);
			info->docs.total++;
		}
	}

} /* static void search_append_cache_doc(nscache_record_t*, search_info_t*) */

static void search_get_cache_docs_from_tree(GuiTreeStore *tree,
					nscache_record_t* nscrec, search_info_t *info)
{
	search_append_cache_doc(nscrec, info);
}

static void search_get_cache_docs(search_data_t *data)
{
	if (mainlist->name == NULL)
	{
		return;
	}

	if (GTK_TOGGLE_BUTTON(data->w.search_all)->active)
	{
		guitl_tree_recursive(&(notebook_list[kTreeList].store.tree), FALSE,
			(GuiTreeStoreRecursiveFunc) search_get_cache_docs_from_tree, &(data->i));
	}

	else if (GTK_TOGGLE_BUTTON(data->w.search_selected)->active)
	{
		if (mainlist->nscrec)
		{
			search_append_cache_doc(mainlist->nscrec, &(data->i));
		}

		else
		{
			guitl_tree_recursive(&(notebook_list[kTreeList].store.tree), TRUE,
				(GuiTreeStoreRecursiveFunc) search_get_cache_docs_from_tree,
											&(data->i));
		}
	}

	else if (GTK_TOGGLE_BUTTON(data->w.search_results)->active)
	{
		guitl_list_foreach_row_data(&(data->search_selection.store.list),
						(GFunc) search_append_cache_doc, &(data->i));
	}

} /* static void search_get_cache_docs(search_data_t*) */

static void search_dlg_free_runtime_data(search_info_t *info)
{
	if (info->idle_id)
	{
		GuiIdleRemove(info->idle_id);
	}

#ifdef HAVE_ICONV
	if (info->hash_locale)
	{
		g_hash_table_foreach(info->hash_locale,
				(GHFunc) search_hash_locale_free, NULL);

		g_hash_table_destroy(info->hash_locale);
	}
#endif

#ifdef HAVE_REGEX
	if (info->pattern.reg_exp)
	{
		re_free(info->pattern.reg_exp);
	}
#endif
	g_free(info->mime_type);
	g_free(info->pattern.str);
	g_slist_free(info->docs.slist);

	memset(info, '\0', sizeof(search_info_t));

} /* static void search_dlg_free_runtime_data(search_info_t *info) */

static void search_dlg_destroy(GtkWidget *w, search_data_t *data)
{
	combo_mime_slist	= g_slist_remove(combo_mime_slist, data->w.combo);
	search_dialog_slist 	= g_slist_remove(search_dialog_slist, data);
	list_selection_slist	= g_slist_remove(list_selection_slist,
							&(data->search_selection));
	search_dlg_free_runtime_data(&(data->i));

#ifndef CONFIG_GTKCLIST_GTKCTREE
	g_free(data->search_selection.name);
#endif
	g_free(data);
}

static char *search_dlg_search_file(search_data_t *data, gchar *beg, gint *blen)
{
#ifdef HAVE_ICONV
#define DATA_I_PATTERN(x)	((data->i.pattern_locale)->x)
#else
#define DATA_I_PATTERN(x)	((data->i.pattern).x)
#endif

	gint		end_offs;
	guint		end_chr;
	gchar		*stop;

	stop     = &beg[*blen];
	end_offs = DATA_I_PATTERN(len);
	end_chr  = '\0';

	while (stop > beg)
	{
		gint	slen;
		gchar	*start;

		if (DATA_I_PATTERN(split))
		{
			gchar *end;

			if (end_chr != '\0')
			{
				/* restore for line counting */
				beg[-1] = end_chr;
			}

			end = DATA_I_PATTERN(funcptr->stopfunc)(beg, stop);
			end_chr = *end,
			*end = '\0';
			slen = end - beg;
		}

		else
		{
			slen = stop - beg;
		}

		start = beg;

		for (;;)
		{
			size_t size = &beg[slen] - start;
#ifdef HAVE_REGEX
			re_entry *reg_exp = DATA_I_PATTERN(reg_exp);

			if (reg_exp)
			{
				gint beg_offs;

				if (size == 0)
				{
					break;
				}

#if defined(HAVE_POSIX_REGEX) || defined(HAVE_V8_REGEX)

				/* see below */
				if (beg[0] == '\0')
				{
					slen = 0;
					break;
				}
#endif
				beg_offs = re_pmatch(reg_exp, start, size, &end_offs);

				if (beg_offs < 0)
				{
#ifndef HAVE_BSD_REGEX
					if (DATA_I_PATTERN(split) == FALSE)
					{

#if defined(HAVE_PCRE_REGEX) || defined(HAVE_GNU_REGEX)

						/* done */
						return NULL;
#else
						/*
							POSIX and V8 function expects a
							null-terminated string, but buffer
							may contain '\0'.
							BSD also, but lines will be always
							splitted.
						*/

						slen = strlen(beg);
#endif
					}
#endif /* ! HAVE_BSD_REGEX */
					break;
				}

				if (end_offs < 0)
				{
					/* no offset information */
					goto _truncate_line_;
				}

				end_offs -= beg_offs;
				start += beg_offs;
			}
			else

#endif /* HAVE_REGEX */
			{
				if (size < DATA_I_PATTERN(len))
				{
					break;
				}
				
				start = DATA_I_PATTERN(strstrfunc)(start, size,
						DATA_I_PATTERN(str), DATA_I_PATTERN(len));

				if (start == NULL)
				{
					if (DATA_I_PATTERN(split) == FALSE)
					{
						/* done */
						return NULL;
					}

					break;
				}

			} /* if (reg_exp) .. else */

			if ( (data->i.full_match == FALSE) ||
					DATA_I_PATTERN(funcptr->alnumfunc)(beg, start, end_offs) )
			{
				stop  = DATA_I_PATTERN(funcptr->stopfunc)(&start[end_offs], &beg[slen]);
				start = DATA_I_PATTERN(funcptr->startfunc)(beg, start);
				slen  = stop - start;
#ifdef HAVE_REGEX
_truncate_line_:
#endif
				if (slen > 255)
				{
					slen = 255;
#ifdef HAVE_ICONV
					if (DATA_I_PATTERN(funcptr->is_utf8))
					{
						stop = UTF8_TOOLS_PREV_CHAR(start, &start[255]);

						if (stop)
						{
							slen = stop - start;
						}
					}
#endif /* HAVE_ICONV */
				}

				*blen = slen;
				start[slen] = '\0';

				return start;
			}

			start++;

		} /* for (;;) */

		beg += slen + 1;

	} /* while (stop > beg) */

	return NULL;

} /* static char *search_dlg_search_file(search_data_t*, gchar*, gint*) */

static void search_disable_results_radiobutton(search_data_t *data)
{
	list_selection_slist = g_slist_remove(list_selection_slist,
						&(data->search_selection));
#ifndef CONFIG_GTKCLIST_GTKCTREE
	GuiListSetSelectionMode(&(data->search_selection.store.list),
							GTK_SELECTION_NONE);
	g_free(data->search_selection.name);
#endif
	data->search_selection.name   = NULL;
	data->search_selection.nscrec = NULL;

	if (GTK_TOGGLE_BUTTON(data->w.search_results)->active)
	{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->w.search_all), TRUE);
	}

	gtk_widget_set_sensitive(data->w.search_results, FALSE);
}

static void search_disable_all_entries(GuiListStore *list, gpointer userdata)
{
	GuiListSetRowData(list, userdata);

#ifdef CONFIG_GTKCLIST_GTKCTREE
	gtk_clist_set_selectable(list->clist, list->row, FALSE);
#endif
}

void search_disable_entries(gpointer d, gpointer u)
{
	/* if there are search dialogs containing results */

	search_data_t	*data;
	GuiListStore	*list;

	data = d;
	list = &(data->search_selection.store.list);

	guitl_list_foreach(list,
		(GuiListStoreForeachFunc) search_disable_all_entries, NULL);

	search_disable_results_radiobutton(data);
}

gint search_update_window_title(gpointer d, gpointer u)
{
	gint	found;
	gchar	*str;
	gchar	pom[256];

	search_data_t *data = d;

#ifdef CONFIG_GTKCLIST_GTKCTREE
	/* also called from Delete() in 'gui.c' */
	if ((data->search_selection.store.list).clist->freeze_count)
	{
		gtk_clist_thaw((data->search_selection.store.list).clist);
	}
#endif

	found = GuiListGetRows(&(data->search_selection.store.list));

	if (found)
	{
		gtk_widget_set_sensitive(data->w.search_results, TRUE);
		g_snprintf(pom, sizeof(pom),
				gettext("MozCache: Find (found %d)"), found);
		str = guitl_convert_window_title_if_enabled(pom);

		if (str == NULL)
		{
			str = tl_replace_non_ascii_chars_in_str(pom);
		}

		gtk_window_set_title(GTK_WINDOW(data->w.window), str);

		if (str != pom)
		{
			g_free(str);
		}
	}

	else
	{
		search_disable_results_radiobutton(data);
		guitl_window_set_title_gettext(data->w.window,
					gettext_nop("MozCache: Find (not found)"));
	}

	return found;
}

static void search_dlg_search_stop(GtkWidget *w, search_data_t *data)
{
	search_dlg_free_runtime_data(&(data->i));

	gtk_widget_hide(data->w.progress_bar);
	gtk_widget_set_sensitive(data->w.do_stop, FALSE);
	gtk_window_set_modal(GTK_WINDOW(data->w.window), FALSE);

	if (search_update_window_title(data, NULL) > 0)
	{
		search_dialog_slist  = g_slist_append(search_dialog_slist, data);
		list_selection_slist = g_slist_append(list_selection_slist,
								&(data->search_selection));

		GuiListSetSelectionMode(&(data->search_selection.store.list),
								GTK_SELECTION_BROWSE);

		GuiListSelectFirstRow(&(data->search_selection.store.list));
	}

	gtk_widget_set_sensitive(data->w.do_search, TRUE);
	gtk_widget_set_sensitive(GTK_WIDGET(data->w.pattern), TRUE);
	gtk_widget_grab_focus(GTK_WIDGET(data->w.pattern));

	gdk_beep();

} /* static void search_dlg_search_stop(GtkWidget*, search_data_t*) */

static gint search_dlg_search_doc(search_data_t *data)
{
	gint	len;
	gchar	*fbuf;

#ifdef HAVE_ICONV
	search_pattern_t *pattern;
	const gchar *charset;
#endif
#ifdef HAVE_UNICODE
	gint	is_utf8;
#else
#define is_utf8	FALSE
#endif
	nscache_record_t *nscrec;

	if (data->i.docs.slist == NULL)
	{
		data->i.idle_id = 0;
		search_dlg_search_stop(NULL, data);

		return FALSE;
	}

	nscrec = data->i.docs.slist->data;
	data->i.docs.slist = g_slist_remove(data->i.docs.slist, nscrec);
	data->i.docs.processed++;

	GuiProgressBarSetValue(GTK_PROGRESS_BAR(data->w.progress_bar),
		((gdouble) data->i.docs.processed) / ((gdouble) data->i.docs.total));

#ifdef HAVE_ICONV

	charset = nscrec->charset;

	if (charset == NULL)
	{
		charset = charconv_iso8859_1_str;
	}

	pattern = g_hash_table_lookup(data->i.hash_locale, charset);

	if (pattern == NULL)
	{
		pattern = search_hash_locale_new(&(data->i), charset);
	}

	if (pattern == (search_pattern_t *) -1)
	{
		return TRUE;
	}

	data->i.pattern_locale = pattern;

#ifdef HAVE_UNICODE
	is_utf8 = (pattern->funcptr)->is_utf8;
#endif

#endif /* HAVE_ICONV */

	fbuf = file_read(nscrec, &len);

	if (fbuf == NULL)
	{
		perror(nscrec->filename);
	}

	else
	{
		gchar *matchln = search_dlg_search_file(data, fbuf, &len);

		if (matchln)
		{
			const PixbufIcon	*icon;
			gchar 			linenr[15], *pp[4];

			pp[kColumnURL]   = nscrec->urlstr;
			pp[kColumnFile]	 = nscrec->filename;
			pp[kColumnFound] = matchln;
			pp[kColumnLine]  = linenr;
			sprintf(linenr, "%d", tl_count_lines(fbuf, matchln));
#ifdef HAVE_ICONV

#if GTK_MAJOR_VERSION == 1
			/* see macro 'Search_CList_Append' */
#else
			if (g_utf8_validate(matchln, len, NULL) == FALSE)
			{
				pp[kColumnFound] =
					charconv(matchln, charconv_utf8_str,
						((nscrec->charset == NULL) ?
						nscrec->charset :
						XWindows949ToCP949(nscrec->charset)),
										&len);

				if (pp[kColumnFound] == NULL || pp[kColumnFound] == matchln)
				{
					pp[kColumnFound] =
						tl_replace_non_ascii_chars_in_str(matchln);
				}
			}
#ifdef PANGO_LOCALE
			/*
				Pango-ERROR **: file pango-1.8.1/pango/break.c: line 780
				(pango_default_break): assertion failed:
				(IN_BREAK_TABLE (break_type)) aborting...
				=> glib: g_assert(expr)
			*/

			{ gchar *str;

			if ((str = charconv(pp[kColumnFound], charconv_locale_str,
							charconv_utf8_str, &len)) != NULL)
			{
				CHARCONV_UTF8_FREE(str, pp[kColumnFound]);
			}

			else
			{
				pp[kColumnFound] =
					utf8tl_replace_non_ascii_chars_in_str(pp[kColumnFound]);
			} }
#endif /* PANGO_LOCALE */

#endif /* #if GTK_MAJOR_VERSION == 1 .. #else */

#endif /* HAVE_ICONV */

			icon = guitl_get_icon_mime(nscrec->content_type);
			SearchListAppend(&(data->search_selection.store.list),
							pp, is_utf8, icon, nscrec);
#ifdef HAVE_ICONV
			if (pp[kColumnFound] != matchln)
			{
				g_free(pp[kColumnFound]);
			}
#endif
		} /* if (matchln) */

		free(fbuf);

	} /* if (fbuf == NULL) .. else */

	return TRUE;

} /* static gint search_dlg_search_doc(search_data_t*) */

static void search_dlg_start(GtkWidget *w, search_data_t *data)
{
	const gchar	*mime_entry, *search_entry;
	gchar		*str;
	gint		case_sensitive;

#ifdef HAVE_REGEX
	gboolean	with_reg_exp;
#endif

	if (data->i.docs.slist)
	{
		return;
	}

	data->i.full_match = GTK_TOGGLE_BUTTON(data->w.full_match)->active;

	mime_entry = gtk_entry_get_text(data->w.mime_type);
	str = NULL;

	if (mime_entry == NULL || *mime_entry == '\0')
	{
		mime_entry = NULL;
	}

	else if (mime_entry[0] != '*' || mime_entry[1] != '\0')
	{
		if ((str = CHARCONV_UTF8_TO_LOCALE_DUP(mime_entry)) == NULL)
		{
			gdk_beep();
			return;
		}

		mime_entry = str;
	}

	search_entry = gtk_entry_get_text(data->w.pattern);

	if (search_entry == NULL || *search_entry == '\0')
	{
		g_free(str);
		gdk_beep();
		return;
	}

	data->i.mime_type = str;
	case_sensitive = GTK_TOGGLE_BUTTON(data->w.case_sensitive)->active;

	if (case_sensitive)
	{
		data->i.pattern.strstrfunc = (StrStrFunc) tl_mem_find_str;
	}

	data->i.pattern.len = strlen(search_entry);

#if GTK_MAJOR_VERSION == 1

	data->i.pattern.str = (case_sensitive == FALSE) ?
		tl_strndup_to_lower(search_entry, data->i.pattern.len) :
				g_strndup(search_entry, data->i.pattern.len);

	data->i.pattern.funcptr = &tl_funcs;

	if (case_sensitive == FALSE)
	{
		data->i.pattern.strstrfunc = (StrStrFunc) tl_mem_find_str_igncase;
	}

#else /* GTK_MAJOR_VERSION > 1 */

	data->i.pattern.str = (case_sensitive == FALSE) ?
		utf8tl_strndup_to_lower(search_entry, data->i.pattern.len) :
				g_strndup(search_entry, data->i.pattern.len);

	data->i.pattern.funcptr = &utf8tl_funcs;

	if (case_sensitive == FALSE)
	{
		if (tl_is_ascii_str(search_entry) == FALSE)
		{
			data->i.pattern.split = TRUE;
			data->i.pattern.strstrfunc = (StrStrFunc) utf8tl_strstr_igncase;
		}

		else
		{
			data->i.pattern.strstrfunc = (StrStrFunc) tl_mem_find_str_igncase;
		}
	}

#endif /* #if GTK_MAJOR_VERSION == 1 .. #else */

#ifdef HAVE_ICONV

#if GTK_MAJOR_VERSION == 1
#define CHARSET_KEY	charconv_locale_str
#else
#define CHARSET_KEY	charconv_utf8_str
#endif
	data->i.pattern.ref = 1; /* avoid freeing memory */
	data->i.hash_locale = g_hash_table_new(g_str_hash, g_str_equal);

	g_hash_table_insert(data->i.hash_locale,
			(gchar *) CHARSET_KEY, &(data->i.pattern));
#endif /* HAVE_ICONV */

#ifdef HAVE_REGEX

	with_reg_exp = GTK_TOGGLE_BUTTON(data->w.reg_exp)->active;

	if (with_reg_exp)
	{
		gchar *error;

		data->i.pattern.reg_exp =
			search_make_regex(&(data->i.pattern), case_sensitive, &error);

		if (data->i.pattern.reg_exp == NULL)
		{
			static const gchar errstr[] =
				gettext_nop("Error compiling regular expression");

			if (error == NULL)
			{
				gui_errfmt("%s : %s",
						gettext(errstr), search_entry);
			}

			else
			{
				gui_errfmt("%s : %s - %s",
						gettext(errstr), search_entry, error);
				g_free(error);
			}

			search_dlg_free_runtime_data(&(data->i));

			return;
		}
	}

	gprop_set_bool(kPropBoolFindWithRegex, with_reg_exp);

#endif /* HAVE_REGEX */

	gprop_set_str(kPropStrFindPattern, search_entry);
	gprop_set_str(kPropStrFindMimeType, mime_entry);
	gprop_set_bool(kPropBoolFindCaseSensitive, case_sensitive);
	gprop_set_bool(kPropBoolFindFullMatch, data->i.full_match);

	guitl_window_set_title_gettext(data->w.window, search_dialog_title);
	gtk_window_set_modal(GTK_WINDOW(data->w.window), TRUE);

	gui_err(NULL);
	search_get_cache_docs(data);

	GuiListClear(&(data->search_selection.store.list));

	search_dialog_slist  = g_slist_remove(search_dialog_slist, data);
	list_selection_slist = g_slist_remove(list_selection_slist,
							&(data->search_selection));
#ifndef CONFIG_GTKCLIST_GTKCTREE
	g_free(data->search_selection.name);
#endif
	data->search_selection.name = NULL;
	data->search_selection.nscrec = NULL;

	if (data->i.docs.total == 0)
	{
		search_dlg_search_stop(NULL, data);
	}

	else
	{
		gtk_widget_set_sensitive(GTK_WIDGET(data->w.pattern), FALSE);
		gtk_widget_set_sensitive(data->w.do_search, FALSE);
		gtk_widget_set_sensitive(data->w.do_stop, TRUE);

		GuiProgressBarSetValue(GTK_PROGRESS_BAR(data->w.progress_bar), 0.0);
		gtk_widget_show(data->w.progress_bar);

		data->i.idle_id = GuiIdleAddPriority(GUI_PRIORITY_REDRAW,
							search_dlg_search_doc, data);
	}

} /* static void search_dlg_start(GtkWidget*, search_data_t*) */

#ifdef CONFIG_GTKCLIST_GTKCTREE
static void search_dlg_result_list_resize_column(GtkWidget *w, gint colnr,
							gint width, gpointer data)
{
	/* data == kPropIntFindColWidthUrl */

	int prop = (int) data;

	gprop_set_int(prop + colnr, width);
}
#endif

#ifndef CONFIG_GTKCLIST_GTKCTREE
static void search_found_row_adjust_height(GtkTreeViewColumn *column,
				GtkCellRenderer *renderer, GtkTreeModel *model,
							GtkTreeIter *iter, gpointer data)
{
	struct SearchAdjustHeight
	{
		GtkTreeView	*view;
		gint		height;
		gint		set;

	} *ptr;

	gint height;

	ptr = data;
	gtk_tree_model_get(model, iter, kColumnRowHeight, &height, -1);

	if (height == 0)
	{
		gchar		*str;
		PangoLayout	*layout;

		gtk_tree_model_get(model, iter, kColumnFound, &str, -1);
		layout = gtk_widget_create_pango_layout(GTK_WIDGET(ptr->view), str);
		pango_layout_get_pixel_size(layout, NULL, &height);

		g_object_unref(layout);
		g_free(str);

		gtk_list_store_set(GTK_LIST_STORE(model), iter, kColumnRowHeight,
				(height > ptr->height) ? height : ptr->height, -1);
	}

	if (ptr->set != height)
	{
		ptr->set = height;
		g_object_set(renderer, "height", height, NULL);
	}

} /* static void search_found_row_adjust_height(GtkTreeViewColumn*,
			GtkCellRenderer*, GtkTreeModel*, GtkTreeIter*, gpointer) */

#endif /* ! CONFIG_GTKCLIST_GTKCTREE */

static GtkWidget *search_tab_add_check_button(GtkTable *table, gchar row,
							gchar *str, gint prop)
{
	gint		sensitive;
	GtkWidget	*check_button;

	check_button = guitl_check_button_new_gettext(str);

	gtk_table_attach(table, check_button, 0, 1, row, row + 1,
						GTK_FILL, GTK_FILL, 2, 2);
	if (gprop_get_bool(prop, &sensitive))
	{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_button), sensitive);
	}

	return check_button;
}

static GtkWidget *search_tab_add_radio_button(GtkTable *table, gchar row,
						gchar *label, GtkWidget *radio_button)
{
	gchar 	*str;
	GSList	*button_group;

	str = CHARCONV_UTF8_GETTEXT(label);

	if ((button_group = (GSList *) radio_button) != NULL)
	{
		button_group = GuiRadioButtonGetGroup(GTK_RADIO_BUTTON(radio_button));
	}

	radio_button = gtk_radio_button_new_with_label(button_group, str);
	CHARCONV_UTF8_DESTROY(str);

	gtk_table_attach(table, radio_button, 1, 2, row, row + 1,
						GTK_FILL, GTK_FILL, 16, 2);
	return radio_button;
}

static void search_build_column(GuiListStore *list, gint colnr, gint height, gchar *title)
{
	gchar	*str;
	gint	prop, width;

	width = 100;

	if (colnr == kColumnURL)
	{
		GuiListColumnPixtextNew(list, kColumnURL, height, kColumnPixbuf);
	}

	else
	{
		GtkJustification justify = GTK_JUSTIFY_LEFT;

		if (colnr == kColumnLine)
		{
			justify = GTK_JUSTIFY_RIGHT;
		}

#ifdef CONFIG_GTKCLIST_GTKCTREE

		gtk_clist_set_column_justification(list->clist, colnr, justify);
#else
		GuiListColumnTextNew(list, colnr, height, justify);

		if (colnr == kColumnFound)
		{
			struct SearchAdjustHeight
			{
				GtkTreeView	*view;
				gint		height;
				gint		set;

			} *ptr;

			GList			*renderer;
			GtkTreeViewColumn	*column;

			ptr = g_malloc(sizeof(struct SearchAdjustHeight));

			ptr->view   = list->view;
			ptr->height = height;
			ptr->set    = height;

			column = gtk_tree_view_get_column(ptr->view, colnr);
			renderer = gtk_tree_view_column_get_cell_renderers(column);

			gtk_tree_view_column_set_cell_data_func(column, renderer->data,
				(GtkTreeCellDataFunc) search_found_row_adjust_height,
						(gpointer) ptr, ( GtkDestroyNotify) g_free);
			g_list_free(renderer);
		}

		else if (colnr == kColumnLine)
		{
			GuiListSetColumnAlignment(list, colnr, 1.0);
		}

#endif /* #ifdef CONFIG_GTKCLIST_GTKCTREE .. #else */

	} /* if (colnr == kColumnURL) .. else */

	prop = kPropIntFindColWidthUrl + colnr;
	gprop_get_int(kPropIntFindColWidthUrl + colnr, &width);

#ifndef CONFIG_GTKCLIST_GTKCTREE
	GuiListSignalConnectResizeColumn(list, colnr,
				gui_list_resize_column, (gpointer) prop);
#endif
	str = CHARCONV_UTF8_GETTEXT(title);
	GuiListSetColumnTitle(list, colnr, str);
	CHARCONV_UTF8_DESTROY(str);

	GuiListSetColumnAutoResize(list, colnr, FALSE);
	GuiListSetColumnResizable(list, colnr, TRUE);
	GuiListSetColumnWidth(list, colnr, width);

} /* static void search_build_column(GuiListStore*, gint, gint, gchar*) */

void search_dialog_open(void)
{
	gint		height;
	gchar		*p, *str;
	GtkBox		*vbox, *hbox, *tbox;
	GtkCombo	*combo;
	GtkTable	*table;
	GtkWidget	*bbox, *frame, *swin, *widget;
	search_data_t	*data;

	/* initialize to '\0' */
	data = g_malloc0(sizeof(search_data_t));

	widget = gtk_vbox_new(FALSE, 0);
	data->w.window = gui_window_new(search_dialog_title, widget, 5, kPropIntFindWinWidth);
	GuiSignalConnect(data->w.window, "destroy", search_dlg_destroy, data);

	vbox = GTK_BOX(widget);
	widget = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(vbox, widget, FALSE, FALSE, 2);
	hbox = GTK_BOX(widget);

	widget = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(widget), 2);
	frame = guitl_frame_new_add_child_gettext(NULL, widget);
	gtk_box_pack_start(hbox, frame, TRUE, TRUE, 0);

	tbox = GTK_BOX(widget);
	widget = gtk_table_new(2, 2, FALSE);
	gtk_box_pack_start(tbox, widget, TRUE, TRUE, 0);
	table = GTK_TABLE(widget);

	data->w.pattern =
		guitl_tab_add_entry(table, gettext_nop("Pattern: "), 0, 0);

	if (gprop_get_str(kPropStrFindPattern, &p) && p != NULL)
	{
		str = CHARCONV_UTF8_FROM_LOCALE(p);

		if (str)
		{
			gtk_entry_set_text(data->w.pattern, str);
			CHARCONV_UTF8_FREE(str, p);
		}
	}

	GuiSignalConnect(data->w.pattern, "activate", search_dlg_start, data);

	combo = guitl_tab_add_combo(table, gettext_nop("MIME type: "),
						mime_types_available, 0, 1);
	data->w.combo = combo;
	data->w.mime_type = GTK_ENTRY(combo->entry);
	combo_mime_slist = g_slist_prepend(combo_mime_slist, combo);

	if (gprop_get_str(kPropStrFindMimeType, &p) && p != NULL &&
			g_list_find_custom(mime_types_available, p,
							(GCompareFunc) g_strcasecmp))
	{
		g_strdown(p);
		str = CHARCONV_UTF8_FROM_LOCALE(p);

		if (str)
		{
			gtk_entry_set_text(data->w.mime_type, str);
			CHARCONV_UTF8_FREE(str, p);
		}
	}

	widget = gtk_table_new(2, 3, FALSE);
	gtk_box_pack_start(tbox, widget, TRUE, TRUE, 0);
	table = GTK_TABLE(widget);

	data->w.case_sensitive = search_tab_add_check_button(table, 0,
		gettext_nop("Case sensitive"), kPropBoolFindCaseSensitive);

	data->w.full_match = search_tab_add_check_button(table, 1,
		gettext_nop("Match whole word"), kPropBoolFindFullMatch);

#ifdef HAVE_REGEX

	data->w.reg_exp = search_tab_add_check_button(table, 2,
		gettext_nop("Regular pattern"), kPropBoolFindWithRegex);
#endif

	data->w.search_all = search_tab_add_radio_button(table, 0,
			gettext_nop("All entries"), NULL);

	data->w.search_selected = search_tab_add_radio_button(table, 1,
			gettext_nop("Selected folder/entry"), data->w.search_all);

	data->w.search_results = search_tab_add_radio_button(table, 2,
			gettext_nop("Results"), data->w.search_selected);

	gtk_widget_set_sensitive(data->w.search_results, FALSE);

	bbox = gtk_vbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_EDGE);
	gtk_box_pack_start(hbox, bbox, FALSE, FALSE, 0);

	data->w.do_search = guitl_button_box_add_icon_button(bbox, kIconButtonSearch);
	GuiSignalConnect(data->w.do_search, "clicked", search_dlg_start, data);

	data->w.do_stop = guitl_button_box_add_icon_button(bbox, kIconButtonStop);
	GuiSignalConnect(data->w.do_stop, "clicked", search_dlg_search_stop, data);
	gtk_widget_set_sensitive(data->w.do_stop, FALSE);

	data->w.do_close = guitl_button_box_add_icon_button(bbox, kIconButtonClose);
	GuiSignalConnectSwapped(data->w.do_close, "clicked", gtk_widget_destroy, data->w.window);

	widget = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(widget), 2);
	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_add(GTK_CONTAINER(widget), swin);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (swin),
				GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);

	frame = guitl_frame_new_add_child_gettext(gettext_nop("Results"), widget);
	gtk_container_add(GTK_CONTAINER(vbox), frame);

	data->search_selection.type = kSortedList;

	SearchListNew(&(data->search_selection.store.list));	
	GuiListContainerAdd(&(data->search_selection.store.list), swin);

	GuiObjectSetUserData(data->w.window, &(data->search_selection));

	height = guitl_list_calculate_row_height(&(data->search_selection.store.list));
	GuiListSetRowHeight(&(data->search_selection.store.list), height);

	search_build_column(&(data->search_selection.store.list),
					kColumnURL, height, gettext_nop("URL"));
	search_build_column(&(data->search_selection.store.list),
					kColumnFound, height, gettext_nop("Found"));
	search_build_column(&(data->search_selection.store.list),
					kColumnFile, height, gettext_nop("Local file"));
	search_build_column(&(data->search_selection.store.list),
					kColumnLine, height, gettext_nop("Line"));

	GuiListColumnTitlesShow(&(data->search_selection.store.list));
	GuiListColumnTitlesPassive(&(data->search_selection.store.list));

	GuiListSignalConnect(&(data->search_selection.store.list),
		"button-press-event", gui_mouse_button_pressed, gui_popup_menu);

#if GTK_MAJOR_VERSION == 1

	GuiSignalConnect(data->w.window, "key-press-event",
				gui_sf10_show_popup_menu, gui_popup_menu);
#else
	GuiSignalConnect(data->w.window, "popup-menu",
				gui_sf10_show_popup_menu, gui_popup_menu);
#endif

#ifdef CONFIG_GTKCLIST_GTKCTREE
	GuiListSignalConnectResizeColumn(&(data->search_selection.store.list), 
					search_dlg_result_list_resize_column,
						(gpointer) kPropIntFindColWidthUrl);
#endif

	GuiListSignalConnectSelectedRow(&(data->search_selection.store.list),
				gui_list_row_selected, &(data->search_selection));
	GuiListSignalConnectUnselectedRow(&(data->search_selection.store.list),
				gui_list_row_unselected, &(data->search_selection));

	gtk_widget_show_all(data->w.window);

	data->w.progress_bar = gtk_progress_bar_new();
	gtk_box_pack_start(vbox, data->w.progress_bar, FALSE, FALSE, 0);

#if GTK_MAJOR_VERSION == 1
	gtk_editable_select_region(GTK_EDITABLE(data->w.pattern), 0, -1);
	gtk_widget_grab_focus(GTK_WIDGET(data->w.pattern));
#endif

} /* void search_dialog_open(void) */

/* EOF */
